﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_TriggerState.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_BitPack.h>

#include "hid_GcAdapterDriver.h"

namespace nn { namespace hid { namespace detail {

namespace
{
const int GcControllerAdapterVid = 0x057E;   //!< Gc コントローラーアダプターの Vid
const int GcControllerAdapterPid = 0x0337;   //!< Gc コントローラーアダプターの Pid
const int OutputReportSizeMax = 5;
const int ControlDataSize = 9;
const int CommandMaxLength = 24;
const int GcTriggerMin = 30;
const int GcTriggerMax = 180;

enum ReportId : uint8_t
{
    ReportId_OutputData = 0x11,
    ReportId_OutputOrigin = 0x12,
    ReportId_OutputActivate = 0x13,
    ReportId_OutputReset = 0x14,
    ReportId_OutputReboot = 0x15,
    ReportId_InputData = 0x21,
    ReportId_InputOrigin = 0x22,
    ReportId_InputActivate = 0x23,
    ReportId_InputReset = 0x24,
    ReportId_Invalid = 0xff,
};

 /**
 * @brief       GC コントローラードライバーの状態を管理するためフラグ群です
 */
struct GcAdapterDriverFlags
{
    typedef ::nn::util::BitFlagSet<32, GcAdapterDriverFlags>::Flag<0>  isAssigned;         //!< Usb のポートに対してコントローラーが割り当てられている
    typedef ::nn::util::BitFlagSet<32, GcAdapterDriverFlags>::Flag<1>  isActivated;        //!< Gc コントローラーアダプターがアクティベートされている
    typedef ::nn::util::BitFlagSet<32, GcAdapterDriverFlags>::Flag<2>  isAcConnected;      //!< Gc コントローラーアダプターの電源が供給されている
    typedef ::nn::util::BitFlagSet<32, GcAdapterDriverFlags>::Flag<3>  isSendMotorCommand; //!< 有効な Motor コマンドが存在する
    typedef ::nn::util::BitFlagSet<32, GcAdapterDriverFlags>::Flag<11> isSetStickOrigin;   //!< アナログスティックの原点補正が行われている
    typedef ::nn::util::BitFlagSet<32, GcAdapterDriverFlags>::Flag<12> isOriginChanged;    //!< アナログスティックの原点が変更されている
    typedef ::nn::util::BitFlagSet<32, GcAdapterDriverFlags>::Flag<13> isWbConnected;      //!< ウェーブバードが接続されている
    typedef ::nn::util::BitFlagSet<32, GcAdapterDriverFlags>::Flag<14> isMotorActivated;   //!< モーターが有効になっている
};

/**
 * @brief       GC コントローラードライバーの状態を管理するためのフラグの集合です
 */
typedef ::nn::util::BitFlagSet<32, GcAdapterDriverFlags> GcAdapterDriverFlagsSet;


//!< GC コントローラー毎の状態
struct GcControllerUnit
{
    GcControllerState state;
    AnalogStickState leftStickCalibration;
    AnalogStickState rightStickCalibration;
    int32_t triggerLOrigin;
    int32_t triggerROrigin;
    GcControllerMotorState currentMotorState;
    GcControllerMotorState nextMotorState;
    GcAdapterDriverFlagsSet flags;
    GcPortDeviceType        deviceType;
    int32_t autoVibrate;
};

//!< GC コントローラーアダプター毎のクラス
class GcAdapter
{
private:
    struct Command {
        int64_t samplingNumber;
        size_t  length;
        uint8_t data[OutputReportSizeMax];
    };

    RingFIFO<Command, CommandMaxLength> m_CommandRingBuffer;  //!< コマンドを格納するための FIFO

    GcControllerUnit       m_Controllers[GcPortPerAdapterCountMax];
    size_t                 m_UsbPort;
    nn::ahid::Ahid*        m_pAhid;
    GcAdapterIndex         m_AdapterIndex;
    GcAdapterDriverFlagsSet m_Flags;

public:
    //!< 割り当てが可能であれば割り当てを行う
    bool TryAssignDevice(size_t port, nn::ahid::Ahid& ahid, GcAdapterIndex adapterIndex) NN_NOEXCEPT;

    //!< Port 番号が一致するかどうかを検証する
    bool IsSamePort(size_t port) NN_NOEXCEPT;

    //!< アダプターのインデックス番号番号を取得する
    bool GetAdapterIndex(GcAdapterIndex* pOutValue) NN_NOEXCEPT;

    //!< デバイスの切断処理を行う
    void DetachDevice() NN_NOEXCEPT;

    //!< InputReport をパースする
    void ParseInputReport(const uint8_t* pReport, size_t length) NN_NOEXCEPT;

    //!< OutputReport を Queue に追加する
    void SetOutputReport(uint8_t reportId) NN_NOEXCEPT;

    //!< OutputReport を送信する
    bool SendOutputReport() NN_NOEXCEPT;

    //!< デバイスの状態を取得します
    bool GetState(GcControllerState* pOutValue, GcPortIndex index) NN_NOEXCEPT;

    //!< モーターの制御を行う
    Result ControlMotor(GcPortIndex index, GcControllerMotorState motor, bool forceUpdate) NN_NOEXCEPT;

    //!< 接続時のモーターの制御を行う
    Result ControlMotorOnConnect(GcPortIndex index) NN_NOEXCEPT;

    //!< モーターの状態を取得する
    GcControllerMotorState GetMotorState(GcPortIndex index) NN_NOEXCEPT;

    //!< 左スティックのキャリブレーション値を取得します
    Result GetLeftAnalogStickCalibration(AnalogStickState* pOutCenter,
                                         GcPortIndex index) NN_NOEXCEPT;

    //!< 右スティックのキャリブレーション値を取得します
    Result GetRightAnalogStickCalibration(AnalogStickState* pOutCenter,
                                          GcPortIndex index) NN_NOEXCEPT;

private:
    //!< コントローラーが新たに接続されたときの処理を行う
    void OnControllerConnected(GcPortIndex index) NN_NOEXCEPT;

    //!< コントローラーが新たに接続されたときの処理を行う
    void OnAcAdapterConnected() NN_NOEXCEPT;
};

//!< GC コントローラーアダプターのマネージャー
class GcAdapterManager final
{
    NN_DISALLOW_COPY(GcAdapterManager);
    NN_DISALLOW_MOVE(GcAdapterManager);
private:
    bool m_IsEnabled;
    GcAdapter m_Adapter[GcAdapterCountMax];
    GcAdapterIndex m_LastIndex;   // 最後に割り当てた Index

public:
    GcAdapterManager() NN_NOEXCEPT;

    //!< GcAdapterDriver を有効にする
    void Enable() NN_NOEXCEPT;

    //!< 新規にデバイスを登録する
    bool Attach(size_t port, nn::ahid::Ahid& ahid) NN_NOEXCEPT;

    //!< 登録済みのデバイスを切断する
    void Detach(size_t port) NN_NOEXCEPT;

    //!< Input Report をパースします
    void ParseInputReport(size_t port, const uint8_t* pReport, size_t length) NN_NOEXCEPT;

    //!< Output Report を送信します
    bool SendOutputReport(size_t port) NN_NOEXCEPT;

    //!< 接続されている Gc コントローラーアダプターを列挙する
    GcAdapterList ListAdapters() NN_NOEXCEPT;

    //!< デバイスの状態を取得します
    bool GetState(GcControllerState* pOutValue, const GcControllerIndex& index) NN_NOEXCEPT;

    //!< モーターの制御を行う
    Result ControlMotor(const GcControllerIndex& index, GcControllerMotorState motor) NN_NOEXCEPT;

    //!< 接続時のモーターの制御を行う
    Result ControlMotorOnConnect(const GcControllerIndex& index) NN_NOEXCEPT;

    //!< モーターの状態を取得する
    GcControllerMotorState GetMotorState(const GcControllerIndex& index) NN_NOEXCEPT;

    //!< 左スティックのキャリブレーション値を取得します
    Result GetLeftAnalogStickCalibration(AnalogStickState* pOutCenter,
                                         const GcControllerIndex& index) NN_NOEXCEPT;

    //!< 右スティックのキャリブレーション値を取得します
    Result GetRightAnalogStickCalibration(AnalogStickState* pOutCenter,
                                          const GcControllerIndex& index) NN_NOEXCEPT;
private:
    //!< port 番号に対応する GcAdapter を取得する
    bool GetGcAdapterFromUsbPort(GcAdapter** ppOutValue, size_t port) NN_NOEXCEPT;

    //!< アダプターのインデックス番号に対応するGcAdapter を取得する
    bool GetGcAdapterFromAdapterIndex(GcAdapter** ppOutValue, GcAdapterIndex index) NN_NOEXCEPT;
};

//!< GcAdapter の管理を担うマネージャーを返します
GcAdapterManager& GetGcAdapterManager() NN_NOEXCEPT;

//!< Trigger の値を正規化します
inline int32_t ClampTrigger(const int16_t& value, const int16_t& origin) NN_NOEXCEPT
{
    auto output = value - origin;
    if (output < GcTriggerMin)
    {
        return 0;
    }
    if (output >= GcTriggerMax)
    {
        return TriggerMax;
    }
    // スケーリング
    return (output - GcTriggerMin) * TriggerMax / (GcTriggerMax - GcTriggerMin);
}

} // namespace

GcAdapterDriver::GcAdapterDriver() NN_NOEXCEPT
{
    // 何もしない
}

bool GcAdapterDriver::IsGcAdapter(int vid, int pid) NN_NOEXCEPT
{
    return vid == GcControllerAdapterVid && pid == GcControllerAdapterPid;
}

void GcAdapterDriver::Enable() NN_NOEXCEPT
{
    GetGcAdapterManager().Enable();
}

bool GcAdapterDriver::Attach(size_t port, ::nn::ahid::Ahid& ahid) NN_NOEXCEPT
{
    return GetGcAdapterManager().Attach(port, ahid);
}

void GcAdapterDriver::Detach(size_t port) NN_NOEXCEPT
{
    GetGcAdapterManager().Detach(port);
}

void GcAdapterDriver::ParseInputReport(size_t port, const uint8_t* pReport, size_t length) NN_NOEXCEPT
{
    GetGcAdapterManager().ParseInputReport(port, pReport, length);
}

bool GcAdapterDriver::SendOutputReport(size_t port) NN_NOEXCEPT
{
    return GetGcAdapterManager().SendOutputReport(port);
}

GcAdapterList GcAdapterDriver::ListAdapters() NN_NOEXCEPT
{
    return GetGcAdapterManager().ListAdapters();
}

bool GcAdapterDriver::GetState(GcControllerState* pOutValue, const GcControllerIndex& index) NN_NOEXCEPT
{
    return GetGcAdapterManager().GetState(pOutValue, index);
}

Result GcAdapterDriver::ControlMotor(const GcControllerIndex& index, GcControllerMotorState motor) NN_NOEXCEPT
{
    NN_RESULT_DO(GetGcAdapterManager().ControlMotor(index, motor));
    NN_RESULT_SUCCESS;
}

Result GcAdapterDriver::ControlMotorOnConnect(const GcControllerIndex& index) NN_NOEXCEPT
{
    NN_RESULT_DO(GetGcAdapterManager().ControlMotorOnConnect(index));
    NN_RESULT_SUCCESS;
}

GcControllerMotorState GcAdapterDriver::GetMotorState(const GcControllerIndex& index) NN_NOEXCEPT
{
    return GetGcAdapterManager().GetMotorState(index);
}

Result GcAdapterDriver::GetLeftAnalogStickCalibration(AnalogStickState* pOutCenter,
                                                      int16_t* pOutOriginPlay,
                                                      int16_t* pOutRange,
                                                      const GcControllerIndex& index) NN_NOEXCEPT
{
    NN_RESULT_DO(GetGcAdapterManager().GetLeftAnalogStickCalibration(pOutCenter, index));
    *pOutOriginPlay = 15;
    *pOutRange = 56;

    NN_RESULT_SUCCESS;
}

Result GcAdapterDriver::GetRightAnalogStickCalibration(AnalogStickState* pOutCenter,
                                                       int16_t* pOutOriginPlay,
                                                       int16_t* pOutRange,
                                                       const GcControllerIndex& index) NN_NOEXCEPT
{
    NN_RESULT_DO(GetGcAdapterManager().GetRightAnalogStickCalibration(pOutCenter, index));
    *pOutOriginPlay = 15;
    *pOutRange = 44;

    NN_RESULT_SUCCESS;
}

namespace
{
bool GcAdapter::TryAssignDevice(size_t port, ::nn::ahid::Ahid& ahid, GcAdapterIndex adapterIndex) NN_NOEXCEPT
{
    if (m_Flags.Test<GcAdapterDriverFlags::isAssigned>() == false)
    {
        m_Flags.Set<GcAdapterDriverFlags::isAssigned>();
        m_UsbPort = port;
        m_pAhid = &ahid;
        m_AdapterIndex = adapterIndex;

        this->SetOutputReport(ReportId_OutputActivate);

        ahid.SetProtocol(1).IsFailure();
        return true;
    }

    return false;
}

bool GcAdapter::IsSamePort(size_t port) NN_NOEXCEPT
{
    return (m_Flags.Test<GcAdapterDriverFlags::isAssigned>() == true &&
        m_UsbPort == port);
}

bool GcAdapter::GetAdapterIndex(GcAdapterIndex* pOutValue) NN_NOEXCEPT
{
    if (m_Flags.Test<GcAdapterDriverFlags::isAssigned>() == true)
    {
        *pOutValue = m_AdapterIndex;
        return true;
    }

    return false;
}

void GcAdapter::DetachDevice() NN_NOEXCEPT
{
    if (m_Flags.Test<GcAdapterDriverFlags::isAssigned>() == true)
    {
        m_Flags.Reset();
        m_UsbPort = 0;
        m_pAhid = nullptr;
        m_CommandRingBuffer.Clear();
    }
}

void GcAdapter::ParseInputReport(const uint8_t* pReport, size_t length)
{
    auto reportId = pReport[0];

    if (reportId == ReportId_InputData)
    {
        GcPortIndex index = 0;
        uint8_t payloadIndex = 1; // 2nd Byte of 0x21 report format
        uint8_t zeros[4] = { 0 };
        bool prevIsAcConnected = m_Flags.Test<GcAdapterDriverFlags::isAcConnected>();

        bool isAcConnected = (((pReport[payloadIndex] & 0x04) >> 2) == 0x01); // MAV ( respective channel's MAV bit are the same )
        if (prevIsAcConnected != isAcConnected)
        {
            m_Flags.Set<GcAdapterDriverFlags::isAcConnected>(isAcConnected);
            if (isAcConnected == true)
            {
                this->OnAcAdapterConnected();
            }
        }

        for (index = 0; index < GcPortPerAdapterCountMax; index++)
        {
            GcControllerUnit& controller = m_Controllers[index];
            GcPortDeviceType prevDeviceType = controller.deviceType;

            controller.deviceType = static_cast<GcPortDeviceType>((pReport[payloadIndex] & 0xf0) >> 4); // DEVICE_TYPE
            controller.flags.Set<GcAdapterDriverFlags::isOriginChanged>(((pReport[payloadIndex] & 0x08) >> 3) == 0x01); // ORG_CH
            controller.flags.Set<GcAdapterDriverFlags::isWbConnected>(((pReport[payloadIndex] & 0x02) >> 1) == 0x01); // ORG_CH

            if (controller.deviceType == GcPortDeviceType_WaveBird &&
                controller.flags.Test<GcAdapterDriverFlags::isWbConnected>() == false)
            {
                controller.deviceType = GcPortDeviceType_None;
            }

            if (controller.deviceType == GcPortDeviceType_GcController ||
                controller.deviceType == GcPortDeviceType_WaveBird)
            {
                // コントローラーが新たに接続された
                if (prevDeviceType != controller.deviceType)
                {
                    this->OnControllerConnected(index);
                }

                if (memcmp(&pReport[payloadIndex + 3], zeros, sizeof(zeros)))
                {
                    auto bit = nn::util::BitPack8();

                    bit.SetMaskedBits(0xFF, pReport[payloadIndex + 1]);
                    controller.state.buttons.Set<GcControllerButton::A>(bit.GetBit(0));
                    controller.state.buttons.Set<GcControllerButton::B>(bit.GetBit(1));
                    controller.state.buttons.Set<GcControllerButton::X>(bit.GetBit(2));
                    controller.state.buttons.Set<GcControllerButton::Y>(bit.GetBit(3));
                    controller.state.buttons.Set<GcControllerButton::Left>(bit.GetBit(4));
                    controller.state.buttons.Set<GcControllerButton::Right>(bit.GetBit(5));
                    controller.state.buttons.Set<GcControllerButton::Down>(bit.GetBit(6));
                    controller.state.buttons.Set<GcControllerButton::Up>(bit.GetBit(7));

                    bit.SetMaskedBits(0x0f, pReport[payloadIndex + 2]);
                    controller.state.buttons.Set<GcControllerButton::Start>(bit.GetBit(0));
                    controller.state.buttons.Set<GcControllerButton::Z>(bit.GetBit(1));
                    controller.state.buttons.Set<GcControllerButton::R>(bit.GetBit(2));
                    controller.state.buttons.Set<GcControllerButton::L>(bit.GetBit(3));

                    // Analog stick & Trigger
                    controller.state.analogStickL.x = pReport[payloadIndex + 3];
                    controller.state.analogStickL.y = pReport[payloadIndex + 4];
                    controller.state.analogStickR.x = pReport[payloadIndex + 5];
                    controller.state.analogStickR.y = pReport[payloadIndex + 6];
                    controller.state.triggerL = pReport[payloadIndex + 7];
                    controller.state.triggerR = pReport[payloadIndex + 8];

                    // Set the origin
                    if (controller.flags.Test<GcAdapterDriverFlags::isSetStickOrigin>() == false)
                    {
                        controller.flags.Set<GcAdapterDriverFlags::isSetStickOrigin>();
                        controller.leftStickCalibration.x = controller.state.analogStickL.x;
                        controller.leftStickCalibration.y = controller.state.analogStickL.y;
                        controller.rightStickCalibration.x = controller.state.analogStickR.x;
                        controller.rightStickCalibration.y = controller.state.analogStickR.y;
                        controller.triggerLOrigin = controller.state.triggerL;
                        controller.triggerROrigin = controller.state.triggerR;
                    }

                    // 正規化
                    controller.state.triggerL = ClampTrigger(controller.state.triggerL, controller.triggerLOrigin);
                    controller.state.triggerR = ClampTrigger(controller.state.triggerR, controller.triggerROrigin);
               }
            }

            payloadIndex += ControlDataSize;
        }
    }
}

void GcAdapter::SetOutputReport(uint8_t reportId) NN_NOEXCEPT
{
    Command command;
    std::memset(command.data, 0, sizeof(command.data));
    command.data[0] = reportId;
    if( reportId == ReportId_OutputData ){
        for(int i = 0; i < GcPortPerAdapterCountMax; ++i){
            if (m_Controllers[i].autoVibrate > 0)
            {
                command.data[i + 1] = (m_Controllers[i].autoVibrate == 1) ? GcControllerMotorState_Off : GcControllerMotorState_On;
                m_Controllers[i].autoVibrate--;
            }
            else
            {
                command.data[i + 1] = static_cast<uint8_t>(m_Controllers[i].nextMotorState);
            }
        }
        command.length = OutputReportSizeMax;
    }
    else
    {
        command.length = 1;
    }
    m_CommandRingBuffer.Push(command);
}

bool GcAdapter::SendOutputReport() NN_NOEXCEPT
{
    Command command;
    auto bytesSent = uint32_t();

    if( m_Flags.Test<GcAdapterDriverFlags::isAssigned>() == false)
    {
        // デバイスが割り当てられていないので何もしない
        return false;
    }

    bool autoVibrationEnabled = false;
    for (auto& controller : m_Controllers)
    {
        if (controller.autoVibrate > 0)
        {
            autoVibrationEnabled = true;
            break;
        }
    }

    // モーターに更新があればコマンドを追加する
    if (m_Flags.Test<GcAdapterDriverFlags::isSendMotorCommand>() == true ||
        autoVibrationEnabled == true)
    {
        this->SetOutputReport(ReportId_OutputData);
        m_Flags.Set<GcAdapterDriverFlags::isSendMotorCommand>(false);

        for (auto& controller : m_Controllers)
        {
            controller.currentMotorState = controller.nextMotorState;
        }
    }

    // バッファからコマンドを取得する
    if (m_CommandRingBuffer.Pop(&command).IsFailure())
    {
        // コマンドが取得できない場合は何もしない
        return true;
    }
    nn::Result result = m_pAhid->WriteWithTimeout(
            &bytesSent, reinterpret_cast<void*>(command.data),
            command.length, nn::TimeSpan::FromMilliSeconds(15));
    // timeout is only a little bit crazy, not yet insane
    if (result.IsFailure())
    {
        if (nn::ahid::ResultTimeout::Includes(result))
        {
            return true;
        }
        else
        {
            return false;
        }
    }
    // 送信に成功した場合は true
    return true;
}


bool GcAdapter::GetState(GcControllerState* pOutValue, GcPortIndex index) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(index, 0, GcPortPerAdapterCountMax);
    if (m_Flags.Test<GcAdapterDriverFlags::isAssigned>() == false)
    {
        return false;
    }

    if (m_Controllers[index].deviceType != GcPortDeviceType_GcController && m_Controllers[index].deviceType != GcPortDeviceType_WaveBird)
    {
        return false;
    }

    *pOutValue = m_Controllers[index].state;
    return true;
}


Result GcAdapter::ControlMotor(GcPortIndex index, GcControllerMotorState motor, bool forceUpdate) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(index, 0, GcPortPerAdapterCountMax);
    NN_RESULT_THROW_UNLESS(m_Flags.Test<GcAdapterDriverFlags::isAssigned>(), ResultGcControllerNotConnected());
    NN_RESULT_THROW_UNLESS(m_Controllers[index].deviceType == GcPortDeviceType_GcController || m_Controllers[index].deviceType == GcPortDeviceType_WaveBird,
        ResultGcControllerNotConnected());

    if (motor == GcControllerMotorState_On &&
        m_Flags.Test<GcAdapterDriverFlags::isAcConnected>() == false)
    {
        // モーターを制御するときに、AC アダプターが ON である必要がある
        NN_RESULT_THROW(ResultGcAdapterNotPowered());
    }

    if (m_Controllers[index].nextMotorState != motor ||
        forceUpdate)
    {
        m_Controllers[index].nextMotorState = motor;
        m_Flags.Set<GcAdapterDriverFlags::isSendMotorCommand>();
    }

    NN_RESULT_SUCCESS;
}

Result GcAdapter::ControlMotorOnConnect(GcPortIndex index) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(index, 0, GcPortPerAdapterCountMax);
    NN_RESULT_THROW_UNLESS(m_Flags.Test<GcAdapterDriverFlags::isAssigned>(), ResultGcControllerNotConnected());
    NN_RESULT_THROW_UNLESS(m_Controllers[index].deviceType == GcPortDeviceType_GcController || m_Controllers[index].deviceType == GcPortDeviceType_WaveBird,
        ResultGcControllerNotConnected());


    if (m_Flags.Test<GcAdapterDriverFlags::isAcConnected>() == false)
    {
        // モーターを制御するときに、AC アダプターが ON である必要がある
        NN_RESULT_THROW(ResultGcAdapterNotPowered());
    }

    m_Controllers[index].autoVibrate = 16;

    NN_RESULT_SUCCESS;
}

GcControllerMotorState GcAdapter::GetMotorState(GcPortIndex index) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(index, 0, GcPortPerAdapterCountMax);
    if (m_Flags.Test<GcAdapterDriverFlags::isAssigned>() == false ||
        (m_Controllers[index].deviceType != GcPortDeviceType_GcController && m_Controllers[index].deviceType != GcPortDeviceType_WaveBird))
    {
        return GcControllerMotorState_Off;
    }

    return m_Controllers[index].currentMotorState;
}

Result GcAdapter::GetLeftAnalogStickCalibration(AnalogStickState* pOutCenter, GcPortIndex index) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(index, 0, GcPortPerAdapterCountMax);
    NN_RESULT_THROW_UNLESS(m_Flags.Test<GcAdapterDriverFlags::isAssigned>(), ResultGcControllerNotConnected());
    NN_RESULT_THROW_UNLESS(m_Controllers[index].deviceType == GcPortDeviceType_GcController || m_Controllers[index].deviceType == GcPortDeviceType_WaveBird,
        ResultGcControllerNotConnected());

    *pOutCenter = m_Controllers[index].leftStickCalibration;

    NN_RESULT_SUCCESS;
}

Result GcAdapter::GetRightAnalogStickCalibration(AnalogStickState* pOutCenter, GcPortIndex index) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(index, 0, GcPortPerAdapterCountMax);
    NN_RESULT_THROW_UNLESS(m_Flags.Test<GcAdapterDriverFlags::isAssigned>(), ResultGcControllerNotConnected());
    NN_RESULT_THROW_UNLESS(m_Controllers[index].deviceType == GcPortDeviceType_GcController || m_Controllers[index].deviceType == GcPortDeviceType_WaveBird,
        ResultGcControllerNotConnected());

    *pOutCenter = m_Controllers[index].rightStickCalibration;

    NN_RESULT_SUCCESS;
}

void GcAdapter::OnControllerConnected(GcPortIndex index) NN_NOEXCEPT
{
    m_Controllers[index].currentMotorState = GcControllerMotorState_Off;
    this->ControlMotor(index, GcControllerMotorState_Off, true);
}

void GcAdapter::OnAcAdapterConnected() NN_NOEXCEPT
{
    // 電源が入った際にモーターを更新する
    for (auto& controller : m_Controllers)
    {
        controller.nextMotorState = GcControllerMotorState_Off;
    }
    m_Controllers->flags.Set<GcAdapterDriverFlags::isSendMotorCommand>();
}

GcAdapterManager::GcAdapterManager() NN_NOEXCEPT
    : m_IsEnabled(true)
    , m_LastIndex(std::numeric_limits<uint64_t>::min())
{
    // 何もしない
}

void GcAdapterManager::Enable() NN_NOEXCEPT
{
    m_IsEnabled = true;
}

bool GcAdapterManager::Attach(size_t port, ::nn::ahid::Ahid& ahid) NN_NOEXCEPT
{
    if (m_IsEnabled == false)
    {
        // ドライバが有効でないときは割り当てをせずにスキップ
        return false;
    }

    uint32_t adapterIndex = ++m_LastIndex;
    for (auto& adapter : m_Adapter)
    {
        if (adapter.TryAssignDevice(port, ahid, adapterIndex) == true)
        {
            // 割り当てに成功
            return true;
        }
    }

    return false;
}

void GcAdapterManager::Detach(size_t port) NN_NOEXCEPT
{
    GcAdapter* pAdapter;
    if (this->GetGcAdapterFromUsbPort(&pAdapter, port) == true)
    {
        pAdapter->DetachDevice();
    }
}

void GcAdapterManager::ParseInputReport(size_t port, const uint8_t* pReport, size_t length) NN_NOEXCEPT
{
    GcAdapter* pAdapter;
    if (this->GetGcAdapterFromUsbPort(&pAdapter, port) == true)
    {
        return pAdapter->ParseInputReport(pReport, length);
    }
}

bool GcAdapterManager::SendOutputReport(size_t port) NN_NOEXCEPT
{
    GcAdapter* pAdapter;
    if (this->GetGcAdapterFromUsbPort(&pAdapter, port) == true)
    {
        return pAdapter->SendOutputReport();
    }

    return false;
}

GcAdapterList GcAdapterManager::ListAdapters() NN_NOEXCEPT
{
    GcAdapterList list;
    list.count = 0;

    for (auto& adapter : m_Adapter)
    {
        GcAdapterIndex index;
        if (adapter.GetAdapterIndex(&index) == true)
        {
            list.indexes[list.count] = index;
            list.count++;
        }
    }

    return list;
}

bool GcAdapterManager::GetState(GcControllerState* pOutValue, const GcControllerIndex& index) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(index.portIndex, 0, GcPortPerAdapterCountMax);

    GcAdapter* pAdapter;
    if (this->GetGcAdapterFromAdapterIndex(&pAdapter, index.adapterIndex) == false)
    {
        return false;
    }

    return pAdapter->GetState(pOutValue, index.portIndex);
}

Result GcAdapterManager::ControlMotor(const GcControllerIndex& index, GcControllerMotorState motor) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(index.portIndex, 0, GcPortPerAdapterCountMax);

    GcAdapter* pAdapter;

    NN_RESULT_THROW_UNLESS(this->GetGcAdapterFromAdapterIndex(&pAdapter, index.adapterIndex), ResultGcControllerNotConnected());

    NN_RESULT_DO(pAdapter->ControlMotor(index.portIndex, motor, false));

    NN_RESULT_SUCCESS;
}

Result GcAdapterManager::ControlMotorOnConnect(const GcControllerIndex& index) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(index.portIndex, 0, GcPortPerAdapterCountMax);

    GcAdapter* pAdapter;

    NN_RESULT_THROW_UNLESS(this->GetGcAdapterFromAdapterIndex(&pAdapter, index.adapterIndex), ResultGcControllerNotConnected());

    NN_RESULT_DO(pAdapter->ControlMotorOnConnect(index.portIndex));

    NN_RESULT_SUCCESS;
}

GcControllerMotorState GcAdapterManager::GetMotorState(const GcControllerIndex& index) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(index.portIndex, 0, GcPortPerAdapterCountMax);


    GcAdapter* pAdapter;
    if (this->GetGcAdapterFromAdapterIndex(&pAdapter, index.adapterIndex) == false)
    {
        return GcControllerMotorState_Off;
    }

    return pAdapter->GetMotorState(index.portIndex);
}



Result GcAdapterManager::GetLeftAnalogStickCalibration(AnalogStickState* pOutCenter, const GcControllerIndex& index) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(index.portIndex, 0, GcPortPerAdapterCountMax);

    GcAdapter* pAdapter;

    NN_RESULT_THROW_UNLESS(this->GetGcAdapterFromAdapterIndex(&pAdapter, index.adapterIndex), ResultGcControllerNotConnected());

    NN_RESULT_DO(pAdapter->GetLeftAnalogStickCalibration(pOutCenter, index.portIndex));

    NN_RESULT_SUCCESS;
}

Result GcAdapterManager::GetRightAnalogStickCalibration(AnalogStickState* pOutCenter, const GcControllerIndex& index) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(index.portIndex, 0, GcPortPerAdapterCountMax);

    GcAdapter* pAdapter;

    NN_RESULT_THROW_UNLESS(this->GetGcAdapterFromAdapterIndex(&pAdapter, index.adapterIndex), ResultGcControllerNotConnected());

    NN_RESULT_DO(pAdapter->GetRightAnalogStickCalibration(pOutCenter, index.portIndex));

    NN_RESULT_SUCCESS;
}

bool GcAdapterManager::GetGcAdapterFromUsbPort(GcAdapter** ppOutValue, size_t port) NN_NOEXCEPT
{
    for (auto& adapter : m_Adapter)
    {
        if (adapter.IsSamePort(port) == true)
        {
            *ppOutValue = &adapter;
            return true;
        }
    }

    return false;
}

bool GcAdapterManager::GetGcAdapterFromAdapterIndex(GcAdapter** ppOutValue, GcAdapterIndex index) NN_NOEXCEPT
{
    for (auto& adapter : m_Adapter)
    {
        GcAdapterIndex targetIndex;
        if (adapter.GetAdapterIndex(&targetIndex) == true)
        {
            if (targetIndex == index)
            {
                *ppOutValue = &adapter;
                return true;
            }
        }
    }

    return false;
}

GcAdapterManager& GetGcAdapterManager() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(GcAdapterManager, s_GcAdapterManager);
    return s_GcAdapterManager;
}

} // namespace

}}} // namespace nn::hid::detail
