﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_DebugPad.h>
#include <nn/hid/hid_IHidDebugServer.sfdl.h>
#include <nn/hid/hid_IHidServer.sfdl.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/debug/hid_DebugPad.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/util/util_TypedStorage.h>

#include "hid_AppletResourceUserId.h"
#include "hid_DebugPadImpl.h"
#include "hid_DebugPadLifo.h"
#include "hid_HidDebugServer.h"
#include "hid_HidServer.h"
#include "hid_SharedMemoryAccessor.h"
#include "hid_SharedMemoryFormat.h"
#include "hid_StaticObject.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< DebugPad の共有メモリへのアクセスを扱うクラスです。
class DebugPadSharedMemoryAccessor final : public SharedMemoryAccessor
{
private:
    ::std::atomic<DebugPadLifo*> m_pLifo;

public:
    DebugPadSharedMemoryAccessor() NN_NOEXCEPT
        : m_pLifo(nullptr)
    {
        this->SetResultActivationUpperLimitOver(
            ::nn::hid::ResultDebugPadActivationUpperLimitOver());
        this->SetResultDeactivationLowerLimitOver(
            ::nn::hid::ResultDebugPadDeactivationLowerLimitOver());
    }

    virtual ~DebugPadSharedMemoryAccessor() NN_NOEXCEPT NN_OVERRIDE
    {
        // 何もしない
    }

    //!< DebugPad の LIFO を返します。
    const DebugPadLifo* GetLifo() const NN_NOEXCEPT
    {
        return m_pLifo;
    }

protected:
    virtual ::nn::Result Attach(SharedMemoryFormat* ptr) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(ptr);
        ::nn::sf::SharedPointer<IHidServer> pHidServer;
        NN_RESULT_DO(CreateHidServerProxy(&pHidServer));
        NN_RESULT_DO(pHidServer->ActivateDebugPad(GetAppletResourceUserId()));
        m_pLifo = &::nn::util::Get(ptr->debugPad.lifo);
        NN_RESULT_SUCCESS;
    }

    virtual ::nn::Result Detach() NN_NOEXCEPT NN_OVERRIDE
    {
        ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
        NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
        NN_RESULT_DO(pHidDebugServer->DeactivateDebugPad());
        m_pLifo = nullptr;
        NN_RESULT_SUCCESS;
    }
};

//!< DebugPad の共有メモリアクセッサを返します。
DebugPadSharedMemoryAccessor& GetDebugPadSharedMemoryAccessor() NN_NOEXCEPT
{
    return StaticObject<DebugPadSharedMemoryAccessor>::Get();
}

} // namespace

::nn::Result InitializeDebugPad() NN_NOEXCEPT
{
    DebugPadSharedMemoryAccessor& accessor = GetDebugPadSharedMemoryAccessor();
    NN_RESULT_DO(accessor.SetAppletResourceUserId(GetAppletResourceUserId()));
    NN_RESULT_DO(accessor.Activate());
    NN_RESULT_SUCCESS;
}

::nn::Result FinalizeDebugPad() NN_NOEXCEPT
{
    NN_RESULT_DO(GetDebugPadSharedMemoryAccessor().Deactivate());
    NN_RESULT_SUCCESS;
}

::nn::Result GetDebugPadState(DebugPadState* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                           ResultDebugPadNullStateBuffer());
    const DebugPadLifo* pLifo = GetDebugPadSharedMemoryAccessor().GetLifo();
    NN_RESULT_THROW_UNLESS(pLifo != nullptr, ResultDebugPadNotInitialized());
    int count = pLifo->Read(pOutValue, 1);
    NN_SDK_REQUIRES_EQUAL(count, 1);
    NN_UNUSED(count);
    NN_RESULT_SUCCESS;
}

::nn::Result GetDebugPadStates(
    int* pOutCount, DebugPadState outValues[], int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_RESULT_THROW_UNLESS(outValues != nullptr,
                           ResultDebugPadNullStateBuffer());
    NN_RESULT_THROW_UNLESS(count >= 0, ResultDebugPadInvalidStateCount());
    const DebugPadLifo* pLifo = GetDebugPadSharedMemoryAccessor().GetLifo();
    NN_RESULT_THROW_UNLESS(pLifo != nullptr, ResultDebugPadNotInitialized());
    NN_SDK_REQUIRES(!pLifo->IsEmpty());
    *pOutCount = pLifo->Read(outValues, count);
    NN_RESULT_SUCCESS;
}

::nn::Result SetDebugPadAutoPilotState(
    const ::nn::hid::debug::DebugPadAutoPilotState& value) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
    NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
    NN_RESULT_DO(pHidDebugServer->SetDebugPadAutoPilotState(value));
    NN_RESULT_SUCCESS;
}

::nn::Result UnsetDebugPadAutoPilotState() NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
    NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
    NN_RESULT_DO(pHidDebugServer->UnsetDebugPadAutoPilotState());
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
