﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   コントローラーサポートの UI 呼び出しに関する内部 API の定義
 */

#include <nn/nn_Result.h>
#include <nn/applet/applet.h>
#include <nn/hid/hid_ResultControllerSupport.h>
#include <nn/hid/system/hid_ControllerSupportTypesPrivate.h>
#include <nn/la/la_Api.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "hid_ControllerSupportImpl.h"

//---------------------------------------------------------------------------
// C++ 関数の定義
//---------------------------------------------------------------------------

namespace nn { namespace hid { namespace detail {

//---------------------------------------------------------------------------
//
class ScopedLibraryApplet final
{
public:
    ScopedLibraryApplet()
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::CreateLibraryApplet(
            &m_Handle,
            nn::applet::AppletId_LibraryAppletController,
            nn::applet::LibraryAppletMode_AllForeground ) );
    }

    ~ScopedLibraryApplet()
    {
        nn::applet::CloseLibraryApplet( m_Handle );
    }

    nn::applet::LibraryAppletHandle& GetHandle()
    {
        return m_Handle;
    }

    void PushCommonArg( const system::ControllerSupportArgPrivate& argPrivate )
    {
        nn::la::CommonArgumentsWriter commonArg( nn::hid::system::c_CurrentMajorVersion, nn::hid::system::c_CurrentMinorVersion );
        commonArg.SetPlayStartupSound( argPrivate.isSystem && argPrivate.showConnectionStatus );
        commonArg.PushToInChannel( GetHandle() );
    }

    void PushArgPrivate( const system::ControllerSupportArgPrivate& argPrivate )
    {
        nn::la::PushToInChannel( GetHandle(), &argPrivate, sizeof( decltype( argPrivate ) ) );
    }

    template< typename T >
    void PushArg( const T& arg )
    {
        nn::la::PushToInChannel( GetHandle(), &arg, sizeof( decltype( arg ) ) );
    }

    nn::Result Start()
    {
        return nn::la::StartLibraryApplet( GetHandle(), true /* 末端 */ );
    }

    void PopReturnArg( system::ControllerSupportReturnArgPrivate* outReturnArg )
    {
        nn::la::PopFromOutChannel( GetHandle(), outReturnArg, sizeof( decltype( *outReturnArg ) ) );
    }

private:
    nn::applet::LibraryAppletHandle m_Handle;
};

//---------------------------------------------------------------------------
//
nn::Result ShowControllerSupportCore(
    ControllerSupportResultInfo* pOutValue,
    const ControllerSupportArg& arg,
    const system::ControllerSupportArgPrivate& argPrivate ) NN_NOEXCEPT
{
    // FW 更新モードでないことを確認
    NN_ABORT_UNLESS( argPrivate.mode != decltype( argPrivate.mode )::ControllerSupportMode_FirmwareUpdate );

    std::memset( pOutValue, 0, sizeof( ControllerSupportResultInfo ) );

    // コントローラーサポート生成
    ScopedLibraryApplet la;
    la.PushCommonArg( argPrivate );

    // コントローラーサポートの起動パラメータ送信
    la.PushArgPrivate( argPrivate );
    la.PushArg( arg );

    // コントローラーサポート起動
    auto result = la.Start();

    // Result判定
    if( result.IsSuccess() )
    {
        // 結果を取得
        system::ControllerSupportReturnArgPrivate returnArg;
        la.PopReturnArg( &returnArg );

        // 呼び出し元に返す結果
        *pOutValue = returnArg.controllerSupportResultInfo;

        switch( returnArg.result )
        {
        case system::ControllerSupportResult_Success:
            NN_RESULT_SUCCESS;
        case system::ControllerSupportResult_NotSupportedNpadStyle:
            return nn::hid::ResultControllerSupportNotSupportedNpadStyle();
        default:
            break;
        }
    }

    return nn::hid::ResultControllerSupportCanceled();
}

//---------------------------------------------------------------------------
//
nn::Result ShowControllerFirmwareUpdateCore(
    const ControllerFirmwareUpdateArg& arg,
    const system::ControllerSupportArgPrivate& argPrivate ) NN_NOEXCEPT
{
    // FW 更新モードかどうか確認
    NN_ABORT_UNLESS( argPrivate.mode == decltype( argPrivate.mode )::ControllerSupportMode_FirmwareUpdate );

    // コントローラーサポート生成
    ScopedLibraryApplet la;
    la.PushCommonArg( argPrivate );

    // コントローラーサポートの起動パラメータ送信
    la.PushArgPrivate( argPrivate );
    la.PushArg( arg );

    // コントローラーサポート起動
    auto result = la.Start();

    // Result判定
    if( result.IsSuccess() )
    {
        // 結果を取得
        system::ControllerSupportReturnArgPrivate returnArg;
        la.PopReturnArg( &returnArg );

        switch( returnArg.result )
        {
        case system::ControllerSupportResult_Success:
            NN_RESULT_SUCCESS;
        default:
            break;
        }
    }

    return nn::hid::ResultControllerFirmwareUpdateFailed();
}

}}} // namespace nn::hid::detail

