﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <new>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_ConsoleSixAxisSensor.h>
#include <nn/hid/detail/hid_Log.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_TypedStorage.h>

#include "hid_AppletResourceManager.h"
#include "hid_HandheldManager.h"
#include "hid_IConsoleSixAxisSensorDriver.h"
#include "hid_SharedMemoryFormat.h"
#include "hid_ConsoleSixAxisSensorHandle.h"
#include "hid_ConsoleSixAxisSensorLifo.h"
#include "hid_ConsoleSixAxisSensorManager.h"

namespace nn { namespace hid { namespace detail {

namespace {

bool IsSixAxisSensorStateChanged(const ::nn::util::Float3& angularVelocity) NN_NOEXCEPT
{
    const int AxisCountMax = 3; //!< 軸の数
    const float AngularVelocityThreshold = 0.3f; //!< 実験に基づく無入力判定閾値[1.0 = 360dps]

    for (int i = 0; i < AxisCountMax; i++)
    {
        if(::std::abs(angularVelocity.v[i]) > AngularVelocityThreshold)
        {
            return true;
        }
    }
    return false;
}

const int NpadHandheldSamplingIntervalMilliSeconds = 5;

//!< 本体 6 軸センサーの加速度バイアス値のデフォルト値
const ::nn::settings::system::ConsoleSixAxisSensorAccelerationBias
    DefaultConsoleSixAxisSensorAccelerationBias =
{
    0.f, 0.f, 0.f
};

//!< 本体 6 軸センサーの角速度バイアス値のデフォルト値
const ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityBias
    DefaultConsoleSixAxisSensorAngularVelocityBias =
{
    0.f, 0.f, 0.f
};

//!< 本体 6 軸センサーの加速度バイアス値のデフォルト値
const ::nn::settings::system::ConsoleSixAxisSensorAccelerationGain
    DefaultConsoleSixAxisSensorAccelerationGain =
{
    1.f, 0.f, 0.f,
    0.f, 1.f, 0.f,
    0.f, 0.f, 1.f,
};

//!< 本体 6 軸センサーの角速度バイアス値のデフォルト値
const ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityGain
    DefaultConsoleSixAxisSensorAngularVelocityGain =
{
    1.f, 0.f, 0.f,
    0.f, 1.f, 0.f,
    0.f, 0.f, 1.f,
};

//!< 本体 6 軸センサーの角速度の時間バイアス値のデフォルト値
const ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityTimeBias
    DefaultConsoleSixAxisSensorAngularVelocityTimeBias =
{
    0.f, 0.f, 0.f
};

//!< 本体 6 軸センサーの角加速度の係数のデフォルト値
const ::nn::settings::system::ConsoleSixAxisSensorAngularAcceleration
    DefaultConsoleSixAxisSensorAngularAcceleration =
{
    1.f, 0.f, 0.f,
    0.f, 1.f, 0.f,
    0.f, 0.f, 1.f,
};

const ::nn::hid::detail::ConsoleSixAxisSensorCalibrationParameters
    DefaultConsoleSixAxisSensorCalibrationParameters =
{
    DefaultConsoleSixAxisSensorAccelerationBias,
    DefaultConsoleSixAxisSensorAngularVelocityBias,
    DefaultConsoleSixAxisSensorAccelerationGain,
    DefaultConsoleSixAxisSensorAngularVelocityGain,
    DefaultConsoleSixAxisSensorAngularVelocityTimeBias,
    DefaultConsoleSixAxisSensorAngularAcceleration
};

} // namespace

const ::nn::TimeSpan ConsoleSixAxisSensorManager::SamplingInterval =
    ::nn::TimeSpan::FromMilliSeconds(5);

ConsoleSixAxisSensorManager::ConsoleSixAxisSensorManager() NN_NOEXCEPT
    : m_CommonActivationCount()
    , m_ConsoleSixAxisSensorActivationCount()
    , m_InFocusAruid(::nn::applet::AppletResourceUserId::GetInvalidId())
    , m_IsAwake(false)
    , m_SamplingFrequency(0)
    , m_IsSevenSamplingStarted(false)
    , m_IsNpadHandheldSamplingStarted(false)
    , m_CalibrationState(CalibrationState::CalibrationState_Off)
    , m_pTimerEvent(nullptr)
    , m_pDriver(nullptr)
    , m_pAppletResourceManager(nullptr)
    , m_pAppletResourceManagerMutex(nullptr)
    , m_pHandheldManager(nullptr)
    , m_pInputDetectorManager(nullptr)
    , m_pInputDetectorManagerMutex(nullptr)
    , m_pConsoleSixAxisSensorProcessor(nullptr)
    , m_pSixAxisSensorFilter(nullptr)
    , m_pConsoleSixAxisSensorSetting(nullptr)
    , m_CoordinateNumber(0)
    , m_SamplingStartedTick(::nn::os::GetSystemTick())
    , m_ConsoleSixAxisSensorCalibrationValues()
    , m_AppendStateCount(0)
    , m_SixAxisSensorLastSamplingNumber(0)
    , m_Temp()
    , m_SevenSixAxisSensorIsAtRest(false)
    , m_VerticalizationError(0.f)
    , m_GyroBias()
    , m_ConsoleSixAxisSensorCalibrationParameters()
{
    for (ConsoleSixAxisSensorAppletResourceEntry& extraEntry : m_AppletResourceEntries)
    {
        extraEntry = ConsoleSixAxisSensorAppletResourceEntry();
    }

    for (auto& state : m_SixAxisSensorCountStates)
    {
        state = tmp::SixAxisSensorCountState();
    }

    for (auto& state : m_SixAxisSensorStates)
    {
        state = SixAxisSensorState();
    }

    for (auto& state : m_SevenSixAxisSensorStates)
    {
        state = SevenSixAxisSensorState();
    }

    for (auto& samplingNumber : m_SamplingNumbers)
    {
        samplingNumber = 0;
    }
}

void ConsoleSixAxisSensorManager::SetTimerEvent(::nn::os::TimerEventType* pTimerEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pTimerEvent);
    NN_SDK_REQUIRES(m_CommonActivationCount.IsZero());
    NN_SDK_REQUIRES(m_ConsoleSixAxisSensorActivationCount.IsZero());
    m_pTimerEvent = pTimerEvent;
}

void ConsoleSixAxisSensorManager::SetDriver(IConsoleSixAxisSensorDriver* pDriver) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDriver);
    NN_SDK_REQUIRES(m_CommonActivationCount.IsZero());
    NN_SDK_REQUIRES(m_ConsoleSixAxisSensorActivationCount.IsZero());
    m_pDriver = pDriver;
}

void ConsoleSixAxisSensorManager::SetAppletResourceManager(
    AppletResourceManager* pManager, ::nn::os::SdkRecursiveMutex* pMutex
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    NN_SDK_REQUIRES_NOT_NULL(pMutex);
    NN_SDK_REQUIRES(m_CommonActivationCount.IsZero());
    NN_SDK_REQUIRES(m_ConsoleSixAxisSensorActivationCount.IsZero());
    m_pAppletResourceManager = pManager;
    m_pAppletResourceManagerMutex = pMutex;
}

void ConsoleSixAxisSensorManager::SetSixAxisSensorAppletSettingManager(SixAxisSensorAppletSettingManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    m_pSixAxisSensorAppletSettingManager = pManager;
}

void ConsoleSixAxisSensorManager::SetConsoleSixAxisSensorAppletSettingManager(ConsoleSixAxisSensorAppletSettingManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    m_pConsoleSixAxisSensorAppletSettingManager = pManager;
}

void ConsoleSixAxisSensorManager::SetHandheldManager(
    HandheldManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    NN_SDK_REQUIRES(m_CommonActivationCount.IsZero());
    NN_SDK_REQUIRES(m_ConsoleSixAxisSensorActivationCount.IsZero());
    m_pHandheldManager = pManager;
}

void ConsoleSixAxisSensorManager::SetInputDetectorManager(
    InputDetectorManager* pManager, ::nn::os::SdkMutex* pMutex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    NN_SDK_REQUIRES_NOT_NULL(pMutex);
    NN_SDK_REQUIRES(m_CommonActivationCount.IsZero());
    NN_SDK_REQUIRES(m_ConsoleSixAxisSensorActivationCount.IsZero());
    m_pInputDetectorManager = pManager;
    m_pInputDetectorManagerMutex = pMutex;
}

void ConsoleSixAxisSensorManager::SetConsoleSixAxisSensorProcessor(
    ConsoleSixAxisSensorProcessor* pProcessor) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pProcessor);
    NN_SDK_REQUIRES(m_CommonActivationCount.IsZero());
    NN_SDK_REQUIRES(m_ConsoleSixAxisSensorActivationCount.IsZero());
    m_pConsoleSixAxisSensorProcessor = pProcessor;
}

void ConsoleSixAxisSensorManager::SetSixAxisSensorFilter(
    ISixAxisSensorFilter* pFilter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFilter);
    NN_SDK_REQUIRES(m_CommonActivationCount.IsZero());
    NN_SDK_REQUIRES(m_ConsoleSixAxisSensorActivationCount.IsZero());
    m_pSixAxisSensorFilter = pFilter;
}

void ConsoleSixAxisSensorManager::UpdateSixAxisSensorSetting() NN_NOEXCEPT
{
    // 本体6軸の設定値を注入
    {
        // No need to memset - fully filled in by function call.
        ConsoleSixAxisSensorHandle handle;
        detail::GetConsoleSixAxisSensorHandle(&handle);
        AttachConsoleSixAxisSensorSetting(handle);
    }

    // Handheld 操作形態の設定値を注入
    {
        // No need to memset - fully filled in by function call.
        SixAxisSensorHandle handle;
        int count;
        detail::GetSixAxisSensorHandle(&handle, &count, 1, NpadId::Handheld, NpadStyleHandheld::Mask);
        AttachSixAxisSensorSetting(handle);
    }
}

void ConsoleSixAxisSensorManager::AttachSixAxisSensorSetting(const nn::hid::SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    SixAxisSensorSetting* pSetting = nullptr;
    m_pSixAxisSensorAppletSettingManager->GetSixAxisSensorSetting(&pSetting,
                                                                  m_pAppletResourceManager->GetAppletResourceUserId(),
                                                                  handle);
    m_pConsoleSixAxisSensorProcessor->SetSixAxisSensorSetting(pSetting);
}

void ConsoleSixAxisSensorManager::AttachConsoleSixAxisSensorSetting(const nn::hid::ConsoleSixAxisSensorHandle& handle) NN_NOEXCEPT
{
    ConsoleSixAxisSensorSetting* pSetting = nullptr;
    m_pConsoleSixAxisSensorAppletSettingManager->GetConsoleSixAxisSensorSetting(&pSetting,
                                                                                m_pAppletResourceManager->GetAppletResourceUserId(),
                                                                                handle);
    m_pConsoleSixAxisSensorSetting = pSetting;
    m_IsSevenSamplingStarted = pSetting->IsSamplingEnabled();

    // nerd::Imu に設定値を注入
    m_pSixAxisSensorFilter->SetFilterSetting(pSetting->GetFilterSetting());
}

::nn::Result ConsoleSixAxisSensorManager::ActivateForConsoleSixAxisSensor() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_CommonActivationCount.IsMax(),
                           ResultConsoleSixAxisSensorActivationUpperLimitOver());
    NN_RESULT_THROW_UNLESS(!m_ConsoleSixAxisSensorActivationCount.IsMax(),
                           ResultConsoleSixAxisSensorActivationUpperLimitOver());

    if (m_ConsoleSixAxisSensorActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        // LIFO を初期化
        this->ProcessSharedMemory([] (
            ConsoleSixAxisSensorManager* that,
            ConsoleSixAxisSensorSharedMemoryFormat* ConsoleSixAxisSensorAddress,
            ::nn::applet::AppletResourceUserId aruid,
            bool enablesInput,
            bool isConsoleSixAxisSensorActivated) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(ConsoleSixAxisSensorAddress);
            NN_UNUSED(that);
            NN_UNUSED(aruid);
            NN_UNUSED(enablesInput);
            NN_UNUSED(isConsoleSixAxisSensorActivated);
            ::nn::util::Get(ConsoleSixAxisSensorAddress->internalState).Clear();
        });

        if (m_CommonActivationCount.IsZero())
        {
            NN_RESULT_DO(this->ActivateCommon());
        }
    }

    ++m_ConsoleSixAxisSensorActivationCount;

    ++m_CommonActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorManager::DeactivateForConsoleSixAxisSensor() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ConsoleSixAxisSensorActivationCount.IsZero(),
                           ResultConsoleSixAxisSensorDeactivationLowerLimitOver());
    NN_RESULT_THROW_UNLESS(!m_CommonActivationCount.IsZero(),
                           ResultConsoleSixAxisSensorDeactivationLowerLimitOver());

    --m_CommonActivationCount;

    --m_ConsoleSixAxisSensorActivationCount;

    if (m_ConsoleSixAxisSensorActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施

        if (m_CommonActivationCount.IsZero())
        {
            NN_RESULT_DO(this->DeactivateCommon());
        }
    }

    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorManager::EnsureAppletResourceForConsoleSixAxisSensor(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    this->ProcessSharedMemory([] (
        ConsoleSixAxisSensorManager* that,
        ConsoleSixAxisSensorSharedMemoryFormat* ConsoleSixAxisSensorAddress,
        ::nn::applet::AppletResourceUserId aruid,
        bool enablesInput,
        bool isConsoleSixAxisSensorActivated) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(that);
        NN_SDK_REQUIRES_NOT_NULL(ConsoleSixAxisSensorAddress);
        NN_UNUSED(that);
        NN_UNUSED(ConsoleSixAxisSensorAddress);
        NN_UNUSED(aruid);
        NN_UNUSED(enablesInput);
        NN_UNUSED(isConsoleSixAxisSensorActivated);
    });

    NN_RESULT_SUCCESS;
}

void ConsoleSixAxisSensorManager::Sample() NN_NOEXCEPT
{
    if (!m_CommonActivationCount.IsZero())
    {
        this->Update();

        this->ProcessSharedMemory([] (
            ConsoleSixAxisSensorManager* that,
            ConsoleSixAxisSensorSharedMemoryFormat* ConsoleSixAxisSensorAddress,
            ::nn::applet::AppletResourceUserId aruid,
            bool enablesInput,
            bool isConsoleSixAxisSensorActivated) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(that);
            NN_SDK_REQUIRES_NOT_NULL(ConsoleSixAxisSensorAddress);
            NN_UNUSED(enablesInput);

            if (isConsoleSixAxisSensorActivated)
            {
                // TransferMemory の更新
                {
                    if (that->m_pSixAxisSensorFilter->IsWorkBufferAllocated())
                    {
                        ConsoleSixAxisSensorSetting* pSetting = nullptr;
                        that->m_pConsoleSixAxisSensorAppletSettingManager->GetConsoleSixAxisSensorSetting(
                            &pSetting,
                            aruid,
                            ::nn::hid::ConsoleSixAxisSensorHandle());
                        if (pSetting != nullptr &&
                            pSetting->IsSamplingEnabled())
                        {
                            ConsoleSixAxisSensorTansferMemoryStatus* pStatus;
                            that->m_pAppletResourceManager->GetConsoleSixAxisSensorTransferMemoryStateBuffer(aruid, &pStatus);
                            auto pLifo = reinterpret_cast<SevenSixAxisSensorStateLifo*>(pStatus->address);

                            if (pLifo != nullptr)
                            {
                                for (int i = that->m_AppendStateCount - 1; i >= 0; i--)
                                {
                                    pLifo->Append(that->m_SevenSixAxisSensorStates[i]);
                                }
                            }
                        }
                    }
                }
                // SharedMemory を更新
                {
                    ConsoleSixAxisSensorSharedMemoryFormat* address = ConsoleSixAxisSensorAddress;
                    ConsoleSixAxisSensorInternalState& internalState = ::nn::util::Get(address->internalState);
                    internalState.SetSevenSixAxisSensorIsAtRest(that->m_SevenSixAxisSensorIsAtRest);
                    internalState.SetVerticalizationError(that->m_VerticalizationError);
                    internalState.SetGyroBias(that->m_GyroBias);
                }
            }
        });
    }
}

::nn::Result ConsoleSixAxisSensorManager::WakeDeviceUp() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);
    NN_SDK_REQUIRES_NOT_NULL(m_pHandheldManager);

    NN_RESULT_THROW_UNLESS(!m_CommonActivationCount.IsZero(),
                           ResultConsoleSixAxisSensorNotInitialized());

    if (m_pHandheldManager->IsHandheldEnabled())
    {
        NN_RESULT_DO(m_pDriver->Wake());
    }

    m_IsAwake = true;

    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorManager::PutDeviceToSleep() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    NN_RESULT_THROW_UNLESS(!m_CommonActivationCount.IsZero(),
                           ResultConsoleSixAxisSensorNotInitialized());

    NN_RESULT_DO(m_pDriver->Sleep());

    m_IsAwake = false;

    NN_RESULT_SUCCESS;
}

int ConsoleSixAxisSensorManager::GetSixAxisSensorState(SixAxisSensorState* pOutStates,
                                                       int count) const NN_NOEXCEPT
{
    const auto StateToCopyCount = (count < ::nn::hid::ConsoleSixAxisSensorStateCountMax) ? count
                                                                                         : ::nn::hid::ConsoleSixAxisSensorStateCountMax;
    for (int i = 0; i < StateToCopyCount; i++)
    {
        pOutStates[i] = m_SixAxisSensorStates[i];
    }

    return StateToCopyCount;
}

int ConsoleSixAxisSensorManager::GetSevenSixAxisSensorState(SevenSixAxisSensorState* pOutStates,
                                                         int count) const NN_NOEXCEPT
{
    const auto StateToCopyCount = (count < ::nn::hid::ConsoleSixAxisSensorStateCountMax) ? count
                                                                                         : ::nn::hid::ConsoleSixAxisSensorStateCountMax;
    for (int i = 0; i < StateToCopyCount; i++)
    {
        pOutStates[i] = m_SevenSixAxisSensorStates[i];
    }

    return StateToCopyCount;
}

void ConsoleSixAxisSensorManager::GetCalibrationParameters(
    ConsoleSixAxisSensorCalibrationParameters* pOutParameters) const NN_NOEXCEPT
{
    ::nn::settings::system::GetConsoleSixAxisSensorAccelerationBias(
        &pOutParameters->consoleSixAxisSensorAccelerationBias);
    ::nn::settings::system::GetConsoleSixAxisSensorAngularVelocityBias(
        &pOutParameters->consoleSixAxisSensorAngularVelocityBias);
    ::nn::settings::system::GetConsoleSixAxisSensorAccelerationGain(
        &pOutParameters->consoleSixAxisSensorAccelerationGain);
    ::nn::settings::system::GetConsoleSixAxisSensorAngularVelocityGain(
        &pOutParameters->consoleSixAxisSensorAngularVelocityGain);
    ::nn::settings::system::GetConsoleSixAxisSensorAngularVelocityTimeBias(
        &pOutParameters->consoleSixAxisSensorAngularVelocityTimeBias);
    ::nn::settings::system::GetConsoleSixAxisSensorAngularAcceleration(
        &pOutParameters->consoleSixAxisSensorAngularAcceleration);

    NN_DETAIL_HID_TRACE("%s from NAND\n", NN_CURRENT_FUNCTION_NAME);
    pOutParameters->Print();
}

void ConsoleSixAxisSensorManager::SetCalibrationParameters(
    const ConsoleSixAxisSensorCalibrationParameters& parameters) NN_NOEXCEPT
{
    ::nn::settings::system::SetConsoleSixAxisSensorAccelerationBias(
        parameters.consoleSixAxisSensorAccelerationBias);
    ::nn::settings::system::SetConsoleSixAxisSensorAngularVelocityBias(
        parameters.consoleSixAxisSensorAngularVelocityBias);
    ::nn::settings::system::SetConsoleSixAxisSensorAccelerationGain(
        parameters.consoleSixAxisSensorAccelerationGain);
    ::nn::settings::system::SetConsoleSixAxisSensorAngularVelocityGain(
        parameters.consoleSixAxisSensorAngularVelocityGain);
    ::nn::settings::system::SetConsoleSixAxisSensorAngularVelocityTimeBias(
        parameters.consoleSixAxisSensorAngularVelocityTimeBias);
    ::nn::settings::system::SetConsoleSixAxisSensorAngularAcceleration(
        parameters.consoleSixAxisSensorAngularAcceleration);

    NN_DETAIL_HID_TRACE("%s to NAND\n", NN_CURRENT_FUNCTION_NAME);
    parameters.Print();
}

const SixAxisSensorProcessor& ConsoleSixAxisSensorManager::GetSixAxisSensorProcessor() const NN_NOEXCEPT
{
    return m_pConsoleSixAxisSensorProcessor->GetSixAxisSensorProcessor();
}

::nn::Result ConsoleSixAxisSensorManager::GetConsoleSixAxisSensorCalibrationValues(tmp::ConsoleSixAxisSensorCalibrationValues* pOutValues) const NN_NOEXCEPT
{
    *pOutValues = m_ConsoleSixAxisSensorCalibrationValues;
    NN_RESULT_SUCCESS;
}

int ConsoleSixAxisSensorManager::GetSamplingFrequency() const NN_NOEXCEPT
{
    if (m_IsSevenSamplingStarted)
    {
        return ConsoleSixAxisSensorSamplingFrequency::ConsoleSixAxisSensorSamplingFrequency_Seven;
    }
    else if (m_IsNpadHandheldSamplingStarted)
    {
        return ConsoleSixAxisSensorSamplingFrequency::ConsoleSixAxisSensorSamplingFrequency_Npad;
    }
    else
    {
        // 少なくとも 1 つのアプリがサンプリングを有効化しているときは低周波数でサンプリングする
        const AppletResourceEntry(&entries)[AppletResourceEntryCountMax] =
            m_pAppletResourceManager->GetAppletResourceEntries();

        for (size_t i = 0; i < AppletResourceEntryCountMax; ++i)
        {
            const AppletResourceEntry& entry = entries[i];

            if (entry.flags.Test<AppletResourceFlag::IsAvailable>() &&
                entry.consoleSixAxisSensorSetting.console.IsSamplingEnabled())
            {
                return ConsoleSixAxisSensorSamplingFrequency::ConsoleSixAxisSensorSamplingFrequency_System;
            }
        }
    }

    return ConsoleSixAxisSensorSamplingFrequency::ConsoleSixAxisSensorSamplingFrequency_Off;
}

void ConsoleSixAxisSensorManager::EnableNpadHandheldSampling(bool enable) NN_NOEXCEPT
{
    m_IsNpadHandheldSamplingStarted = enable;
}

::nn::Result ConsoleSixAxisSensorManager::IsConsoleSixAxisSensorUserCalibrationSupported(bool* pOutIsSupported) NN_NOEXCEPT
{
    *pOutIsSupported = true; // 非サポートデバイスが存在しないため true
    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorManager::ResetConsoleSixAxisSensorCalibrationValues() NN_NOEXCEPT
{
    SetCalibrationParameters(DefaultConsoleSixAxisSensorCalibrationParameters);
    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorManager::StartConsoleSixAxisSensorUserCalibration() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pSixAxisSensorFilter);

    m_pSixAxisSensorFilter->StartCalibration();

    ConsoleSixAxisSensorSetting* pSetting = nullptr;
    m_pConsoleSixAxisSensorAppletSettingManager->GetConsoleSixAxisSensorSetting(&pSetting,
                                                                                m_pAppletResourceManager->GetAppletResourceUserId(),
                                                                                ::nn::hid::ConsoleSixAxisSensorHandle());
    pSetting->isSamplingEnabled = true;
    m_IsSevenSamplingStarted = pSetting->IsSamplingEnabled();
    m_CalibrationState = CalibrationState::CalibrationState_On;
    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorManager::CancelConsoleSixAxisSensorUserCalibration() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pSixAxisSensorFilter);

    m_pSixAxisSensorFilter->StopCalibration();
    m_CalibrationState = CalibrationState::CalibrationState_Off;
    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorManager::GetSixAxisSensorAccurateUserCalibrationState(::nn::hid::system::SixAxisSensorAccurateUserCalibrationState* pOutState) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutState);

    // 異常終了した場合
    NN_RESULT_THROW_UNLESS(
        m_CalibrationState != CalibrationState::CalibrationState_Invalid,
        system::ResultSixAxisSensorAccurateInvalidTermination());

    m_pSixAxisSensorFilter->GetAccurateUserCalibrationState(pOutState);

    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorManager::SetStateBufferMemoryHandle(const ::nn::applet::AppletResourceUserId& aruid,
                                                                     ::nn::os::NativeHandle&& transferMemoryHandle,
                                                                     size_t size,
                                                                     bool isManaged) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManager);

    auto needsRollback = true;

    // TransferMemory をアタッチ
    ::nn::os::TransferMemoryType* pTransferMemory = nullptr;
    NN_RESULT_DO(
        m_pAppletResourceManager->GetConsoleSixAxisSensorStateBufferTransferMemoryType(
            &pTransferMemory,
            aruid)
    );

    ::nn::os::AttachTransferMemory(
        pTransferMemory, size, transferMemoryHandle, isManaged);

    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            ::nn::os::DestroyTransferMemory(pTransferMemory);
        }
    };

    // TransferMemory をマップ
    void* address = nullptr;
    NN_RESULT_TRY(::nn::os::MapTransferMemory(
        &address,
        pTransferMemory,
        ::nn::os::MemoryPermission_ReadOnly))
        NN_RESULT_CATCH_ALL
        {
            return ResultSevenSixAxisSensorNullStateBuffer(); // toriaezu
        }
    NN_RESULT_END_TRY

    auto stateBuffer = ConsoleSixAxisSensorTansferMemoryStatus{
        std::move(transferMemoryHandle),
        isManaged,
        *pTransferMemory,
        address,
        size
    };
    m_pAppletResourceManager->SetConsoleSixAxisSensorTransferMemoryStateBuffer(aruid, std::move(stateBuffer));
    m_pSixAxisSensorFilter->SetStateBuffer(aruid, address, size);

    needsRollback = false;
    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorManager::SetWorkBufferMemoryHandle(const ::nn::applet::AppletResourceUserId& aruid,
                                                                    ::nn::os::NativeHandle&& transferMemoryHandle,
                                                                    size_t size,
                                                                    bool isManaged) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManager);

    auto needsRollback = true;

    // TransferMemory をアタッチ
    ::nn::os::TransferMemoryType* pTransferMemory = nullptr;
    m_pAppletResourceManager->GetConsoleSixAxisSensorWorkBufferTransferMemoryType(&pTransferMemory, aruid);

    ::nn::os::AttachTransferMemory(
        pTransferMemory, size, transferMemoryHandle, isManaged);

    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            ::nn::os::DestroyTransferMemory(pTransferMemory);
        }
    };

    // TransferMemory をマップ
    void* address = nullptr;
    NN_RESULT_TRY(::nn::os::MapTransferMemory(
        &address,
        pTransferMemory,
        ::nn::os::MemoryPermission_None))
        NN_RESULT_CATCH_ALL
        {
            return ResultSevenSixAxisSensorNullStateBuffer();
        }
    NN_RESULT_END_TRY

    auto workBuffer = ConsoleSixAxisSensorTansferMemoryStatus{
        std::move(transferMemoryHandle),
        isManaged,
        *pTransferMemory,
        address,
        size
    };
    m_pAppletResourceManager->SetConsoleSixAxisSensorTransferMemoryWorkBuffer(aruid, std::move(workBuffer));
    m_pSixAxisSensorFilter->SetWorkBuffer(aruid, address, size);
    m_pSixAxisSensorFilter->SetCalibrationParameters(m_ConsoleSixAxisSensorCalibrationParameters);

    needsRollback = false;
    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorManager::InitializeSevenSixAxisSensor(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManager);
    m_pAppletResourceManager->InitializeSevenSixAxisSensor(aruid, &m_pSixAxisSensorFilter);
    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorManager::FinalizeSevenSixAxisSensor(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManager);

    m_IsSevenSamplingStarted = false;
    m_pAppletResourceManager->FinalizeSevenSixAxisSensor(aruid);
    NN_RESULT_SUCCESS;
}

int ConsoleSixAxisSensorManager::GetConsoleSixAxisSensorCountStates(::nn::hid::debug::ConsoleSixAxisSensorCountState* pOutValues,
                                                                    int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValues);
    int gotCount = count;
    if (m_SamplingNumbers[ConsoleSixAxisSensorIdx_Count] < gotCount)
    {
        gotCount = static_cast<int>(m_SamplingNumbers[ConsoleSixAxisSensorIdx_Count]);
    }

    for (int i = 0; i < gotCount; i++)
    {
        auto pValue = &pOutValues[i];
        const auto& pState = &m_SixAxisSensorCountStates[i];

        pValue->deltaTime         = pState->deltaTime;
        pValue->samplingNumber    = pState->samplingNumber;
        pValue->tick              = pState->tick;
        pValue->acceleration.x    = pState->acceleration.x;
        pValue->acceleration.y    = pState->acceleration.y;
        pValue->acceleration.z    = pState->acceleration.z;
        pValue->angularVelocity.x = pState->angularVelocity.x;
        pValue->angularVelocity.y = pState->angularVelocity.y;
        pValue->angularVelocity.z = pState->angularVelocity.z;
    }

    return gotCount;
}

void ConsoleSixAxisSensorManager::Update() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);
    NN_SDK_REQUIRES_NOT_NULL(m_pHandheldManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pInputDetectorManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pInputDetectorManagerMutex);
    NN_SDK_REQUIRES_NOT_NULL(m_pSixAxisSensorFilter);

    // サンプル数を初期化
    m_AppendStateCount = 0;

    UpdateSixAxisSensorSetting();

    // サンプリング状態の更新
    const auto SamplingFrequency = GetSamplingFrequency();
    if (m_SamplingFrequency != SamplingFrequency)
    {
        if (SamplingFrequency == 0)
        {
            m_pDriver->Stop();
        }
        else
        {
            const int64_t SamplingIntervalMilliSeconds = 1000 / SamplingFrequency;
            m_pDriver->Stop();
            m_pDriver->Start(::nn::TimeSpan::FromMilliSeconds(SamplingIntervalMilliSeconds));
        }

        if (m_SamplingFrequency == 0)
        {
            m_SamplingStartedTick = ::nn::os::GetSystemTick();
            m_CoordinateNumber++;
        }

        m_SamplingFrequency = SamplingFrequency;
    }

    // Wake, Sleep のハンドリング
    if (m_IsAwake && m_pHandheldManager->IsHandheldEnabled())
    {
        // 起床中かつドックアウト状態ならばデバイスも起床
        m_pDriver->Wake();
    }
    else
    {
        // それ以外の状態ではデバイスはスリープ
        m_pDriver->Sleep();
    }

    // 入力状態を取得
    int validSampleCount = m_pDriver->GetStates(m_SixAxisSensorCountStates,
                                                SixAxisSensorStateCountMax);

    // 更新サンプル数を計算
    validSampleCount = (ConsoleSixAxisSensorStateCountMax < validSampleCount) ? ConsoleSixAxisSensorStateCountMax
                                                                              : validSampleCount;

    auto deltaSamplingNumber = m_SixAxisSensorCountStates[0].samplingNumber - m_SixAxisSensorLastSamplingNumber;

    // サンプリング番号がクリアされたときに負になることがあるため
    if (deltaSamplingNumber < 0)
    {
        deltaSamplingNumber = 0;
    }

    m_AppendStateCount = (validSampleCount < deltaSamplingNumber) ? validSampleCount
                                                                  : static_cast<int>(deltaSamplingNumber);

    m_SixAxisSensorLastSamplingNumber = m_SixAxisSensorCountStates[0].samplingNumber;

    // センサー値の加工計算を実行
    CalculateSixAxisSensorStates();

    // ワークバッファ割り当て済の場合は Filter にデータを注入
    if(m_pSixAxisSensorFilter->IsWorkBufferAllocated())
    {
        CalculateSevenSixAxisSensorStates();

        // オンラインでキャリブレーション値を更新
        m_pSixAxisSensorFilter->GetCalibrationParameters(&m_ConsoleSixAxisSensorCalibrationParameters);
    }


    // 生値のサンプル番号を更新
    for (int i = m_AppendStateCount - 1; i >= 0; i--)
    {
        m_SamplingNumbers[ConsoleSixAxisSensorIdx_Count]++;
        m_SixAxisSensorCountStates[i].samplingNumber = m_SamplingNumbers[ConsoleSixAxisSensorIdx_Count];
    }


    // 入力変化を通知
    ::std::lock_guard<decltype(*m_pInputDetectorManagerMutex)
                      > locker(*m_pInputDetectorManagerMutex);

    if (IsSixAxisSensorStateChanged(m_SixAxisSensorStates[0].angularVelocity) ||
        IsSixAxisSensorStateChanged(m_SevenSixAxisSensorStates[0].angularVelocity))
    {
        m_pInputDetectorManager->Notify(
            ::nn::hid::system::InputSourceId::ConsoleSixAxisSensor::Mask);
    }

    // 状態を更新
    m_SevenSixAxisSensorIsAtRest = m_pSixAxisSensorFilter->GetIsStill();
    m_VerticalizationError = m_pSixAxisSensorFilter->GetVerticalizationError();
    m_GyroBias = m_pSixAxisSensorFilter->GetGyroBias();

    // キャリブレーション状態を更新
    if (m_CalibrationState == CalibrationState::CalibrationState_On)
    {
        ::nn::hid::system::SixAxisSensorAccurateUserCalibrationState state;
        auto result = GetSixAxisSensorAccurateUserCalibrationState(&state);

        // 完了していたら NAND に書き込む
        if (result.IsSuccess() &&
            state.stage == system::SixAxisSensorAccurateUserCalibrationStage_Completed)
        {
            m_pSixAxisSensorFilter->GetCalibrationParameters(&m_ConsoleSixAxisSensorCalibrationParameters);
            SetCalibrationParameters(m_ConsoleSixAxisSensorCalibrationParameters);

            CancelConsoleSixAxisSensorUserCalibration(); // 完了したので止める
        }

    }

    // ARUID の更新
    const bool IsInFocusAruidChanged = (m_InFocusAruid != m_pAppletResourceManager->GetAppletResourceUserId());

    if (IsInFocusAruidChanged &&
        m_pSixAxisSensorFilter->IsWorkBufferAllocated())
    {
        CancelConsoleSixAxisSensorUserCalibration(); // アプリが切り替わったので止める
        m_CalibrationState = CalibrationState::CalibrationState_Invalid;
    }

    m_InFocusAruid = m_pAppletResourceManager->GetAppletResourceUserId(); // インフォーカス状態にある ARUID を更新
}

void ConsoleSixAxisSensorManager::ProcessSharedMemory(
    void (*processor)(
        ConsoleSixAxisSensorManager* that,
        ConsoleSixAxisSensorSharedMemoryFormat* ConsoleSixAxisSensorAddress,
        ::nn::applet::AppletResourceUserId aruid,
        bool enablesInput,
        bool isConsoleSixAxisSensorActivated) NN_NOEXCEPT) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(processor);
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManagerMutex);

    const bool isConsoleSixAxisSensorActivated = !m_ConsoleSixAxisSensorActivationCount.IsZero();

    ::std::lock_guard<decltype(*m_pAppletResourceManagerMutex)
                      > locker(*m_pAppletResourceManagerMutex);

    const AppletResourceEntry (&entries)[AppletResourceEntryCountMax] =
        m_pAppletResourceManager->GetAppletResourceEntries();

    for (size_t i = 0; i < AppletResourceEntryCountMax; ++i)
    {
        const AppletResourceEntry& entry = entries[i];

        ConsoleSixAxisSensorAppletResourceEntry& extraEntry = m_AppletResourceEntries[i];

        if (!entry.flags.Test<AppletResourceFlag::IsAvailable>())
        {
            extraEntry = ConsoleSixAxisSensorAppletResourceEntry();
        }
        else
        {
            NN_SDK_ASSERT_NOT_NULL(entry.address);

            if (entry.aruid != extraEntry.aruid)
            {
                extraEntry = ConsoleSixAxisSensorAppletResourceEntry();

                extraEntry.aruid = entry.aruid;
            }

            processor(this,
                      &entry.address->consoleSixAxisSensor,
                      entry.aruid,
                      entry.flags.Test<AppletResourceFlag::EnablesInput>(),
                      isConsoleSixAxisSensorActivated);
        }
    }
}

::nn::Result ConsoleSixAxisSensorManager::ActivateCommon() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTimerEvent);
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    // ドライバをアクティブ化
    NN_RESULT_DO(m_pDriver->Activate());

    // 起床状態に設定
    m_IsAwake = true;

    // タイマーイベントをアクティブ化
    ::nn::os::StartPeriodicTimerEvent(
        m_pTimerEvent, SamplingInterval, SamplingInterval);

    for (auto& state : m_SixAxisSensorCountStates)
    {
        state = tmp::SixAxisSensorCountState();
    }

    UpdateSixAxisSensorSetting();

    // NAND 上の 本体 6 軸センサーのキャリブレーション値を取得
    GetCalibrationParameters(&m_ConsoleSixAxisSensorCalibrationParameters);

    // キャリブレーション値を取得
    NN_RESULT_DO(m_pDriver->GetConsoleSixAxisSensorCalibrationValues(&m_ConsoleSixAxisSensorCalibrationValues));

    // 加工クラスにキャリブレーション値を注入
    m_pConsoleSixAxisSensorProcessor->SetCalibrationValues(m_ConsoleSixAxisSensorCalibrationValues);

    // 入力状態を更新
    this->Update();

    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorManager::DeactivateCommon() NN_NOEXCEPT
{
    // タイマーイベントを非アクティブ化
    ::nn::os::StopTimerEvent(m_pTimerEvent);

    // ドライバを非アクティブ化
    NN_RESULT_DO(m_pDriver->Deactivate());

    // 起床状態を解除
    m_IsAwake = false;

    NN_RESULT_SUCCESS;
}

void ConsoleSixAxisSensorManager::CalculateSixAxisSensorStates() NN_NOEXCEPT
{
    // JoyCon 相当の加工計算クラスに注入するサンプル数
    int stateCountToInject = static_cast<int>(SamplingInterval.GetMilliSeconds()) / NpadHandheldSamplingIntervalMilliSeconds; // 1000Hz to 200Hz にダウンサンプリング
    stateCountToInject = (m_AppendStateCount < stateCountToInject) ? m_AppendStateCount
                                                                   : stateCountToInject;
    // 生値を加工クラスに注入
    for (int i = stateCountToInject - 1; i >= 0; i--)
    {
        m_pConsoleSixAxisSensorProcessor->Sampling(m_SixAxisSensorCountStates[i]);
    }

    // 加工値を取得
    int count = m_pConsoleSixAxisSensorProcessor->GetSixAxisSensorStates(m_SixAxisSensorStates,
                                                                         stateCountToInject);
    NN_SDK_ASSERT_LESS_EQUAL(count, stateCountToInject);

    // 入力状態を更新
    for (int i = count - 1; i >= 0; i--)
    {
        m_SamplingNumbers[ConsoleSixAxisSensorIdx_JoyCon]++;
        m_SixAxisSensorStates[i].attributes.Set<nn::hid::SixAxisSensorAttribute::IsConnected>(true);
        m_SixAxisSensorStates[i].samplingNumber = m_SamplingNumbers[ConsoleSixAxisSensorIdx_JoyCon];
    }
}

void ConsoleSixAxisSensorManager::CalculateSevenSixAxisSensorStates() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pDriver);
    NN_SDK_ASSERT_NOT_NULL(m_pSixAxisSensorFilter);

    // JoyCon 相当の加工計算クラスに注入するサンプル数
    int stateCountToInject = m_AppendStateCount;

    // 生値を加工クラスに注入
    ::nn::util::Float3 acceleration;
    ::nn::util::Float3 angularVelocity;
    ::nn::os::Tick tick;

    for (int i = stateCountToInject - 1; i >= 0; i--)
    {
        m_pDriver->ConvertAcceleration(&acceleration, m_SixAxisSensorCountStates[i].acceleration);
        m_pDriver->ConvertAngularVelocity(&angularVelocity, m_SixAxisSensorCountStates[i].angularVelocity);
        tick = ::nn::os::Tick(m_SixAxisSensorCountStates[i].tick);
        m_pSixAxisSensorFilter->Update(acceleration,
                                       angularVelocity,
                                       tick);
    }

    // 加工値を取得
    int count = m_pSixAxisSensorFilter->GetStates(m_SevenSixAxisSensorStates,
                                                  stateCountToInject,
                                                  &tick);

    // 入力状態を更新
    for (int i = count - 1; i >= 0; i--)
    {
        m_SamplingNumbers[ConsoleSixAxisSensorIdx_Seven]++;

        m_SevenSixAxisSensorStates[i].samplingNumber = m_SamplingNumbers[ConsoleSixAxisSensorIdx_Seven];
        m_SevenSixAxisSensorStates[i].coordinateNumber = m_CoordinateNumber;
        m_SevenSixAxisSensorStates[i].timeStamp -= ::nn::os::ConvertToTimeSpan(m_SamplingStartedTick);
    }
}

}}} // namespace nn::hid::detail
