﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/settings_Xpad.h>
#include <nn/settings/settings_GenericPadAxis.h>
#include <nn/settings/settings_GenericPadButton.h>

#include "hid_AbstractedGenericPad-os.win.h"
#include "hid_CommonStateUtility.h"
#include "hid_HidShellPortFile-os.win.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< コントローラー共通の電源状態
const system::PowerInfo StaticGeneircPadPowerInfo = { false, false, system::BatteryLevel_High };

//!< 基本的な構成を持つ Xpad のデジタルボタンの押下状態を返します。
AbstractedPadButtonSet GetButtons(const WindowsGenericPad& genericPad) NN_NOEXCEPT
{
    WindowsGenericPadButtonSet buttons = genericPad.GetButtons();
    AbstractedPadButtonSet value = {};
    value.Set<AbstractedPadButton::A>(
        buttons.Test<WindowsGenericPadButton::B>());
    value.Set<AbstractedPadButton::B>(
        buttons.Test<WindowsGenericPadButton::A>());
    value.Set<AbstractedPadButton::X>(
        buttons.Test<WindowsGenericPadButton::Y>());
    value.Set<AbstractedPadButton::Y>(
        buttons.Test<WindowsGenericPadButton::X>());
    value.Set<AbstractedPadButton::StickL>(
        buttons.Test<WindowsGenericPadButton::ThumbL>());
    value.Set<AbstractedPadButton::StickR>(
        buttons.Test<WindowsGenericPadButton::ThumbR>());
    value.Set<AbstractedPadButton::L>(
        buttons.Test<WindowsGenericPadButton::L>());
    value.Set<AbstractedPadButton::R>(
        buttons.Test<WindowsGenericPadButton::R>());
    value.Set<AbstractedPadButton::ZL>(
        buttons.Test<WindowsGenericPadButton::TriggerL>());
    value.Set<AbstractedPadButton::ZR>(
        buttons.Test<WindowsGenericPadButton::TriggerR>());
    value.Set<AbstractedPadButton::Start>(
        buttons.Test<WindowsGenericPadButton::Start>());
    value.Set<AbstractedPadButton::Select>(
        buttons.Test<WindowsGenericPadButton::Back>());
    value.Set<AbstractedPadButton::Left>(
        buttons.Test<WindowsGenericPadButton::Left>());
    value.Set<AbstractedPadButton::Up>(
        buttons.Test<WindowsGenericPadButton::Up>());
    value.Set<AbstractedPadButton::Right>(
        buttons.Test<WindowsGenericPadButton::Right>());
    value.Set<AbstractedPadButton::Down>(
        buttons.Test<WindowsGenericPadButton::Down>());
    return value;
}

//!< 指定の割り当てに基づいた Xpad のデジタルボタンの押下状態を返します。
AbstractedPadButtonSet GetButtons(
    const WindowsGenericPad& genericPad,
    const ::nn::settings::BasicXpadGenericPadMap& map) NN_NOEXCEPT
{
    AbstractedPadButtonSet value = {};
    value.Set<AbstractedPadButton::A>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonA)));
    value.Set<AbstractedPadButton::B>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonB)));
    value.Set<AbstractedPadButton::X>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonX)));
    value.Set<AbstractedPadButton::Y>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonY)));
    value.Set<AbstractedPadButton::StickL>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonStickL)));
    value.Set<AbstractedPadButton::StickR>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonStickR)));
    value.Set<AbstractedPadButton::L>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonL)));
    value.Set<AbstractedPadButton::R>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonR)));
    value.Set<AbstractedPadButton::ZL>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonZL)));
    value.Set<AbstractedPadButton::ZR>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonZR)));
    value.Set<AbstractedPadButton::Start>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonStart)));
    value.Set<AbstractedPadButton::Select>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonSelect)));
    value.Set<AbstractedPadButton::Left>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonLeft)));
    value.Set<AbstractedPadButton::Up>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonUp)));
    value.Set<AbstractedPadButton::Right>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonRight)));
    value.Set<AbstractedPadButton::Down>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonDown)));
    return value;
}

//!< NpadFullKey の状態を取得します。
void GetAbstractedPadState(
    AbstractedPadState* pOutValue,
    const ::nn::hid::debug::BasicXpadAutoPilotState& state) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    pOutValue->buttons = ConvertBasicXpadButtonSetToAbstracted(state.buttons);
    pOutValue->analogStickR = LimitAnalogStick(
        state.analogStickR, 0, AnalogStickMax, AnalogStickMax);
    pOutValue->analogStickL = LimitAnalogStick(
        state.analogStickL, 0, AnalogStickMax, AnalogStickMax);
}
}

const ::nn::TimeSpan AbstractedGenericPad::SamplingInterval =
    ::nn::TimeSpan::FromMilliSeconds(5);

AbstractedGenericPad::AbstractedGenericPad() NN_NOEXCEPT
    : m_pTimerEvent(nullptr)
    , m_pIdPublisher(nullptr)
    , m_IsConnected(false)
    , m_pMappingManager(nullptr)
{
    // Usb 接続
    m_InterfaceType = system::InterfaceType_Usb;
    // Switch Pro Controller 互換
    m_DeviceType = system::DeviceType::SwitchProController::Mask;

    // FeatureSet
    m_FeatureSet.Reset();
    m_FeatureSet.Set<AbstractedPadFeature::AnalogStick>();
}

AbstractedGenericPad::~AbstractedGenericPad() NN_NOEXCEPT
{
    // 何もしない
}

AbstractedPadType AbstractedGenericPad::GetType() NN_NOEXCEPT
{
    return s_Type;
}

bool AbstractedGenericPad::IsConnected() NN_NOEXCEPT
{
    return m_IsConnected;
}

system::PowerInfo AbstractedGenericPad::GetPowerInfo() NN_NOEXCEPT
{
    return StaticGeneircPadPowerInfo;
}

AbstractedPadState AbstractedGenericPad::GetPadState() NN_NOEXCEPT
{
    return m_PadState;
}

bool AbstractedGenericPad::GetButtonTriggerElapsedTime(nn::os::Tick* pOutTick, AbstractedPadButtonSet button) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(button.CountPopulation(), 1);
    NN_UNUSED(pOutTick);
    NN_UNUSED(button);

    // ボタン押しをさかのぼる機能は提供しない
    return false;
}

int AbstractedGenericPad::GetSensorStates(nn::xcd::SixAxisSensorState* pOutValue, int count) NN_NOEXCEPT
{
    NN_UNUSED(pOutValue);
    NN_UNUSED(count);
    return 0;
}

void AbstractedGenericPad::SetIndicator(uint8_t pattern, bool blink) NN_NOEXCEPT
{
    NN_UNUSED(blink);

    if (m_IsConnected)
    {
        m_Pattern = pattern;
    }
}

uint8_t AbstractedGenericPad::GetIndicator() NN_NOEXCEPT
{
    if (m_IsConnected)
    {
        return m_Pattern;
    }

    return 0;
}

bool AbstractedGenericPad::IsWired() NN_NOEXCEPT
{
    if (m_IsConnected == false)
    {
        return false;
    }
    // USB コントローラーは常時 Wired 状態
    return true;
}

bool AbstractedGenericPad::IsUsbConnected() NN_NOEXCEPT
{
    if (m_IsConnected == false)
    {
        return false;
    }
    // USB コントローラーは常時 USB 接続状態
    return true;
}

void AbstractedGenericPad::Connect() NN_NOEXCEPT
{
    if (m_IsAttached == false)
    {
        return;
    }

    // 接続状態に変更
    if (m_IsConnected == false)
    {
        m_IsConnected = true;
    }
}

void AbstractedGenericPad::Detach() NN_NOEXCEPT
{
    m_IsConnected = false;
}

bool AbstractedGenericPad::HasBattery() NN_NOEXCEPT
{
    return false;
}

bool AbstractedGenericPad::SetDeviceInfoOnPlayReportControllerUsage(system::PlayReportControllerUsage* pOutValue) NN_NOEXCEPT
{
    // TODO : Gc 用の情報を追加する
    pOutValue->deviceType = system::PlayReportDeviceType_SwitchProController;
    for (auto& byte : pOutValue->raw)
    {
        byte = 0;
    }
    return true;
}

void AbstractedGenericPad::SetTimerEvent(::nn::os::TimerEventType* pTimerEvent
                                ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pTimerEvent);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pTimerEvent = pTimerEvent;
}

void AbstractedGenericPad::ActivateTimerEvent() NN_NOEXCEPT
{
    if (m_pTimerEvent != nullptr &&
        m_ActivationCount.IsZero())
    {
        ::nn::os::StartPeriodicTimerEvent(
            m_pTimerEvent, SamplingInterval, SamplingInterval);
    }
}

void AbstractedGenericPad::DeactivateTimerEvent() NN_NOEXCEPT
{
    if (m_pTimerEvent != nullptr &&
        m_ActivationCount.IsZero())
    {
        ::nn::os::StopTimerEvent(m_pTimerEvent);
    }
}

void AbstractedGenericPad::SetAbstractedPadIdPublisher(AbstractedPadIdPublisher* pPublisher) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPublisher);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pIdPublisher = pPublisher;
}

void AbstractedGenericPad::SetHidShellPort(int port) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValidHidShellXpadPort(port));
    m_HidShellPort = port;
}

void AbstractedGenericPad::SetMappingManager(XpadMappingManager* pMappingManager
                                   ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pMappingManager);
    m_pMappingManager = pMappingManager;
}

bool AbstractedGenericPad::IsActivated() NN_NOEXCEPT
{
    return !m_ActivationCount.IsZero();
}

::nn::Result AbstractedGenericPad::Activate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultGamePadDriverActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施
        m_WindowsGenericPad.Reset();

        // 汎用ゲームパッドとのマッピングを取得
        m_pMappingManager->GetBasicXpadGenericPadMap(&m_GenericPadMap);
        if (m_pMappingManager->IsValidBasicXpadGenericPadMap(m_GenericPadMap))
        {
            // マッピングされた識別子が有効ならば対象とする汎用コントローラに設定
            m_WindowsGenericPad.SetGenericPadId(m_GenericPadMap.genericPadId);
        }

        m_WindowsKeyboard.Reset();

        // タイマーイベントをアクティブ化
        this->ActivateTimerEvent();
    }

    // アクティブ化した回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result AbstractedGenericPad::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultGamePadDriverDeactivationLowerLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // タイマーイベントを非アクティブ化
        this->DeactivateTimerEvent();
    }

    // アクティブ化した回数をデクリメント
    --m_ActivationCount;

    NN_RESULT_SUCCESS;
}

void AbstractedGenericPad::Update() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    bool isAttached = false;
    auto oldButtons = m_PadState.buttons;

    // Win コントローラーでは更新処理が走るたびに新規サンプル受信相当とする
    m_IsNewSampleReceived = true;

    // 汎用ゲームパッドとのマッピングを取得
    m_pMappingManager->GetBasicXpadGenericPadMap(&m_GenericPadMap);

    if (!m_pMappingManager->IsValidBasicXpadGenericPadMap(m_GenericPadMap))
    {
        // マッピングされた識別子が無効ならば内部状態をリセット
        m_WindowsGenericPad.Reset();
    }
    else
    {
        // マッピングされた識別子が有効ならば対象とする汎用コントローラに設定
        m_WindowsGenericPad.SetGenericPadId(m_GenericPadMap.genericPadId);

        // 汎用コントローラの内部状態を更新
        m_WindowsGenericPad.Update();

        // Xpad の入力状態を決定
        isAttached = m_WindowsGenericPad.IsDetected();
        if (m_WindowsGenericPad.IsXInput())
        {
            m_PadState.buttons = GetButtons(m_WindowsGenericPad);
            m_PadState.analogStickR = LimitAnalogStick(
                m_WindowsGenericPad.GetStickR(),
                0, XInputAnalogStickMaxCount,
                XInputAnalogStickTheoreticalCount);
            m_PadState.analogStickL = LimitAnalogStick(
                m_WindowsGenericPad.GetStickL(),
                0, XInputAnalogStickMaxCount,
                XInputAnalogStickTheoreticalCount);
        }
        else
        {
            m_PadState.buttons = GetButtons(m_WindowsGenericPad, m_GenericPadMap);
            m_PadState.analogStickR = LimitAnalogStick(
                m_WindowsGenericPad.GetStickR(m_GenericPadMap),
                XInputAnalogStickRDeadZoneCount, XInputAnalogStickMaxCount,
                XInputAnalogStickTheoreticalCount);
            m_PadState.analogStickL = LimitAnalogStick(
                m_WindowsGenericPad.GetStickL(m_GenericPadMap),
                XInputAnalogStickLDeadZoneCount, XInputAnalogStickMaxCount,
                XInputAnalogStickTheoreticalCount);
        }

        // アナログスティックの十字ボタンエミュレーション
        SetCrossStickEmulationButtonsOnAbstractedPad(&m_PadState);
    }

    // キーボードとのマッピングを取得
    m_pMappingManager->GetBasicXpadKeyboardMap(&m_KeyboardMap);

    if (!m_pMappingManager->IsValidBasicXpadKeyboardMap(m_KeyboardMap))
    {
        // マッピングが無効ならば内部状態をリセット
        m_WindowsKeyboard.Reset();
    }
    else
    {
        // Keyboard のキーの状態について内部状態を更新
        m_WindowsKeyboard.UpdateKeys();

        // マッピングが有効ならば接続状態
        isAttached = true;

        // キーボードによるエミュレーション結果をマージ
        m_PadState.buttons = MergeNpadButtons(
            m_PadState.buttons, ConvertBasicXpadButtonSetToAbstracted(
                m_WindowsKeyboard.GetButtons(m_KeyboardMap)));
    }

    //!< 基本的な構成を持つ Xpad の HidShell 向けに共有される自動操作状態を取得
    ::nn::hid::debug::BasicXpadAutoPilotState basicXpadAutoPilotState = {};
    bool isBasicXpadAutoPilotEnabled = false;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        GetHidShellXpadState(&isBasicXpadAutoPilotEnabled,
                             &basicXpadAutoPilotState, m_HidShellPort));
    if (isBasicXpadAutoPilotEnabled)
    {
        GetAbstractedPadState(&m_PadState, basicXpadAutoPilotState);
        isAttached = basicXpadAutoPilotState.powerState != PowerState_Disconnected;
    }

    // 離されたボタンは無効化を解除
    m_ButtonMask |= ~m_PadState.buttons;
    // 無効化
    m_PadState.buttons &= m_ButtonMask;

    // 何かボタンが押されたら接続状態へと変更する
    bool isButtonTriggered = ((oldButtons ^ m_PadState.buttons) & m_PadState.buttons).IsAnyOn();

    UpdateConnectionState(isAttached, isButtonTriggered);
}

void AbstractedGenericPad::UpdateConnectionState(bool isAttached, bool isKeyOn) NN_NOEXCEPT
{
    if (m_IsAttached != isAttached)
    {
        if (isAttached)
        {
            m_Id = m_pIdPublisher->PublishId();
            m_ButtonMask.Reset();
        }
        else
        {
            m_Id = AbstractedPadId();
            m_PadState = AbstractedPadState();
            m_IsConnected = false;
        }
        m_IsAttached = isAttached;
    }

    if (m_IsConnected == false && isKeyOn == true)
    {
        m_IsConnected = true;
        // 入力無効化するボタンを追加
        m_ButtonMask &= ~m_PadState.buttons;
    }
}

}}} // namespace nn::hid::detail
