﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "grcsrv_SyncpointEventManager.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <nvos.h>
#include <hos/nvhost_devctl.h>

#include "../grcsrv_SyncpointWaiter.h"
#include "../grcsrv_SyncpointLog.h"

#define NVHOST_CTRL_DEVNODE "/dev/nvhost-ctrl"

enum EventType {
    EventTypeSystem = 0x1,
    EventTypeTimer = 0x2,
    EventTypeUser = 0x3,
};

typedef struct NvOsSystemEventRec
{
    int fd;
    uint32_t id;
    nn::os::SystemEventType data;
} NvOsSystemEvent;

typedef struct NvOsTimerEventRec
{
    nn::os::TimerEventType data;
} NvOsTimerEvent;

typedef struct NvOsUserEventRec
{
    nn::os::EventType data;
} NvOsUserEvent;

typedef struct NvOsEventRec
{
    EventType type;
    union {
        NvOsSystemEvent sysEvent;
        NvOsTimerEvent timerEvent;
        NvOsUserEvent userEvent;
    } eventTypeData;
    nn::os::MultiWaitHolderType holder;
} NvOsEvent;


namespace nn{ namespace grcsrv{ namespace offscreen{ namespace native{ namespace detail{

    const int SyncpointEventManager::Size;
    const int SyncpointEventManager::EventSlotOffset;

    SyncpointEventManager::SyncpointEventManager() NN_NOEXCEPT
    {
        m_DeviceFd = 0;
        std::memset(m_SlotList, 0, sizeof(m_SlotList));
    }

    void SyncpointEventManager::Initialize() NN_NOEXCEPT
    {
        m_DeviceFd = NvOsDrvOpen(NVHOST_CTRL_DEVNODE);
        if(m_DeviceFd < 0)
        {
            NN_LOG_SYNCPOINT_ERR("Failed to open ctrl driver (%d)\n", m_DeviceFd);
        }
        NN_ABORT_UNLESS_GREATER_EQUAL(m_DeviceFd, 0);

        for(int i = 0; i < Size; i++)
        {
            m_SlotList[i].state = SlotState_Empty;
        }
    }

    void SyncpointEventManager::Finalize() NN_NOEXCEPT
    {
        for(int i = 0; i < Size; i++)
        {
            if(m_SlotList[i].state != SlotState_Empty)
            {
                DestroySlotImpl(i);
            }
        }

        NvOsDrvClose(m_DeviceFd);
    }

    bool SyncpointEventManager::IsFull() const NN_NOEXCEPT
    {
        for(int i = 0; i < Size; i++)
        {
            if(m_SlotList[i].state != SlotState_Waiting)
            {
                return false;
            }
        }
        return true;
    }

    bool SyncpointEventManager::IsEmpty() const NN_NOEXCEPT
    {
        for(int i = 0; i < Size; i++)
        {
            auto state = m_SlotList[i].state;
            if(state != SlotState_Empty && state != SlotState_Unused)
            {
                return false;
            }
        }
        return true;
    }

    bool SyncpointEventManager::AcquireWaitSlot(int* pOutSlotIndex, const NativeRmFence& fence, SyncpointEntry* pEntry) NN_NOEXCEPT
    {
        int slotIndex = -1;
        for(int i = 0; i < Size; i++)
        {
            if(m_SlotList[i].state != SlotState_Waiting)
            {
                slotIndex = i;
                break;
            }
        }
        NN_ABORT_UNLESS_GREATER_EQUAL(slotIndex, 0);
        if(m_SlotList[slotIndex].state == SlotState_Empty)
        {
            CreateSlotImpl(slotIndex);
        }

        nvhost_ctrl_syncpt_waitex_args args = {};
        args.id      = fence.syncpointId;
        args.thresh  = fence.value;
        args.timeout = NVHOST_NO_TIMEOUT;
        args.value   = slotIndex + EventSlotOffset;

        NvError err = NvOsDrvIoctl(m_DeviceFd, NVHOST_DEVCTL_CTRL_SYNCPT_WAITEX3, &args, sizeof(args));
        if(err == NvSuccess)
        {
            // already expired
            *pOutSlotIndex = -1;
            return true;
        }
        NN_ABORT_UNLESS_EQUAL(err, NvError_Timeout);

        m_SlotList[slotIndex].state = SlotState_Waiting;
        m_SlotList[slotIndex].pEntry = pEntry;
        *pOutSlotIndex = slotIndex;
        return false;
    }

    void SyncpointEventManager::ReleaseWaitSlot(int slotIndex) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(slotIndex, 0, Size);
        auto& slot = m_SlotList[slotIndex];
        NN_SDK_REQUIRES_EQUAL(slot.state, SlotState_Waiting);
        NN_SDK_REQUIRES(nn::os::TryWaitSystemEvent(&slot.pEvent->eventTypeData.sysEvent.data));

        nn::os::ClearSystemEvent(&slot.pEvent->eventTypeData.sysEvent.data);
        slot.state = SlotState_Unused;
        slot.pEntry = nullptr;
    }

    void SyncpointEventManager::CancelWaitSlot(int slotIndex) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(slotIndex, 0, Size);
        auto& slot = m_SlotList[slotIndex];
        NN_SDK_REQUIRES_EQUAL(slot.state, SlotState_Waiting);

        nvhost_ctrl_syncpt_clear_waitex_args args = {};
        args.event_id = NVHOST_WAITEX3_GEN_EVENT_ID(0, slotIndex + EventSlotOffset);
        NvOsDrvIoctl(m_DeviceFd, NVHOST_DEVCTL_CTRL_SYNCPT_CLEAR_WAITEX, &args, sizeof(args));

        slot.state = SlotState_Unused;
        slot.pEntry = nullptr;
    }

    SyncpointEntry* SyncpointEventManager::GetEntry(int slotIndex) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(slotIndex, 0, Size);
        auto& slot = m_SlotList[slotIndex];
        return slot.pEntry;
    }

    nn::os::MultiWaitHolderType* SyncpointEventManager::GetMultiWaitHolder(int slotIndex) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(slotIndex, 0, Size);
        auto& slot = m_SlotList[slotIndex];
        NN_SDK_REQUIRES_EQUAL(slot.state, SlotState_Waiting);

        return &slot.pEvent->holder;
    }


    void SyncpointEventManager::CreateSlotImpl(int slotIndex) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(slotIndex, 0, Size);
        auto& slot = m_SlotList[slotIndex];
        NN_SDK_REQUIRES_EQUAL(slot.state, SlotState_Empty);

        nvhost_ctrl_syncpt_event_mgmt_args args = {};
        args.slot = slotIndex + EventSlotOffset;

        NvError err;
        err = NvOsDrvIoctl(m_DeviceFd, NVHOST_DEVCTL_CTRL_SYNCPT_EVENT_CREATE, &args, sizeof(args));
        NN_ABORT_UNLESS_EQUAL(err, NvSuccess);

        err = NvOsSystemEventCreate(m_DeviceFd, NVHOST_WAITEX3_GEN_EVENT_ID(0, args.slot), &slot.pEvent);
        NN_ABORT_UNLESS_EQUAL(err, NvSuccess);

        slot.state = SlotState_Unused;
        slot.pEntry = nullptr;
    }

    void SyncpointEventManager::DestroySlotImpl(int slotIndex) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(slotIndex, 0, Size);
        auto& slot = m_SlotList[slotIndex];
        NN_SDK_REQUIRES_NOT_EQUAL(slot.state, SlotState_Empty);
        NvOsEventDestroy(slot.pEvent);
        slot.state = SlotState_Empty;
        slot.pEvent = nullptr;
        slot.pEntry = nullptr;
    }

}}}}}
