﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "grcsrvOffscreen_VicCopyCaptureImageBuffer.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/grc/grc_Result.h>
#include <nvnflinger_service.h>
#include <nvgr.h>

#include "grcsrvOffscreen_Config.h"
#include "../../capsrv/capture/capsrv_CaptureModule.h"
#include "../../capsrv/capture/capsrv_CaptureModule-module.nvnflinger.h"

namespace nn{ namespace grcsrv{ namespace offscreen{

    namespace {
        nn::Result VicCopyCaptureImageBufferImpl(
            android::sp<android::Fence>* pOutFence,
            const android::sp<android::GraphicBuffer>& dst,
            const NvRect& dstRect,
            const android::sp<android::GraphicBuffer>& src,
            const NvRectF32& srcRect,
            nn::capsrv::capture::CaptureModule* pModule,
            bool flipY
        ) NN_NOEXCEPT
        {
            static const int InputSlotIndex = 0;

            NvError err = {};

            NvDdkVicConfigParameters config;
            std::memset(&config, 0, sizeof(config));
            config.BackgroundColor = NvDdkVicFloatColorRec{ 0, 0, 0, 1 };
            config.BackgroundColorSpace = NvDdkVicColorSpace_LegacyRGB;

            NvRmSurface* dstSurfList = nullptr;
            size_t dstSurfCount = 0;
            nvgr_get_surfaces(dst->handle, const_cast<const NvRmSurface**>(&dstSurfList), &dstSurfCount);
            NN_RESULT_THROW_UNLESS(dstSurfCount > 0 && dstSurfCount <= 3, nn::grc::ResultInvalidCall());

            NvRmSurface* srcSurfList = nullptr;
            size_t srcSurfCount = 0;
            nvgr_get_surfaces(src->handle, const_cast<const NvRmSurface**>(&srcSurfList), &srcSurfCount);
            NN_RESULT_THROW_UNLESS(srcSurfCount > 0 && srcSurfCount <= 3, nn::grc::ResultInvalidCall());

            // configure target
            err = NvDdkVicConfigureTargetSurface(
                pModule->GetVicSession(),
                &config,
                dstSurfList,
                static_cast<NvU32>(dstSurfCount),
                &dstRect
            );
            NN_RESULT_THROW_UNLESS(err == NvSuccess, nn::grc::ResultInvalidState());

            // configure source
            err = NvDdkVicConfigureSourceSurface(
                pModule->GetVicSession(),
                &config,
                InputSlotIndex,
                srcSurfList,
                static_cast<NvU32>(srcSurfCount),
                &srcRect,
                &dstRect
            );
            NN_RESULT_THROW_UNLESS(err == NvSuccess, nn::grc::ResultInvalidState());

            {
                auto& slot = config.Slot[InputSlotIndex];

                // setup downsample method
                slot.FilterX = NvDdkVicFilter_Nicest;
                slot.FilterY = NvDdkVicFilter_Nicest;

                // setup blend parameter
                auto& blend = slot.Blend;
                blend.K1 = 1;
                blend.SrcFactC = NvDdkVicBlendSrcFactC_K1;
                blend.DstFactC = NvDdkVicBlendDstFactC_Zero;
                blend.SrcFactA = NvDdkVicBlendSrcFactA_K1;
                blend.DstFactA = NvDdkVicBlendDstFactA_Zero;
                blend.UseOverrideA = NV_TRUE;

                // setup orientation
                config.OutputFlipY = flipY;
            }

            err = NvDdkVicConfigure(pModule->GetVicSession(), &config);
            NN_RESULT_THROW_UNLESS(err == NvSuccess, nn::grc::ResultInvalidState());

            NvDdkVicExecParameters params;
            std::memset(&params, 0, sizeof(params));
            params.OutputSurface = dstSurfList;
            for(size_t p = 0; p < srcSurfCount; p++)
            {
                params.InputSurfaces[InputSlotIndex][p] = &srcSurfList[p];
            }

            int fenceFd = -1;
            err = NvDdkVicExecuteFd(
                pModule->GetVicSession(),
                &params,
                -1,
                &fenceFd,
                "grccopy"
            );
            NN_RESULT_THROW_UNLESS(err == NvSuccess, nn::grc::ResultInvalidState());

            android::sp<android::Fence> outFence;
            if(fenceFd >= 0)
            {
                outFence = new android::Fence(fenceFd);
                NN_RESULT_THROW_UNLESS(outFence != nullptr && outFence->isValid(), nn::grc::ResultOutOfResource());
            }
            else
            {
                outFence = android::Fence::NO_FENCE;
            }

            *pOutFence = outFence;
            NN_RESULT_SUCCESS;
        }
    }

    nn::Result VicCopyCaptureImageBuffer(
        android::sp<android::Fence>* pOutFence,
        const android::sp<android::GraphicBuffer>& dst,
        const android::sp<android::GraphicBuffer>& src,
        nn::capsrv::capture::CaptureModule* pModule
    ) NN_NOEXCEPT
    {
        NvRect dstRect = {};
        dstRect.left   = 0;
        dstRect.top    = 0;
        dstRect.right  = static_cast<NvS32>(dst->getWidth());
        dstRect.bottom = static_cast<NvS32>(dst->getHeight());

        NvRectF32 srcRect = {};
        srcRect.left   = 0;
        srcRect.top    = 0;
        srcRect.right  = static_cast<float>(src->getWidth());
        srcRect.bottom = static_cast<float>(src->getHeight());

        bool flipY = false;

        return VicCopyCaptureImageBufferImpl(pOutFence, dst, dstRect, src, srcRect, pModule, flipY);
    }

    nn::Result VicCopyCaptureImageBuffer(
        android::sp<android::Fence>* pOutFence,
        const android::sp<android::GraphicBuffer>& dst,
        const android::sp<android::GraphicBuffer>& src,
        const android::IGraphicBufferConsumer::BufferItem& srcItem,
        nn::capsrv::capture::CaptureModule* pModule
    ) NN_NOEXCEPT
    {
        NvRect dstRect = {};
        dstRect.left   = 0;
        dstRect.top    = 0;
        dstRect.right  = static_cast<NvS32>(dst->getWidth());
        dstRect.bottom = static_cast<NvS32>(dst->getHeight());

        NvRectF32 srcRect = {};
        if(srcItem.mCrop.width() > 0 && srcItem.mCrop.height() > 0)
        {
            srcRect.left   = srcItem.mCrop.left;
            srcRect.top    = srcItem.mCrop.top;
            srcRect.right  = srcItem.mCrop.right;
            srcRect.bottom = srcItem.mCrop.bottom;
        }
        else
        {
            srcRect.left   = 0;
            srcRect.top    = 0;
            srcRect.right  = static_cast<float>(src->getWidth());
            srcRect.bottom = static_cast<float>(src->getHeight());
        }

        bool flipY =  srcItem.mTransform & NATIVE_WINDOW_TRANSFORM_FLIP_V;

        return VicCopyCaptureImageBufferImpl(pOutFence, dst, dstRect, src, srcRect, pModule, flipY);
    }



}}}
