﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include "grcsrvOffscreen_Config.h"

#include "../../capsrv/capture/capsrv_MemoryPool.h"
#include "../../capsrv/capture/capsrv_MemoryPool-module.nvnflinger.h"
#include "../../capsrv/capture/capsrv_ImageBuffer.h"
#include "../../capsrv/capture/capsrv_ImageBuffer-module.nvnflinger.h"

#include <nvnflinger_service.h>

namespace nn{ namespace grcsrv{ namespace offscreen{

    class CaptureImageBuffer
    {
    public:
        static const int BufferCountMax = VideoCaptureImageBufferCount;

    public:
        static size_t GetRequiredMemorySize(int width, int height, int bufferCount) NN_NOEXCEPT
        {
            size_t v =
                GetRequiredBufferMemorySizeNv12YImpl(width, height) +
                GetRequiredBufferMemorySizeNv12UvImpl(width, height);
            return v * bufferCount;
        }

        static size_t GetRequiredMemoryAlignment() NN_NOEXCEPT
        {
            return 4096 * 2;
        }

    private:
        static size_t GetRequiredBufferMemorySizeNv12YImpl(int width, int height) NN_NOEXCEPT
        {
            if(width == 1280 && height == 720)
            {
                return 1048576; // 8 * 128KB
            }
            else if(width == 1920 && height == 1080)
            {
                return 2097152; // 16 * 128KB
            }
            else
            {
                NN_ABORT("unsupported image size");
            }
        }

        static size_t GetRequiredBufferMemorySizeNv12UvImpl(int width, int height) NN_NOEXCEPT
        {
            if(width == 1280 && height == 720)
            {
                return 524288; // 4 * 128KB
            }
            else if(width == 1920 && height == 1080)
            {
                return 1048576; // 8 * 128KB
            }
            else
            {
                NN_ABORT("unsupported image size");
            }
        }

    public:

        CaptureImageBuffer() NN_NOEXCEPT;

        // @pre memory % GetRequiredMemoryAlignment == 0
        // @pre size >= GetRequiredMemorySize
        nn::Result InitializeMemoryPool(
            nn::capsrv::capture::CaptureModule* pModule,
            void* memory,
            size_t size,
            int widthMax,
            int heightMax
        ) NN_NOEXCEPT;
        void FinalizeMemoryPool() NN_NOEXCEPT;

        nn::Result InitializeBuffer(int index, int width, int height) NN_NOEXCEPT;
        void FinalizeBuffer(int index) NN_NOEXCEPT;

        bool IsBufferInitialized(int index) const NN_NOEXCEPT;
        int GetBufferWidth(int index) const NN_NOEXCEPT;
        int GetBufferHeight(int index) const NN_NOEXCEPT;
        void* GetBufferMemoryNv12Y(int index) NN_NOEXCEPT;
        void* GetBufferMemoryNv12Uv(int index) NN_NOEXCEPT;
        size_t GetBufferMemorySizeNv12Y(int index) const NN_NOEXCEPT;
        size_t GetBufferMemorySizeNv12Uv(int index) const NN_NOEXCEPT;

        android::sp<android::GraphicBuffer>* GetGraphicBufferListRef() NN_NOEXCEPT;

        // メモリプール全体の CPU キャッシュを Flush する。
        void FlushDataCache() NN_NOEXCEPT;

    private:
        // MemoryPool
        char* m_Memory;
        size_t m_MemorySize;
        nn::capsrv::capture::CaptureModule* m_pModule;
        nn::capsrv::capture::MemoryPool     m_MemoryPool;
        int m_WidthMax;
        int m_HeightMax;
        ptrdiff_t m_Nv12YBufferOffsetList[BufferCountMax];
        ptrdiff_t m_Nv12UvBufferOffsetList[BufferCountMax];

        // BufferList
        int m_WidthList[BufferCountMax];
        int m_HeightList[BufferCountMax];
        nn::capsrv::capture::ImageBuffer    m_Nv12PlaneList    [BufferCountMax * 2];
        NvRmSurface                         m_SurfaceList      [BufferCountMax * 2];
        buffer_handle_t                     m_BufferHandleList [BufferCountMax];
        android::sp<android::GraphicBuffer> m_GraphicBufferList[BufferCountMax];
    };

}}}
