﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "grcsrvOffscreen_VideoRendererHandler.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/grc/grc_Result.h>
#include <nn/grc/grc_ResultPrivate.h>

#include "../grcsrvOffscreen_Macro.h"
#include "../grcsrvOffscreen_ResourceId.h"
#include "../grcsrvOffscreen_MultiWaitHandler.h"
#include "../grcsrvOffscreen_VicCopyCaptureImageBuffer.h"

namespace nn{ namespace grcsrv{ namespace offscreen{ namespace detail{

    VideoRendererHandler::VideoRendererHandler() NN_NOEXCEPT
        : m_State(State::NotInitialized)
        , m_pConsumerBufferAcquirableEvent(nullptr)
        , m_pAcquirableOutputBufferQueue(nullptr)
        , m_pQueuedOutputBufferQueue(nullptr)
        , m_FrameRate(0)
        , m_FrameCount(0)
    {
    }

    bool VideoRendererHandler::IsInitialized() const NN_NOEXCEPT
    {
        return m_State != State::NotInitialized;
    }

    nn::util::optional<nn::Result> VideoRendererHandler::GetExitResult() const NN_NOEXCEPT
    {
        return m_ExitResult;
    }

    nn::Result VideoRendererHandler::Initialize(
        const android::sp<android::IGraphicBufferConsumer>& pConsumer,
        nn::os::SystemEventType* pConsumerBufferAcquirableEvent,
        AcquirableOutputBufferQueue* pAcquirableOutputBufferQueue,
        QueuedOutputBufferQueue* pQueuedOutputBufferQueue
    ) NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("initializing video-renderer...\n");
        NN_SDK_REQUIRES(!IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(pConsumerBufferAcquirableEvent);

        NN_GRCSRV_PROCESS_START();

        NN_GRCSRV_PROCESS_SUCCESS(); // 以降失敗しない
        nn::os::InitializeEvent(&m_ExitEvent, false, nn::os::EventClearMode_ManualClear);

        m_pConsumer = pConsumer;
        for(auto& e : m_ConsumerBufferList){ e.clear(); }
        m_pConsumerBufferAcquirableEvent = pConsumerBufferAcquirableEvent;
        m_pAcquirableOutputBufferQueue = pAcquirableOutputBufferQueue;
        m_pQueuedOutputBufferQueue = pQueuedOutputBufferQueue;

        m_EncodingBufferList.Initialize();

        nn::os::InitializeMultiWaitHolder(m_InputAcquirableWaitHandler.GetHolder(), pConsumerBufferAcquirableEvent);
        m_pAcquirableOutputBufferQueue->InitializeDequeueableMultiWaitHolder(m_OutputDequeueableWaitHandler.GetHolder());

        m_ExitResult = nn::util::nullopt;
        m_FrameRate = 0;
        m_FrameCount = 0;
        m_State = State::Initializing;
        ChangeStateIdleImpl();
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("initialized video-renderer\n");
        NN_RESULT_SUCCESS;
    }

    void VideoRendererHandler::Finalize() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("finalizing video-renderer...\n");
        NN_SDK_REQUIRES(IsInitialized());

        ChangeStateFinalizingImpl();

        nn::os::FinalizeMultiWaitHolder(m_InputAcquirableWaitHandler.GetHolder());
        m_pAcquirableOutputBufferQueue->FinalizeDequeueableMultiWaitHolder(m_OutputDequeueableWaitHandler.GetHolder());

        m_EncodingBufferList.Finalize();

        nn::os::FinalizeEvent(&m_ExitEvent);

        m_ExitResult = nn::util::nullopt;
        m_pConsumer.clear();
        for(auto& e : m_ConsumerBufferList){ e.clear(); }
        m_pConsumerBufferAcquirableEvent = nullptr;
        m_pAcquirableOutputBufferQueue = nullptr;
        m_pQueuedOutputBufferQueue = nullptr;
        m_FrameRate = 0;
        m_FrameCount = 0;
        m_State = State::NotInitialized;
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("finalized video-renderer\n");
    }

    //------------------------------------------------

    nn::Result VideoRendererHandler::Start(int frameRate) NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("starting video-renderer...\n");
        NN_SDK_REQUIRES(IsInitialized());

        NN_RESULT_THROW_UNLESS(m_State == State::Idle, nn::grc::ResultInvalidState());

        // Start 前に Present されたバッファをすべて返却する。
        while(nn::os::TryWaitSystemEvent(m_pConsumerBufferAcquirableEvent))
        {
            android::sp<android::GraphicBuffer> buf;
            android::IGraphicBufferConsumer::BufferItem item;
            if(AcquireConsumerBufferImpl(&buf, &item).IsFailure())
            {
                break;
            }
            NN_ABORT_UNLESS_RESULT_SUCCESS(ReleaseConsumerBufferImpl(item, item.mFence));
        }

        m_ExitResult = nn::util::nullopt;
        m_FrameRate = frameRate;
        m_FrameCount = 0;
        ChangeStateEncodingImpl();
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("started video-renderer\n");
        NN_RESULT_SUCCESS;
    }

    void VideoRendererHandler::Abort() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("aborting video-renderer...\n");
        NN_SDK_REQUIRES(IsInitialized());

        // 強制的に全バッファを返却
        ForceReleaseAllEncodingBuffersImpl();

        ChangeStateIdleImpl();
        m_FrameRate  = 0;
        m_FrameCount = 0;
        m_ExitResult = nn::util::nullopt;
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("aborted video-renderer\n");
    }

    void VideoRendererHandler::Reset() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("resetting video-renderer...\n");
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_EQUAL(m_State, State::Exited);
        NN_SDK_ASSERT_EQUAL(m_EncodingBufferList.GetCount(), 0);

        ChangeStateIdleImpl();
        m_FrameRate  = 0;
        m_FrameCount = 0;
        m_ExitResult = nn::util::nullopt;
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("reset video-renderer\n");
    }

    void VideoRendererHandler::RequestFinish() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("requesting finish video-renderer...\n");
        NN_SDK_REQUIRES(IsInitialized());

        NN_GRCSRV_ASSERT_ABORT(m_State == State::Encoding);

        // Present 済のバッファをすべてエンコーダに送る
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("  queueing presented buffers\n");
        while(nn::os::TryWaitSystemEvent(m_pConsumerBufferAcquirableEvent))
        {
            if(AcquireInputBufferAndSendToEncoderImpl(false).IsFailure())
            {
                break;
            }
        }

        // エンコーダに Finish リクエストを送る
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("  queueing finish request\n");
        {
            QueuedOutputBufferQueue::ValueType e = {};
            e.isFinishRequested = true;
            m_pQueuedOutputBufferQueue->Enqueue(e);
        }

        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("requested finish video-renderer\n");

        if(m_EncodingBufferList.GetCount() == 0)
        {
            return ChangeStateExitedImpl(nn::ResultSuccess());
        }
        else
        {
            return ChangeStateFinishingImpl();
        }
    }

    void VideoRendererHandler::NotifyError() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("notifying error..\n");
        NN_SDK_REQUIRES(IsInitialized());
        if(m_State == State::Encoding || m_State == State::Finishing)
        {
            ForceReleaseAllEncodingBuffersImpl();
            ChangeStateExitedImpl(nn::grc::ResultInternalOffscreenErrorReported());
        }
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("notifyed error..\n");
    }


    void VideoRendererHandler::InitializeExitWaitHolder(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        nn::os::InitializeMultiWaitHolder(pHolder, &m_ExitEvent);
    }

    void VideoRendererHandler::FinalizeExitWaitHolder(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        nn::os::FinalizeMultiWaitHolder(pHolder);
    }

    //------------------------------------------------

    nn::Result VideoRendererHandler::AcquireConsumerBufferImpl(
        android::sp<android::GraphicBuffer>* pOutBuffer,
        android::IGraphicBufferConsumer::BufferItem* pOutItem
    ) NN_NOEXCEPT
    {
        android::IGraphicBufferConsumer::BufferItem item = {};

        android::status_t status = m_pConsumer->acquireBuffer(&item, 0);
        if(status == android::IGraphicBufferConsumer::NO_BUFFER_AVAILABLE)
        {
            // W/A
            // GraphicBufferConsumer の不具合なのかアプリが nvnWindow を破棄した際に
            // FrameAvailableEvent がシグナルしっぱなしになることがある。
            // この場合、 NO_BUFFER_AVAILABLE が返ってくる。
            //
            // GraphicBufferQueue の操作はすべて同一スレッドで行われるので
            // ここでイベントをクリアして無限に NO_BUFFER_AVAILABLE が返ってくる状態を抜ける。
            nn::os::ClearSystemEvent(m_pConsumerBufferAcquirableEvent);
            NN_RESULT_THROW(nn::grc::ResultInternalOffscreenVideoRendererNoBufferAvailable());
        }

        if(status != android::NO_ERROR)
        {
            NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("failed to acquire buffer(status=%d)\n", status);
            NN_ABORT();
        }

        NN_ABORT_UNLESS_RANGE(item.mBuf, 0, ClientImageBufferCountMax);

        if(!item.mAcquireCalled)
        {
            m_ConsumerBufferList[item.mBuf] = item.mGraphicBuffer;
        }

        *pOutBuffer = m_ConsumerBufferList[item.mBuf];
        *pOutItem = item;
        NN_RESULT_SUCCESS;
    }

    nn::Result VideoRendererHandler::ReleaseConsumerBufferImpl(
        const android::IGraphicBufferConsumer::BufferItem& item,
        const android::sp<android::Fence>& fence
    ) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_EQUAL(android::NO_ERROR,
            m_pConsumer->releaseBuffer(
                item.mBuf,
                item.mFrameNumber,
                EGL_NO_DISPLAY,
                EGL_NO_SYNC_KHR,
                fence
            )
        );
        NN_RESULT_SUCCESS;
    }

    nn::Result VideoRendererHandler::AcquireInputBufferAndReleaseImpl() NN_NOEXCEPT
    {
        // acquire
        android::sp<android::GraphicBuffer> buf;
        android::IGraphicBufferConsumer::BufferItem item;
        NN_RESULT_TRY(AcquireConsumerBufferImpl(&buf, &item))
            NN_RESULT_CATCH(nn::grc::ResultInternalOffscreenVideoRendererNoBufferAvailable)
            {
                // 無視
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH_ALL
            {
                NN_RESULT_THROW(nn::grc::ResultInternalOffscreenVideoRendererError());
            }
        NN_RESULT_END_TRY;

        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("  passthrough buffer\n");

        // release
        NN_RESULT_TRY(ReleaseConsumerBufferImpl(item, item.mFence))
            NN_RESULT_CATCH_ALL
            {
                NN_RESULT_THROW(nn::grc::ResultInternalOffscreenVideoRendererError());
            }
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    }

    nn::Result VideoRendererHandler::AcquireInputBufferAndSendToEncoderImpl(bool isNoBufferAvailableErrorIgnored) NN_NOEXCEPT
    {
        // acquire
        android::sp<android::GraphicBuffer> buf;
        android::IGraphicBufferConsumer::BufferItem item;
        NN_RESULT_TRY(AcquireConsumerBufferImpl(&buf, &item))
            NN_RESULT_CATCH(nn::grc::ResultInternalOffscreenVideoRendererNoBufferAvailable)
            {
                if(isNoBufferAvailableErrorIgnored)
                {
                    NN_RESULT_SUCCESS;
                }
                else
                {
                    NN_RESULT_RETHROW;
                }
            }
            NN_RESULT_CATCH_ALL
            {
                NN_RESULT_THROW(nn::grc::ResultInternalOffscreenVideoRendererError());
            }
        NN_RESULT_END_TRY;

        NN_SDK_ASSERT_NOT_NULL(buf);

        // send to capture
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("  sending to video-capture\n");
        {
            VideoRendererToVideoCaptureValue e = {};
            e.index = item.mBuf;
            e.item = item;
            e.buffer = buf;
            e.timestampUs = 1000000 * m_FrameCount / m_FrameRate;
            m_pQueuedOutputBufferQueue->Enqueue(e);
        }

        // remember encoding buffer
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("  registering to encoding buffer list\n");
        {
            EncodingBufferEntry e = {};
            e.item = item;
            e.frameNumber = m_FrameCount;
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_EncodingBufferList.Register(e));
        }

        m_FrameCount++;
        NN_RESULT_SUCCESS;
    }

    nn::Result VideoRendererHandler::DequeueOutputBufferAndReleaseToConsumerImpl() NN_NOEXCEPT
    {
        // dequeue
        AcquirableOutputBufferQueue::ValueType e;
        m_pAcquirableOutputBufferQueue->Dequeue(&e);

        // recall&remove encoding buffer
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("  recalling from encoding buffer list\n");
        EncodingBufferEntry bufEntry = {};

        int k = m_EncodingBufferList.RemoveIf([&](const EncodingBufferEntry& v)->bool{
            if(v.item.mBuf == e.index)
            {
                bufEntry = v;
                return true;
            }
            return false;
        });
        if(k != 1)
        {
            // Abort 後に返ってきたバッファは無視する
            NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("  buffer not found. ignore\n");
            NN_RESULT_SUCCESS;
        }

        // send to client
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("  releasing to consumer\n");
        NN_RESULT_TRY(ReleaseConsumerBufferImpl(bufEntry.item, android::Fence::NO_FENCE))
            NN_RESULT_CATCH_ALL
            {
                NN_RESULT_THROW(nn::grc::ResultInternalOffscreenVideoRendererError());
            }
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    }

    void VideoRendererHandler::ForceReleaseAllEncodingBuffersImpl() NN_NOEXCEPT
    {
        // エンコード中のバッファを強制的にクライアントに返却する
        while(m_EncodingBufferList.GetCount() > 0)
        {
            EncodingBufferEntry minValue = {};
            minValue.frameNumber = std::numeric_limits<int64_t>::max();

            m_EncodingBufferList.Foreach([&](const EncodingBufferEntry& e)->void
                {
                    if(e.frameNumber <= minValue.frameNumber)
                    {
                        minValue = e;
                    }
                }
            );

            NN_ABORT_UNLESS_RESULT_SUCCESS(ReleaseConsumerBufferImpl(minValue.item, minValue.item.mFence));
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_EncodingBufferList.Unregister(minValue));
        }
    }

    void VideoRendererHandler::OnInputAcquirableCallbackFunction(MultiWaitHandler* pHandler, void* userPtr) NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("input acquirable\n");
        // link again
        pHandler->Link(OnInputAcquirableCallbackFunction, userPtr);
        auto pSelf = reinterpret_cast<VideoRendererHandler*>(userPtr);

        nn::Result result = nn::ResultSuccess();

        switch(pSelf->m_State)
        {
        case State::Encoding:
            result = pSelf->AcquireInputBufferAndSendToEncoderImpl(true);
            break;
        default:
            result = pSelf->AcquireInputBufferAndReleaseImpl();
        }

        if(result.IsFailure())
        {
            pSelf->ForceReleaseAllEncodingBuffersImpl();
            pSelf->ChangeStateExitedImpl(result);
        }
    }

    void VideoRendererHandler::OnOutputDequeueableCallbackFunction(MultiWaitHandler* pHandler, void* userPtr) NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("output acquirable\n");
        // link again
        pHandler->Link(OnOutputDequeueableCallbackFunction, userPtr);
        auto pSelf = reinterpret_cast<VideoRendererHandler*>(userPtr);

        auto result = pSelf->DequeueOutputBufferAndReleaseToConsumerImpl();
        if(result.IsFailure())
        {
            pSelf->ForceReleaseAllEncodingBuffersImpl();
            pSelf->ChangeStateExitedImpl(result);
        }

        // check finished
        if(pSelf->m_State == State::Finishing)
        {
            if(pSelf->m_EncodingBufferList.GetCount() == 0)
            {
                pSelf->ChangeStateExitedImpl(nn::ResultSuccess());
            }
        }
    }

    //--------------------------------------------------------------------

    void VideoRendererHandler::ChangeStateIdleImpl() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("changing state to idle\n");

        ExitCurrentStateImpl();
        m_InputAcquirableWaitHandler.Link(OnInputAcquirableCallbackFunction, this);
        m_OutputDequeueableWaitHandler.Link(OnOutputDequeueableCallbackFunction, this);
        m_State = State::Idle;
        nn::os::ClearEvent(&m_ExitEvent);
    }

    void VideoRendererHandler::ChangeStateEncodingImpl() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("changing state to encoding\n");

        ExitCurrentStateImpl();
        m_InputAcquirableWaitHandler.Link(OnInputAcquirableCallbackFunction, this);
        m_OutputDequeueableWaitHandler.Link(OnOutputDequeueableCallbackFunction, this);
        m_State = State::Encoding;
        nn::os::ClearEvent(&m_ExitEvent);
    }

    void VideoRendererHandler::ChangeStateFinishingImpl() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("changing state to finishing\n");

        ExitCurrentStateImpl();
        // Input は止める
        m_OutputDequeueableWaitHandler.Link(OnOutputDequeueableCallbackFunction, this);
        m_State = State::Finishing;
        nn::os::ClearEvent(&m_ExitEvent);
    }

    void VideoRendererHandler::ChangeStateExitedImpl(nn::Result exitResult) NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("changing state to exited(%d-%d)\n", exitResult.GetModule(), exitResult.GetDescription());

        // エンコード完了するか、 ForceRelease を呼び出して
        // 全バッファがアプリに返却されていなければならない。
        NN_SDK_ASSERT_EQUAL(m_EncodingBufferList.GetCount(), 0);

        ExitCurrentStateImpl();
        m_InputAcquirableWaitHandler.Link(OnInputAcquirableCallbackFunction, this);
        m_OutputDequeueableWaitHandler.Link(OnOutputDequeueableCallbackFunction, this);
        m_State = State::Exited;
        m_ExitResult = exitResult;
        nn::os::SignalEvent(&m_ExitEvent);
    }

    void VideoRendererHandler::ChangeStateFinalizingImpl() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEORENDERER("changing state to finalizing\n");

        ExitCurrentStateImpl();
        m_State = State::Finalizing;
        nn::os::ClearEvent(&m_ExitEvent);
    }

    void VideoRendererHandler::ExitCurrentStateImpl() NN_NOEXCEPT
    {
        m_InputAcquirableWaitHandler.Unlink();
        m_OutputDequeueableWaitHandler.Unlink();
    }

    //-----------------------------------------------------------------------------------------

}}}}

