﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "grcsrvOffscreen_MuxerHandler.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/grc/grc_Result.h>
#include <nn/grc/grc_ResultPrivate.h>

namespace nn{ namespace grcsrv{ namespace offscreen{ namespace detail{

    MuxerMpeg4Writer::MuxerMpeg4Writer() NN_NOEXCEPT
        : m_FileHandle()
        , m_CurrentPosition(0)
        , m_LastError(nn::ResultSuccess())
    {
    }

    void MuxerMpeg4Writer::Initialize(nn::fs::FileHandle hFile) NN_NOEXCEPT
    {
        m_FileHandle = hFile;
        m_CurrentPosition = 0;
        m_LastError = nn::ResultSuccess();
    }

    void MuxerMpeg4Writer::Finalize() NN_NOEXCEPT
    {
        m_FileHandle = {};
        m_CurrentPosition = 0;
        m_LastError = nn::ResultSuccess();
    }

    nn::Result MuxerMpeg4Writer::GetLastError() const NN_NOEXCEPT
    {
        return m_LastError;
    }

    void MuxerMpeg4Writer::SetPosition(int64_t offset) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0);
        m_CurrentPosition = offset;
    }

    void MuxerMpeg4Writer::Write(const void* data, uint32_t size) NN_NOEXCEPT
    {
        if(size == 0)
        {
            return;
        }
        // 既にエラー状態になっていたら無視
        if(m_LastError.IsFailure())
        {
            return;
        }
        NN_SDK_REQUIRES_NOT_NULL(data);
        m_LastError = nn::fs::WriteFile(m_FileHandle, m_CurrentPosition, data, size, nn::fs::WriteOption::MakeValue(0));
        m_CurrentPosition += size;
    }

    //----------------------------------------------------------------------------------

    MuxerHandler::MuxerHandler() NN_NOEXCEPT
        : m_State()
        , m_VideoTrackId(-1)
        , m_AudioTrackId(-1)
    {
    }

    nn::Result MuxerHandler::Initialize() NN_NOEXCEPT
    {
        NN_GRCSRV_ASSERT_THROW(m_State == State::NotInitialized, nn::grc::ResultInvalidCall());
        nn::os::InitializeMutex(&m_Mutex, false, 0);
        nn::os::InitializeEvent(&m_WritingEvent, true, nn::os::EventClearMode_ManualClear);
        m_State = State::Initialized;
        NN_RESULT_SUCCESS;
    }

    void MuxerHandler::Finalize() NN_NOEXCEPT
    {
        NN_GRCSRV_ASSERT_ABORT(m_State == State::Initialized);
        nn::os::FinalizeEvent(&m_WritingEvent);
        nn::os::FinalizeMutex(&m_Mutex);
        m_State = State::NotInitialized;
    }

    nn::Result MuxerHandler::Start(nn::fs::FileHandle hFile) NN_NOEXCEPT
    {
        nn::os::LockMutex(&m_Mutex);
        NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

        NN_GRCSRV_ASSERT_THROW(m_State == State::Initialized, nn::grc::ResultInvalidCall());

        NN_GRCSRV_PROCESS_START();

        m_Mpeg4Writer.Initialize(hFile);
        NN_GRCSRV_PROCESS_ROLLBACK(m_Mpeg4Writer.Finalize());

        m_pMuxer = new android::MediaMuxer(&m_Mpeg4Writer, android::MediaMuxer::OUTPUT_FORMAT_MPEG_4);
        NN_RESULT_THROW_UNLESS(m_pMuxer != nullptr, nn::grc::ResultOutOfMemory());
        NN_GRCSRV_PROCESS_ROLLBACK(m_pMuxer.clear());

        NN_GRCSRV_PROCESS_SUCCESS();
        m_VideoTrackId = -1;
        m_AudioTrackId = -1;
        m_State = State::Starting;
        nn::os::ClearEvent(&m_WritingEvent);
        NN_RESULT_SUCCESS;
    }

    void MuxerHandler::Stop() NN_NOEXCEPT
    {
        nn::os::LockMutex(&m_Mutex);
        NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

        NN_GRCSRV_ASSERT_ABORT(m_State == State::Starting || m_State == State::Writing);

        m_pMuxer->stop();
        m_pMuxer.clear();
        m_Mpeg4Writer.Finalize();

        m_VideoTrackId = -1;
        m_AudioTrackId = -1;
        m_State = State::Initialized;
        nn::os::SignalEvent(&m_WritingEvent);
    }

    nn::Result MuxerHandler::SetVideoFormat(const android::sp<android::AMessage>& pFormat) NN_NOEXCEPT
    {
        nn::os::LockMutex(&m_Mutex);
        NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

        NN_RESULT_THROW_UNLESS(m_State == State::Starting, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(m_VideoTrackId < 0, nn::grc::ResultInvalidCall());

        auto trackId = m_pMuxer->addTrack(pFormat);
        NN_RESULT_THROW_UNLESS(trackId >= 0, nn::grc::ResultOutOfResource());
        NN_RESULT_DO(m_Mpeg4Writer.GetLastError());

        m_VideoTrackId = trackId;

        NN_RESULT_DO(StartWritingIfAllTrackReadyImpl());
        NN_RESULT_SUCCESS;
    }

    void MuxerHandler::SetVideoOrientation(uint32_t orientation) NN_NOEXCEPT
    {
        nn::os::LockMutex(&m_Mutex);
        NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

        int videoOrientation = (360 - orientation * 90) % 360;
        m_pMuxer->setOrientationHint(videoOrientation);
    }

    nn::Result MuxerHandler::SetAudioFormat(const android::sp<android::AMessage>& pFormat) NN_NOEXCEPT
    {
        nn::os::LockMutex(&m_Mutex);
        NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

        NN_RESULT_THROW_UNLESS(m_State == State::Starting, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(m_AudioTrackId < 0, nn::grc::ResultInvalidCall());

        auto trackId = m_pMuxer->addTrack(pFormat);
        NN_RESULT_THROW_UNLESS(trackId >= 0, nn::grc::ResultOutOfResource());
        NN_RESULT_DO(m_Mpeg4Writer.GetLastError());

        m_AudioTrackId = trackId;

        NN_RESULT_DO(StartWritingIfAllTrackReadyImpl());
        NN_RESULT_SUCCESS;
    }

    nn::Result MuxerHandler::StartWritingIfAllTrackReadyImpl() NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_VideoTrackId >= 0, nn::ResultSuccess());
        NN_RESULT_THROW_UNLESS(m_AudioTrackId >= 0, nn::ResultSuccess());

        NN_RESULT_THROW_UNLESS(m_pMuxer->start() == android::NO_ERROR, nn::grc::ResultInvalidState());
        NN_RESULT_DO(m_Mpeg4Writer.GetLastError());

        m_State = State::Writing;
        nn::os::SignalEvent(&m_WritingEvent);
        NN_RESULT_SUCCESS;
    }

    nn::Result MuxerHandler::WaitForWriting() NN_NOEXCEPT
    {
        // 全トラックが設定されて Muxer が start するか、 Stop() が呼び出されるとシグナルする。
        nn::os::WaitEvent(&m_WritingEvent);

        {
            nn::os::LockMutex(&m_Mutex);
            NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

            // Stop() の呼び出しで起床した場合 AbortRequested を返す。
            NN_RESULT_THROW_UNLESS(m_State == State::Writing, nn::grc::ResultInternalOffscreenAbortRequested());
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result MuxerHandler::StoreVideoFrame(
        const android::sp<android::ABuffer>& buffer,
        int64_t timestamp,
        uint32_t flags
    ) NN_NOEXCEPT
    {
        nn::os::LockMutex(&m_Mutex);
        NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

        NN_RESULT_THROW_UNLESS(m_State == State::Writing, nn::grc::ResultInvalidState());

        uint32_t muxFlags = flags & (android::MediaCodec::BUFFER_FLAG_SYNCFRAME | android::MediaCodec::BUFFER_FLAG_EOS);
        NN_RESULT_THROW_UNLESS(m_pMuxer->writeSampleData(buffer, m_VideoTrackId, timestamp, muxFlags) == android::NO_ERROR, nn::grc::ResultInvalidState());

        NN_RESULT_DO(m_Mpeg4Writer.GetLastError());

        NN_RESULT_SUCCESS;
    }

    nn::Result MuxerHandler::StoreAudioFrame(
        const android::sp<android::ABuffer>& buffer,
        int64_t timestamp,
        uint32_t flags
    ) NN_NOEXCEPT
    {
        nn::os::LockMutex(&m_Mutex);
        NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

        NN_RESULT_THROW_UNLESS(m_State == State::Writing, nn::grc::ResultInvalidState());

        uint32_t muxFlags = flags & (android::MediaCodec::BUFFER_FLAG_SYNCFRAME | android::MediaCodec::BUFFER_FLAG_EOS);
        NN_RESULT_THROW_UNLESS(m_pMuxer->writeSampleData(buffer, m_AudioTrackId, timestamp, muxFlags) == android::NO_ERROR, nn::grc::ResultInvalidState());

        NN_RESULT_DO(m_Mpeg4Writer.GetLastError());

        NN_RESULT_SUCCESS;
    }

}}}}
