﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/util/util_Optional.h>
#include <nn/sf/sf_NativeHandleFwd.h>
#include <nn/grc/grc_CommonTypes.h>
#include "../grcsrvOffscreen_Config.h"
#include "../grcsrvOffscreen_MultiWaitHandler.h"
#include "grcsrvOffscreen_QueueType.h"

namespace nn{ namespace grcsrv{ namespace offscreen{ namespace detail{

    class AudioRendererHandler
    {
    public:
        enum class State
        {
            NotInitialized,
            Initializing, // Initialize 中の状態。即座に Idle に変更される。
            Finalizing,   // Finalize 中の状態。即座に NotInitialized に変更される。
            Idle,
            Encoding,
            Finishing,
            Exited,
        };

        typedef AudioEncoderToAudioRendererQueue AcquirableOutputBufferQueue;
        typedef AudioRendererToAudioEncoderQueue QueuedOutputBufferQueue;

    public:
        AudioRendererHandler() NN_NOEXCEPT;

        bool IsInitialized() const NN_NOEXCEPT;

        nn::util::optional<nn::Result> GetExitResult() const NN_NOEXCEPT;

        nn::Result Initialize(
            AcquirableOutputBufferQueue* pAcquirableOutputBufferQueue,
            QueuedOutputBufferQueue* pQueuedOutputBufferQueue
        ) NN_NOEXCEPT;

        void Finalize() NN_NOEXCEPT;

        nn::Result Start(const nn::grc::OffscreenRecordingParameter& param) NN_NOEXCEPT;
        void Abort() NN_NOEXCEPT;
        void Reset() NN_NOEXCEPT;

        void RequestFinish() NN_NOEXCEPT;
        void NotifyError() NN_NOEXCEPT;

        void InitializeExitWaitHolder(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT;
        void FinalizeExitWaitHolder(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT;

        void GetEncodeReadyEvent(nn::sf::NativeHandle& outHandle) NN_NOEXCEPT;
        nn::Result EncodeAudio(size_t* pOutEncodedSize, const void* buffer, size_t size) NN_NOEXCEPT;

    private:
        void SubmitFinishRequestImpl() NN_NOEXCEPT;

    private:
        void ChangeStateIdleImpl() NN_NOEXCEPT;
        void ChangeStateEncodingImpl() NN_NOEXCEPT;
        void ChangeStateFinishingImpl() NN_NOEXCEPT;
        void ChangeStateExitedImpl(nn::Result exitResult) NN_NOEXCEPT;
        void ChangeStateFinalizingImpl() NN_NOEXCEPT;

        // ChangeState 内部で使用。 MultiWait の解除を行う。
        void ExitCurrentStateImpl() NN_NOEXCEPT;

        void CheckEncodeReadyImpl() NN_NOEXCEPT;

    private:
        State m_State;

        nn::util::optional<nn::Result> m_ExitResult;
        nn::os::EventType m_ExitEvent;

        AcquirableOutputBufferQueue* m_pAcquirableOutputBufferQueue;
        QueuedOutputBufferQueue* m_pQueuedOutputBufferQueue;

        // (State != Encoding) || (State == Encoding && !AcquirableOutputBufferQueue.IsEmpty) のときシグナルする。
        nn::os::SystemEventType m_EncodeReadyEvent;

        // EncodeReadyEvent がシグナルしていない間 Link される。
        // AcquirableOutputBufferQueue に何か入ってくるのを待つ
        MultiWaitHandler m_OutputDequeueableWaitHandler;

        int64_t m_SampleRate;
        int64_t m_ChannelCount;
        int64_t m_ChannelSize;

        int64_t m_TotalByteCount;
    };

}}}}
