﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "grcsrv_OffscreenRecorderImpl.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/grc/grc_Result.h>

#include <nn/grcsrv/grcsrv_Offscreen.h>
#include "grcsrv_MovieMakerImpl.h"

namespace nn{ namespace grcsrv{
    namespace {
        const uint64_t InvalidHandle = 0;

        struct DestroyingLayerCallback
        {
            void (*m_Function)(uint64_t, void*);
            void* m_UserPointer;
        };

        DestroyingLayerCallback g_DestroyingLayerCallbackList[MovieMakerClientCountMax] = {};
    }

    //------------------------------------------

    void RegisterOffscreenLayerDestroyingCallback(int* pOutSlot, void (*func)(uint64_t, void*), void* userPtr) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutSlot);
        NN_SDK_ASSERT_NOT_NULL(func);

        int slot = -1;
        for(int i = 0; i < MovieMakerClientCountMax; i++)
        {
            if(g_DestroyingLayerCallbackList[i].m_Function == nullptr)
            {
                slot = i;
                break;
            }
        }
        NN_ABORT_UNLESS_RANGE(slot, 0, MovieMakerClientCountMax);

        auto p = &g_DestroyingLayerCallbackList[slot];
        p->m_Function = func;
        p->m_UserPointer = userPtr;
        *pOutSlot = slot;
    }

    void UnregisterOffscreenLayerDestroyingCallback(int slot) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RANGE(slot, 0, MovieMakerClientCountMax);
        NN_ABORT_UNLESS_NOT_NULL(g_DestroyingLayerCallbackList[slot].m_Function);

        g_DestroyingLayerCallbackList[slot] = {};
    }

    static void InvokeOffscreenLayerDestroyingCallback(uint64_t layerHandle) NN_NOEXCEPT
    {
        for(int i = 0; i < MovieMakerClientCountMax; i++)
        {
            auto p = &g_DestroyingLayerCallbackList[i];
            if(p->m_Function != nullptr)
            {
                p->m_Function(layerHandle, p->m_UserPointer);
            }
        }
    }

    //------------------------------------------

    OffscreenRecorderImpl::OffscreenRecorderImpl() NN_NOEXCEPT
        : m_CreatedLayerHandle(InvalidHandle)
        , m_IsUnbindingTransferMemoryRequired(false)
    {
    }

    Result OffscreenRecorderImpl::Initialize(sf::NativeHandle&& workMemoryHandle, uint64_t workMemorySize) NN_NOEXCEPT
    {
        auto success = false;
        void* workMemory;
        NN_RESULT_THROW_UNLESS(!offscreen::IsInitialized(), nn::grc::ResultGrcBusy());

        NN_RESULT_DO(offscreen::BindTransferMemory(&workMemory, std::move(workMemoryHandle), workMemorySize));
        NN_UTIL_SCOPE_EXIT
        {
            if (!success)
            {
                offscreen::UnbindTransferMemory();
            }
        };

        NN_RESULT_DO(offscreen::Initialize(workMemory, workMemorySize));
        NN_UTIL_SCOPE_EXIT
        {
            if (!success)
            {
                offscreen::Finalize();
            }
        };

        success = true;
        this->m_IsUnbindingTransferMemoryRequired = true;
        NN_RESULT_SUCCESS;
    }

    OffscreenRecorderImpl::~OffscreenRecorderImpl() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(nn::grcsrv::offscreen::IsInitialized());

        // 未破棄のレイヤがあれば破棄
        if(m_CreatedLayerHandle != InvalidHandle)
        {
            DestroyOffsreenLayerImpl(m_CreatedLayerHandle);
        }

        nn::grcsrv::offscreen::Finalize();

        if(m_IsUnbindingTransferMemoryRequired)
        {
            nn::grcsrv::offscreen::UnbindTransferMemory();
        }
    }

    nn::Result OffscreenRecorderImpl::CreateOffscreenLayer(nn::sf::Out<std::uint64_t> outHandle, nn::applet::AppletResourceUserId rendererAruid) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(nn::grcsrv::offscreen::IsInitialized());

        // 既にレイヤを作っていたら拒否
        NN_RESULT_THROW_UNLESS(m_CreatedLayerHandle == InvalidHandle, nn::grc::ResultOutOfResource());

        uint64_t handle = {};
        NN_RESULT_DO(nn::grcsrv::offscreen::CreateLayer(&handle, rendererAruid));

        m_CreatedLayerHandle = handle;
        outHandle.Set(handle);
        NN_RESULT_SUCCESS;
    }

    nn::Result OffscreenRecorderImpl::DestroyOffscreenLayer(std::uint64_t handle) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(nn::grcsrv::offscreen::IsInitialized());

        // 作成済のレイヤでなければ拒否
        NN_RESULT_THROW_UNLESS(handle != InvalidHandle, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(handle == m_CreatedLayerHandle, nn::grc::ResultInvalidCall());

        DestroyOffsreenLayerImpl(handle);
        NN_RESULT_SUCCESS;
    }

    void OffscreenRecorderImpl::DestroyOffsreenLayerImpl(uint64_t handle) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_EQUAL(handle, m_CreatedLayerHandle);
        NN_SDK_REQUIRES_NOT_EQUAL(handle, InvalidHandle);

        InvokeOffscreenLayerDestroyingCallback(handle);
        nn::grcsrv::offscreen::DestroyLayer(handle);

        m_CreatedLayerHandle = InvalidHandle;
    }


}}
