﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "grcsrv_MovieMakerImpl.h"
#include <nn/grcsrv/grcsrv_Service.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_LmemUtility.h>

#include <nn/grc/grc_Result.h>
#include <nn/grc/grc_ResultPrivate.h>

#define NN_GRCSRV_CHECK_ALIVE() \
    NN_RESULT_THROW_UNLESS(m_pClient->IsAlive(), nn::grc::ResultInvalidCall());

#define NN_GRCSRV_RETURN_UNLESS_ALIVE() if(!m_pClient->IsAlive()){ return; }

namespace nn{ namespace grcsrv{

    namespace {
        MemoryResource* GetGrcObjectMemoryResource() NN_NOEXCEPT
        {
            static Bit64 g_HeapMemory[256];
            NN_FUNCTION_LOCAL_STATIC(lmem::HeapHandle, g_HeapHandle, = lmem::CreateExpHeap(g_HeapMemory, sizeof(g_HeapMemory), lmem::CreationOption_ThreadSafe));
            NN_FUNCTION_LOCAL_STATIC(sf::ExpHeapMemoryResource, g_MemoryResource, {g_HeapHandle});
            return &g_MemoryResource;
        }

        nn::sf::UnmanagedServiceObject<IMovieMakerService, MovieMakerServiceImpl> g_MovieMakerServiceImpl;
    }

    MovieMakerImpl::MovieMakerImpl(MovieMakerClient* pClient) NN_NOEXCEPT
        : m_pClient(pClient)
    {
        NN_SDK_ASSERT(m_pClient->IsAlive());
    }

    MovieMakerImpl::~MovieMakerImpl() NN_NOEXCEPT
    {
        m_pClient->Invalidate();
        g_MovieMakerServiceImpl.GetImpl().Unlock(m_pClient, MovieMakerServiceImpl::LockFlag_MovieMaker);
        m_pClient = nullptr;
    }

    nn::Result MovieMakerImpl::CreateVideoProxy(nn::sf::Out<nn::sf::SharedPointer<nns::hosbinder::IHOSBinderDriver>> outProxy) NN_NOEXCEPT
    {
        NN_GRCSRV_CHECK_ALIVE();

        NN_RESULT_THROW_UNLESS(!g_MovieMakerServiceImpl.GetImpl().IsAnyLocked(MovieMakerServiceImpl::LockFlag_VideoProxy), nn::grc::ResultOutOfResource());

        sf::ScopedCurrentMemoryResourceSetter sc(GetGrcObjectMemoryResource());
        auto p = sf::CreateSharedObjectEmplaced<nns::hosbinder::IHOSBinderDriver, MovieMakerVideoProxyImpl>(m_pClient);
        NN_RESULT_THROW_UNLESS(p, grc::ResultOutOfMemory());

        // 作成側がカウントアップ
        g_MovieMakerServiceImpl.GetImpl().Lock(m_pClient, MovieMakerServiceImpl::LockFlag_VideoProxy);

        *outProxy = std::move(p);
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieMakerImpl::OpenOffscreenLayer(nn::sf::Out<std::int32_t> outProducerHandle, std::uint64_t layerHandle) NN_NOEXCEPT
    {
        NN_GRCSRV_CHECK_ALIVE();
        int32_t producerHandle = 0;
        NN_RESULT_DO(m_pClient->OpenOffscreenLayer(&producerHandle, layerHandle));
        outProducerHandle.Set(producerHandle);
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieMakerImpl::CloseOffscreenLayer(std::uint64_t layerHandle) NN_NOEXCEPT
    {
        NN_GRCSRV_CHECK_ALIVE();
        return m_pClient->CloseOffscreenLayer(layerHandle);
    }

    nn::Result MovieMakerImpl::StartOffscreenRecording(std::uint64_t layerHandle) NN_NOEXCEPT
    {
        NN_GRCSRV_CHECK_ALIVE();
        auto param = nn::grc::OffscreenRecordingParameter::GetDefaultValue();
        return m_pClient->StartOffscreenRecording(layerHandle, param);
    }

    nn::Result MovieMakerImpl::StartOffscreenRecordingEx(std::uint64_t layerHandle, const nn::grcsrv::OffscreenRecordingParameter& paramData) NN_NOEXCEPT
    {
        NN_GRCSRV_CHECK_ALIVE();
        nn::grc::OffscreenRecordingParameter param = {};
        NN_STATIC_ASSERT(sizeof(param) == sizeof(paramData));
        std::memcpy(&param, &paramData, sizeof(param));
        return m_pClient->StartOffscreenRecording(layerHandle, param);
    }

    nn::Result MovieMakerImpl::AbortOffscreenRecording(std::uint64_t layerHandle) NN_NOEXCEPT
    {
        NN_GRCSRV_CHECK_ALIVE();
        return m_pClient->AbortOffscreenRecording(layerHandle);
    }

    nn::Result MovieMakerImpl::RequestOffscreenRecordingFinishReady(std::uint64_t layerHandle) NN_NOEXCEPT
    {
        NN_GRCSRV_CHECK_ALIVE();
        return m_pClient->RequestOffscreenRecordingFinishReady(layerHandle);
    }

    nn::Result MovieMakerImpl::CompleteOffscreenRecordingFinish(std::uint64_t layerHandle, const sf::InBuffer& userData) NN_NOEXCEPT
    {
        NN_GRCSRV_CHECK_ALIVE();

        auto size = userData.GetSize();
        capsrv::ApplicationData applicationData = {};
        NN_RESULT_THROW_UNLESS(size <= sizeof(applicationData.value), grc::ResultInvalidCall());
        std::memcpy(applicationData.value, userData.GetPointerUnsafe(), size);
        applicationData.size = size;
        sf::InBuffer thumbnailImage(nullptr, 0);
        return m_pClient->CompleteOffscreenRecordingFinish(layerHandle, applicationData, thumbnailImage);
    }

    nn::Result MovieMakerImpl::CompleteOffscreenRecordingFinishEx0(std::uint64_t layerHandle, const sf::InBuffer& userData, const sf::InBuffer& thumbnailImage, int thumbnailImageWidth, int thumbnailImageHeight) NN_NOEXCEPT
    {
        NN_GRCSRV_CHECK_ALIVE();
        NN_RESULT_THROW_UNLESS(thumbnailImageWidth == 1280, grc::ResultAlbumInvalidParameter());
        NN_RESULT_THROW_UNLESS(thumbnailImageHeight == 720, grc::ResultAlbumInvalidParameter());

        auto size = userData.GetSize();
        capsrv::ApplicationData applicationData = {};
        NN_RESULT_THROW_UNLESS(size <= sizeof(applicationData.value), grc::ResultInvalidCall());
        std::memcpy(applicationData.value, userData.GetPointerUnsafe(), size);
        applicationData.size = size;
        return m_pClient->CompleteOffscreenRecordingFinish(layerHandle, applicationData, thumbnailImage);
    }

    nn::Result MovieMakerImpl::GetOffscreenLayerError(std::uint64_t layerHandle) NN_NOEXCEPT
    {
        NN_GRCSRV_CHECK_ALIVE();
        return m_pClient->GetOffscreenLayerError(layerHandle);
    }

    nn::Result MovieMakerImpl::EncodeOffscreenLayerAudioSample(nn::sf::Out<std::uint64_t> outEncodedSize, std::uint64_t layerHandle, const nn::sf::InBuffer& buffer) NN_NOEXCEPT
    {
        NN_GRCSRV_CHECK_ALIVE();
        return m_pClient->EncodeOffscreenLayerAudioSample(outEncodedSize, layerHandle, buffer);
    }

    nn::Result MovieMakerImpl::GetOffscreenLayerRecordingFinishReadyEvent(nn::sf::Out<nn::sf::NativeHandle>& outHandle, std::uint64_t layerHandle) NN_NOEXCEPT
    {
        NN_GRCSRV_CHECK_ALIVE();
        return m_pClient->GetOffscreenLayerRecordingFinishReadyEvent(outHandle, layerHandle);
    }

    nn::Result MovieMakerImpl::GetOffscreenLayerAudioEncodeReadyEvent(nn::sf::Out<nn::sf::NativeHandle>& outHandle, std::uint64_t layerHandle) NN_NOEXCEPT
    {
        NN_GRCSRV_CHECK_ALIVE();
        return m_pClient->GetOffscreenLayerAudioEncodeReadyEvent(outHandle, layerHandle);
    }

    //---------------------------------------------------

    MovieMakerVideoProxyImpl::MovieMakerVideoProxyImpl(MovieMakerClient* pClient) NN_NOEXCEPT
        : m_pClient(pClient)
    {
    }

    MovieMakerVideoProxyImpl::~MovieMakerVideoProxyImpl() NN_NOEXCEPT
    {
        g_MovieMakerServiceImpl.GetImpl().Unlock(m_pClient, MovieMakerServiceImpl::LockFlag_VideoProxy);
        m_pClient = nullptr;
    }

    void MovieMakerVideoProxyImpl::TransactParcel(std::int32_t handle, std::uint32_t code, const nn::sf::InBuffer& requestBuffer, const nn::sf::OutBuffer& replyBuffer, std::uint32_t flags) NN_NOEXCEPT
    {
        NN_GRCSRV_RETURN_UNLESS_ALIVE();
        return m_pClient->TransactParcel(handle, code, requestBuffer, replyBuffer, flags);
    }

    void MovieMakerVideoProxyImpl::AdjustRefcount(std::int32_t handle, std::int32_t diff, std::int32_t isStrong) NN_NOEXCEPT
    {
        NN_GRCSRV_RETURN_UNLESS_ALIVE();
        return m_pClient->AdjustRefcount(handle, diff, isStrong);
    }

    void MovieMakerVideoProxyImpl::GetNativeHandle(std::int32_t handle, std::uint32_t code, nn::sf::Out<nn::sf::NativeHandle> result) NN_NOEXCEPT
    {
        NN_GRCSRV_RETURN_UNLESS_ALIVE();
        return m_pClient->GetNativeHandle(handle, code, result);
    }

    void MovieMakerVideoProxyImpl::TransactParcelAuto(std::int32_t handle, std::uint32_t code, const nn::sf::InBuffer& requestBuffer, const nn::sf::OutBuffer& replyBuffer, std::uint32_t flags) NN_NOEXCEPT
    {
        NN_GRCSRV_RETURN_UNLESS_ALIVE();
        return m_pClient->TransactParcel(handle, code, requestBuffer, replyBuffer, flags);
    }

    //---------------------------------------------------

    MovieMakerServiceImpl::MovieMakerServiceImpl() NN_NOEXCEPT
        : m_Client()
        , m_Lock(0)
    {
    }

    nn::Result MovieMakerServiceImpl::CreateMovieMaker(nn::sf::Out<nn::sf::SharedPointer<nn::grcsrv::IMovieMaker>> outProxy, nn::applet::AppletResourceUserId aruid, ncm::ApplicationId applicationId) NN_NOEXCEPT
    {
        bool isSuccess = false;
        MovieMakerClient* pClient = nullptr;
        NN_RESULT_DO(AcquireAndLock(&pClient, aruid, applicationId, LockFlag_MovieMaker));
        NN_UTIL_SCOPE_EXIT{
            if(!isSuccess)
            {
                pClient->Invalidate();
                Unlock(pClient, LockFlag_MovieMaker);
            }
        };

        sf::ScopedCurrentMemoryResourceSetter sc(GetGrcObjectMemoryResource());
        auto p = sf::CreateSharedObjectEmplaced<nn::grcsrv::IMovieMaker, MovieMakerImpl>(pClient);
        NN_RESULT_THROW_UNLESS(p, grc::ResultOutOfMemory());

        isSuccess = true;
        *outProxy = std::move(p);
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieMakerServiceImpl::AcquireAndLock(MovieMakerClient** pOutClient, nn::applet::AppletResourceUserId aruid, ncm::ApplicationId applicationId, uint64_t lockMask) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_Client == nn::util::nullopt, nn::grc::ResultOutOfResource());
        NN_SDK_ASSERT_EQUAL(m_Lock, 0u);

        m_Client.emplace(aruid, applicationId);
        m_Lock = lockMask;

        *pOutClient = &*m_Client;
        NN_RESULT_SUCCESS;
    }

    bool MovieMakerServiceImpl::IsAnyLocked(uint64_t lockMask) const NN_NOEXCEPT
    {
        return (m_Lock & lockMask) != 0;
    }

    void MovieMakerServiceImpl::Lock(MovieMakerClient* pClient, uint64_t lockMask) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(m_Client != nn::util::nullopt);
        NN_ABORT_UNLESS_EQUAL(&*m_Client, pClient);
        NN_ABORT_UNLESS((m_Lock & lockMask) == 0);

        m_Lock |= lockMask;
    }

    void MovieMakerServiceImpl::Unlock(MovieMakerClient* pClient, uint64_t lockMask) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(m_Client != nn::util::nullopt);
        NN_ABORT_UNLESS_EQUAL(&*m_Client, pClient);
        NN_ABORT_UNLESS((m_Lock & lockMask) == lockMask);

        m_Lock &= ~lockMask;
        if(m_Lock == 0)
        {
            m_Client = nn::util::nullopt;
        }
    }

    nn::sf::SharedPointer<IMovieMakerService> GetMovieMakerService() NN_NOEXCEPT
    {
        return g_MovieMakerServiceImpl.GetShared();
    }

}}

