﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "grcsrv_ContinuousRecorderImpl.h"

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Exchange.h>
#include <nn/util/util_IntUtil.h>
#include <memory>
#include <atomic>
#include <mutex>
#include <nn/grc/grc_CommonTypes.h>
#include <nn/grc/grc_Result.h>
#include <nn/os/os_TransferMemoryApi.h>
#include <nn/grcsrv/grcsrv_MovieFileWriter.h>
#include <nn/util/util_Optional.h>
#include <nn/grcsrv/grcsrv_SetAllocatorForMalloc.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/grcsrv/grcsrv_RecordingNotificator.h>
#include <nn/vi/vi_LayerStack.h>
#include <nerd/scoop/video.h>
#include <nn/capsrv/capsrv_AlbumControl.h>
#include <nn/capsrv/capsrv_ScreenShotControl.h>
#include <nn/fs/fs_PriorityPrivate.h>

namespace nn { namespace grcsrv {

namespace {

std::atomic_bool g_IsUsed{false};
util::optional<mem::StandardAllocator> g_pAllocator;
NN_OS_ALIGNAS_THREAD_STACK char g_FlushThreadStack[1024 * 16];

}

ContinuousRecorderImpl::ContinuousRecorderImpl(MovieFileWriter* pMovieFileWriter, RecordingNotificator* pRecordingNotificator) NN_NOEXCEPT
    : m_pMovieFileWriter(pMovieFileWriter)
    , m_pRecordingNotificator(pRecordingNotificator)
{
}

namespace {

util::optional<nerd::scoop::VideoFrameRate> GetVideoFrameRate(uint8_t fps) NN_NOEXCEPT
{
    switch (fps)
    {
        case 30: return nerd::scoop::VideoFrameRate::Fps30;
        case 60: return nerd::scoop::VideoFrameRate::Fps60;
        default: return util::nullopt;
    }
}

}

Result ContinuousRecorderImpl::Initialize(sf::NativeHandle&& workMemoryHandle, uint64_t workMemorySize, const grcsrv::ContinuousRecordingParameter& parameter, int flushThreadPriority, const char* flushThreadName) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_Initialized, grc::ResultInvalidState());

    NN_RESULT_DO(nerd::scoop::InitializeResourcesOnce());

    grc::ContinuousRecordingParameter grcParameter = {};
    std::memcpy(&grcParameter, &parameter, sizeof(grcParameter));
    // TODO:(Security) grcParameter のチェック

    auto pFrameRate = GetVideoFrameRate(grcParameter.fps);
    NN_RESULT_THROW_UNLESS(pFrameRate, grc::ResultInvalidCall());

    auto success = false;

    NN_RESULT_THROW_UNLESS(!g_IsUsed.exchange(true), grc::ResultOutOfContinuousRecorder());
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            g_IsUsed = false;
        }
    };

    NN_RESULT_DO(m_MemoryHolder.Initialize(std::move(workMemoryHandle), workMemorySize));
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            m_MemoryHolder.Finalize();
        }
    };

    const size_t Mi = 1024 * 1024;

    size_t mallocBufferSize = 10 * Mi;
    auto malloBuffer = m_MemoryHolder.AlignedAllocate(2 * Mi, mallocBufferSize);
    NN_RESULT_THROW_UNLESS(malloBuffer, grc::ResultOutOfMemory());
    g_pAllocator.emplace(malloBuffer, mallocBufferSize);
    SetStandardAllocatorForMalloc(&*g_pAllocator);
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            SetStandardAllocatorForMalloc(nullptr);
        }
    };

    size_t metaBufferSize = 512 * 1024;
    auto metaBuffer = m_MemoryHolder.AlignedAllocate(4096, metaBufferSize);
    size_t lastFrameYBufferSize = util::align_up(1280 * 720, capsrv::OverlayMovieImageBufferMemoryUnitSize);
    auto lastFrameYBuffer = m_MemoryHolder.AlignedAllocate(capsrv::OverlayMovieImageBufferMemoryAlignment, lastFrameYBufferSize);
    size_t lastFrameUVBufferSize = util::align_up(1280 * 720 / 2, capsrv::OverlayMovieImageBufferMemoryUnitSize);
    auto lastFrameUVBuffer = m_MemoryHolder.AlignedAllocate(capsrv::OverlayMovieImageBufferMemoryAlignment, lastFrameUVBufferSize);

    nerd::scoop::RecordingConfiguration configuration = {};
    configuration.nFramesBetweenIDR = grcParameter.frameCountBetweenIDR - 1;
    configuration.layerStack = vi::LayerStack_Recording;
    NN_RESULT_THROW_UNLESS(util::IsIntValueRepresentable<decltype(configuration.recordBufferSize)>(grcParameter.recordBufferSize), grc::ResultInvalidCall());
    configuration.recordBufferSize = grcParameter.recordBufferSize;
    configuration.videoBitRate = grcParameter.videoBitRate;
    configuration.frameRate = *pFrameRate;
    size_t coreHeapSize = 50 * Mi + configuration.recordBufferSize;
    auto coreHeap = m_MemoryHolder.AlignedAllocate(2 * Mi, coreHeapSize);
    NN_RESULT_THROW_UNLESS(coreHeap, grc::ResultOutOfMemory());
    NN_RESULT_DO(nerd::scoop::InitializeScoopVideoFromUserMem(configuration, coreHeap, coreHeapSize));
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nerd::scoop::FinalizeScoopVideo());
        }
    };

    NN_RESULT_DO(SubTaskThread::StartThread(g_FlushThreadStack, sizeof(g_FlushThreadStack), flushThreadPriority, flushThreadName));

    success = true;
    this->m_ApplicationId = grcParameter.applicationId;
    this->m_Initialized = true;
    this->m_Fps = grcParameter.fps;
    this->m_KeyFrameIntervalFrames = grcParameter.frameCountBetweenIDR;
    this->m_MetaBuffer = metaBuffer;
    this->m_MetaBufferSize = metaBufferSize;
    this->m_LastFrameYBuffer = lastFrameYBuffer;
    this->m_LastFrameYBufferSize = lastFrameYBufferSize;
    this->m_LastFrameUVBuffer = lastFrameUVBuffer;
    this->m_LastFrameUVBufferSize = lastFrameUVBufferSize;
    this->m_FlushParameter = {};
    this->m_IsRecording = false;
    this->m_MaxRecordingTime = TimeSpan::FromMilliSeconds(grcParameter.maxTimeInMilliSeconds);
    this->m_MinRecordingTime = TimeSpan::FromMilliSeconds(grcParameter.minTimeInMilliSeconds);
    this->m_ForDebug = grcParameter.forDebug;
    NN_RESULT_SUCCESS;
}

void ContinuousRecorderImpl::FinalizeImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);

    SubTaskThread::StopThread();

    if (m_IsRecording)
    {
        StopRecordingImpl();
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(nerd::scoop::FinalizeScoopVideo());

    SetStandardAllocatorForMalloc(nullptr);
    g_pAllocator->WalkAllocatedBlocks([](void*, size_t, void*) -> int
    {
        #if !defined(NN_SDK_BUILD_RELEASE)
        g_pAllocator->Dump();
        #endif
        NN_ABORT();
    }, nullptr);
    g_pAllocator = util::nullopt;

    m_MemoryHolder.Finalize();
    g_IsUsed = false;
}

ContinuousRecorderImpl::~ContinuousRecorderImpl() NN_NOEXCEPT
{
    if (m_Initialized)
    {
        FinalizeImpl();
    }
}

void ContinuousRecorderImpl::StartRecordingImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_IsRecording);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nerd::scoop::StartContinuousRecording());
}

void ContinuousRecorderImpl::StopRecordingImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsRecording);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nerd::scoop::StopContinuousRecording());
}

Result ContinuousRecorderImpl::StartRecording() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Initialized, grc::ResultInvalidState());
    NN_RESULT_THROW_UNLESS(!m_IsRecording, grc::ResultInvalidState());
    StartRecordingImpl();
    this->m_RecordingStartTick = os::GetSystemTick();
    ResetFlushStartTick();
    this->m_IsRecording = true;
    NN_RESULT_SUCCESS;
}

Result ContinuousRecorderImpl::StopRecording() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Initialized, grc::ResultInvalidState());
    NN_RESULT_THROW_UNLESS(m_IsRecording, grc::ResultInvalidState());
    StopRecordingImpl();
    this->m_IsRecording = false;
    NN_RESULT_SUCCESS;
}

void ContinuousRecorderImpl::ResetFlushStartTickImpl(os::Tick additionalTick) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_FlushMutex.IsLockedByCurrentThread());
    auto newTick =os::GetSystemTick() + additionalTick;
    if (newTick > m_FlushStartTick)
    {
        this->m_FlushStartTick = newTick;
    }
}

void ContinuousRecorderImpl::ResetFlushStartTick() NN_NOEXCEPT
{
    std::unique_lock<decltype(m_FlushMutex)> lk(m_FlushMutex);
    ResetFlushStartTickImpl();
}

TimeSpan ContinuousRecorderImpl::GetFlushTimeImpl() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_FlushMutex.IsLockedByCurrentThread());
    return (os::GetSystemTick() - m_FlushStartTick).ToTimeSpan();
}

Result ContinuousRecorderImpl::WriteDataImpl(int* pFrameCount, fs::FileHandle f, TimeSpan timeSpan, int videoOrientation) NN_NOEXCEPT
{
    bool canceled;
    NN_RESULT_DO(nerd::scoop::FlushContinuousRecordingToFile(&canceled, pFrameCount, f, timeSpan, videoOrientation));
    NN_RESULT_DO(fs::FlushFile(f));
    NN_RESULT_THROW_UNLESS(!canceled, grc::ResultFlushCanceled());
    NN_RESULT_SUCCESS;
}

void ContinuousRecorderImpl::FlushImpl() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_FlushMutex.IsLockedByCurrentThread());

    auto minRecoringTime = m_MinRecordingTime;
    auto maxRecoringTime = m_MaxRecordingTime;
    auto cutsPrevious = !m_ForDebug;

    if (m_FlushParameter.overrides)
    {
        minRecoringTime = m_FlushParameter.overrideParameter.minTime;
        maxRecoringTime = m_FlushParameter.overrideParameter.maxTime;
        cutsPrevious = m_FlushParameter.overrideParameter.cutsPrevious;
    }

    TimeSpan timeSpan;
    {
        std::unique_lock<decltype(m_FlushMutex)> lk(m_FlushMutex);
        timeSpan = cutsPrevious ? GetFlushTimeImpl() : (os::GetSystemTick() - m_RecordingStartTick).ToTimeSpan();
        if (!(timeSpan >= minRecoringTime))
        {
            lk.unlock();
            m_pRecordingNotificator->OnDataError();
            return;
        }
        if (cutsPrevious)
        {
            ResetFlushStartTickImpl();
        }
    }

    if (maxRecoringTime > 0)
    {
        if (!(timeSpan <= maxRecoringTime))
        {
            timeSpan = maxRecoringTime;
        }
    }

    capsrv::AlbumFileId albumFileId;
    NN_ABORT_UNLESS_RESULT_SUCCESS(capsrv::GenerateCurrentAlbumFileId(&albumFileId, m_ApplicationId, capsrv::AlbumFileContents_Movie));

    auto preProcessSuccess = false;
    NN_RESULT_IGNORING_BLOCK
    {
        NN_RESULT_DO(nerd::scoop::GetCopyLastRecordedFrame(m_LastFrameYBuffer, m_LastFrameYBufferSize, m_LastFrameUVBuffer, m_LastFrameUVBufferSize));
        NN_RESULT_DO(capsrv::SetupOverlayMovieThumbnail(m_LastFrameYBuffer, m_LastFrameYBufferSize, m_LastFrameUVBuffer, m_LastFrameUVBufferSize, 1280, 720, albumFileId));
        preProcessSuccess = true;
        NN_RESULT_SUCCESS;
    };
    if (!preProcessSuccess)
    {
        m_pRecordingNotificator->OnDataError();
        return;
    }

    m_pRecordingNotificator->BeforeFlush(albumFileId);

    capsrv::movie::MovieMetaDataBuilder metaBuilder;
    metaBuilder.Initialize(nn::capsrv::movie::MovieMetaDataVersion_1, m_MetaBuffer, m_MetaBufferSize, albumFileId);
    NN_UTIL_SCOPE_EXIT
    {
        metaBuilder.Finalize();
    };
    metaBuilder.SetDescription(capsrv::AlbumFileDescription_MovieContinuous);

    auto impl = MakeFunctorDataWriter([&](fs::FileHandle handle) -> Result
    {
        // SIGLO-80085:
        // ここで再度最終フレーム用の画像をキャプチャする。
        // GetCopyLastRecordedFrame() でエラーが返った場合は、
        // 既にキャプチャ済みの画像をそのままサムネイルに適用する。
        (void)nerd::scoop::GetCopyLastRecordedFrame(m_LastFrameYBuffer, m_LastFrameYBufferSize, m_LastFrameUVBuffer, m_LastFrameUVBufferSize);
        metaBuilder.SetImageDataNv12(m_LastFrameYBuffer, m_LastFrameYBufferSize, m_LastFrameUVBuffer, m_LastFrameUVBufferSize, 1280, 720);

        int frameCount;
        int videoOrientation = (360 - m_FlushParameter.imageOrientation * 90) % 360;
        NN_RESULT_DO(this->WriteDataImpl(&frameCount, handle, timeSpan, videoOrientation));
        metaBuilder.SetAttribute(frameCount, 1, m_Fps, static_cast<int>(int64_t{frameCount} * 1000 / m_Fps), m_KeyFrameIntervalFrames, m_FlushParameter.isCopyrightImageComposited, capsrv::ScreenShotSize_1280x720, m_FlushParameter.imageOrientation);
        NN_RESULT_SUCCESS;
    });
    auto result = m_pMovieFileWriter->Write(albumFileId, &impl, &metaBuilder);
    m_pRecordingNotificator->AfterFlush(result, albumFileId);
}

void ContinuousRecorderImpl::CancelFlushImpl() NN_NOEXCEPT
{
    nerd::scoop::CancelFlushContinuousRecordingToFile();
}

void ContinuousRecorderImpl::DoTask() NN_NOEXCEPT
{
    fs::SetPriorityRawOnCurrentThread(fs::PriorityRaw_Background);
    FlushImpl();
    if (m_pFlushCompletedEvent)
    {
        m_pFlushCompletedEvent->Signal();
        this->m_pFlushCompletedEvent = util::nullopt;
    }
}

void ContinuousRecorderImpl::DoCancel() NN_NOEXCEPT
{
    CancelFlushImpl();
}

Result ContinuousRecorderImpl::GetNotFlushingEvent(sf::Out<sf::NativeHandle> pOutSystemEventHandle) NN_NOEXCEPT
{
    return SubTaskThread::GetNotRunningEvent(pOutSystemEventHandle);
}

Result ContinuousRecorderImpl::StartFlush(sf::NativeHandle* pOut, const grcsrv::ContinuousRecordingFlushParameter& parameter) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Initialized, grc::ResultInvalidState());
    grc::ContinuousRecordingFlushParameter grcParameter;
    NN_STATIC_ASSERT(sizeof(parameter) >= sizeof(grcParameter));
    std::memcpy(&grcParameter, &parameter, sizeof(grcParameter));
    this->m_FlushParameter = grcParameter;
    sf::NativeHandle readableEvent;
    NN_RESULT_DO(SubTaskThread::Start([&]()
    {
        if (pOut)
        {
            this->m_pFlushCompletedEvent.emplace(os::EventClearMode_ManualClear, true);
            readableEvent = sf::NativeHandle(m_pFlushCompletedEvent->DetachReadableHandle(), true);
        }
    }));
    if (pOut)
    {
        *pOut = std::move(readableEvent);
    }
    NN_RESULT_SUCCESS;
}

Result ContinuousRecorderImpl::CancelFlush() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Initialized, grc::ResultInvalidState());
    return SubTaskThread::Cancel();
}

Result ContinuousRecorderImpl::ResetFlushTime(int64_t additionalTimeNs) NN_NOEXCEPT
{
    std::unique_lock<decltype(m_FlushMutex)> lk(m_FlushMutex);
    ResetFlushStartTickImpl(os::Tick(TimeSpan::FromNanoSeconds(additionalTimeNs)));
    NN_RESULT_SUCCESS;
}

bool ContinuousRecorderImpl::IsFlushing() NN_NOEXCEPT
{
    return this->IsRunning();
}

ContinuousRecorderImpl2::ContinuousRecorderImpl2(MovieFileWriter* pMovieFileWriter, RecordingNotificator* pRecordingNotificator, DeactivatableHolder* pDeactivatableHolder) NN_NOEXCEPT
    : m_pMovieFileWriter(pMovieFileWriter)
    , m_pRecordingNotificator(pRecordingNotificator)
    , m_pDeactivatableHolder(pDeactivatableHolder)
{
}

Result ContinuousRecorderImpl2::ActivateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_pImpl);
    m_pImpl.emplace(m_pMovieFileWriter, m_pRecordingNotificator);
    NN_RESULT_DO(m_pImpl->Initialize(sf::NativeHandle{m_WorkMemoryHandle.GetOsHandle(), false}, m_WorkMemorySize, m_Parameter, m_FlushThreadPriority, m_FlushThreadName));
    m_pDeactivatableHolder->Set(this);
    NN_RESULT_SUCCESS;
}

void ContinuousRecorderImpl2::DeactivateImpl() NN_NOEXCEPT
{
    m_pImpl = util::nullopt;
    m_pDeactivatableHolder->Set(nullptr);
}

bool ContinuousRecorderImpl2::Deactivate() NN_NOEXCEPT
{
    if (m_pImpl && m_pImpl->IsFlushing())
    {
        return false;
    }
    DeactivateImpl();
    return true;
}

Result ContinuousRecorderImpl2::Initialize(sf::NativeHandle&& workMemoryHandle, uint64_t workMemorySize, const grcsrv::ContinuousRecordingParameter& parameter, int flushThreadPriority, const char* flushThreadName) NN_NOEXCEPT
{
    this->m_WorkMemoryHandle = std::move(workMemoryHandle);
    this->m_WorkMemorySize = workMemorySize;
    this->m_Parameter = parameter;
    this->m_FlushThreadPriority = flushThreadPriority;
    this->m_FlushThreadName = flushThreadName;
    return ActivateImpl();
}

ContinuousRecorderImpl2::~ContinuousRecorderImpl2() NN_NOEXCEPT
{
    DeactivateImpl();
}

Result ContinuousRecorderImpl2::StartRecording() NN_NOEXCEPT
{
    if (!m_pImpl)
    {
        NN_RESULT_DO(ActivateImpl());
    }
    return m_pImpl->StartRecording();
}

Result ContinuousRecorderImpl2::StopRecording() NN_NOEXCEPT
{
    if (m_pImpl)
    {
        NN_RESULT_DO(m_pImpl->StopRecording());
    }
    NN_RESULT_SUCCESS;
}

Result ContinuousRecorderImpl2::GetNotFlushingEvent(sf::Out<sf::NativeHandle> pOutSystemEventHandle) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_pImpl, grc::ResultInvalidState());
    return m_pImpl->GetNotFlushingEvent(pOutSystemEventHandle);
}

Result ContinuousRecorderImpl2::StartFlush(const grcsrv::ContinuousRecordingFlushParameter& parameter) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_pImpl, grc::ResultInvalidState());
    return m_pImpl->StartFlush(nullptr, parameter);
}

Result ContinuousRecorderImpl2::StartFlushWithEvent(sf::Out<sf::NativeHandle> pOut, const grcsrv::ContinuousRecordingFlushParameter& parameter) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_pImpl, grc::ResultInvalidState());
    sf::NativeHandle ret;
    NN_RESULT_DO(m_pImpl->StartFlush(&ret, parameter));
    *pOut = std::move(ret);
    NN_RESULT_SUCCESS;
}

Result ContinuousRecorderImpl2::CancelFlush() NN_NOEXCEPT
{
    if (m_pImpl)
    {
        NN_RESULT_DO(m_pImpl->CancelFlush());
    }
    NN_RESULT_SUCCESS;
}

Result ContinuousRecorderImpl2::ResetFlushTime(int64_t additionalTimeNs) NN_NOEXCEPT
{
    if (m_pImpl)
    {
        NN_RESULT_DO(m_pImpl->ResetFlushTime(additionalTimeNs));
    }
    NN_RESULT_SUCCESS;
}

}}
