﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdarg>
#include <fstream>
#include <iomanip>

#if !defined( WIN32_LEAN_AND_MEAN )
    #define WIN32_LEAN_AND_MEAN
#endif
#if !defined( NOMINMAX )
    #define NOMINMAX
#endif
#include <nn/nn_Windows.h>
#include <nn/cstd/cstd_CStdArg.h>
#include <nn/util/util_StringView.h>

#include <nn/gfxTool/gfxTool_Util.h>

namespace nn {
namespace gfxTool {

Custom< std::string >::Type ReadTextFile( const char* pFilename )
{
    std::ifstream ifs( pFilename );
    if( ifs.fail() )
    {
        return Custom< std::string >::Type();
    }

    return Custom< std::string >::Type( ( std::istreambuf_iterator< char >( ifs ) ),
        ( std::istreambuf_iterator< char >() ) );
}

bool WriteTextFile( const char* pFilename, const nn::util::string_view& text )
{
    std::ofstream ofs( pFilename );
    if( ofs.fail() )
    {
        return false;
    }

    ofs.write( text.data(), text.length() );
    return true;
}

Custom< std::vector< char > >::Type Format( const char* pFormat, ... )
{
    va_list arg;
    va_start( arg, pFormat );
    auto ret = Format( pFormat, arg );
    va_end( arg );

    return std::move( ret );
}

Custom< std::vector< char > >::Type Format( const char* pFormat, va_list arg )
{
    Custom< std::vector< char > >::Type ret;

    va_list arg2;
    NN_CSTD_VA_COPY( arg2, arg );

    auto count = _vscprintf( pFormat, arg ) + 1;
    ret.resize( count );
    _vsnprintf_s( &ret[ 0 ], count, count, pFormat, arg2 );

    va_end( arg2 );

    return std::move( ret );
}

Custom< std::string >::Type GetLastErrorString()
{
    void* pMessage;
    FormatMessageA( FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM
        | FORMAT_MESSAGE_IGNORE_INSERTS, nullptr, GetLastError(),
        MAKELANGID( LANG_NEUTRAL, SUBLANG_DEFAULT ), reinterpret_cast< LPSTR >( &pMessage ), 0, nullptr );
    Custom< std::string >::Type ret = static_cast< char* >( pMessage );
    LocalFree( pMessage );
    return std::move( ret );
}

Custom< std::string >::Type GetModulePath( void* hModule )
{
    char path[ MAX_PATH ];
    GetModuleFileNameA( static_cast< HMODULE >( hModule ), path, MAX_PATH );
    return Custom< std::string >::Type( path );
}

Custom< std::string >::Type ConvertEncoding(
    const Custom< std::string >::Type& value, int fromCodePage, int toCodePage )
{
    auto utf16Length = MultiByteToWideChar( StaticCastAuto( fromCodePage ),
        0, value.c_str(), StaticCastAuto( value.length() ), nullptr, 0 );
    Custom< std::unique_ptr< wchar_t[] > >::Type utf16( new wchar_t[ utf16Length ] );
    utf16Length = MultiByteToWideChar( StaticCastAuto( fromCodePage ),
        0, value.c_str(), StaticCastAuto( value.length() ), utf16.get(), utf16Length );

    auto resultLength = WideCharToMultiByte( StaticCastAuto( toCodePage ),
        0, utf16.get(), utf16Length, nullptr, 0, nullptr, nullptr );
    Custom< std::unique_ptr< char[] > >::Type result( new char[ resultLength ] );
    resultLength = WideCharToMultiByte( StaticCastAuto( toCodePage ),
        0, utf16.get(), utf16Length, result.get(), resultLength, nullptr, nullptr );

    return Custom< std::string >::Type( result.get(), resultLength );
}

bool CaseInsensiveEquals( const char* lhs, const char* rhs )
{
    if( lhs == nullptr || rhs == nullptr )
    {
        return false;
    }
    size_t lhsLength = strlen( lhs );
    if( strlen( rhs ) != lhsLength )
    {
        return false;
    }
    for( int idx = 0; idx < static_cast< int >( lhsLength ); ++idx )
    {
        if( tolower( lhs[ idx ] ) != tolower( rhs[ idx ] ) )
        {
            return false;
        }
    }
    return true;
}

Custom< std::string >::Type AddLineNumber( const nn::util::string_view& source )
{
    int lineCount = 0;
    for( auto pChar = source.data(), pEnd = source.data() + source.length(); pChar < pEnd; ++pChar )
    {
        if( *pChar == '\n' )
        {
            ++lineCount;
        }
    }
    auto lineDigit = static_cast< int >( std::log10( static_cast< float >( lineCount ) ) ) + 1;
    Custom< std::stringstream >::Type ss;
    auto pBefore = source.data();
    auto pChar = pBefore ;
    for( int idxLine = 0; idxLine < lineCount; ++idxLine, pBefore = pChar + 1 )
    {
        pChar = strchr( pBefore, '\n' );
        ss << std::dec << std::setfill( ' ' ) << std::setw( lineDigit ) << idxLine + 1 << "|";
        ss.write( pBefore, pChar - pBefore + 1 );
    }
    ss << std::dec << std::setfill( ' ' ) << std::setw( lineDigit ) << lineCount << "|";
    ss.write( pBefore, ( source.data() + source.length() ) - pBefore );
    return ss.str();
}

}
}
