﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <fstream>

#include <nn/nn_SdkLog.h>
#if !defined( WIN32_LEAN_AND_MEAN )
    #define WIN32_LEAN_AND_MEAN
#endif
#if !defined( NOMINMAX )
    #define NOMINMAX
#endif
#include <nn/nn_Windows.h>
#include <shellapi.h>

#include <nn/gfxTool/gfxTool_CommandLineParser.h>

namespace nn {
namespace gfxTool {

CommandLineParser::CommandLineArg::CommandLineArg( const nn::util::string_view& commandLine )
{
    m_Args.clear();
    m_ArgStrings.clear();

    if( commandLine.size() < 1 )
    {
        return;
    }

    auto lengthw = MultiByteToWideChar( CP_UTF8, 0, commandLine.data(),
        StaticCastAuto( commandLine.length() ), nullptr, 0 );
    Custom< std::vector< wchar_t > >::Type bufw;
    const wchar_t* dummy = L"dummy "; // 第一引数だけ解釈が異なるようなので回避策としてダミー引数を挿入
    bufw.resize( lstrlen( dummy ) + lengthw );
    lstrcpy( &bufw[ 0 ], dummy );
    MultiByteToWideChar( CP_UTF8, 0, commandLine.data(),
        StaticCastAuto( commandLine.length() ), &bufw[ lstrlen( dummy ) ], lengthw );
    bufw.push_back( '\0' );
    int argc;
    auto argvw = CommandLineToArgvW( &bufw[ 0 ], &argc );
    --argc;
    ++argvw;

    m_ArgStrings.resize( argc );
    m_Args.reserve( argc );
    for( int idxArg = 0; idxArg < argc; ++idxArg )
    {
        auto length = WideCharToMultiByte( CP_UTF8, 0, argvw[ idxArg ], -1, nullptr, 0, nullptr, nullptr );
        m_ArgStrings[ idxArg ].resize( length );
        WideCharToMultiByte( CP_UTF8, 0, argvw[ idxArg ], -1,
            &m_ArgStrings[ idxArg ][ 0 ], length, nullptr, nullptr );
        m_Args.push_back( &m_ArgStrings[ idxArg ][ 0 ] );
    }

    LocalFree( argvw );
}

void CommandLineParser::Parse( int argc, const char* const* argv )
{
    bool isArgMode = false;
    for( int idxArg = 0; idxArg < argc; ++idxArg )
    {
        auto arg = argv[ idxArg ];
        auto argLength = strlen( arg );
        if( argLength < 1 )
        {
            continue;
        }
        if( !isArgMode && arg[ 0 ] == '-' )
        {
            if( argLength < 2 )
            {
                NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument, "%s is invalid.", arg );
            }

            CommandLineOptionBase* pOption = nullptr;
            const char* pValue = nullptr;
            if( arg[ 1 ] == '-' )
            {
                if( argLength < 3 )
                {
                    isArgMode = true;
                }

                auto found = std::find_if( m_Options.begin(), m_Options.end(),
                    [&]( CommandLineOptionBase* pOption )
                {
                    auto& definition = pOption->GetDefinition();
                    return definition.pLongName && strncmp( &arg[ 2 ], definition.pLongName,
                        definition.longNameLength ) == 0 && ( arg[ 2 + definition.longNameLength ] == '='
                        || arg[ 2 + definition.longNameLength ] == '\0' );
                } );
                if( found == m_Options.end() )
                {
                    if (m_pUndefinedOptionCallback)
                    {
                        if (m_pUndefinedOptionCallback(argc, argv, idxArg, m_pUndefinedOptionCallback))
                        {
                            continue;
                        }
                    }
                    NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument, "%s is invalid.", arg );
                }
                pOption = *found;

                if( arg[ 2 + pOption->GetDefinition().longNameLength ] == '=' )
                {
                    pValue = &arg[ 3 + pOption->GetDefinition().longNameLength ];
                }
            }
            else
            {
                auto found = std::find_if( m_Options.begin(), m_Options.end(),
                    [ & ]( CommandLineOptionBase* pOption )
                {
                    return pOption->GetDefinition().shortName == arg[ 1 ];
                } );
                if( found == m_Options.end() )
                {
                    if (m_pUndefinedOptionCallback)
                    {
                        if (m_pUndefinedOptionCallback(argc, argv, idxArg, m_pUndefinedOptionCallback))
                        {
                            continue;
                        }
                    }
                    NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument, "%s is invalid.", arg );
                }
                pOption = *found;
                if( argLength > 2 )
                {
                    pValue = arg[ 2 ] == '=' ? &arg[ 3 ] : &arg[ 2 ];
                }
            }

            if( pOption->GetOptionValueType() != OptionValueType::None )
            {
                if( pValue == nullptr )
                {
                    auto argValue = argv[ ++idxArg ];
                    if( idxArg >= argc || argValue[ 0 ] == '-' )
                    {
                        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument, "%s is invalid.", arg );
                    }
                    pValue = argValue;
                }
                pOption->SetValue( pValue );
            }
            else if( pValue )
            {
                NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument, "%s is invalid.", arg );
            }

            pOption->SetExisting( true );
            if( pOption == &m_ArgsFile )
            {
                ExpandExternalFile( pValue );
            }
        }
        else
        {
            if( m_pArgs == nullptr )
            {
                NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument, "%s is invalid.", arg );
            }
            m_pArgs->push_back( arg );
        }
    }
} // NOLINT(impl/function_size)

void CommandLineParser::PrintHelp( const char* pToolName, const char* pToolDescription )
{
    // TODO
    NN_GFXTOOL_PRINT( "%s: %s\n", pToolName, pToolDescription );
    // TODO : C++ ツールのローカライズ方法が確立したら、元の日本語に置き換える : "Usage" -> "使用方法"
    NN_GFXTOOL_PRINT( "Usage: %s", pToolName );
    for( auto& pOption : m_Options )
    {
        auto& definition = pOption->GetDefinition();
        if( definition.shortName )
        {
            NN_GFXTOOL_PRINT( " " );
            if( !definition.isRequired )
            {
                NN_GFXTOOL_PRINT( "[" );
            }
            NN_GFXTOOL_PRINT( "-%c", definition.shortName );
            if( pOption->GetOptionValueType() != OptionValueType::None )
            {
                NN_GFXTOOL_PRINT( "<arg>" );
                if( pOption->GetOptionValueType() == OptionValueType::Multi )
                {
                    NN_GFXTOOL_PRINT( "..." );
                }
            }
            if( !definition.isRequired )
            {
                NN_GFXTOOL_PRINT( "]" );
            }
        }
        if( definition.pLongName )
        {
            NN_GFXTOOL_PRINT( " " );
            if( !definition.isRequired )
            {
                NN_GFXTOOL_PRINT( "[" );
            }
            NN_GFXTOOL_PRINT( "--%s", definition.pLongName );
            if( pOption->GetOptionValueType() != OptionValueType::None )
            {
                NN_GFXTOOL_PRINT( "<arg>" );
                if( pOption->GetOptionValueType() == OptionValueType::Multi )
                {
                    NN_GFXTOOL_PRINT( "..." );
                }
            }
            if( !definition.isRequired )
            {
                NN_GFXTOOL_PRINT( "]" );
            }
        }
    }
    if( m_pArgs )
    {
        NN_GFXTOOL_PRINT( " <inputFile>..." );
    }

    // TODO : C++ ツールのローカライズ方法が確立したら、元の日本語に置き換える : "Acceptable Arguments" -> "使用可能な引数"
    NN_GFXTOOL_PRINT( "\n\nAcceptable Arguments:\n" );
    for( auto& pOption : m_Options )
    {
        auto& definition = pOption->GetDefinition();
        NN_GFXTOOL_PRINT( "  " );
        std::string optionString;
        optionString.reserve( 256 );
        if( definition.shortName )
        {
            optionString.push_back( '-' );
            optionString.push_back( definition.shortName );
            if( definition.pLongName )
            {
                optionString.append( ", " );
            }
        }
        if( definition.pLongName )
        {
            optionString.append( "--" );
            optionString.append( definition.pLongName );
        }
        if( pOption->GetOptionValueType() != OptionValueType::None )
        {
            optionString.append( " <arg>" );
            if( pOption->GetOptionValueType() == OptionValueType::Multi )
            {
                optionString.append( "..." );
            }
        }
        if( optionString.length() < 30 )
        {
            NN_GFXTOOL_PRINT( "%-30s", optionString.data() );
        }
        else
        {
            NN_GFXTOOL_PRINT( "%s\n", optionString.data() );
            NN_GFXTOOL_PRINT( "  %30s", "" );
        }
        NN_GFXTOOL_PRINT( definition.pDescription );
        NN_GFXTOOL_PRINT( "\n" );
    }
    if( m_pArgs )
    {
        NN_GFXTOOL_PRINT( "  %-30sSpecifies the input filename.\n", "<inputPath>..." );
    }
}

void CommandLineParser::ExpandExternalFile( const char* pPath )
{
    const char* pFile = nullptr;
    Custom< std::vector< char > >::Type fileStr;
    size_t fileSize = 0;
    if( m_pCallback )
    {
        void* pFileData = nullptr;
        m_pCallback( &pFileData, &fileSize, pPath, m_pCallbackParam );
        pFile = static_cast< char* >( pFileData );
    }
    else
    {
        std::ifstream ifs( pPath );
        if( ifs.fail() )
        {
            NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_FailedToLoadFile, "Failed to load %s.", pPath );
        }
        fileStr = Custom< std::vector< char > >::Type(
            ( std::istreambuf_iterator< std::string::value_type >( ifs ) ),
            ( std::istreambuf_iterator< std::string::value_type >() ) );
        pFile = fileStr.size() > 0 ? &fileStr[ 0 ] : nullptr;
        fileSize = fileStr.size();
    }

    if( pFile == nullptr )
    {
        pFile = "";
    }

    int fileReadOffsetByte = 0;
    if( fileSize >= 3 && IsBom( pFile ) )
    {
        fileReadOffsetByte = 3;
    }

    Custom< std::string >::Type commandLine;
    commandLine.reserve( fileStr.size() );
    bool isComment = *( pFile + fileReadOffsetByte ) == '#';
    for( const char* pStr = pFile + fileReadOffsetByte, *pFileEnd = pFile + fileSize; pStr < pFileEnd; ++pStr )
    {
        if( *pStr == '\n' )
        {
            if( !isComment )
            {
                commandLine.push_back( ' ' );
            }
            isComment = *( pStr + 1 ) == '#';
        }
        else if( !isComment && *pStr != '\r' )
        {
            commandLine.push_back( *pStr );
        }
    }

    CommandLineArg args( nn::util::string_view( commandLine.data(), commandLine.size() ) );

    Parse( args.GetArgC(), args.GetArgv() );
}

void CommandLineParser::CheckRequiredOption()
{
    for( auto& pOption : m_Options )
    {
        if( pOption->GetDefinition().isRequired && !pOption->IsExisting() )
        {
            auto& definition = pOption->GetDefinition();
            if( definition.pLongName )
            {
                NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_LackArgument,
                    "--%s is required.", definition.pLongName );
            }
            else
            {
                NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_LackArgument,
                    "-%c is required.", definition.shortName );
            }
        }
    }
}

}
}
