﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx/util/detail/gfx_TextureCompressor-common.h>

namespace nn {
namespace gfx {
namespace util {

namespace detail {

TextureCompressorCommonState::TextureCompressorCommonState() NN_NOEXCEPT:
m_IsInitialized(false),
m_pDevice(NULL)
{}

TextureCompressorCommonState::~TextureCompressorCommonState() NN_NOEXCEPT
{}

void TextureCompressorCommonState::Initialize(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDevice);
    NN_SDK_REQUIRES(nn::gfx::IsInitialized(*pDevice));
    NN_SDK_ASSERT(m_IsInitialized == false);

    this->m_pDevice = pDevice;

    //Initialize Gfx Objects
    {
        InitializeRasterizerState();
        InitializeBlendState();
        InitializeDepthStencilState();
    }

    m_IsInitialized = true;
}

void TextureCompressorCommonState::Finalize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    m_DepthStencilState.Finalize(m_pDevice);
    m_BlendStateHolder.Finalize(m_pDevice);
    m_RasterizerState.Finalize(m_pDevice);

    m_IsInitialized = false;
}

void TextureCompressorCommonState::InitializeRasterizerState() NN_NOEXCEPT
{
    nn::gfx::RasterizerState::InfoType info;
    {
        info.SetDefault();
        info.SetCullMode(nn::gfx::CullMode_None);
        info.SetScissorEnabled(true);
    }
    m_RasterizerState.Initialize(m_pDevice,info);
}

void TextureCompressorCommonState::InitializeBlendState() NN_NOEXCEPT
{
    nn::gfx::BlendState::InfoType info;
    {
        info.SetDefault();

        nn::gfx::BlendTargetStateInfo targetInfo;
        targetInfo.SetDefault();
        info.SetBlendTargetStateInfoArray(&targetInfo,1);
    }
    m_BlendStateHolder.Initialize(m_pDevice,info);
}

void TextureCompressorCommonState::InitializeDepthStencilState() NN_NOEXCEPT
{
    nn::gfx::DepthStencilState::InfoType info;
    info.SetDefault();
    m_DepthStencilState.Initialize(m_pDevice,info);
}

void TextureCompressorCommonState::MakeCommand(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::gfx::DescriptorSlot& textureDescriptorSlot,
    const nn::gfx::DescriptorSlot& samplerDescriptorSlot
    ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pCommandBuffer);
    NN_SDK_REQUIRES(nn::gfx::IsInitialized(*pCommandBuffer));
    NN_SDK_REQUIRES(IsInitialized());

    pCommandBuffer->SetRasterizerState(&m_RasterizerState);
    pCommandBuffer->SetBlendState(m_BlendStateHolder.Get());
    pCommandBuffer->SetDepthStencilState(&m_DepthStencilState);
    pCommandBuffer->SetTextureAndSampler(0,nn::gfx::ShaderStage_Pixel,textureDescriptorSlot,samplerDescriptorSlot);
}

bool TextureCompressorCommonState::IsInitialized() const NN_NOEXCEPT
{
    return m_IsInitialized;
}

nn::gfx::Device* TextureCompressorCommonState::GetDevice() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized == true);
    return this->m_pDevice;
}

}

}
}
}
