﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx/util/detail/gfx_OffsetRingBuffer.h>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

namespace nn{ namespace gfx{ namespace util{ namespace detail{

    OffsetRingBuffer::OffsetRingBuffer() NN_NOEXCEPT
    {
        m_BaseOffset = 0;
        m_Size = 0;
        m_HeadLocalOffset.store(0, std::memory_order_relaxed);
        m_TailLocalOffset.store(0, std::memory_order_relaxed);
        m_LastLocalOffset = 0;
        m_IsInitialized = false;
        m_IsRecording.store(false, std::memory_order_relaxed);
    }

    void OffsetRingBuffer::Initialize(
        ptrdiff_t base,
        ptrdiff_t size
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!m_IsInitialized);
        NN_SDK_REQUIRES_GREATER_EQUAL(base, 0);
        NN_SDK_REQUIRES_GREATER_EQUAL(size, 1);
        NN_SDK_REQUIRES_LESS_EQUAL(size, static_cast<ptrdiff_t>(SizeMax));
        NN_SDK_REQUIRES_LESS_EQUAL(base, PTRDIFF_MAX - size + 1);

        m_BaseOffset = base;
        m_Size = size;

        m_HeadLocalOffset.store(0, std::memory_order_relaxed);
        m_TailLocalOffset.store(size - 1, std::memory_order_relaxed);
        m_LastLocalOffset = 0;

        m_IsInitialized = true;
        m_IsRecording = false;
    }

    void OffsetRingBuffer::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        m_IsInitialized = false;
    }

    bool OffsetRingBuffer::IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

    bool OffsetRingBuffer::IsRecording() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        return m_IsRecording;
    }

    ptrdiff_t OffsetRingBuffer::GetBaseOffset() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        return m_BaseOffset;
    }
    ptrdiff_t OffsetRingBuffer::GetSize() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        return m_Size;
    }

    void OffsetRingBuffer::Begin() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        NN_SDK_REQUIRES(!m_IsRecording);

        NN_SDK_ASSERT_RANGE(m_HeadLocalOffset, 0, m_Size);
        NN_SDK_ASSERT_RANGE(m_TailLocalOffset, m_HeadLocalOffset, m_HeadLocalOffset + m_Size);
        NN_SDK_ASSERT_EQUAL(m_HeadLocalOffset, m_LastLocalOffset);

        m_IsRecording.store(true);
    }

    void OffsetRingBuffer::End(OffsetRange* pOutRange) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        NN_SDK_REQUIRES(m_IsRecording);
        NN_SDK_REQUIRES_NOT_NULL(pOutRange);

        m_IsRecording.store(false);

        ptrdiff_t head = m_HeadLocalOffset.load(std::memory_order_relaxed);
        ptrdiff_t tail = m_TailLocalOffset.load(std::memory_order_relaxed);
        ptrdiff_t last = m_LastLocalOffset;

        pOutRange->base  = last;
        pOutRange->size = head - last;

        NN_SDK_REQUIRES_RANGE(pOutRange->base, 0, m_Size);
        NN_SDK_REQUIRES_RANGE(pOutRange->size, 0, m_Size);

        // 1 周していたら戻す
        if(head >= m_Size)
        {
            head -= m_Size;
            tail -= m_Size;
            m_HeadLocalOffset.store(head, std::memory_order_relaxed);
            m_TailLocalOffset.store(tail, std::memory_order_relaxed);
        }

        NN_SDK_ASSERT_RANGE(head, 0, m_Size);
        NN_SDK_ASSERT_RANGE(tail, head, head + m_Size);

        m_LastLocalOffset = head;
    }

    void OffsetRingBuffer::ReleaseOffsetRange(const OffsetRange* pRange) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        NN_SDK_REQUIRES_NOT_NULL(pRange);

        // ゼロ初期化の値の場合は何もしない
        if(pRange->base == 0 && pRange->size == 0)
        {
            return;
        }

        NN_SDK_REQUIRES_RANGE(pRange->base , 0, m_Size);
        NN_SDK_REQUIRES_RANGE(pRange->size, 0, m_Size);
        // 順番通りに解放していることを確認
        NN_SDK_REQUIRES_EQUAL((m_TailLocalOffset + 1) % m_Size, pRange->base);
        // Last の位置を追い越さないことを確認
        NN_SDK_REQUIRES_LESS(m_TailLocalOffset + pRange->size, m_LastLocalOffset + m_Size);

        m_TailLocalOffset.fetch_add(pRange->size, std::memory_order_release);
    }

    namespace {

        // head を alignment の整数倍に切り上げる。 head % alignment == 0。
        // ただし head >= ringSize の場合リングバッファの 2 周目なので (head - ringSize) % alignment == 0 が満たすべき条件。
        ptrdiff_t GetAlignedHead(ptrdiff_t head, ptrdiff_t alignment, ptrdiff_t ringSize) NN_NOEXCEPT
        {
            const ptrdiff_t InvalidOffset = OffsetRingBuffer::InvalidOffset;
            ptrdiff_t alignedHead;

            if(alignment >= ringSize)
            {
                // alignment が ringSize より大きい場合、リングの先頭からしかとれない
                if(head == 0)
                {
                    // 丁度 head が 0 なら 1 周目の最初から取る
                    alignedHead = 0;
                }
                else if(head <= ringSize)
                {
                    // 1 周目なら 2 周目の最初から取る
                    alignedHead = ringSize;
                }
                else
                {
                    // 2 周目の途中からではとれない
                    return InvalidOffset;
                }
            }
            else if(head < ringSize)
            {
                // 1 周目の場合。
                //  * alignment < ringSize
                //  * head < ringSize

                alignedHead = ((head + alignment - 1) / alignment) * alignment;
                // NOTE:
                //   * head + alignment < 2 * ringSize ( < MAX ) なのでオーバーフローしない

                if(alignedHead >= ringSize)
                {
                    // 2 周目に突入していたら 2 周目の最初から取る
                    alignedHead = ringSize;
                }
            }
            else
            {
                // 2 周目の場合
                //  * alignment < ringSize
                //  * head >= ringSize

                ptrdiff_t m = (head - ringSize) % alignment;
                if(m == 0)
                {
                    alignedHead = head;
                }
                else
                {
                    ptrdiff_t v = alignment - m;

                    if(head <= PTRDIFF_MAX - v)
                    {
                        alignedHead = head + v;
                    }
                    else
                    {
                        return InvalidOffset;
                    }

                    if(alignedHead >= 2 * ringSize)
                    {
                        // 2 周目の終わりを超えていたら確保できない
                        return InvalidOffset;
                    }
                }
            }

            return alignedHead;
        }

    }

    ptrdiff_t OffsetRingBuffer::Allocate(ptrdiff_t size, ptrdiff_t alignment) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsRecording);
        NN_SDK_REQUIRES(m_IsInitialized);
        NN_SDK_REQUIRES_GREATER_EQUAL(size, 0);
        NN_SDK_REQUIRES_GREATER(alignment, 0);
        NN_SDK_REQUIRES(m_BaseOffset % alignment == 0);

        // * 確保する長さは 1 以上。
        // * 空の状態でも (m_Size - 1) バイトしか確保できない。
        if(size == 0 || size >= m_Size)
        {
            return InvalidOffset;
        }

        ptrdiff_t head = m_HeadLocalOffset.load(std::memory_order_acquire);
        ptrdiff_t tail;
        ptrdiff_t newHead;
        ptrdiff_t resultLocal;
        do{
            tail = m_TailLocalOffset.load(std::memory_order_acquire);

            // head の位置を alignment の整数倍に合わせる
            // NOTE: この処理の結果 head が tail より大きくなる可能性がある
            ptrdiff_t alignedHead = GetAlignedHead(head, alignment, m_Size);

            // 切り上げ出来なかったら失敗
            if(alignedHead == InvalidOffset)
            {
                return InvalidOffset;
            }

            NN_SDK_ASSERT_RANGE(alignedHead, 0, 2 * m_Size);
            if(alignedHead < m_Size)
            {
                NN_SDK_ASSERT(alignedHead % alignment == 0);
            }
            else
            {
                NN_SDK_ASSERT((alignedHead - m_Size) % alignment == 0);
            }

            // * alignedHead == 0 としても size > tail なら確保できない。
            // * alignedHead + size > tail なら確保できない。
            //
            // alignedHead が tail より大きくなっていたとしてもここで確保失敗になる。
            if(size > tail || alignedHead > tail - size)
            {
                return InvalidOffset;
            }

            newHead = alignedHead + size;
            resultLocal = alignedHead;
            // 取得した範囲がリングバッファの終端を跨いでいたら 2 周目の先頭から取る
            //   * 最初から 2 周目から確保していれば alignedHead >= m_Size。
            //   * 1 周目の最後までに収まったら newHead <= m_Size。
            if(alignedHead < m_Size && newHead > m_Size)
            {
                // * {m_Size * 2 < MAX} で、 {size < m_Size} なので {m_Size + size < MAX}。
                newHead = m_Size + size;
                resultLocal = m_Size;
                if(newHead > tail)
                {
                    return InvalidOffset;
                }
            }
        }while(!m_HeadLocalOffset.compare_exchange_weak(head, newHead));

        NN_SDK_ASSERT_LESS(head, tail);

        ptrdiff_t result = m_BaseOffset + (resultLocal % m_Size);
        return result;
    }

}}}}
