﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/os.h>
#include <nn/nn_SdkLog.h>

#include <nn/util/util_BitPack.h>
#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_StateData-api.vk.1.h>
#include <nn/gfx/gfx_CommandBufferData-api.vk.1.h>

#include <nn/gfx/detail/gfx_CommandBuffer-api.vk.1.h>
#include <nn/gfx/detail/gfx_State-api.vk.1.h>

#include <nn/gfx/detail/gfx_Misc.h>

#include "gfx_VkCommand.h"
#include "gfx_VkHelper.h"
#include "gfx_VkManager.h"
#include "gfx_CommonHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationVk1 Target;

namespace {

void BindPipeline( CommandContext* pCommandContext,
    GfxVkCommandInfo* pCommandInfo, GfxVkCommandInfo* pPrevCommandInfo, bool isDraw ) NN_NOEXCEPT
{
    VkPipeline pipeline = pCommandInfo->pPipelineCache->GetPipeline();

    if ( pPrevCommandInfo && pPrevCommandInfo->pPipelineCache->GetPipeline() == pipeline )
    {
        return;
    }

    NN_GFX_CALL_VK_FUNCTION( vkCmdBindPipeline( pCommandContext->GetVkDrawCommandBuffer(),
        isDraw ? VK_PIPELINE_BIND_POINT_GRAPHICS : VK_PIPELINE_BIND_POINT_COMPUTE, pipeline ) );
}

void BindAndUpdateDescriptorSet( CommandContext* pCommandContext,
    GfxVkCommandInfo* pCommandInfo, GfxVkCommandInfo* pPrevCommandInfo, bool isDraw ) NN_NOEXCEPT
{
    VkDescriptorSet descriptorSet = pCommandInfo->pDescriptorSetCacheNode->GetDescriptorSet();
    VkPipelineLayout layout = pCommandInfo->pPipelineCache->GetPipelineLayout();
    GfxVkGpuState* pGpuState = pCommandContext->GetGpuState();

    uint32_t dynamicOffsets[ GfxVkGpuState::maxDescriptorSlot ];
    uint32_t dynamicOffsetCount = 0;

    if ( pPrevCommandInfo && pPrevCommandInfo->pDescriptorSetCacheNode->GetDescriptorSet() == descriptorSet )
    {
        for ( uint32_t idxSlotState = 0; idxSlotState <=
            pGpuState->descriptorSetLayoutState.maxDescriptorIndex; ++idxSlotState )
        {
            const GfxVkGpuState::DescriptorSetLayoutState::SlotState* pSrcSlotState =
                &pGpuState->descriptorSetLayoutState.slotState[ idxSlotState ];

            if ( !pSrcSlotState->flag.GetBit( GfxVkGpuState::DescriptorSetLayoutState::SlotState::Flag_IsSet )
                || !( pCommandInfo->pPipelineCache->GetValidSlotMask() & ( 1ULL << idxSlotState ) ) )
            {
                continue;
            }

            if ( pSrcSlotState->type == VK_DESCRIPTOR_TYPE_UNIFORM_BUFFER_DYNAMIC ||
                pSrcSlotState->type == VK_DESCRIPTOR_TYPE_STORAGE_BUFFER_DYNAMIC )
            {
                NN_SDK_ASSERT( idxSlotState < GfxVkGpuState::maxDynamicDescriptorSlot );
                dynamicOffsets[ dynamicOffsetCount++ ] = pCommandInfo->dynamicOffsets[ idxSlotState ];
            }
        }
    }
    else if ( VkWriteDescriptorSet* pWriteDescriptorSet = pCommandInfo->pDescriptorSetCacheNode->GetWriteDescriptorSet() )
    {
        NN_SDK_ASSERT( pGpuState->descriptorSetLayoutState.availableDescriptorCount > 0 );

        uint32_t idxAvailableBinding = 0;
        for ( uint32_t idxSlotState = 0; idxSlotState <=
            pGpuState->descriptorSetLayoutState.maxDescriptorIndex; ++idxSlotState )
        {
            const GfxVkGpuState::DescriptorSetLayoutState::SlotState* pSrcSlotState =
                &pGpuState->descriptorSetLayoutState.slotState[ idxSlotState ];

            if ( !pSrcSlotState->flag.GetBit( GfxVkGpuState::DescriptorSetLayoutState::SlotState::Flag_IsSet )
                || !( pCommandInfo->pPipelineCache->GetValidSlotMask() & ( 1ULL << idxSlotState ) ) )
            {
                continue;
            }
            VkWriteDescriptorSet* pDstState = &pWriteDescriptorSet[ idxAvailableBinding ];
            const GfxVkGpuState::DescriptorResourceState::ResourceState* pSrcResourceState =
                &pGpuState->descriptorResourceState.resourceState[ idxSlotState ];

            pDstState->dstSet = descriptorSet;
            switch ( static_cast< VkDescriptorType >( pSrcSlotState->type ) )
            {
            case VK_DESCRIPTOR_TYPE_SAMPLER:
            case VK_DESCRIPTOR_TYPE_COMBINED_IMAGE_SAMPLER:
            case VK_DESCRIPTOR_TYPE_SAMPLED_IMAGE:
            case VK_DESCRIPTOR_TYPE_STORAGE_IMAGE:
            case VK_DESCRIPTOR_TYPE_INPUT_ATTACHMENT:
                {
                    pDstState->pImageInfo = &pSrcResourceState->vkImageInfo;
                }
                break;
            case VK_DESCRIPTOR_TYPE_UNIFORM_BUFFER:
            case VK_DESCRIPTOR_TYPE_STORAGE_BUFFER:
                {
                    pDstState->pBufferInfo = &pSrcResourceState->vkBufferInfo;
                }
                break;
            case VK_DESCRIPTOR_TYPE_UNIFORM_BUFFER_DYNAMIC:
            case VK_DESCRIPTOR_TYPE_STORAGE_BUFFER_DYNAMIC:
                {
                    NN_SDK_ASSERT( idxSlotState < GfxVkGpuState::maxDynamicDescriptorSlot );
                    dynamicOffsets[ dynamicOffsetCount++ ] = pCommandInfo->dynamicOffsets[ idxSlotState ];
                    pDstState->pBufferInfo = &pSrcResourceState->vkBufferInfo;
                }
                break;
            case VK_DESCRIPTOR_TYPE_UNIFORM_TEXEL_BUFFER:
            case VK_DESCRIPTOR_TYPE_STORAGE_TEXEL_BUFFER:
                {
                    pDstState->pTexelBufferView = &pSrcResourceState->vkBufferViewInfo;
                }
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }

            ++idxAvailableBinding;
        }

        VkDevice device = CastToVkDispatchableObject< VkDevice >(
            pCommandContext->GetGfxCommandBuffer()->ToData()->pGfxDevice->ToData()->hDevice );
        NN_GFX_CALL_VK_FUNCTION( vkUpdateDescriptorSets( device,
            idxAvailableBinding, pWriteDescriptorSet, 0, NULL ) );
    }

    if ( descriptorSet != VK_NULL_HANDLE )
    {
        NN_GFX_CALL_VK_FUNCTION( vkCmdBindDescriptorSets( pCommandContext->GetVkDrawCommandBuffer(),
            isDraw ? VK_PIPELINE_BIND_POINT_GRAPHICS : VK_PIPELINE_BIND_POINT_COMPUTE,
            layout, 0, 1, &descriptorSet, dynamicOffsetCount, dynamicOffsets ) );
    }
}

void PrepareCommandOutsideRenderPass( CommandContext* pCommandContext ) NN_NOEXCEPT
{
    if ( pCommandContext->IsInsideRenderPass() )
    {
        NN_GFX_CALL_VK_FUNCTION( vkCmdEndRenderPass( pCommandContext->GetVkDrawCommandBuffer() ) );
        pCommandContext->SetInsideRenderPass( false );
    }
}

template< bool IsRenderPass >
void PrepareLayoutTransition( CommandContext* pCommandContext, TextureManageData* pManager, VkImageLayout layout,
    uint32_t baseLayer, uint32_t layerCount, uint32_t baseMip, uint32_t mipCount, VkImageAspectFlags imageAspectFlags )
{
    NN_SDK_ASSERT( Vk::IsLayoutManagementEnabled() );
    {
        GfxVkImageLayoutState* pLayoutTransitionInfo = pCommandContext->GetImageLayoutState( pManager );
        NN_SDK_ASSERT_NOT_NULL( pLayoutTransitionInfo );

        const uint32_t MaxBarrier = 32;
        VkImageMemoryBarrier barriers[ MaxBarrier ];
        VkPipelineStageFlags srcStages = 0;
        VkPipelineStageFlags dstStages = 0;
        uint32_t count = 0;
        pLayoutTransitionInfo->PrepareLayoutTransition( layout,
            &count, &barriers[ 0 ], MaxBarrier,
            baseLayer, layerCount, baseMip, mipCount,
            &srcStages, &dstStages, imageAspectFlags );

        if ( count > 0 )
        {
            // This occurs when a copy/clear operation happens between draws.
            PrepareCommandOutsideRenderPass( pCommandContext );

            NN_GFX_CALL_VK_FUNCTION( vkCmdPipelineBarrier( pCommandContext->GetVkDrawCommandBuffer(), srcStages, dstStages, 0,
                0, NULL, 0, NULL, count, &barriers[ 0 ] ) );
        }

        if ( IsRenderPass )
        {
            // The spec says that finalLayout applies to the "entire image subresource"
            VkImageSubresourceRange range;
            range.baseArrayLayer = baseLayer;
            range.layerCount = layerCount;
            range.baseMipLevel = baseMip;
            range.levelCount = mipCount;
            pLayoutTransitionInfo->SetImageLayout( layout, range );
        }
    }
}

void PrepareCommandInsideRenderPass( CommandContext* pCommandContext, GfxVkCommandInfo* pCommandInfo ) NN_NOEXCEPT
{
    if ( !pCommandContext->IsInsideRenderPass() )
    {
        if ( Vk::IsLayoutManagementEnabled() )
        {
            GfxVkGpuState::RenderTargetState& renderPassRenderTargetState = *pCommandInfo->pPipelineCache->GetRenderTargetState();
            GfxVkGpuState::RenderTargetState& frameBufferRenderTargetState = *pCommandInfo->pFramebufferCache->GetRenderTargetState();

            if ( frameBufferRenderTargetState.flag.GetBit( GfxVkGpuState::RenderTargetState::Flag_DepthAttached ) )
            {
                NN_SDK_REQUIRES( frameBufferRenderTargetState.pDepthStencilTextureManagerData != NULL );
                const GfxVkGpuState::RenderTargetState::AttachmentState& framebufferDepthStencilAttachment = frameBufferRenderTargetState.depthStencilAttachment;
                const GfxVkGpuState::RenderTargetState::AttachmentState& renderPassDepthStencilAttachment = renderPassRenderTargetState.depthStencilAttachment;
                const auto& depthStencilState = pCommandInfo->pPipelineCache->GetConstSizedPipelineState()->depthStencilState;
                const bool isDepthWriteEnabled = depthStencilState.flag.GetBit( GfxVkGpuState::ConstSizedPipelineState::DepthStencilState::Flag_DepthWriteEnable );
                const VkImageLayout depthLayout = ( renderPassDepthStencilAttachment.loadOp == VK_ATTACHMENT_LOAD_OP_CLEAR || isDepthWriteEnabled ) ? Vk::LayoutDepthStencilAttachment : Vk::LayoutTextureDepthStencilRead;
                if ( renderPassDepthStencilAttachment.loadOp == VK_ATTACHMENT_LOAD_OP_LOAD )
                {
                    PrepareLayoutTransition< true >( pCommandContext, frameBufferRenderTargetState.pDepthStencilTextureManagerData, depthLayout,
                        framebufferDepthStencilAttachment.layer, framebufferDepthStencilAttachment.layerCount, framebufferDepthStencilAttachment.mip, 1, VK_IMAGE_ASPECT_DEPTH_BIT |
                        ( Vk::IsDepthOnlyFormat( static_cast<VkFormat>( framebufferDepthStencilAttachment.format ) ) ? 0 : VK_IMAGE_ASPECT_STENCIL_BIT ) );
                }
                else
                {
                    TextureManageData* pManager = frameBufferRenderTargetState.pDepthStencilTextureManagerData;
                    GfxVkImageLayoutState* pLayoutTransitionInfo = pCommandContext->GetImageLayoutState( pManager );
                    NN_SDK_ASSERT_NOT_NULL( pLayoutTransitionInfo );

                    VkImageSubresourceRange range;
                    range.aspectMask = VK_IMAGE_ASPECT_DEPTH_BIT |
                        ( Vk::IsDepthOnlyFormat( static_cast<VkFormat>( framebufferDepthStencilAttachment.format ) ) ? 0 : VK_IMAGE_ASPECT_STENCIL_BIT );
                    range.baseArrayLayer = framebufferDepthStencilAttachment.layer;
                    range.baseMipLevel = framebufferDepthStencilAttachment.mip;
                    range.layerCount = framebufferDepthStencilAttachment.layerCount;
                    range.levelCount = 1;
                    pLayoutTransitionInfo->SetImageLayout( depthLayout, range );
                }
            }

            for ( int32_t idxColor = 0; idxColor < frameBufferRenderTargetState.colorAttachmentCount; idxColor++ )
            {
                if ( frameBufferRenderTargetState.colorTargetView[ idxColor ] != 0 )
                {
                    NN_SDK_REQUIRES( frameBufferRenderTargetState.pColorTextureManagerData[ idxColor ] != NULL );
                    GfxVkGpuState::RenderTargetState::AttachmentState& colorAttachment = frameBufferRenderTargetState.colorAttachment[ idxColor ];
                    if ( renderPassRenderTargetState.colorAttachment[ idxColor ].loadOp == VK_ATTACHMENT_LOAD_OP_LOAD )
                    {
                        PrepareLayoutTransition< true >( pCommandContext, frameBufferRenderTargetState.pColorTextureManagerData[ idxColor ], Vk::LayoutColorAttachment,
                            colorAttachment.layer, colorAttachment.layerCount, colorAttachment.mip, 1, VK_IMAGE_ASPECT_COLOR_BIT );
                    }
                    else
                    {
                        TextureManageData* pManager = frameBufferRenderTargetState.pColorTextureManagerData[ idxColor ];
                        GfxVkImageLayoutState* pLayoutTransitionInfo = pCommandContext->GetImageLayoutState( pManager );
                        NN_SDK_ASSERT_NOT_NULL( pLayoutTransitionInfo );

                        VkImageSubresourceRange range;
                        range.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
                        range.baseArrayLayer = colorAttachment.layer;
                        range.baseMipLevel = colorAttachment.mip;
                        range.layerCount = colorAttachment.layerCount;
                        range.levelCount = 1;
                        pLayoutTransitionInfo->SetImageLayout( Vk::LayoutColorAttachment, range );
                    }
                }
            }
        }

        GfxVkGpuState::RenderTargetState& renderTargetState = *pCommandInfo->pFramebufferCache->GetRenderTargetState();
        const int clearValueCount = GfxVkGpuState::maxColorAttachments + 1;
        VkClearValue clearValues[ clearValueCount ];

        GfxVkGpuState::RenderPassState& renderPassState = pCommandContext->GetGpuState()->renderPassState;

        for ( int32_t idxColor = 0; idxColor < renderTargetState.colorAttachmentCount; ++idxColor )
        {
            VkClearColorValue& clearColor = clearValues[ idxColor ].color;
            for ( int idx = 0; idx < 4; ++idx )
            {
                clearColor.float32[ idx ] = renderPassState.clearColors[ idxColor ][ idx ];
            }
        }

        VkClearDepthStencilValue& clearDepthStencil = clearValues[ renderTargetState.colorAttachmentCount ].depthStencil;
        clearDepthStencil.depth = renderPassState.clearDepth;
        clearDepthStencil.stencil = renderPassState.clearStencil;

        VkRenderPassBeginInfo beginInfo;
        beginInfo.sType = VK_STRUCTURE_TYPE_RENDER_PASS_BEGIN_INFO;
        beginInfo.pNext = NULL;
        beginInfo.renderPass = pCommandInfo->pPipelineCache->GetRenderPass();
        beginInfo.framebuffer = pCommandInfo->pFramebufferCache->GetFramebuffer();
        beginInfo.renderArea.offset.x = 0;
        beginInfo.renderArea.offset.y = 0;
        beginInfo.renderArea.extent.width = renderTargetState.width;
        beginInfo.renderArea.extent.height = renderTargetState.height;
        beginInfo.clearValueCount = clearValueCount;
        beginInfo.pClearValues = clearValues;

        NN_GFX_CALL_VK_FUNCTION( vkCmdBeginRenderPass(
            pCommandContext->GetVkDrawCommandBuffer(), &beginInfo, VK_SUBPASS_CONTENTS_INLINE ) );

        pCommandContext->SetInsideRenderPass( true );
    }
}

void PrepareDispatch( CommandContext* pCommandContext ) NN_NOEXCEPT
{
    VkManager* pVkManager = pCommandContext->GetVkManager();

    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkGpuState* pGpuState = pCommandContext->GetGpuState();

        GfxVkCommandInfo* pCommandInfo = pVkManager->UpdateCommandInfo< false >(
            *pGpuState, pCommandContext->GetCurrentCommand(), pCommandContext->IsCurrentStartCommand() );
        NN_SDK_ASSERT_NOT_NULL( pCommandInfo );
        NN_UNUSED( pCommandInfo );

        if ( pCommandInfo->updatedState.GetBit( GfxVkCommandInfo::UpdatedStateFlag_SetUniform ) )
        {
            pVkManager->AdvanceUniformBufferOffset();
        }
    }
    else if ( pCommandContext->IsCurrentAssignedCommand() )
    {
        PrepareCommandOutsideRenderPass( pCommandContext );

        GfxVkCommandInfo* pCommandInfo = pVkManager->GetUpdatedCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo->info.GetBit( GfxVkCommandInfo::CommandInfoFlag_IsStateUpdated ) ||
            pCommandContext->IsCurrentAssignedFirstCommand() )
        {
            GfxVkCommandInfo* pPrevCommandInfo = pCommandContext->IsCurrentAssignedFirstCommand()
                ? NULL : pVkManager->GetCommandInfo( pCommandContext->GetCurrentCommand() - 1 );

            BindPipeline( pCommandContext, pCommandInfo, pPrevCommandInfo, false );

            BindAndUpdateDescriptorSet( pCommandContext, pCommandInfo, pPrevCommandInfo, false );
        }
    }
}

void PrepareDraw( CommandContext* pCommandContext, VkEnum primitiveTopology ) NN_NOEXCEPT
{
    VkManager* pVkManager = pCommandContext->GetVkManager();

    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkGpuState* pGpuState = pCommandContext->GetGpuState();
        GfxVkGpuState::ConstSizedPipelineState::InputAssemblyState& inputAssemblyState =
            pGpuState->constSizedPipelineState.inputAssemblyState;
        GfxVkGpuState::ConstSizedPipelineState::DepthStencilState& depthStencilState =
            pGpuState->constSizedPipelineState.depthStencilState;
        GfxVkGpuState::RenderTargetState& renderTargetState = pGpuState->renderTargetState;

        GfxVkCommandInfo* pCommandInfo = pVkManager->GetCommandInfo( pCommandContext->GetCurrentCommand() );
        NN_SDK_ASSERT_NOT_NULL( pCommandInfo );

        const int isDepthUsedMask =
            ( 1 << GfxVkGpuState::ConstSizedPipelineState::DepthStencilState::Flag_DepthTestEnable ) |
            ( 1 << GfxVkGpuState::ConstSizedPipelineState::DepthStencilState::Flag_DepthWriteEnable ) |
            ( 1 << GfxVkGpuState::ConstSizedPipelineState::DepthStencilState::Flag_StencilTestEnable );

        bool isDepthAttached =
            ( renderTargetState.depthStencilTargetView != 0 ) && depthStencilState.flag.GetMaskedBits( isDepthUsedMask );

        if ( renderTargetState.flag.GetBit( GfxVkGpuState::RenderTargetState::Flag_DepthAttached ) != isDepthAttached )
        {
            pCommandInfo->info.SetBit( GfxVkCommandInfo::CommandInfoFlag_ForceNewRenderPass, true );
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_SetRenderTarget, true );
            renderTargetState.flag.SetBit( GfxVkGpuState::RenderTargetState::Flag_DepthAttached, isDepthAttached );
        }

        if ( inputAssemblyState.primitiveTopology != primitiveTopology )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_SetInputAssemblyState, true );
            inputAssemblyState.primitiveTopology = primitiveTopology;
        }

        pVkManager->UpdateCommandInfo< true >(
            *pGpuState, pCommandContext->GetCurrentCommand(), pCommandContext->IsCurrentStartCommand() );

        if ( pCommandInfo->updatedState.GetBit( GfxVkCommandInfo::UpdatedStateFlag_SetUniform ) )
        {
            pVkManager->AdvanceUniformBufferOffset();
        }

        for ( int32_t idxColor = 0; idxColor < renderTargetState.colorAttachmentCount; ++idxColor )
        {
            renderTargetState.colorAttachment[ idxColor ].loadOp = VK_ATTACHMENT_LOAD_OP_LOAD;
        }

        renderTargetState.depthStencilAttachment.loadOp = VK_ATTACHMENT_LOAD_OP_LOAD;
    }
    else if ( pCommandContext->IsCurrentAssignedCommand() )
    {
        GfxVkCommandInfo* pCommandInfo = pVkManager->GetUpdatedCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo->info.GetBit( GfxVkCommandInfo::CommandInfoFlag_ForceNewRenderPass ) )
        {
            PrepareCommandOutsideRenderPass( pCommandContext );
        }

        PrepareCommandInsideRenderPass( pCommandContext, pCommandInfo );

        if ( pCommandInfo->info.GetBit( GfxVkCommandInfo::CommandInfoFlag_IsStateUpdated ) ||
            pCommandContext->IsCurrentAssignedFirstCommand() )
        {
            GfxVkCommandInfo* pPrevCommandInfo = pCommandContext->IsCurrentAssignedFirstCommand()
                ? NULL : pVkManager->GetCommandInfo( pCommandContext->GetCurrentCommand() - 1 );

            BindPipeline( pCommandContext, pCommandInfo, pPrevCommandInfo, true );

            BindAndUpdateDescriptorSet( pCommandContext, pCommandInfo, pPrevCommandInfo, true );
        }
    }
}

void ClearColorImage( CommandContext* pCommandContext, TextureManageData* pManager, VkImage image,
    VkClearColorValue* clearColor, uint32_t baseLayer, uint32_t layerCount, uint32_t baseMip, uint32_t mipCount )
{
    if ( Vk::IsLayoutManagementEnabled() )
    {
        PrepareLayoutTransition< false >( pCommandContext, pManager, Vk::LayoutClearImage,
            baseLayer, layerCount, baseMip, mipCount, VK_IMAGE_ASPECT_COLOR_BIT );
    }
    else
    {
        VkImageMemoryBarrier imageMemoryBarrier;
        imageMemoryBarrier.sType = VK_STRUCTURE_TYPE_IMAGE_MEMORY_BARRIER;
        imageMemoryBarrier.pNext = NULL;
        imageMemoryBarrier.oldLayout = VK_IMAGE_LAYOUT_UNDEFINED;
        imageMemoryBarrier.newLayout = VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL;
        imageMemoryBarrier.srcAccessMask = 0;
        imageMemoryBarrier.dstAccessMask = Vk::GetLayoutAccessFlags( VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL );
        imageMemoryBarrier.srcQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
        imageMemoryBarrier.dstQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
        imageMemoryBarrier.image = image;
        imageMemoryBarrier.subresourceRange.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
        imageMemoryBarrier.subresourceRange.baseArrayLayer = baseLayer;
        imageMemoryBarrier.subresourceRange.layerCount = layerCount;
        imageMemoryBarrier.subresourceRange.baseMipLevel = baseMip;
        imageMemoryBarrier.subresourceRange.levelCount = mipCount;

        NN_GFX_CALL_VK_FUNCTION( vkCmdPipelineBarrier( pCommandContext->GetVkDrawCommandBuffer(),
            VK_PIPELINE_STAGE_COLOR_ATTACHMENT_OUTPUT_BIT, VK_PIPELINE_STAGE_TRANSFER_BIT, 0,
            0, NULL, 0, NULL, 1, &imageMemoryBarrier ) );
    }

    VkImageSubresourceRange range;
    range.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
    range.baseArrayLayer = baseLayer;
    range.baseMipLevel = baseMip;
    range.layerCount = layerCount;
    range.levelCount = mipCount;

    NN_GFX_CALL_VK_FUNCTION( vkCmdClearColorImage( pCommandContext->GetVkDrawCommandBuffer(),
        image, Vk::LayoutClearImage, clearColor, 1, &range ) );

    if ( !Vk::IsLayoutManagementEnabled() )
    {
        VkImageMemoryBarrier imageMemoryBarrier;
        imageMemoryBarrier.sType = VK_STRUCTURE_TYPE_IMAGE_MEMORY_BARRIER;
        imageMemoryBarrier.pNext = NULL;
        imageMemoryBarrier.oldLayout = VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL;
        imageMemoryBarrier.newLayout = VK_IMAGE_LAYOUT_GENERAL;
        imageMemoryBarrier.srcAccessMask = Vk::GetLayoutAccessFlags( VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL );
        imageMemoryBarrier.dstAccessMask = Vk::GetLayoutAccessFlags( VK_IMAGE_LAYOUT_GENERAL );
        imageMemoryBarrier.srcQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
        imageMemoryBarrier.dstQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
        imageMemoryBarrier.image = image;
        imageMemoryBarrier.subresourceRange.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
        imageMemoryBarrier.subresourceRange.baseArrayLayer = baseLayer;
        imageMemoryBarrier.subresourceRange.layerCount = layerCount;
        imageMemoryBarrier.subresourceRange.baseMipLevel = baseMip;
        imageMemoryBarrier.subresourceRange.levelCount = mipCount;

        NN_GFX_CALL_VK_FUNCTION( vkCmdPipelineBarrier( pCommandContext->GetVkDrawCommandBuffer(),
            VK_PIPELINE_STAGE_TRANSFER_BIT, VK_PIPELINE_STAGE_FRAGMENT_SHADER_BIT, 0,
            0, NULL, 0, NULL, 1, &imageMemoryBarrier ) );
    }
}

void ClearDepthStencilImage( CommandContext* pCommandContext, TextureManageData* pManager, VkImage image,
    VkClearDepthStencilValue* clearDepthStencil, uint32_t baseLayer, uint32_t layerCount, uint32_t baseMipLevel, uint32_t mipLevelCount, VkImageAspectFlags aspectMask )
{
    if ( Vk::IsLayoutManagementEnabled() )
    {
        PrepareLayoutTransition< false >( pCommandContext, pManager, Vk::LayoutClearImage,
            baseLayer, layerCount, baseMipLevel, mipLevelCount, aspectMask );
    }
    else
    {
        VkImageMemoryBarrier imageMemoryBarrier;
        imageMemoryBarrier.sType = VK_STRUCTURE_TYPE_IMAGE_MEMORY_BARRIER;
        imageMemoryBarrier.pNext = NULL;
        imageMemoryBarrier.oldLayout = VK_IMAGE_LAYOUT_UNDEFINED;
        imageMemoryBarrier.newLayout = VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL;
        imageMemoryBarrier.srcAccessMask = 0;
        imageMemoryBarrier.dstAccessMask = Vk::GetLayoutAccessFlags( VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL );
        imageMemoryBarrier.srcQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
        imageMemoryBarrier.dstQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
        imageMemoryBarrier.image = image;
        imageMemoryBarrier.subresourceRange.aspectMask = aspectMask;
        imageMemoryBarrier.subresourceRange.baseArrayLayer = baseLayer;
        imageMemoryBarrier.subresourceRange.layerCount = layerCount;
        imageMemoryBarrier.subresourceRange.baseMipLevel = baseMipLevel;
        imageMemoryBarrier.subresourceRange.levelCount = mipLevelCount;

        NN_GFX_CALL_VK_FUNCTION( vkCmdPipelineBarrier( pCommandContext->GetVkDrawCommandBuffer(),
            VK_PIPELINE_STAGE_LATE_FRAGMENT_TESTS_BIT, VK_PIPELINE_STAGE_TRANSFER_BIT, 0,
            0, NULL, 0, NULL, 1, &imageMemoryBarrier ) );
    }

    VkImageSubresourceRange range;
    range.aspectMask = aspectMask;
    range.baseArrayLayer = baseLayer;
    range.baseMipLevel = baseMipLevel;
    range.layerCount = layerCount;
    range.levelCount = mipLevelCount;

    NN_GFX_CALL_VK_FUNCTION( vkCmdClearDepthStencilImage( pCommandContext->GetVkDrawCommandBuffer(),
        image, Vk::LayoutClearImage, clearDepthStencil, 1, &range ) );

    if ( !Vk::IsLayoutManagementEnabled() )
    {
        VkImageMemoryBarrier imageMemoryBarrier;
        imageMemoryBarrier.sType = VK_STRUCTURE_TYPE_IMAGE_MEMORY_BARRIER;
        imageMemoryBarrier.pNext = NULL;
        imageMemoryBarrier.oldLayout = VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL;
        imageMemoryBarrier.newLayout = VK_IMAGE_LAYOUT_GENERAL;
        imageMemoryBarrier.srcAccessMask = Vk::GetLayoutAccessFlags( VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL );
        imageMemoryBarrier.dstAccessMask = Vk::GetLayoutAccessFlags( VK_IMAGE_LAYOUT_GENERAL );
        imageMemoryBarrier.srcQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
        imageMemoryBarrier.dstQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
        imageMemoryBarrier.image = image;
        imageMemoryBarrier.subresourceRange.aspectMask = aspectMask;
        imageMemoryBarrier.subresourceRange.baseArrayLayer = baseLayer;
        imageMemoryBarrier.subresourceRange.layerCount = layerCount;
        imageMemoryBarrier.subresourceRange.baseMipLevel = baseMipLevel;
        imageMemoryBarrier.subresourceRange.levelCount = mipLevelCount;

        NN_GFX_CALL_VK_FUNCTION( vkCmdPipelineBarrier( pCommandContext->GetVkDrawCommandBuffer(),
            VK_PIPELINE_STAGE_TRANSFER_BIT, VK_PIPELINE_STAGE_EARLY_FRAGMENT_TESTS_BIT, 0,
            0, NULL, 0, NULL, 1, &imageMemoryBarrier ) );
    }
}

}

template<>
void VkCommandProc< VkDispatchParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    PrepareDispatch( pCommandContext );

    if ( !pCommandContext->IsContextTypeStateUpdater() && pCommandContext->IsCurrentAssignedCommand() )
    {
        const VkDispatchParam& param = *static_cast< const VkDispatchParam* >( pParam );

        NN_GFX_CALL_VK_FUNCTION( vkCmdDispatch( pCommandContext->GetVkDrawCommandBuffer(),
            param.groupCountX, param.groupCountY, param.groupCountZ ) );
    }

    pCommandContext->IncrementCurrentCommand();
}

template<>
void VkCommandProc< VkDispatchIndirectParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    PrepareDispatch( pCommandContext );

    if ( !pCommandContext->IsContextTypeStateUpdater() && pCommandContext->IsCurrentAssignedCommand() )
    {
        const VkDispatchIndirectParam& param = *static_cast< const VkDispatchIndirectParam* >( pParam );

        VkBuffer indirectBuffer = CastToVkNonDispatchableObject< VkBuffer >( param.buffer );

        NN_GFX_CALL_VK_FUNCTION( vkCmdDispatchIndirect(
            pCommandContext->GetVkDrawCommandBuffer(), indirectBuffer, param.offset ) );
    }

    pCommandContext->IncrementCurrentCommand();
}

template<>
void VkCommandProc< VkDrawParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    const VkDrawParam& param = *static_cast< const VkDrawParam* >( pParam );

    PrepareDraw( pCommandContext, param.topology );

    if ( !pCommandContext->IsContextTypeStateUpdater() && pCommandContext->IsCurrentAssignedCommand() )
    {
        NN_GFX_CALL_VK_FUNCTION( vkCmdDraw( pCommandContext->GetVkDrawCommandBuffer(), param.vertexCount,
            param.instanceCount, param.firstVertex, param.firstInstance ) );
    }

    pCommandContext->IncrementCurrentCommand();
}

template<>
void VkCommandProc< VkDrawIndexedParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    const VkDrawIndexedParam& param = *static_cast< const VkDrawIndexedParam* >( pParam );

    PrepareDraw( pCommandContext, param.topology );

    if ( !pCommandContext->IsContextTypeStateUpdater() && pCommandContext->IsCurrentAssignedCommand() )
    {
        VkBuffer indexBuffer = CastToVkNonDispatchableObject< VkBuffer >( param.indexBuffer );

        NN_GFX_CALL_VK_FUNCTION( vkCmdBindIndexBuffer( pCommandContext->GetVkDrawCommandBuffer(),
            indexBuffer, param.indexOffset, static_cast< VkIndexType >( param.indexType ) ) );

        NN_GFX_CALL_VK_FUNCTION( vkCmdDrawIndexed( pCommandContext->GetVkDrawCommandBuffer(),
            param.indexCount, param.instanceCount, 0, param.vertexOffset, param.firstInstance ) );
    }

    pCommandContext->IncrementCurrentCommand();
}

template<>
void VkCommandProc< VkDrawIndirectParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    const VkDrawIndirectParam& param = *static_cast< const VkDrawIndirectParam* >( pParam );

    PrepareDraw( pCommandContext, param.topology );

    if ( !pCommandContext->IsContextTypeStateUpdater() && pCommandContext->IsCurrentAssignedCommand() )
    {
        VkBuffer indirectBuffer = CastToVkNonDispatchableObject< VkBuffer >( param.buffer );

        NN_GFX_CALL_VK_FUNCTION( vkCmdDrawIndirect( pCommandContext->GetVkDrawCommandBuffer(), indirectBuffer, param.offset, 1, 0 ) );
    }

    pCommandContext->IncrementCurrentCommand();
}

template<>
void VkCommandProc< VkDrawIndexedIndirectParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    const VkDrawIndexedIndirectParam& param = *static_cast< const VkDrawIndexedIndirectParam* >( pParam );

    PrepareDraw( pCommandContext, param.topology );

    if ( !pCommandContext->IsContextTypeStateUpdater() && pCommandContext->IsCurrentAssignedCommand() )
    {
        VkBuffer indexBuffer = CastToVkNonDispatchableObject< VkBuffer >( param.indexBuffer );
        NN_GFX_CALL_VK_FUNCTION( vkCmdBindIndexBuffer( pCommandContext->GetVkDrawCommandBuffer(),
            indexBuffer, param.indexOffset, static_cast< VkIndexType >( param.indexType ) ) );

        VkBuffer indirectBuffer = CastToVkNonDispatchableObject< VkBuffer >( param.indirectBuffer );
        NN_GFX_CALL_VK_FUNCTION( vkCmdDrawIndexedIndirect( pCommandContext->GetVkDrawCommandBuffer(),
            indirectBuffer, param.indirectOffset, 1, 0 ) );
    }

    pCommandContext->IncrementCurrentCommand();
}

template<>
void VkCommandProc< VkSetRenderTargetsParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkGpuState* pGpuState = pCommandContext->GetGpuState();

        const VkSetRenderTargetsParam& param =
            *static_cast< const VkSetRenderTargetsParam* >( pParam );

        GfxVkGpuState::RenderTargetState& renderTargetState =
            pGpuState->renderTargetState;

        renderTargetState.colorAttachmentCount = param.colorAttachmentCount;
        renderTargetState.width = param.width;
        renderTargetState.height = param.height;
        renderTargetState.layers = param.layers;

        renderTargetState.depthStencilTargetView = param.depthAttachment.imageView;
        renderTargetState.depthStencilAttachment.format = param.depthAttachment.format;
        renderTargetState.depthStencilAttachment.samples = param.depthAttachment.samples;
        renderTargetState.depthStencilAttachment.layer = param.depthAttachment.layer;
        renderTargetState.depthStencilAttachment.layerCount = param.depthAttachment.layerCount;
        renderTargetState.depthStencilAttachment.mip = param.depthAttachment.mip;
        renderTargetState.depthStencilAttachment.loadOp = VK_ATTACHMENT_LOAD_OP_LOAD;

        renderTargetState.pDepthStencilTextureManagerData = param.depthAttachment.pManager;

        int idxColor;
        for ( idxColor = 0; idxColor < renderTargetState.colorAttachmentCount; ++idxColor )
        {
            renderTargetState.colorTargetView[ idxColor ] = param.colorAttachment[ idxColor ].imageView;
            renderTargetState.colorAttachment[ idxColor ].format = param.colorAttachment[ idxColor ].format;
            renderTargetState.colorAttachment[ idxColor ].samples = param.colorAttachment[ idxColor ].samples;
            renderTargetState.colorAttachment[ idxColor ].layer = param.colorAttachment[ idxColor ].layer;
            renderTargetState.colorAttachment[ idxColor ].layerCount = param.colorAttachment[ idxColor ].layerCount;
            renderTargetState.colorAttachment[ idxColor ].mip = param.colorAttachment[ idxColor ].mip;
            renderTargetState.colorAttachment[ idxColor ].loadOp = VK_ATTACHMENT_LOAD_OP_LOAD;

            renderTargetState.pColorTextureManagerData[ idxColor ] = param.colorAttachment[ idxColor ].pManager;
        }
        for ( ; idxColor < GfxVkGpuState::maxColorAttachments; ++idxColor )
        {
            renderTargetState.colorTargetView[ idxColor ] = 0;
            renderTargetState.pColorTextureManagerData[ idxColor ] = NULL;
            renderTargetState.colorAttachment[ idxColor ].format = 0;
            renderTargetState.colorAttachment[ idxColor ].samples = 0;
            renderTargetState.colorAttachment[ idxColor ].layer = 0;
            renderTargetState.colorAttachment[ idxColor ].layerCount = 0;
            renderTargetState.colorAttachment[ idxColor ].mip = 0;
            renderTargetState.colorAttachment[ idxColor ].loadOp = VK_ATTACHMENT_LOAD_OP_LOAD;
        }

        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_SetRenderTarget, true );
        }
    }
    else if ( pCommandContext->IsCurrentAssignedCommand() )
    {
        PrepareCommandOutsideRenderPass( pCommandContext );
    }
}

template<>
void VkCommandProc< VkSetVertexBufferParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    if ( !pCommandContext->IsContextTypeStateUpdater() )
    {
        const VkSetVertexBufferParam& param =
            *static_cast< const VkSetVertexBufferParam* >( pParam );

        VkBuffer vertexBuffer = CastToVkNonDispatchableObject< VkBuffer >( param.buffer );
        VkDeviceSize offset = param.offset;
        NN_GFX_CALL_VK_FUNCTION( vkCmdBindVertexBuffers( pCommandContext->GetVkDrawCommandBuffer(),
            param.bufferIndex, 1, &vertexBuffer, &offset ) );
    }
}

template<>
void VkCommandProc< ViewportScissorStateImplData< ApiVariationVk1 > >(
    CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    const ViewportScissorStateImplData< ApiVariationVk1 >& param =
        *static_cast< const ViewportScissorStateImplData< ApiVariationVk1 >* >( pParam );

    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkGpuState* pGpuState = pCommandContext->GetGpuState();

        GfxVkGpuState::ConstSizedPipelineState::ViewportState& viewportState =
            pGpuState->constSizedPipelineState.viewportState;
        viewportState.viewportCount = param.viewportCount;

        #if defined( NN_SDK_BUILD_DEBUG )
        GfxVkGpuState::DynamicState::ViewportScissorState& dynamicViewportScissorState =
            pGpuState->dynamicState.viewportScissorState;
        memcpy( &dynamicViewportScissorState.viewport[ 0 ][ 0 ], param.viewport, sizeof( float ) * 6 );
        memcpy( &dynamicViewportScissorState.scissor[ 0 ][ 0 ], param.scissor, sizeof( int32_t ) * 4 );
        if( param.viewportCount > 1 )
        {
            nn::util::ConstBytePtr ptr( &param.pWorkMemory );
            int extraViewportCount = param.viewportCount - 1;
            const float* pViewportArray = ptr.Get< float >();
            const int32_t* pScissorArray = ptr.Advance( sizeof( float ) * 6 * extraViewportCount ).Get< int32_t >();

            memcpy( &dynamicViewportScissorState.viewport[ 1 ][ 0 ],
                pViewportArray, sizeof( float ) * 6 * extraViewportCount );
            memcpy( &dynamicViewportScissorState.scissor[ 1 ][ 0 ],
                pScissorArray, sizeof( int32_t ) * 4 * extraViewportCount );
        }
        #endif
    }
    else
    {
        NN_GFX_CALL_VK_FUNCTION( vkCmdSetViewport( pCommandContext->GetVkDrawCommandBuffer(), 0, 1,
            reinterpret_cast< const VkViewport* >( param.viewport ) ) );

        if ( param.flag.GetBit( ViewportScissorStateImplData< ApiVariationVk1 >::Flag_ScissorEnable ) )
        {
            NN_GFX_CALL_VK_FUNCTION( vkCmdSetScissor( pCommandContext->GetVkDrawCommandBuffer(), 0, 1,
                reinterpret_cast< const VkRect2D* >( param.scissor ) ) );
        }
        else
        {
            // Vulkan always enables scissoring, so set scissor area same as viewport when scissoring is disabled.
            VkRect2D rect;
            rect.offset.x = static_cast< int32_t >( param.viewport[ 0 ] );
            rect.offset.y = static_cast< int32_t >( param.viewport[ 1 ] );
            rect.extent.width = static_cast< uint32_t >( param.viewport[ 2 ] );
            rect.extent.height = static_cast< uint32_t >( param.viewport[ 3 ] );
            NN_GFX_CALL_VK_FUNCTION( vkCmdSetScissor( pCommandContext->GetVkDrawCommandBuffer(), 0, 1, &rect ) );
        }

        if ( param.viewportCount > 1 )
        {
            nn::util::ConstBytePtr ptr( &param.pWorkMemory );
            int extraViewportCount = param.viewportCount - 1;
            const VkViewport* pViewportArray = ptr.Get< VkViewport >();
            NN_GFX_CALL_VK_FUNCTION( vkCmdSetViewport(
                pCommandContext->GetVkDrawCommandBuffer(), 1, extraViewportCount, pViewportArray ) );

            if ( param.flag.GetBit( ViewportScissorStateImplData< ApiVariationVk1 >::Flag_ScissorEnable ) )
            {
                const VkRect2D* pScissorArray = ptr.Advance(
                    sizeof( float ) * 6 * extraViewportCount ).Get< VkRect2D >();
                NN_GFX_CALL_VK_FUNCTION( vkCmdSetScissor(
                    pCommandContext->GetVkDrawCommandBuffer(), 1, extraViewportCount, pScissorArray ) );
            }
            else
            {
                // Vulkan always enables scissoring, so set scissor area same as viewport when scissoring is disabled.
                for ( int idxViewport = 0; idxViewport < extraViewportCount; ++idxViewport )
                {
                    VkRect2D rect;
                    rect.offset.x = static_cast< int32_t >( pViewportArray[ idxViewport ].x );
                    rect.offset.y = static_cast< int32_t >( pViewportArray[ idxViewport ].y );
                    rect.extent.width = static_cast< uint32_t >( pViewportArray[ idxViewport ].width );
                    rect.extent.height = static_cast< uint32_t >( pViewportArray[ idxViewport ].height );

                    NN_GFX_CALL_VK_FUNCTION( vkCmdSetScissor( pCommandContext->GetVkDrawCommandBuffer(), idxViewport + 1, 1, &rect ) );
                }
            }
        }
    }
}

template<>
void VkCommandProc< VkCopyBufferParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_OutsideRenderPass, true );
        }
    }
    else if ( pCommandContext->IsCurrentAssignedCommand() )
    {
        PrepareCommandOutsideRenderPass( pCommandContext );

        const VkCopyBufferParam& param = *static_cast< const VkCopyBufferParam* >( pParam );

        VkBuffer dstBuffer = CastToVkNonDispatchableObject< VkBuffer >( param.dstBuffer );
        VkBuffer srcBuffer = CastToVkNonDispatchableObject< VkBuffer >( param.srcBuffer );
        VkBufferCopy region;
        region.srcOffset = param.srcOffset;
        region.dstOffset = param.dstOffset;
        region.size = param.size;

        NN_GFX_CALL_VK_FUNCTION( vkCmdCopyBuffer( pCommandContext->GetVkDrawCommandBuffer(),
            srcBuffer, dstBuffer, 1, &region ) );
    }
}

template<>
void VkCommandProc< VkCopyImageParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_OutsideRenderPass, true );
        }
    }
    else if ( pCommandContext->IsCurrentAssignedCommand() )
    {
        const VkCopyImageParam& param = *static_cast< const VkCopyImageParam* >( pParam );

        PrepareCommandOutsideRenderPass( pCommandContext );

        if ( Vk::IsLayoutManagementEnabled() )
        {
            PrepareLayoutTransition< false >( pCommandContext, param.pSrcManager, Vk::LayoutCopyImageSrc,
                param.srcBaseLayer, param.layerCount, param.srcMipLevel, 1, param.srcAspect );

            PrepareLayoutTransition< false >( pCommandContext, param.pDstManager, Vk::LayoutCopyImageDst,
                param.dstBaseLayer, param.layerCount, param.dstMipLevel, 1, param.dstAspect );
        }

        VkImage dstImage = CastToVkNonDispatchableObject< VkImage >( param.dstImage );
        VkImage srcImage = CastToVkNonDispatchableObject< VkImage >( param.srcImage );
        VkImageCopy region;
        region.srcSubresource.aspectMask = param.srcAspect;
        region.srcSubresource.baseArrayLayer = param.srcBaseLayer;
        region.srcSubresource.mipLevel = param.srcMipLevel;
        region.srcSubresource.layerCount = param.layerCount;
        region.srcOffset.x = param.srcX;
        region.srcOffset.y = param.srcY;
        region.srcOffset.z = param.srcZ;
        region.dstSubresource.aspectMask = param.dstAspect;
        region.dstSubresource.baseArrayLayer = param.dstBaseLayer;
        region.dstSubresource.mipLevel = param.dstMipLevel;
        region.dstSubresource.layerCount = param.layerCount;
        region.dstOffset.x = param.dstX;
        region.dstOffset.y = param.dstY;
        region.dstOffset.z = param.dstZ;
        region.extent.width = param.width;
        region.extent.height = param.height;
        region.extent.depth = param.depth;

        NN_GFX_CALL_VK_FUNCTION( vkCmdCopyImage( pCommandContext->GetVkDrawCommandBuffer(),
            srcImage, Vk::LayoutCopyImageSrc, dstImage, Vk::LayoutCopyImageDst, 1, &region ) );
    }
}

template<>
void VkCommandProc< VkCopyBufferToImageParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_OutsideRenderPass, true );
        }
    }
    else if ( pCommandContext->IsCurrentAssignedCommand() )
    {
        const VkBufferImageCopyParam& param = ( *static_cast< const VkCopyBufferToImageParam* >( pParam ) ).param;

        PrepareCommandOutsideRenderPass( pCommandContext );

        if ( Vk::IsLayoutManagementEnabled() )
        {
            PrepareLayoutTransition< false >( pCommandContext, param.pManager, Vk::LayoutCopyImageDst,
                param.imageBaseLayer, param.imageLayerCount, param.imageMipLevel, 1, param.imageAspect );
        }

        VkBuffer srcBuffer = CastToVkNonDispatchableObject< VkBuffer >( param.buffer );
        VkImage dstImage = CastToVkNonDispatchableObject< VkImage >( param.image );
        VkBufferImageCopy region;

        region.bufferOffset = param.bufferOffset;
        region.bufferRowLength = param.bufferRowLength;
        region.bufferImageHeight = param.bufferImageHeight;
        region.imageSubresource.aspectMask = param.imageAspect;
        region.imageSubresource.baseArrayLayer = param.imageBaseLayer;
        region.imageSubresource.layerCount = param.imageLayerCount;
        region.imageSubresource.mipLevel = param.imageMipLevel;
        region.imageExtent.width = param.imageWidth;
        region.imageExtent.height = param.imageHeight;
        region.imageExtent.depth = param.imageDepth;
        region.imageOffset.x = param.imageOffsetX;
        region.imageOffset.y = param.imageOffsetY;
        region.imageOffset.z = param.imageOffsetZ;

        NN_GFX_CALL_VK_FUNCTION( vkCmdCopyBufferToImage( pCommandContext->GetVkDrawCommandBuffer(),
            srcBuffer, dstImage, Vk::LayoutCopyImageDst, 1, &region ) );
    }
}

template<>
void VkCommandProc< VkCopyImageToBufferParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_OutsideRenderPass, true );
        }
    }
    else if ( pCommandContext->IsCurrentAssignedCommand() )
    {
        const VkBufferImageCopyParam& param = ( *static_cast< const VkCopyImageToBufferParam* >( pParam ) ).param;

        PrepareCommandOutsideRenderPass( pCommandContext );

        if ( Vk::IsLayoutManagementEnabled() )
        {
            PrepareLayoutTransition< false >( pCommandContext, param.pManager, Vk::LayoutCopyImageSrc,
                param.imageBaseLayer, param.imageLayerCount, param.imageMipLevel, 1, param.imageAspect );
        }

        VkBuffer dstBuffer = CastToVkNonDispatchableObject< VkBuffer >( param.buffer );
        VkImage srcImage = CastToVkNonDispatchableObject< VkImage >( param.image );
        VkBufferImageCopy region;

        region.bufferOffset = param.bufferOffset;
        region.bufferRowLength = param.bufferRowLength;
        region.bufferImageHeight = param.bufferImageHeight;
        region.imageSubresource.aspectMask = param.imageAspect;
        region.imageSubresource.baseArrayLayer = param.imageBaseLayer;
        region.imageSubresource.layerCount = param.imageLayerCount;
        region.imageSubresource.mipLevel = param.imageMipLevel;
        region.imageExtent.width = param.imageWidth;
        region.imageExtent.height = param.imageHeight;
        region.imageExtent.depth = param.imageDepth;
        region.imageOffset.x = param.imageOffsetX;
        region.imageOffset.y = param.imageOffsetY;
        region.imageOffset.z = param.imageOffsetZ;

        NN_GFX_CALL_VK_FUNCTION( vkCmdCopyImageToBuffer( pCommandContext->GetVkDrawCommandBuffer(),
            srcImage, Vk::LayoutCopyImageSrc, dstBuffer, 1, &region ) );
    }
}

template<>
void VkCommandProc< VkBlitImageParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_OutsideRenderPass, true );
        }
    }
    else if ( pCommandContext->IsCurrentAssignedCommand() )
    {
        const VkBlitImageParam& param = *static_cast< const VkBlitImageParam* >( pParam );

        PrepareCommandOutsideRenderPass( pCommandContext );

        if ( Vk::IsLayoutManagementEnabled() )
        {
            PrepareLayoutTransition< false >( pCommandContext, param.pSrcManager, Vk::LayoutBlitImageSrc,
                param.srcBaseLayer, param.srcLayerCount, param.srcMipLevel, 1, param.srcAspect );

            PrepareLayoutTransition< false >( pCommandContext, param.pDstManager, Vk::LayoutBlitImageDst,
                param.dstBaseLayer, param.dstLayerCount, param.dstMipLevel, 1, param.dstAspect );
        }

        VkImage srcImage = CastToVkNonDispatchableObject< VkImage >( param.srcImage );
        VkImage dstImage = CastToVkNonDispatchableObject< VkImage >( param.dstImage );
        VkFilter filter = static_cast< VkFilter >( param.filter );
        VkImageBlit region;

        region.srcOffsets[ 0 ].x = param.srcOffsetX;
        region.srcOffsets[ 0 ].y = param.srcOffsetY;
        region.srcOffsets[ 0 ].z = param.srcOffsetZ;
        region.srcOffsets[ 1 ].x = param.srcOffsetX + param.srcWidth;
        region.srcOffsets[ 1 ].y = param.srcOffsetY + param.srcHeight;
        region.srcOffsets[ 1 ].z = param.srcOffsetZ + param.srcDepth;
        region.srcSubresource.aspectMask = param.srcAspect;
        region.srcSubresource.baseArrayLayer = param.srcBaseLayer;
        region.srcSubresource.layerCount = param.srcLayerCount;
        region.srcSubresource.mipLevel = param.srcMipLevel;

        region.dstOffsets[ 0 ].x = param.dstOffsetX;
        region.dstOffsets[ 0 ].y = param.dstOffsetY;
        region.dstOffsets[ 0 ].z = param.dstOffsetZ;
        region.dstOffsets[ 1 ].x = param.dstOffsetX + param.dstWidth;
        region.dstOffsets[ 1 ].y = param.dstOffsetY + param.dstHeight;
        region.dstOffsets[ 1 ].z = param.dstOffsetZ + param.dstDepth;
        region.dstSubresource.aspectMask = param.dstAspect;
        region.dstSubresource.baseArrayLayer = param.dstBaseLayer;
        region.dstSubresource.layerCount = param.dstLayerCount;
        region.dstSubresource.mipLevel = param.dstMipLevel;

        NN_GFX_CALL_VK_FUNCTION( vkCmdBlitImage( pCommandContext->GetVkDrawCommandBuffer(),
            srcImage, Vk::LayoutBlitImageSrc, dstImage, Vk::LayoutBlitImageDst, 1, &region, filter ) );
    }
}

template<>
void VkCommandProc< VkClearBufferParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_OutsideRenderPass, true );
        }
    }
    else if ( pCommandContext->IsCurrentAssignedCommand() )
    {
        PrepareCommandOutsideRenderPass( pCommandContext );

        const VkClearBufferParam& param = *static_cast< const VkClearBufferParam* >( pParam );

        VkBuffer buffer = CastToVkNonDispatchableObject< VkBuffer >( param.buffer );

        NN_GFX_CALL_VK_FUNCTION( vkCmdFillBuffer( pCommandContext->GetVkDrawCommandBuffer(),
            buffer, param.offset, param.size, param.data ) );
    }
}

template<>
void VkCommandProc< VkClearColorParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    const VkClearColorParam& param = *static_cast< const VkClearColorParam* >( pParam );

    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkGpuState::RenderTargetState& renderTargetState = pCommandContext->GetGpuState()->renderTargetState;

        for ( int32_t idxColor = 0; idxColor < renderTargetState.colorAttachmentCount; ++idxColor )
        {
            GfxVkGpuState::RenderTargetState::AttachmentState& colorAttachment =
                renderTargetState.colorAttachment[ idxColor ];

            if ( renderTargetState.colorTargetView[ idxColor ] == param.imageView &&
                colorAttachment.layer == param.baseLayer &&
                colorAttachment.mip == param.baseMipLevel &&
                colorAttachment.layerCount == param.layerCount )
            {
                colorAttachment.loadOp = VK_ATTACHMENT_LOAD_OP_CLEAR;
            }
        }

        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_OutsideRenderPass, true );
        }
    }
    else if ( pCommandContext->IsCurrentAssignedCommand() )
    {
        PrepareCommandOutsideRenderPass( pCommandContext );

        bool isPassClear = false;

        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetUpdatedCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            GfxVkGpuState::RenderTargetState& framebufferRenderTargetState = *pCommandInfo->pFramebufferCache->GetRenderTargetState();
            GfxVkGpuState::RenderTargetState& renderpassRenderTargetState = *pCommandInfo->pPipelineCache->GetRenderTargetState();

            for ( int32_t idxColor = 0; idxColor < framebufferRenderTargetState.colorAttachmentCount; ++idxColor )
            {
                if ( renderpassRenderTargetState.colorAttachment[ idxColor ].loadOp == VK_ATTACHMENT_LOAD_OP_CLEAR &&
                    framebufferRenderTargetState.colorTargetView[ idxColor ] == param.imageView &&
                    framebufferRenderTargetState.colorAttachment[ idxColor ].layer == param.baseLayer &&
                    framebufferRenderTargetState.colorAttachment[ idxColor ].mip == param.baseMipLevel &&
                    framebufferRenderTargetState.colorAttachment[ idxColor ].layerCount == param.layerCount )
                {
                    GfxVkGpuState::RenderPassState& renderPassState = pCommandContext->GetGpuState()->renderPassState;
                    for ( int idx = 0; idx < 4; ++idx )
                    {
                        renderPassState.clearColors[ idxColor ][ idx ] = param.color[ idx ];
                    }

                    isPassClear = true;
                }
            }
        }

        if ( !isPassClear )
        {
            VkClearColorValue clearColor;
            for ( int idx = 0; idx < 4; ++idx )
            {
                clearColor.float32[ idx ] = param.color[ idx ];
            }

            ClearColorImage( pCommandContext, param.pManager, CastToVkNonDispatchableObject< VkImage >( param.image ), &clearColor,
               param.baseLayer, param.layerCount, param.baseMipLevel, param.mipLevelCount );
        }
    }
}

template<>
void VkCommandProc< VkClearDepthStencilParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    const VkClearDepthStencilParam& param =
        *static_cast< const VkClearDepthStencilParam* >( pParam );

    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkGpuState::RenderTargetState& renderTargetState = pCommandContext->GetGpuState()->renderTargetState;
        GfxVkGpuState::RenderTargetState::AttachmentState& depthStencilAttachment =
            renderTargetState.depthStencilAttachment;

        if ( renderTargetState.depthStencilTargetView == param.imageView &&
            depthStencilAttachment.layer == param.baseLayer &&
            depthStencilAttachment.mip == param.baseMipLevel &&
            depthStencilAttachment.layerCount == param.layerCount )
        {
            depthStencilAttachment.loadOp = VK_ATTACHMENT_LOAD_OP_CLEAR;
        }

        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_OutsideRenderPass, true );
        }
    }
    else if ( pCommandContext->IsCurrentAssignedCommand() )
    {
        PrepareCommandOutsideRenderPass( pCommandContext );

        bool isPassClear = false;

        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetUpdatedCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            GfxVkGpuState::RenderTargetState& framebufferRenderTargetState = *pCommandInfo->pFramebufferCache->GetRenderTargetState();
            GfxVkGpuState::RenderTargetState& renderpassRenderTargetState = *pCommandInfo->pPipelineCache->GetRenderTargetState();

            if ( framebufferRenderTargetState.flag.GetBit( GfxVkGpuState::RenderTargetState::Flag_DepthAttached ) &&
                renderpassRenderTargetState.depthStencilAttachment.loadOp == VK_ATTACHMENT_LOAD_OP_CLEAR &&
                framebufferRenderTargetState.depthStencilTargetView == param.imageView &&
                framebufferRenderTargetState.depthStencilAttachment.layer == param.baseLayer &&
                framebufferRenderTargetState.depthStencilAttachment.mip == param.baseMipLevel &&
                framebufferRenderTargetState.depthStencilAttachment.layerCount == param.layerCount )
            {
                GfxVkGpuState::RenderPassState& renderPassState = pCommandContext->GetGpuState()->renderPassState;
                renderPassState.clearDepth = param.depth;
                renderPassState.clearStencil = param.stencil;

                isPassClear = true;
            }
        }

        if ( !isPassClear )
        {
            VkClearDepthStencilValue clearDepthStencil;
            clearDepthStencil.depth = param.depth;
            clearDepthStencil.stencil = param.stencil;

            ClearDepthStencilImage( pCommandContext, param.pManager, CastToVkNonDispatchableObject< VkImage >( param.image ), &clearDepthStencil,
                param.baseLayer, param.layerCount, param.baseMipLevel, param.mipLevelCount, param.aspectMask );
        }
    }
}

template<>
void VkCommandProc< VkResolveParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    const VkResolveParam& param =
        *static_cast< const VkResolveParam* >( pParam );

    if ( !pCommandContext->IsContextTypeStateUpdater() && pCommandContext->IsCurrentAssignedCommand() )
    {
        if ( Vk::IsLayoutManagementEnabled() )
        {
            PrepareLayoutTransition< false >( pCommandContext, param.pSrcManager, Vk::LayoutResolveImageSrc,
                param.srcBaseLayer, param.layerCount, 0, 1, VK_IMAGE_ASPECT_COLOR_BIT );

            PrepareLayoutTransition< false >( pCommandContext, param.pDstManager, Vk::LayoutResolveImageDst,
                param.dstBaseLayer, param.layerCount, param.dstMipLevel, 1, VK_IMAGE_ASPECT_COLOR_BIT );
        }

        VkImage srcImage = CastToVkNonDispatchableObject< VkImage >( param.srcImage );
        VkImage dstImage = CastToVkNonDispatchableObject< VkImage >( param.dstImage );
        VkImageResolve region;

        region.srcOffset.x = 0;
        region.srcOffset.y = 0;
        region.srcOffset.z = 0;
        region.srcSubresource.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
        region.srcSubresource.baseArrayLayer = param.srcBaseLayer;
        region.srcSubresource.layerCount = param.layerCount;
        region.srcSubresource.mipLevel = 0;
        region.dstOffset.x = 0;
        region.dstOffset.y = 0;
        region.dstOffset.z = 0;
        region.dstSubresource.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
        region.dstSubresource.baseArrayLayer = param.dstBaseLayer;
        region.dstSubresource.layerCount = param.layerCount;
        region.dstSubresource.mipLevel = param.dstMipLevel;
        region.extent.width = param.width;
        region.extent.height = param.height;
        region.extent.depth = param.depth;

        NN_GFX_CALL_VK_FUNCTION( vkCmdResolveImage( pCommandContext->GetVkDrawCommandBuffer(),
            srcImage, Vk::LayoutResolveImageSrc, dstImage, Vk::LayoutResolveImageDst, 1, &region ) );
    }
}

template<>
void VkCommandProc< VkDiscardColorTargetParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        const VkDiscardColorTargetParam& param =
            *static_cast< const VkDiscardColorTargetParam* >( pParam );

        GfxVkGpuState::RenderTargetState::AttachmentState& colorAttachment =
            pCommandContext->GetGpuState()->renderTargetState.colorAttachment[ param.index ];

        colorAttachment.loadOp = VK_ATTACHMENT_LOAD_OP_DONT_CARE;
    }
}

template<>
void VkCommandProc< VkDiscardDepthStencilTargetParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        const VkDiscardDepthStencilTargetParam& param =
            *static_cast< const VkDiscardDepthStencilTargetParam* >( pParam );

        GfxVkGpuState::RenderTargetState::AttachmentState& depthStencilAttachment =
            pCommandContext->GetGpuState()->renderTargetState.depthStencilAttachment;

        VkBool32 hasStencil = Vk::IsDepthOnlyFormat( static_cast< VkFormat >( depthStencilAttachment.format ) );

        if ( Vk::IsDepth( param.aspectMask ) && ( !hasStencil || Vk::IsStencil( param.aspectMask ) ) )
        {
            depthStencilAttachment.loadOp = VK_ATTACHMENT_LOAD_OP_DONT_CARE;
        }
    }
}

template<>
void VkCommandProc< VkMemoryBarrierParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_OutsideRenderPass, true );
        }
    }
    else if ( pCommandContext->IsCurrentAssignedCommand() )
    {
        PrepareCommandOutsideRenderPass( pCommandContext );

        const VkMemoryBarrierParam& param = *static_cast< const VkMemoryBarrierParam* >( pParam );

        VkMemoryBarrier barrier;
        barrier.sType = VK_STRUCTURE_TYPE_MEMORY_BARRIER;
        barrier.pNext = NULL;
        barrier.srcAccessMask = param.srcAccessMask;
        barrier.dstAccessMask = param.dstAccessMask;

        NN_GFX_CALL_VK_FUNCTION( vkCmdPipelineBarrier( pCommandContext->GetVkDrawCommandBuffer(),
            VK_PIPELINE_STAGE_ALL_COMMANDS_BIT, VK_PIPELINE_STAGE_ALL_COMMANDS_BIT, 0,
            1, &barrier, 0, NULL, 0, NULL ) );
    }
}

template<>
void VkCommandProc< VkCallCommandListParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    const VkCallCommandListParam& param = *static_cast< const VkCallCommandListParam* >( pParam );

    ParseVkCommandList( pCommandContext, param.pCommandList );
}

template<>
void VkCommandProc< VkSetBufferStateTransitionParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_OutsideRenderPass, true );
        }
    }
    else if ( pCommandContext->IsCurrentAssignedCommand() )
    {
        PrepareCommandOutsideRenderPass( pCommandContext );

        const VkSetBufferStateTransitionParam& param =
            *static_cast< const VkSetBufferStateTransitionParam* >( pParam );

        VkBuffer buffer = CastToVkNonDispatchableObject< VkBuffer >( param.buffer );
        VkBufferMemoryBarrier bufferMemoryBarrier;
        bufferMemoryBarrier.sType = VK_STRUCTURE_TYPE_BUFFER_MEMORY_BARRIER;
        bufferMemoryBarrier.pNext = NULL;
        bufferMemoryBarrier.srcAccessMask = param.srcAccessMask;
        bufferMemoryBarrier.dstAccessMask = param.dstAccessMask;
        bufferMemoryBarrier.srcQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
        bufferMemoryBarrier.dstQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
        bufferMemoryBarrier.buffer = buffer;
        bufferMemoryBarrier.offset = 0;
        bufferMemoryBarrier.size = VK_WHOLE_SIZE;

        NN_GFX_CALL_VK_FUNCTION( vkCmdPipelineBarrier( pCommandContext->GetVkDrawCommandBuffer(),
            param.srcStageFlag, param.dstStageFlag, 0, 0, NULL, 1, &bufferMemoryBarrier, 0, NULL ) );
    }
}

template<>
void VkCommandProc< VkSetTextureStateTransitionParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    const VkSetTextureStateTransitionParam& param =
        *static_cast< const VkSetTextureStateTransitionParam* >( pParam );

    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_OutsideRenderPass, true );
        }
    }
    else if ( pCommandContext->IsCurrentAssignedCommand() )
    {
        if ( Vk::IsLayoutManagementEnabled() )
        {
            PrepareLayoutTransition< false >( pCommandContext, param.pManager, static_cast< VkImageLayout >( param.newLayout ),
                param.imageBaseLayer, param.imageLayerCount, param.imageBaseMipLevel, param.imageMipCount,
                param.imageAspect );
        }
        else
        {
            PrepareCommandOutsideRenderPass( pCommandContext );

            VkImage image = CastToVkNonDispatchableObject< VkImage >( param.image );
            VkImageMemoryBarrier imageMemoryBarrier;
            imageMemoryBarrier.sType = VK_STRUCTURE_TYPE_IMAGE_MEMORY_BARRIER;
            imageMemoryBarrier.pNext = NULL;
            imageMemoryBarrier.srcAccessMask = param.srcAccessMask;
            imageMemoryBarrier.dstAccessMask = param.dstAccessMask;
            imageMemoryBarrier.oldLayout = static_cast< VkImageLayout >( param.oldLayout );
            imageMemoryBarrier.newLayout = static_cast< VkImageLayout >( param.newLayout );
            imageMemoryBarrier.srcQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
            imageMemoryBarrier.dstQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
            imageMemoryBarrier.image = image;
            imageMemoryBarrier.subresourceRange.aspectMask = param.imageAspect;
            imageMemoryBarrier.subresourceRange.baseMipLevel = param.imageBaseMipLevel;
            imageMemoryBarrier.subresourceRange.levelCount = param.imageMipCount;
            imageMemoryBarrier.subresourceRange.baseArrayLayer = param.imageBaseLayer;
            imageMemoryBarrier.subresourceRange.layerCount = param.imageLayerCount;

            NN_GFX_CALL_VK_FUNCTION( vkCmdPipelineBarrier( pCommandContext->GetVkDrawCommandBuffer(),
                param.srcStageFlag, param.dstStageFlag, 0, 0, NULL, 0, NULL, 1, &imageMemoryBarrier ) );
        }
    }
}

template<>
void VkCommandProc< VkBeginQueryParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_OutsideRenderPass, true );
        }
    }
    else if ( pCommandContext->IsUseQuery() && pCommandContext->IsCurrentAssignedCommand() )
    {
        PrepareCommandOutsideRenderPass( pCommandContext );

        const VkBeginQueryParam& param =
            *static_cast< const VkBeginQueryParam* >( pParam );

        VkQueryPool queryPool = CastToVkNonDispatchableObject< VkQueryPool >( param.query );
        NN_GFX_CALL_VK_FUNCTION( vkCmdResetQueryPool( pCommandContext->GetVkDrawCommandBuffer(), queryPool, 0, 1 ) );
        NN_GFX_CALL_VK_FUNCTION( vkCmdBeginQuery( pCommandContext->GetVkDrawCommandBuffer(), queryPool, 0, param.flags ) );

        pCommandContext->GetGpuState()->queryState.queryStartedFlag.SetBit( param.queryTarget, true );
    }
}

template<>
void VkCommandProc< VkEndQueryParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_OutsideRenderPass, true );
        }
    }
    else if ( pCommandContext->IsUseQuery() && pCommandContext->IsCurrentAssignedCommand() )
    {
        const VkEndQueryParam& param =
            *static_cast< const VkEndQueryParam* >( pParam );

        if ( !pCommandContext->GetGpuState()->queryState.queryStartedFlag.GetBit( param.queryTarget ) )
        {
            // BeginQueryされていない、または既にEndQueryされた後である場合は
            // コマンドを追加しません。
            return;
        }

        PrepareCommandOutsideRenderPass( pCommandContext );

        VkQueryPool queryPool = CastToVkNonDispatchableObject< VkQueryPool >( param.query );
        NN_GFX_CALL_VK_FUNCTION( vkCmdEndQuery( pCommandContext->GetVkDrawCommandBuffer(), queryPool, 0 ) );

        VkBuffer buffer = CastToVkNonDispatchableObject< VkBuffer >( param.buffer );
        VkQueryResultFlags flags = VK_QUERY_RESULT_64_BIT | VK_QUERY_RESULT_WAIT_BIT;
        NN_GFX_CALL_VK_FUNCTION( vkCmdCopyQueryPoolResults( pCommandContext->GetVkDrawCommandBuffer(),
            queryPool, 0, 1, buffer, param.offset, param.stride, flags ) );

        VkBufferMemoryBarrier bufferBarrier;
        bufferBarrier.sType = VK_STRUCTURE_TYPE_BUFFER_MEMORY_BARRIER;
        bufferBarrier.pNext = NULL;
        bufferBarrier.srcAccessMask = VK_ACCESS_TRANSFER_WRITE_BIT;
        bufferBarrier.dstAccessMask = VK_ACCESS_HOST_READ_BIT;
        bufferBarrier.srcQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
        bufferBarrier.dstQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
        bufferBarrier.buffer = buffer;
        bufferBarrier.offset = param.offset;
        bufferBarrier.size = param.stride;
        NN_GFX_CALL_VK_FUNCTION( vkCmdPipelineBarrier( pCommandContext->GetVkDrawCommandBuffer(),
            VK_PIPELINE_STAGE_TRANSFER_BIT, VK_PIPELINE_STAGE_HOST_BIT, 0, 0, NULL, 1, &bufferBarrier, 0, NULL ) );

        pCommandContext->GetGpuState()->queryState.queryStartedFlag.SetBit( param.queryTarget, false );
    }
}

template<>
void VkCommandProc< VkBeginFastQueryParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_OutsideRenderPass, true );
        }
    }
    else if ( pCommandContext->IsUseQuery() && pCommandContext->IsCurrentAssignedCommand() )
    {
        PrepareCommandOutsideRenderPass( pCommandContext );

        const VkBeginFastQueryParam& param =
            *static_cast< const VkBeginFastQueryParam* >( pParam );

        VkQueryPool queryPool = CastToVkNonDispatchableObject< VkQueryPool >( param.query );
        NN_GFX_CALL_VK_FUNCTION( vkCmdResetQueryPool( pCommandContext->GetVkPrepCommandBuffer(), queryPool, param.queryIndex, 1 ) );
        NN_GFX_CALL_VK_FUNCTION( vkCmdBeginQuery( pCommandContext->GetVkDrawCommandBuffer(), queryPool, param.queryIndex, param.flags ) );

        pCommandContext->GetGpuState()->queryState.queryStartedFlag.SetBit( GfxVkGpuState::QueryState::QueryTargetFlag_GraphicsPipelineStatistics, true );
    }
}

template<>
void VkCommandProc< VkEndFastQueryParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_OutsideRenderPass, true );
        }
    }
    else if ( pCommandContext->IsUseQuery() && pCommandContext->IsCurrentAssignedCommand() )
    {
        if ( !pCommandContext->GetGpuState()->queryState.queryStartedFlag.GetBit( GfxVkGpuState::QueryState::QueryTargetFlag_GraphicsPipelineStatistics ) )
        {
            // BeginQueryされていない、または既にEndQueryされた後である場合は
            // コマンドを追加しません。
            return;
        }

        PrepareCommandOutsideRenderPass( pCommandContext );

        const VkEndFastQueryParam& param =
            *static_cast< const VkEndFastQueryParam* >( pParam );

        VkQueryPool queryPool = CastToVkNonDispatchableObject< VkQueryPool >( param.query );
        NN_GFX_CALL_VK_FUNCTION( vkCmdEndQuery( pCommandContext->GetVkDrawCommandBuffer(), queryPool, param.queryIndex ) );

        VkBuffer buffer = CastToVkNonDispatchableObject< VkBuffer >( param.buffer );
        VkQueryResultFlags flags = VK_QUERY_RESULT_64_BIT | VK_QUERY_RESULT_WAIT_BIT;
        NN_GFX_CALL_VK_FUNCTION( vkCmdCopyQueryPoolResults( pCommandContext->GetVkPostCommandBuffer(),
            queryPool, param.queryIndex, 1, buffer, param.offset, param.stride, flags ) );

        VkBufferMemoryBarrier bufferBarrier;
        bufferBarrier.sType = VK_STRUCTURE_TYPE_BUFFER_MEMORY_BARRIER;
        bufferBarrier.pNext = NULL;
        bufferBarrier.srcAccessMask = VK_ACCESS_TRANSFER_WRITE_BIT;
        bufferBarrier.dstAccessMask = VK_ACCESS_HOST_READ_BIT;
        bufferBarrier.srcQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
        bufferBarrier.dstQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
        bufferBarrier.buffer = buffer;
        bufferBarrier.offset = param.offset;
        bufferBarrier.size = param.stride;
        NN_GFX_CALL_VK_FUNCTION( vkCmdPipelineBarrier( pCommandContext->GetVkPostCommandBuffer(),
            VK_PIPELINE_STAGE_TRANSFER_BIT, VK_PIPELINE_STAGE_HOST_BIT, 0, 0, NULL, 1, &bufferBarrier, 0, NULL ) );

        pCommandContext->GetGpuState()->queryState.queryStartedFlag.SetBit( GfxVkGpuState::QueryState::QueryTargetFlag_GraphicsPipelineStatistics, false );
    }
}

template<>
void VkCommandProc< VkWriteTimestampParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_OutsideRenderPass, true );
        }
    }
    else if ( pCommandContext->IsCurrentAssignedCommand() )
    {
        PrepareCommandOutsideRenderPass( pCommandContext );

        const VkWriteTimestampParam& param =
            *static_cast< const VkWriteTimestampParam* >( pParam );

        VkQueryPool queryPool = CastToVkNonDispatchableObject< VkQueryPool >( param.query );
        NN_GFX_CALL_VK_FUNCTION( vkCmdResetQueryPool( pCommandContext->GetVkDrawCommandBuffer(), queryPool, 0, 1 ) );

        VkBuffer buffer = CastToVkNonDispatchableObject< VkBuffer >( param.buffer );
        NN_GFX_CALL_VK_FUNCTION( vkCmdWriteTimestamp( pCommandContext->GetVkDrawCommandBuffer(),
            VK_PIPELINE_STAGE_ALL_COMMANDS_BIT, queryPool, 0 ) );

        VkQueryResultFlags flags = VK_QUERY_RESULT_64_BIT | VK_QUERY_RESULT_WAIT_BIT;
        NN_GFX_CALL_VK_FUNCTION( vkCmdCopyQueryPoolResults( pCommandContext->GetVkDrawCommandBuffer(),
            queryPool, 0, 1, buffer, param.offset, sizeof( uint64_t ), flags ) );

        VkBufferMemoryBarrier bufferBarrier;
        bufferBarrier.sType = VK_STRUCTURE_TYPE_BUFFER_MEMORY_BARRIER;
        bufferBarrier.pNext = NULL;
        bufferBarrier.srcAccessMask = VK_ACCESS_TRANSFER_WRITE_BIT;
        bufferBarrier.dstAccessMask = VK_ACCESS_HOST_READ_BIT;
        bufferBarrier.srcQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
        bufferBarrier.dstQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
        bufferBarrier.buffer = buffer;
        bufferBarrier.offset = param.offset;
        bufferBarrier.size = sizeof( uint64_t );
        NN_GFX_CALL_VK_FUNCTION( vkCmdPipelineBarrier( pCommandContext->GetVkDrawCommandBuffer(),
            VK_PIPELINE_STAGE_TRANSFER_BIT, VK_PIPELINE_STAGE_HOST_BIT, 0, 0, NULL, 1, &bufferBarrier, 0, NULL ) );
    }
}

template<>
void VkCommandProc< VkWriteFastTimestampParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_OutsideRenderPass, true );
        }
    }
    else if ( pCommandContext->IsCurrentAssignedCommand() )
    {
        PrepareCommandOutsideRenderPass( pCommandContext );

        const VkWriteFastTimestampParam& param =
            *static_cast< const VkWriteFastTimestampParam* >( pParam );

        VkQueryPool queryPool = CastToVkNonDispatchableObject< VkQueryPool >( param.query );
        NN_GFX_CALL_VK_FUNCTION( vkCmdResetQueryPool( pCommandContext->GetVkPrepCommandBuffer(), queryPool, param.queryIndex, 1 ) );

        VkBuffer buffer = CastToVkNonDispatchableObject< VkBuffer >( param.buffer );
        NN_GFX_CALL_VK_FUNCTION( vkCmdWriteTimestamp( pCommandContext->GetVkDrawCommandBuffer(),
            VK_PIPELINE_STAGE_ALL_COMMANDS_BIT, queryPool, param.queryIndex ) );

        VkQueryResultFlags flags = VK_QUERY_RESULT_64_BIT | VK_QUERY_RESULT_WAIT_BIT;
        NN_GFX_CALL_VK_FUNCTION( vkCmdCopyQueryPoolResults( pCommandContext->GetVkPostCommandBuffer(),
            queryPool, param.queryIndex, 1, buffer, param.offset, sizeof( uint64_t ), flags ) );

        VkBufferMemoryBarrier bufferBarrier;
        bufferBarrier.sType = VK_STRUCTURE_TYPE_BUFFER_MEMORY_BARRIER;
        bufferBarrier.pNext = NULL;
        bufferBarrier.srcAccessMask = VK_ACCESS_TRANSFER_WRITE_BIT;
        bufferBarrier.dstAccessMask = VK_ACCESS_HOST_READ_BIT;
        bufferBarrier.srcQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
        bufferBarrier.dstQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
        bufferBarrier.buffer = buffer;
        bufferBarrier.offset = param.offset;
        bufferBarrier.size = sizeof( uint64_t );
        NN_GFX_CALL_VK_FUNCTION( vkCmdPipelineBarrier( pCommandContext->GetVkPostCommandBuffer(),
            VK_PIPELINE_STAGE_TRANSFER_BIT, VK_PIPELINE_STAGE_HOST_BIT, 0, 0, NULL, 1, &bufferBarrier, 0, NULL ) );
    }
}

template<>
void VkCommandProc< VkSetDepthBoundsParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        #if defined( NN_SDK_BUILD_DEBUG )
        const VkSetDepthBoundsParam& param =
            *static_cast< const VkSetDepthBoundsParam* >( pParam );

        GfxVkGpuState* pGpuState = pCommandContext->GetGpuState();
        GfxVkGpuState::DynamicState::DepthStencilState& depthStencilState =
            pGpuState->dynamicState.depthStencilState;

        depthStencilState.minDepthBounds = param.minDepthBounds;
        depthStencilState.maxDepthBounds = param.maxDepthBounds;
        #endif
    }
    else
    {
        const VkSetDepthBoundsParam& param =
            *static_cast< const VkSetDepthBoundsParam* >( pParam );

        NN_GFX_CALL_VK_FUNCTION( vkCmdSetDepthBounds( pCommandContext->GetVkDrawCommandBuffer(),
            param.minDepthBounds, param.maxDepthBounds ) );
    }
}

template<>
void VkCommandProc< VkSetLineWidthParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        #if defined( NN_SDK_BUILD_DEBUG )
        const VkSetLineWidthParam& param =
            *static_cast< const VkSetLineWidthParam* >( pParam );

        GfxVkGpuState* pGpuState = pCommandContext->GetGpuState();
        GfxVkGpuState::DynamicState::RasterizationState& rasterizationState =
            pGpuState->dynamicState.rasterizationState;

        rasterizationState.lineWidth = param.lineWidth;
        #endif
    }
    else
    {
        const VkSetLineWidthParam& param =
            *static_cast< const VkSetLineWidthParam* >( pParam );

        NN_GFX_CALL_VK_FUNCTION( vkCmdSetLineWidth( pCommandContext->GetVkDrawCommandBuffer(), param.lineWidth ) );
    }
}

template<>
void VkCommandProc< VkSetViewportsParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    const VkSetViewportsParam& param =
        *static_cast< const VkSetViewportsParam* >( pParam );

    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkGpuState* pGpuState = pCommandContext->GetGpuState();

        GfxVkGpuState::ConstSizedPipelineState::ViewportState& viewportState =
            pGpuState->constSizedPipelineState.viewportState;
        viewportState.viewportCount = param.count;

        #if defined( NN_SDK_BUILD_DEBUG )
        {
            GfxVkGpuState::DynamicState::ViewportScissorState& viewportScissorState =
                pGpuState->dynamicState.viewportScissorState;

            const float* pViewports = nn::util::ConstBytePtr(
                &param, sizeof( VkSetViewportsParam ) ).Get< float >();
            for ( int idxViewport = param.first; idxViewport < param.first + param.count; ++idxViewport )
            {
                for ( int idxElement = 0; idxElement < 6; ++idxElement )
                {
                    viewportScissorState.viewport[ idxViewport ][ idxElement ] = *pViewports++;
                }
            }
        }
        #endif
    }
    else
    {
        const VkViewport* ppViewports[ GfxVkGpuState::maxViewports ];
        NN_SDK_ASSERT( param.count <= GfxVkGpuState::maxViewports );

        for ( int idxViewport = 0; idxViewport < param.count; ++idxViewport )
        {
            ppViewports[ idxViewport ] = nn::util::ConstBytePtr(
                &param, sizeof( VkSetViewportsParam ) + sizeof( float ) * 6 * idxViewport ).Get< const VkViewport >();
        }

        NN_GFX_CALL_VK_FUNCTION( vkCmdSetViewport( pCommandContext->GetVkDrawCommandBuffer(), param.first, param.count, ppViewports[ 0 ] ) );
    }
}

template<>
void VkCommandProc< VkSetScissorsParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    const VkSetScissorsParam& param =
        *static_cast< const VkSetScissorsParam* >( pParam );

    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        #if defined( NN_SDK_BUILD_DEBUG )
        {
            GfxVkGpuState* pGpuState = pCommandContext->GetGpuState();

            GfxVkGpuState::DynamicState::ViewportScissorState& viewportScissorState =
                pGpuState->dynamicState.viewportScissorState;

            const int32_t* pScissors = nn::util::ConstBytePtr(
                &param, sizeof( VkSetScissorsParam ) ).Get< int32_t >();
            for ( int idxScissor = param.first; idxScissor < param.first + param.count; ++idxScissor )
            {
                for ( int idxElement = 0; idxElement < 4; ++idxElement )
                {
                    viewportScissorState.scissor[ idxScissor ][ idxElement ] = *pScissors++;
                }
            }
        }
        #endif
    }
    else
    {
        const VkRect2D* ppScissors[ GfxVkGpuState::maxViewports ];
        NN_SDK_ASSERT( param.count <= GfxVkGpuState::maxViewports );

        for ( int idxScissor = 0; idxScissor < param.count; ++idxScissor )
        {
            ppScissors[ idxScissor ] = nn::util::ConstBytePtr(
                &param, sizeof( VkSetScissorsParam ) + sizeof( int32_t ) * 4 * idxScissor ).Get< const VkRect2D >();
        }

        NN_GFX_CALL_VK_FUNCTION( vkCmdSetScissor( pCommandContext->GetVkDrawCommandBuffer(), param.first, param.count, ppScissors[ 0 ] ) );
    }
}

template<>
void VkCommandProc< VkSetBufferParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    GfxVkGpuState* pGpuState = pCommandContext->GetGpuState();

    const VkSetBufferParam& param =
        *static_cast< const VkSetBufferParam* >( pParam );

    GfxVkGpuState::DescriptorSetLayoutState& descriptorSetLayoutState =
        pGpuState->descriptorSetLayoutState;
    descriptorSetLayoutState.maxDescriptorIndex =
        std::max NN_PREVENT_MACRO_FUNC( static_cast< uint32_t >( param.slot ),
        descriptorSetLayoutState.maxDescriptorIndex );

    NN_SDK_ASSERT( param.slot < GfxVkGpuState::maxDescriptorSlot );

    GfxVkGpuState::DescriptorSetLayoutState::SlotState& descriptorSlotState =
        pGpuState->descriptorSetLayoutState.slotState[ param.slot ];

    GfxVkGpuState::DescriptorResourceState::ResourceState& descriptorResourceState =
        pGpuState->descriptorResourceState.resourceState[ param.slot ];

    if ( !descriptorSlotState.flag.GetBit(
        GfxVkGpuState::DescriptorSetLayoutState::SlotState::Flag_IsSet ) )
    {
        descriptorSetLayoutState.availableDescriptorCount++;
        descriptorSlotState.flag.SetBit(
            GfxVkGpuState::DescriptorSetLayoutState::SlotState::Flag_IsSet, true );
    }

    if ( descriptorSlotState.type == param.descriptorType
        && descriptorResourceState.vkBufferInfo.buffer == CastToVkNonDispatchableObject< VkBuffer >( param.buffer )
        && descriptorResourceState.vkBufferInfo.offset == param.offset
        && descriptorResourceState.vkBufferInfo.range == param.size )
    {
        descriptorSlotState.shaderStage |= param.shaderStage;
    }
    else
    {
        descriptorSlotState.type = param.descriptorType;
        descriptorSlotState.shaderStage = param.shaderStage;
        descriptorResourceState.vkBufferInfo.buffer = CastToVkNonDispatchableObject< VkBuffer >( param.buffer );
        descriptorResourceState.vkBufferInfo.offset = param.offset;
        descriptorResourceState.vkBufferInfo.range = param.size;
    }

    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_SetDescriptorPool, true );
        }
    }
}

template<>
void VkCommandProc< VkSetTextureAndSamplerParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    GfxVkGpuState* pGpuState = pCommandContext->GetGpuState();

    const VkSetTextureAndSamplerParam& param =
        *static_cast< const VkSetTextureAndSamplerParam* >( pParam );

    GfxVkGpuState::DescriptorSetLayoutState& descriptorSetLayoutState =
        pGpuState->descriptorSetLayoutState;
    descriptorSetLayoutState.maxDescriptorIndex =
        std::max NN_PREVENT_MACRO_FUNC( static_cast< uint32_t >( param.slot ),
        descriptorSetLayoutState.maxDescriptorIndex );

    NN_SDK_ASSERT( param.slot < GfxVkGpuState::maxDescriptorSlot );

    GfxVkGpuState::DescriptorSetLayoutState::SlotState& descriptorSlotState =
        pGpuState->descriptorSetLayoutState.slotState[ param.slot ];

    GfxVkGpuState::DescriptorResourceState::ResourceState& descriptorResourceState =
        pGpuState->descriptorResourceState.resourceState[ param.slot ];

    if ( !descriptorSlotState.flag.GetBit(
        GfxVkGpuState::DescriptorSetLayoutState::SlotState::Flag_IsSet ) )
    {
        descriptorSetLayoutState.availableDescriptorCount++;
        descriptorSlotState.flag.SetBit(
            GfxVkGpuState::DescriptorSetLayoutState::SlotState::Flag_IsSet, true );
    }

    if ( descriptorSlotState.type == param.descriptorType
        && descriptorResourceState.vkImageInfo.imageView == CastToVkNonDispatchableObject< VkImageView >( param.texture )
        && descriptorResourceState.vkImageInfo.sampler == CastToVkNonDispatchableObject< VkSampler >( param.sampler )
        && descriptorResourceState.vkImageInfo.imageLayout == static_cast< VkImageLayout >( param.imageLayout ) )
    {
        descriptorSlotState.shaderStage |= param.shaderStage;
    }
    else
    {
        descriptorSlotState.type = param.descriptorType;
        descriptorSlotState.shaderStage = param.shaderStage;
        descriptorResourceState.vkImageInfo.imageView = CastToVkNonDispatchableObject< VkImageView >( param.texture );
        descriptorResourceState.vkImageInfo.sampler = CastToVkNonDispatchableObject< VkSampler >( param.sampler );
        descriptorResourceState.vkImageInfo.imageLayout = static_cast< VkImageLayout >( param.imageLayout );
    }

    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_SetDescriptorPool, true );
        }
    }
    else if ( Vk::IsLayoutManagementEnabled() && pCommandContext->IsCurrentAssignedCommand() )
    {
        PrepareLayoutTransition< false >( pCommandContext, param.pManager,
            descriptorResourceState.vkImageInfo.imageLayout,
            param.baseLayer, param.layerCount, param.baseMip, param.mipCount,
            param.pManager->GetImageAspectFlags() );
    }
}

template<>
void VkCommandProc< VkSetSeparateShaderParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkGpuState* pGpuState = pCommandContext->GetGpuState();

        const VkSetSeparateShaderParam& param =
            *static_cast< const VkSetSeparateShaderParam* >( pParam );

        GfxVkGpuState::ConstSizedPipelineState::ShaderState& shaderState =
            pGpuState->constSizedPipelineState.shaderState;

        // Recompute valid descriptor slot mask
        uint64_t validSlotMask = 0;

        for ( int idxStage = 0; idxStage < ShaderStage_End; ++idxStage )
        {
            if ( ( 1 << idxStage ) & param.stageBits )
            {
                shaderState.shaderModule[ idxStage ] = param.shaderModule[ idxStage ];
                shaderState.shaderBindingMask[ idxStage ] = param.validMask[ idxStage ];
            }
            validSlotMask |= shaderState.shaderBindingMask[ idxStage ];
        }
        pGpuState->descriptorSetLayoutState.validSlotMask = validSlotMask;

        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_SetShader, true );
        }
    }
}

template<>
void VkCommandProc< RasterizerStateImplData< ApiVariationVk1 > >(
    CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    const RasterizerStateImplData< ApiVariationVk1 >& param =
        *static_cast< const RasterizerStateImplData< ApiVariationVk1 >* >( pParam );

    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkGpuState* pGpuState = pCommandContext->GetGpuState();

        GfxVkGpuState::ConstSizedPipelineState::RasterizationState& rasterizationState =
            pGpuState->constSizedPipelineState.rasterizationState;

        bool polygonOffsetEnabled = rasterizationState.flag.GetBit( RasterizerStateImplData< ApiVariationVk1 >::Flag_PolygonOffsetEnable );
        bool depthClampEnabled = rasterizationState.flag.GetBit( RasterizerStateImplData< ApiVariationVk1 >::Flag_DepthClampEnable );

        rasterizationState.flag = param.flag;

        if ( pCommandContext->GetGfxCommandBuffer()->ToData()->isVulkanDedicatedApiModeEnabled )
        {
            rasterizationState.flag.SetBit( RasterizerStateImplData< ApiVariationVk1 >::Flag_PolygonOffsetEnable, polygonOffsetEnabled );
            rasterizationState.flag.SetBit( RasterizerStateImplData< ApiVariationVk1 >::Flag_DepthClampEnable, depthClampEnabled );
        }

        rasterizationState.polygonMode = param.polygonMode;
        rasterizationState.frontFace = param.frontFace;
        rasterizationState.cullFace = param.cullFace;
        rasterizationState.sampleMask = param.sampleMask;
        rasterizationState.sampleCount = param.sampleCount;

        #if defined( NN_SDK_BUILD_DEBUG )
        GfxVkGpuState::DynamicState::RasterizationState& dynamicRasterizationState =
            pGpuState->dynamicState.rasterizationState;
        dynamicRasterizationState.depthBias = param.depthBias;
        dynamicRasterizationState.depthBiasClamp = param.depthBiasClamp;
        dynamicRasterizationState.slopeScaledDepthBias = param.slopeScaledDepthBias;
        #endif

        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_SetRasterizerState, true );
        }
    }
    else
    {
        if ( !pCommandContext->GetGfxCommandBuffer()->ToData()->isVulkanDedicatedApiModeEnabled )
        {
            NN_GFX_CALL_VK_FUNCTION( vkCmdSetDepthBias( pCommandContext->GetVkDrawCommandBuffer(), param.depthBias,
                param.depthBiasClamp, param.slopeScaledDepthBias ) );
        }
    }
}

template<>
void VkCommandProc< BlendStateImplData< ApiVariationVk1 > >(
    CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    const BlendStateImplData< ApiVariationVk1 >& param =
        *static_cast< const BlendStateImplData< ApiVariationVk1 >* >( pParam );

    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkGpuState* pGpuState = pCommandContext->GetGpuState();

        GfxVkGpuState::ConstSizedPipelineState::BlendState& blendState =
            pGpuState->constSizedPipelineState.blendState;
        blendState.blendTargetCount = param.blendTargetCount;
        blendState.flag = param.flag;
        blendState.logicOp = param.logicOp;

        GfxVkGpuState::VariableSizedPipelineState::BlendTargetState* blendTargetState =
            pGpuState->variableSizedPipelineState.blendTargetState;

        const BlendStateImplData< ApiVariationVk1 >::BlendTargetState* pTargets =
            reinterpret_cast< const BlendStateImplData< ApiVariationVk1 >::BlendTargetState* >( &param.pTargetArray );
        for( int idxTarget = 0; idxTarget < param.blendTargetCount; ++idxTarget )
        {
            const BlendStateImplData< ApiVariationVk1 >::BlendTargetState& target = pTargets[
                param.flag.GetBit( BlendStateImplData< ApiVariationVk1 >::Flag_IndependentBlendEnable ) ? idxTarget : 0 ];
            blendTargetState[ idxTarget ].flag = target.flag;
            blendTargetState[ idxTarget ].colorMask = target.colorMask;
            blendTargetState[ idxTarget ].srcRGB = target.srcRGB;
            blendTargetState[ idxTarget ].dstRGB = target.dstRGB;
            blendTargetState[ idxTarget ].srcAlpha = target.srcAlpha;
            blendTargetState[ idxTarget ].dstAlpha = target.dstAlpha;
            blendTargetState[ idxTarget ].modeRGB = target.modeRGB;
            blendTargetState[ idxTarget ].modeAlpha = target.modeAlpha;
        }

        #if defined( NN_SDK_BUILD_DEBUG )
        GfxVkGpuState::DynamicState::BlendState& dynamicBlendState =
            pGpuState->dynamicState.blendState;
        dynamicBlendState.blendColor[ 0 ] = param.blendColor[ 0 ];
        dynamicBlendState.blendColor[ 1 ] = param.blendColor[ 1 ];
        dynamicBlendState.blendColor[ 2 ] = param.blendColor[ 2 ];
        dynamicBlendState.blendColor[ 3 ] = param.blendColor[ 3 ];
        #endif

        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_SetBlendState, true );
        }
    }
    else
    {
        NN_GFX_CALL_VK_FUNCTION( vkCmdSetBlendConstants( pCommandContext->GetVkDrawCommandBuffer(), param.blendColor ) );
    }
}

template<>
void VkCommandProc< DepthStencilStateImplData< ApiVariationVk1 > >(
    CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    const DepthStencilStateImplData< ApiVariationVk1 >& param =
        *static_cast< const DepthStencilStateImplData< ApiVariationVk1 >* >( pParam );

    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkGpuState* pGpuState = pCommandContext->GetGpuState();

        GfxVkGpuState::ConstSizedPipelineState::DepthStencilState& depthStencilState =
            pGpuState->constSizedPipelineState.depthStencilState;
        depthStencilState.flag = param.flag;
        depthStencilState.depthFunc = param.depthFunc;
        depthStencilState.frontStencil.sfail = param.frontStencil.sfail;
        depthStencilState.frontStencil.dpfail = param.frontStencil.dpfail;
        depthStencilState.frontStencil.dppass = param.frontStencil.dppass;
        depthStencilState.frontStencil.func = param.frontStencil.func;
        depthStencilState.backStencil.sfail = param.backStencil.sfail;
        depthStencilState.backStencil.dpfail = param.backStencil.dpfail;
        depthStencilState.backStencil.dppass = param.backStencil.dppass;
        depthStencilState.backStencil.func = param.backStencil.func;

        #if defined( NN_SDK_BUILD_DEBUG )
        GfxVkGpuState::DynamicState::DepthStencilState& dynamicDepthStencilState =
            pGpuState->dynamicState.depthStencilState;
        dynamicDepthStencilState.frontStencil.ref = param.frontStencil.ref;
        dynamicDepthStencilState.backStencil.ref = param.backStencil.ref;
        dynamicDepthStencilState.stencilWriteMask = param.stencilWriteMask;
        dynamicDepthStencilState.stencilReadMask = param.stencilReadMask;
        #endif

        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_SetDepthStencilState, true );
        }
    }
    else
    {
        VkCommandBuffer commandBuffer = pCommandContext->GetVkDrawCommandBuffer();
        NN_GFX_CALL_VK_FUNCTION( vkCmdSetStencilCompareMask( commandBuffer, VK_STENCIL_FRONT_AND_BACK, param.stencilReadMask ) );
        NN_GFX_CALL_VK_FUNCTION( vkCmdSetStencilWriteMask( commandBuffer, VK_STENCIL_FRONT_AND_BACK, param.stencilWriteMask ) );
        NN_GFX_CALL_VK_FUNCTION( vkCmdSetStencilReference( commandBuffer, VK_STENCIL_FACE_FRONT_BIT, param.frontStencil.ref ) );
        NN_GFX_CALL_VK_FUNCTION( vkCmdSetStencilReference( commandBuffer, VK_STENCIL_FACE_BACK_BIT, param.backStencil.ref ) );
    }
}

template<>
void VkCommandProc< VertexStateImplData< ApiVariationVk1 > >(
    CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        const VertexStateImplData< ApiVariationVk1 >& param =
            *static_cast< const VertexStateImplData< ApiVariationVk1 >* >( pParam );

        GfxVkGpuState* pGpuState = pCommandContext->GetGpuState();

        GfxVkGpuState::ConstSizedPipelineState::VertexInputState& vertexInputState =
            pGpuState->constSizedPipelineState.vertexInputState;
        vertexInputState.attributeStateCount = param.attributeCount;
        vertexInputState.bufferStateCount = param.bufferCount;

        GfxVkGpuState::VariableSizedPipelineState::VertexBufferState* vertexBufferState =
            pGpuState->variableSizedPipelineState.vertexBufferState;

        GfxVkGpuState::VariableSizedPipelineState::VertexAttributeState* vertexAttributeState =
            pGpuState->variableSizedPipelineState.vertexAttributeState;

        nn::util::ConstBytePtr ptr( &param.pWorkMemory );
        const VertexStateImplData< ApiVariationVk1 >::AttributeState* pAttributes =
            ptr.Get< VertexStateImplData< ApiVariationVk1 >::AttributeState >();
        const VertexStateImplData< ApiVariationVk1 >::BufferState* pBuffers = ptr.Advance(
            sizeof( VertexStateImplData< ApiVariationVk1 >::AttributeState ) * param.attributeCount ).Get<
            VertexStateImplData< ApiVariationVk1 >::BufferState >();

        for ( int idxAttribute = 0; idxAttribute < param.attributeCount; ++idxAttribute )
        {
            const VertexStateImplData< ApiVariationVk1 >::AttributeState& attributeState = pAttributes[ idxAttribute ];
            vertexAttributeState[ idxAttribute ].location = attributeState.location;
            vertexAttributeState[ idxAttribute ].bindingIndex = attributeState.bindingIndex;
            vertexAttributeState[ idxAttribute ].format = attributeState.format;
            vertexAttributeState[ idxAttribute ].offset = attributeState.offset;
        }

        for ( int idxBuffer = 0; idxBuffer < param.bufferCount; ++idxBuffer )
        {
            const VertexStateImplData< ApiVariationVk1 >::BufferState& bufferState = pBuffers[ idxBuffer ];
            vertexBufferState[ idxBuffer ].divisor = bufferState.divisor;
            vertexBufferState[ idxBuffer ].stride = bufferState.stride;
        }

        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_SetVertexState, true );
        }
    }
}

template<>
void VkCommandProc< TessellationStateImplData< ApiVariationVk1 > >(
    CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkGpuState* pGpuState = pCommandContext->GetGpuState();

        const TessellationStateImplData< ApiVariationVk1 >& param =
            *static_cast< const TessellationStateImplData< ApiVariationVk1 >* >( pParam );

        GfxVkGpuState::ConstSizedPipelineState::TessellationState& tessellationState =
            pGpuState->constSizedPipelineState.tessellationState;

        tessellationState.patchControlPoints = param.patchControlPointCount;

        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_SetTessellationState, true );
        }
    }
}

template<>
void VkCommandProc< VkCallbackParam >( CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_OutsideRenderPass, true );
        }
    }
    else if ( pCommandContext->IsCurrentAssignedCommand() )
    {
        PrepareCommandOutsideRenderPass( pCommandContext );

        const VkCallbackParam& param = *static_cast< const VkCallbackParam* >( pParam );
        VkCommandBuffer commandBuffer = pCommandContext->GetVkDrawCommandBuffer();

        param.pCallback( commandBuffer, param.pParam );
    }
}

template<>
void VkCommandProc< VkPolygonOffsetParam >(
    CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    const VkPolygonOffsetParam& param =
        *static_cast< const VkPolygonOffsetParam* >( pParam );

    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        #if defined( NN_SDK_BUILD_DEBUG )
        GfxVkGpuState* pGpuState = pCommandContext->GetGpuState();
        GfxVkGpuState::DynamicState::RasterizationState& dynamicRasterizationState =
            pGpuState->dynamicState.rasterizationState;

        dynamicRasterizationState.slopeScaledDepthBias= param.slopeScaledDepthBias;
        dynamicRasterizationState.depthBias = param.depthBias;
        dynamicRasterizationState.depthBiasClamp = param.depthBiasClamp;
        #endif
    }
    else
    {
        NN_GFX_CALL_VK_FUNCTION( vkCmdSetDepthBias( pCommandContext->GetVkDrawCommandBuffer(),
            param.depthBias, param.depthBiasClamp, param.slopeScaledDepthBias ) );
    }
}

template<>
void VkCommandProc< VkPolygonOffsetEnableParam >(
    CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        const VkPolygonOffsetEnableParam& param =
            *static_cast< const VkPolygonOffsetEnableParam* >( pParam );

        GfxVkGpuState* pGpuState = pCommandContext->GetGpuState();
        GfxVkGpuState::ConstSizedPipelineState::RasterizationState& rasterizationState =
            pGpuState->constSizedPipelineState.rasterizationState;
        rasterizationState.flag.SetBit( GfxVkGpuState::ConstSizedPipelineState::RasterizationState::Flag_PolygonOffsetEnable,
            param.enable ? true : false );

        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_SetRasterizerState, true );
        }
    }
}

template<>
void VkCommandProc< VkDepthClampParam >(
    CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        const VkDepthClampParam& param =
            *static_cast< const VkDepthClampParam* >( pParam );

        GfxVkGpuState* pGpuState = pCommandContext->GetGpuState();
        GfxVkGpuState::ConstSizedPipelineState::RasterizationState& rasterizationState =
            pGpuState->constSizedPipelineState.rasterizationState;
        rasterizationState.flag.SetBit(
            GfxVkGpuState::ConstSizedPipelineState::RasterizationState::Flag_DepthClampEnable,
            param.enable != 0 ? true : false );

        GfxVkCommandInfo* pCommandInfo = pCommandContext->GetVkManager()->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_SetRasterizerState, true );
        }
    }
}

template<>
void VkCommandProc< VkActivateUniformBlockParam >(
    CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    GfxVkGpuState* pGpuState = pCommandContext->GetGpuState();

    const VkActivateUniformBlockParam& param =
        *static_cast< const VkActivateUniformBlockParam* >( pParam );

    GfxVkGpuState::DescriptorSetLayoutState& descriptorSetLayoutState =
        pGpuState->descriptorSetLayoutState;
    GfxVkGpuState::DescriptorSetLayoutState::SlotState& descriptorSlotState =
        pGpuState->descriptorSetLayoutState.slotState[ param.slot ];
    GfxVkGpuState::DescriptorResourceState::ResourceState& descriptorResourceState =
        pGpuState->descriptorResourceState.resourceState[ param.slot ];

    descriptorSetLayoutState.maxDescriptorIndex =
        std::max NN_PREVENT_MACRO_FUNC( static_cast< uint32_t >( param.slot ),
            descriptorSetLayoutState.maxDescriptorIndex );

    if ( !descriptorSlotState.flag.GetBit(
        GfxVkGpuState::DescriptorSetLayoutState::SlotState::Flag_IsSet ) )
    {
        descriptorSetLayoutState.availableDescriptorCount++;
        descriptorSlotState.flag.SetBit(
            GfxVkGpuState::DescriptorSetLayoutState::SlotState::Flag_IsSet, true );
    }

    VkBuffer uniformBuffer = CastToVkNonDispatchableObject< VkBuffer >(
        pCommandContext->GetGfxCommandBuffer()->ToData()->hUniformBuffer );

    descriptorSlotState.type = VK_DESCRIPTOR_TYPE_UNIFORM_BUFFER_DYNAMIC;
    descriptorSlotState.shaderStage = Vk::GetShaderStageBits( param.stageBits );
    descriptorResourceState.vkBufferInfo.buffer = uniformBuffer;
    descriptorResourceState.vkBufferInfo.offset = 0;
    descriptorResourceState.vkBufferInfo.range = param.size;

    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        VkManager* pManager = pCommandContext->GetVkManager();

        GfxVkCommandInfo* pCommandInfo = pManager->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_SetDescriptorPool, true );
        }

        pManager->SetUniformBufferBinding( param.slot );
        pManager->SetUniformBlockSize( param.size );
    }
}

template<>
void VkCommandProc< VkSetUniformParam >(
    CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT
{
    if ( pCommandContext->IsContextTypeStateUpdater() )
    {
        const VkSetUniformParam& param =
            *static_cast< const VkSetUniformParam* >( pParam );

        VkManager* pManager = pCommandContext->GetVkManager();

        GfxVkCommandInfo* pCommandInfo = pManager->GetCommandInfo( pCommandContext->GetCurrentCommand() );

        int32_t slot = pManager->GetUniformBufferBinding();
        NN_SDK_REQUIRES( slot >= 0 );
        NN_SDK_REQUIRES( slot < GfxVkGpuState::maxDynamicDescriptorSlot );

        if ( pCommandInfo != NULL )
        {
            pCommandInfo->dynamicOffsets[ slot ] = pManager->GetUniformBufferOffset();
            pCommandInfo->updatedState.SetBit( GfxVkCommandInfo::UpdatedStateFlag_SetUniform, true );

            void *pData = pManager->GetUniformBufferPtr( pCommandInfo->dynamicOffsets[ slot ] + param.offset );
            const void* pSrc = nn::util::ConstBytePtr( &param, sizeof( VkSetUniformParam ) ).Get< const void >();
            memcpy( pData, pSrc, param.size );
        }
    }
}

}
}
}
