﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdlib>

#include <nn/nn_Macro.h>

#include <nn/util/util_BytePtr.h>

#include <nn/nn_SdkLog.h>

#include <nn/os/os_Mutex.h>

#include <nn/gfx/gfx_ShaderInfo.h>
#include <nn/gfx/gfx_ResShaderData.h>
#include <nn/gfx/gfx_ResShaderData-api.nvn.h>

#include <nn/gfx/detail/gfx_Shader-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Device-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Log.h>

#include "gfx_CommonHelper.h"
#include "gfx_NvnHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationNvn8 Target;

namespace {

template< typename TInterface >
int GetShaderSlot( const TInterface& piq, NVNshaderStage stage ) NN_NOEXCEPT
{
    return piq.bindings[ stage ];
}
template<>
int GetShaderSlot< GLSLCprogramInputInfo >(
    const GLSLCprogramInputInfo& piq, NVNshaderStage ) NN_NOEXCEPT
{
    return piq.location;
}
template<>
int GetShaderSlot< GLSLCprogramOutputInfo >(
    const GLSLCprogramOutputInfo& piq, NVNshaderStage ) NN_NOEXCEPT
{
    return piq.location;
}

template< typename TInterface >
bool IsInUbo( const TInterface& ) NN_NOEXCEPT
{
    return false;
}
template<>
bool IsInUbo< GLSLCuniformInfo >( const GLSLCuniformInfo& piq ) NN_NOEXCEPT
{
    return piq.isInUBO != 0;
}

class OnlineCompiledShader
{
public:
    OnlineCompiledShader()
        : m_pMemory()
        , m_pOutput()
    {
    }

    ~OnlineCompiledShader()
    {
        Finalize();
    }

    bool Initialize( DeviceImpl< Target >* pDevice, const GLSLCoutput* pOutput ) NN_NOEXCEPT;

    void Finalize() NN_NOEXCEPT;

    NVNboolean SetShader( NVNprogram* pNvnProgram ) const NN_NOEXCEPT
    {
        return NN_GFX_CALL_NVN_FUNCTION( nvnProgramSetShaders(
            pNvnProgram, m_StageCount, m_NvnShaderData ) );
    }

    int GetInterfaceSlot( ShaderStage stage,
        ShaderInterfaceType interfaceType, const char* pName ) const NN_NOEXCEPT;

    size_t GetScratchMemorySizePerWarp() const NN_NOEXCEPT
    {
        return m_ScratchMemoryPerWarp;
    }

    void GetWorkGroupSize( int* pWorkGroupSizeX, int* pWorkGroupSizeY,
        int* pWorkGroupSizeZ ) const NN_NOEXCEPT;

private:
    template< typename TInterface >
    int SearchInterfaceSlot( const TInterface* pFirst, int count,
        const void* pStringPool, const char* pName, NVNshaderStage stage ) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL( pFirst );

        int stageBits = 0x01 << stage;
        for( int idx = 0; idx < count; ++idx )
        {
            const TInterface& piq = pFirst[ idx ];
            const GLSLCpiqName& nameInfo = piq.nameInfo;
            if( IsInUbo( piq ) )
            {
                continue;
            }
            if( ( piq.stagesReferencedIn & stageBits ) && strncmp( pName, nn::util::ConstBytePtr(
                pStringPool, nameInfo.nameOffset ).Get< const char >() , nameInfo.nameLength ) == 0 )
            {
                int result = GetShaderSlot( piq, stage );
                if( result >= 0 )
                {
                    return result;
                }
            }
        }

        return -1;
    }

    NN_ALIGNAS( 8 ) NVNmemoryPool m_NvnMemoryPool;
    void* m_pMemoryBase;
    void* m_pMemory;
    GLSLCoutput* m_pOutput;
    NVNshaderData m_NvnShaderData[ ShaderStage_End ];
    int m_StageCount;
    size_t m_ScratchMemoryPerWarp;
    const GLSLCprogramReflectionHeader* m_pReflectionHeader;
};

bool OnlineCompiledShader::Initialize( DeviceImpl< Target >* pDevice, const GLSLCoutput* pOutput ) NN_NOEXCEPT
{
    if( ( m_pOutput = static_cast< GLSLCoutput* >( malloc( static_cast< size_t >( pOutput->size ) ) ) ) == NULL )
    {
        return false;
    }
    memcpy( m_pOutput, pOutput, static_cast< size_t >( pOutput->size ) );

    size_t stageDataSize[ ShaderStage_End ] = {};
    const void* pStageData[ ShaderStage_End ] = {};
    m_StageCount = 0;
    size_t memoryPoolSize = 0;
    m_ScratchMemoryPerWarp = 0;
    for( int idxSection = 0; idxSection < static_cast< int >( m_pOutput->numSections ); ++idxSection )
    {
        GLSLCsectionTypeEnum type = m_pOutput->headers[ idxSection ].genericHeader.common.type;
        if( type == GLSLC_SECTION_TYPE_GPU_CODE )
        {
            const GLSLCgpuCodeHeader& gpuCodeHeader = m_pOutput->headers[ idxSection ].gpuCodeHeader;
            memoryPoolSize += gpuCodeHeader.dataSize;
            memoryPoolSize = nn::util::align_up( memoryPoolSize,
                ShaderImpl< Target >::GetBinaryCodeAlignment( pDevice ) );
            const void* pData = nn::util::ConstBytePtr( m_pOutput, gpuCodeHeader.common.dataOffset ).Get();
            stageDataSize[ m_StageCount ] = gpuCodeHeader.dataSize;
            pStageData[ m_StageCount ] =  nn::util::ConstBytePtr( pData, gpuCodeHeader.dataOffset ).Get();
            m_NvnShaderData[ m_StageCount ].control =
                nn::util::ConstBytePtr( pData, gpuCodeHeader.controlOffset ).Get();
            ++m_StageCount;
            m_ScratchMemoryPerWarp = std::max ( m_ScratchMemoryPerWarp,
                static_cast< size_t >( gpuCodeHeader.scratchMemBytesPerWarp ) );
        }
        else if( type == GLSLC_SECTION_TYPE_REFLECTION )
        {
            m_pReflectionHeader = &m_pOutput->headers[ idxSection ].programReflectionHeader;
        }
    }
    memoryPoolSize = nn::util::align_up( memoryPoolSize, NVN_MEMORY_POOL_STORAGE_GRANULARITY );
    if( ( m_pMemoryBase = malloc( memoryPoolSize + NVN_MEMORY_POOL_STORAGE_ALIGNMENT ) ) == NULL )
    {
        return false;
    }
    m_pMemory = nn::util::BytePtr( m_pMemoryBase ).AlignUp(
        NVN_MEMORY_POOL_STORAGE_ALIGNMENT ).Get();

    nn::util::BytePtr pDst( m_pMemory );
    for( int idxStage = 0; idxStage < m_StageCount; ++idxStage )
    {
        memcpy( pDst.Get(), pStageData[ idxStage ], stageDataSize[ idxStage ] );
        pDst.Advance( stageDataSize[ idxStage ] ).AlignUp(
            ShaderImpl< Target >::GetBinaryCodeAlignment( pDevice ) );
    }

    NVNmemoryPoolBuilder memoryPoolBuilder;
    NN_GFX_CALL_NVN_FUNCTION( nvnMemoryPoolBuilderSetDevice(
        &memoryPoolBuilder, pDevice->ToData()->pNvnDevice ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnMemoryPoolBuilderSetDefaults( &memoryPoolBuilder ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnMemoryPoolBuilderSetFlags(
        &memoryPoolBuilder, NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT |
        NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT | NVN_MEMORY_POOL_FLAGS_SHADER_CODE_BIT ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnMemoryPoolBuilderSetStorage(
        &memoryPoolBuilder, m_pMemory, memoryPoolSize ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnMemoryPoolInitialize( &m_NvnMemoryPool, &memoryPoolBuilder ) );
    NVNbufferAddress address = NN_GFX_CALL_NVN_FUNCTION(
        nvnMemoryPoolGetBufferAddress( &m_NvnMemoryPool ) );

    ptrdiff_t offset = 0;
    for( int idxStage = 0; idxStage < m_StageCount; ++idxStage )
    {
        m_NvnShaderData[ idxStage ].data = address + offset;
        offset += stageDataSize[ idxStage ];
        offset = nn::util::align_up( offset, ShaderImpl< Target >::GetBinaryCodeAlignment( pDevice ) );
    }

    return true;
}

void OnlineCompiledShader::Finalize() NN_NOEXCEPT
{
    if( m_pMemoryBase )
    {
        NN_GFX_CALL_NVN_FUNCTION( nvnMemoryPoolFinalize( &m_NvnMemoryPool ) );
        free( m_pMemoryBase );
        m_pMemoryBase = NULL;
        m_pMemory = NULL;
    }

    if( m_pOutput )
    {
        free( m_pOutput );
        m_pOutput = NULL;
    }

    m_pReflectionHeader = NULL;
}

int OnlineCompiledShader::GetInterfaceSlot( ShaderStage stage,
        ShaderInterfaceType interfaceType, const char* pName ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pName );
    NN_SDK_ASSERT_NOT_NULL( m_pOutput );
    NN_SDK_ASSERT_NOT_NULL( m_pReflectionHeader );

    const void* pReflectionData = nn::util::ConstBytePtr(
        m_pOutput, m_pReflectionHeader->common.dataOffset ).Get();
    const void* pStringPool = nn::util::ConstBytePtr(
        pReflectionData, m_pReflectionHeader->stringPoolOffset ).Get();

    NVNshaderStage nvnStage = Nvn::GetShaderStage( stage );

    int ret = -1;
    switch( interfaceType )
    {
    case ShaderInterfaceType_Input:
        {
            ret = SearchInterfaceSlot( nn::util::ConstBytePtr( pReflectionData,
                m_pReflectionHeader->programInputsOffset ).Get< GLSLCprogramInputInfo >(),
                m_pReflectionHeader->numProgramInputs, pStringPool, pName, nvnStage );
        }
        break;
    case ShaderInterfaceType_Output:
        {
            ret = SearchInterfaceSlot( nn::util::ConstBytePtr( pReflectionData,
                m_pReflectionHeader->programOutputsOffset ).Get< GLSLCprogramOutputInfo >(),
                m_pReflectionHeader->numProgramOutputs, pStringPool, pName, nvnStage );
        }
        break;
    case ShaderInterfaceType_ConstantBuffer:
        {
            ret = SearchInterfaceSlot( nn::util::ConstBytePtr( pReflectionData,
                m_pReflectionHeader->uniformBlockOffset ).Get< GLSLCuniformBlockInfo >(),
                m_pReflectionHeader->numUniformBlocks, pStringPool, pName, nvnStage );
        }
        break;
    case ShaderInterfaceType_UnorderedAccessBuffer:
        {
            ret = SearchInterfaceSlot( nn::util::ConstBytePtr( pReflectionData,
                m_pReflectionHeader->ssboOffset ).Get< GLSLCssboInfo >(),
                m_pReflectionHeader->numSsbo, pStringPool, pName, nvnStage );
        }
        break;
    case ShaderInterfaceType_Sampler:
        {
            ret = SearchInterfaceSlot( nn::util::ConstBytePtr( pReflectionData,
                m_pReflectionHeader->uniformOffset ).Get< GLSLCuniformInfo >(),
                m_pReflectionHeader->numUniforms, pStringPool, pName, nvnStage );
        }
        break;
    case ShaderInterfaceType_Image:
        {
            ret = SearchInterfaceSlot( nn::util::ConstBytePtr( pReflectionData,
                m_pReflectionHeader->uniformOffset ).Get< GLSLCuniformInfo >(),
                m_pReflectionHeader->numUniforms, pStringPool, pName, nvnStage );
        }
    default:
        break;
    }

    return ret;
}

void OnlineCompiledShader::GetWorkGroupSize( int* pOutWorkGroupSizeX,
    int* pOutWorkGroupSizeY, int* pOutWorkGroupSizeZ ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pOutWorkGroupSizeX );
    NN_SDK_ASSERT_NOT_NULL( pOutWorkGroupSizeY );
    NN_SDK_ASSERT_NOT_NULL( pOutWorkGroupSizeZ );
    NN_SDK_ASSERT_NOT_NULL( m_pOutput );
    NN_SDK_ASSERT_NOT_NULL( m_pReflectionHeader );

    const void* pReflectionData = nn::util::ConstBytePtr(
        m_pOutput, m_pReflectionHeader->common.dataOffset ).Get();
    const GLSLCperStageShaderInfo* pPerStageShaderInfo = nn::util::ConstBytePtr(
        pReflectionData, m_pReflectionHeader->shaderInfoOffset ).Get< GLSLCperStageShaderInfo >();
    const GLSLCshaderInfoCompute& shaderInfoCompute =
        pPerStageShaderInfo->shaderInfo[ NVN_SHADER_STAGE_COMPUTE ].compute;
    *pOutWorkGroupSizeX = shaderInfoCompute.workGroupSize[ 0 ];
    *pOutWorkGroupSizeY = shaderInfoCompute.workGroupSize[ 1 ];
    *pOutWorkGroupSizeZ = shaderInfoCompute.workGroupSize[ 2 ];
}

ShaderInitializeResult NvnGlslcCompile( ShaderImpl< Target >* pThis,
    const ShaderInfo& info, DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    class CompileObjectFinalizer
    {
    public:
        explicit CompileObjectFinalizer( GLSLCcompileObject* pCompileObject )
            : m_pCompileObject( pCompileObject )
        {
        }
        ~CompileObjectFinalizer()
        {
            GlslcDll::GetInstance().GlslcFinalize( m_pCompileObject );
        }
    private:
        GLSLCcompileObject* m_pCompileObject;
    };

    GLSLCcompileObject compileObject;
    if( GlslcDll::GetInstance().GlslcInitialize( &compileObject ) == 0 ||
        compileObject.initStatus != GLSLC_INIT_SUCCESS )
    {
        return ShaderInitializeResult_SetupFailed;
    }

    CompileObjectFinalizer compileObjectFinalizer( &compileObject );

    GLSLCoptions& options = compileObject.options;
    {
        options.optionFlags.glslSeparable = info.IsSeparationEnabled() ? 1 : 0;
        options.optionFlags.outputAssembly = 0;
        options.optionFlags.outputGpuBinaries = 1;
        options.optionFlags.outputPerfStats = 0;
        options.optionFlags.outputShaderReflection = 1;
        options.optionFlags.language = GLSLC_LANGUAGE_GLSL;
        options.optionFlags.outputDebugInfo = GLSLC_DEBUG_LEVEL_G0;
        options.optionFlags.spillControl = NO_SPILL;
        options.optionFlags.outputThinGpuBinaries = IsThinBinaryAvailable() ? 1 : 0;
        options.optionFlags.tessellationAndPassthroughGS = 0;
        options.includeInfo.numPaths = 0;
        options.includeInfo.paths = NULL;
        options.xfbVaryingInfo.numVaryings = 0; // TODO
        options.xfbVaryingInfo.varyings = NULL;
        options.forceIncludeStdHeader = NULL;
    }

    std::string sources[ ShaderStage_End ];
    const char* ppSources[ ShaderStage_End ];
    NVNshaderStage stages[ ShaderStage_End ];
    compileObject.input.sources = ppSources;
    compileObject.input.stages = stages;
    compileObject.input.count = 0;
    for( int idxStage = 0; idxStage < static_cast< int >( ShaderStage_End ); ++idxStage )
    {
        ShaderStage stage = static_cast< ShaderStage >( idxStage );
        if( const ShaderCode* pShaderCode = static_cast< const ShaderCode* >( info.GetShaderCodePtr( stage ) ) )
        {
            sources[ compileObject.input.count ].assign(
                static_cast< const char* >( pShaderCode->pCode ), pShaderCode->codeSize );
            ppSources[ compileObject.input.count ] = sources[ compileObject.input.count ].c_str();
            stages[ compileObject.input.count ] = Nvn::GetShaderStage( stage );
            ++compileObject.input.count;
        }
    }

    uint8_t compileResult = GlslcDll::GetInstance().GlslcCompile( &compileObject );

    const GLSLCcompilationStatus& compilationStatus = *compileObject.lastCompiledResults->compilationStatus;
    if( compilationStatus.allocError )
    {
        NN_DETAIL_GFX_ERROR( "Allocation error in NvnGlslc" );
        return ShaderInitializeResult_SetupFailed;
    }
    if( compileResult == 0 || !compilationStatus.success )
    {
        if( compilationStatus.infoLog )
        {
            NN_DETAIL_GFX_ERROR( "%s\n", compilationStatus.infoLog );
        }
        return ShaderInitializeResult_SetupFailed;
    }

    pThis->ToData()->pOnlineCompiledShader = malloc( sizeof( OnlineCompiledShader ) );
    OnlineCompiledShader* pOnlineCompiledShader =
        new ( pThis->ToData()->pOnlineCompiledShader.ptr ) OnlineCompiledShader();
    if( !pOnlineCompiledShader->Initialize( pDevice, compileObject.lastCompiledResults->glslcOutput ) )
    {
        return ShaderInitializeResult_SetupFailed;
    }

    return ShaderInitializeResult_Success;
}

ShaderInitializeResult InitializeSourceShader( ShaderImpl< Target >* pThis,
    const ShaderInfo& info, DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
#if !defined( NN_GFX_CONFIG_BUILD_FOR_PCTOOL )
    class MutexLocker
    {
    public:
        explicit MutexLocker( nn::os::Mutex* pMutex )
            : m_pMutex( pMutex )
        {
            NN_GFX_CALL_NNOS_FUNCTION( m_pMutex->Lock() );
        }
        ~MutexLocker()
        {
            NN_GFX_CALL_NNOS_FUNCTION( m_pMutex->Unlock() );
        }
    private:
        nn::os::Mutex* m_pMutex;
    };

    static nn::os::Mutex s_Mutex( false );
    MutexLocker locker( &s_Mutex );
#endif

    NN_SDK_ASSERT_NOT_NULL( pThis );

    GLSLCversion version = GlslcDll::GetInstance().GlslcGetVersion();
    int majorVersionMinimum;
    int majorVersionMaximum;
    int minorVersionMinimum;
    int minorVersionMaximum;
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
        NVN_DEVICE_INFO_GLSLC_MIN_SUPPORTED_GPU_CODE_MAJOR_VERSION, &majorVersionMinimum ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
        NVN_DEVICE_INFO_GLSLC_MAX_SUPPORTED_GPU_CODE_MAJOR_VERSION, &majorVersionMaximum ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
        NVN_DEVICE_INFO_GLSLC_MIN_SUPPORTED_GPU_CODE_MINOR_VERSION, &minorVersionMinimum ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
        NVN_DEVICE_INFO_GLSLC_MAX_SUPPORTED_GPU_CODE_MINOR_VERSION, &minorVersionMaximum ) );
    if( static_cast< int >( version.gpuCodeVersionMajor ) < majorVersionMinimum ||
        static_cast< int >( version.gpuCodeVersionMajor ) > majorVersionMaximum ||
        static_cast< int >( version.gpuCodeVersionMinor ) < minorVersionMinimum ||
        static_cast< int >( version.gpuCodeVersionMinor ) > minorVersionMaximum )
    {
        return ShaderInitializeResult_SetupFailed;
    }

    ShaderInitializeResult initializeResult = NvnGlslcCompile( pThis, info, pDevice );
    if( initializeResult != ShaderInitializeResult_Success )
    {
        return initializeResult;
    }

    NN_SDK_ASSERT_NOT_NULL( pThis->ToData()->pOnlineCompiledShader.ptr );
    if( static_cast< OnlineCompiledShader* >( pThis->ToData()->pOnlineCompiledShader )->SetShader(
        pThis->ToData()->pNvnProgram ) != NVN_TRUE )
    {
        return ShaderInitializeResult_SetupFailed;
    }

    return ShaderInitializeResult_Success;
}

void ReassembleControlSection( void* pDestination,
    const NvnDecomposedControlSection* pDecomposedControlSection ) NN_NOEXCEPT
{
    const struct ReassembleControlSectionInfo
    {
        int offsetDataOffset;
        int offsetDataSize;
        nn::util::BinTPtr< const void > NvnDecomposedControlSection::* pSrcData;
    } ReassembleControlSectionInfoArray[] =
    {
        { 20, 24, &NvnDecomposedControlSection::pAssemblyData }, // アセンブリデータ
        { 32, 28, &NvnDecomposedControlSection::pAssemblyLocalsData }, // 定数データ
        { 1776, 1780, &NvnDecomposedControlSection::pSpecializationData }, // 特殊化情報
        { 1992, 1996, &NvnDecomposedControlSection::pPragmaData }, // pragma 情報
        { 2024, 2028, &NvnDecomposedControlSection::pUniform64InfoData } // 64-bit uniform 情報
    };

    NN_SDK_ASSERT_NOT_NULL( pDestination );
    NN_SDK_ASSERT_NOT_NULL( pDecomposedControlSection );

    const void* const pMetaData = pDecomposedControlSection->pMetaData.Get();
    NN_SDK_ASSERT_NOT_NULL( pMetaData );

    // メタデータの構築
    const int offsetAssemblyDataOffset = 20;
    ptrdiff_t assemblyDataOffset = *nn::util::ConstBytePtr(
        pMetaData, offsetAssemblyDataOffset ).Get< int32_t >();
    memcpy( pDestination, pMetaData, static_cast< size_t >( assemblyDataOffset ) );

    // 各セクションの構築
    for( int idxSection = 0, sectionCount = NN_GFX_ARRAY_LENGTH( ReassembleControlSectionInfoArray );
        idxSection < sectionCount; ++idxSection )
    {
        const ReassembleControlSectionInfo& target = ReassembleControlSectionInfoArray[ idxSection ];
        if( const void* pSrcData = ( pDecomposedControlSection->*target.pSrcData ).Get() )
        {
            ptrdiff_t dataOffset = *nn::util::ConstBytePtr( pMetaData, target.offsetDataOffset ).Get< int32_t >();
            void* pDstData = nn::util::BytePtr( pDestination, dataOffset ).Get();
            size_t dataSize = *nn::util::ConstBytePtr( pMetaData, target.offsetDataSize ).Get< int32_t >();
            memcpy( pDstData, pSrcData, dataSize );
        }
    }
}

ShaderInitializeResult InitializeBinaryShader( ShaderImpl< Target >* pThis, const ShaderInfo& info ) NN_NOEXCEPT
{
    if( info.ToData()->flags.GetBit( ShaderInfo::DataType::Flag_ResShader ) )
    {
        int sourceCount = 0;
        NVNshaderDataExt sources[ ShaderStage_End ];
        for( int idxStage = 0; idxStage < static_cast< int >( ShaderStage_End ); ++idxStage )
        {
            ShaderStage stage = static_cast<ShaderStage>( idxStage );
            if( const NvnShaderCode* pNvnShaderCode = static_cast<
                const NvnShaderCode* >( info.GetShaderCodePtr( stage ) ) )
            {
                NVNshaderDataExt& source = sources[ sourceCount++ ];
                source.data = pNvnShaderCode->dataAddress;
                source.control = pNvnShaderCode->pControl.Get();
                source.debugDataHash = pNvnShaderCode->pNvnDebugDataHash.Get();
                if( const NvnDecomposedControlSection* pDecomposedControlSection =
                    pNvnShaderCode->pDecomposedControlSection.Get() )
                {
                    // コントロールセクションの再構築のために const_cast
                    void* pControl = const_cast<void*>( pNvnShaderCode->pControl.Get() );
                    ReassembleControlSection( pControl, pDecomposedControlSection );
                }
            }
        }

        if( NN_GFX_CALL_NVN_FUNCTION( nvnProgramSetShadersExt(
            pThis->ToData()->pNvnProgram, sourceCount, sources ) ) == NVN_FALSE )
        {
            // たとえば実機版のシェーダーを PC 上で初期化しようとするなど、構成が異なる場合はここにくる
            return ShaderInitializeResult_SetupFailed;
        }

        if( info.ToData()->flags.GetBit( ShaderInfo::DataType::Flag_ResShader ) )
        {
            const ResShaderProgramData* pResShaderProgram = nn::util::ConstBytePtr( &info,
                -static_cast< ptrdiff_t >( offsetof( ResShaderProgramData, info ) ) ).Get< ResShaderProgramData >();
            pThis->ToData()->pReflection = pResShaderProgram->pShaderReflection.Get();
        }
    }
    else
    {
        int sourceCount = 0;
        NVNshaderData sources[ ShaderStage_End ];
        for( int idxStage = 0; idxStage < static_cast< int >( ShaderStage_End ); ++idxStage )
        {
            ShaderStage stage = static_cast<ShaderStage>( idxStage );
            if( const NVNshaderData* shaderData = static_cast<
                const NVNshaderData* >( info.GetShaderCodePtr( stage ) ) )
            {
                sources[ sourceCount++ ] = *shaderData;
            }
        }

        if( NN_GFX_CALL_NVN_FUNCTION( nvnProgramSetShaders(
            pThis->ToData()->pNvnProgram, sourceCount, sources ) ) == NVN_FALSE )
        {
            return ShaderInitializeResult_SetupFailed;
        }
    }

    return ShaderInitializeResult_Success;
}

}

// we don't actually have the intermediate languge compiler yet
// so instead, we're handed an NVNprogram that the user constructed

size_t ShaderImpl< Target >::GetBinaryCodeAlignment( DeviceImpl< Target >* ) NN_NOEXCEPT
{
    return 256;
}

ShaderImpl< Target >::ShaderImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

ShaderImpl< Target >::~ShaderImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

ShaderInitializeResult ShaderImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_STATIC_ASSERT( sizeof( nvnProgram ) == sizeof( NVNprogram ) );

    this->pNvnProgram = &this->nvnProgram;
    this->flags = info.ToData()->flags;

    NVNboolean isProgramOK = NN_GFX_CALL_NVN_FUNCTION(
        nvnProgramInitialize( this->pNvnProgram , pDevice->ToData()->pNvnDevice ) );
    NN_SDK_ASSERT( isProgramOK );
    NN_UNUSED( isProgramOK );

    ShaderInitializeResult result = ShaderInitializeResult_Success;

    switch( info.GetCodeType() )
    {
    case ShaderCodeType_Source:
        {
            result = InitializeSourceShader( this, info, pDevice );
        }
        break;
        case ShaderCodeType_Binary:
        {
            result = InitializeBinaryShader( this, info );
        }
        break;
        default:
        {
            result = ShaderInitializeResult_InvalidType;
        }
        break;
    }

    if ( result != ShaderInitializeResult_Success )
    {
        return result;
    }

    if( !info.IsSeparationEnabled() )
    {
        this->nvnShaderStageBits = info.GetShaderCodePtr( ShaderStage_Compute ) ?
            NVN_SHADER_STAGE_COMPUTE_BIT : NVN_SHADER_STAGE_ALL_GRAPHICS_BITS;
    }

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;

    return result;
}

void ShaderImpl< Target >::Finalize( DeviceImpl< Target >* ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnProgramFinalize( this->pNvnProgram ) );
    this->pNvnProgram = NULL;

    if( this->pOnlineCompiledShader )
    {
        static_cast< OnlineCompiledShader* >( this->pOnlineCompiledShader )->~OnlineCompiledShader();
        free( this->pOnlineCompiledShader );
        this->pOnlineCompiledShader = NULL;
    }

    this->state = State_NotInitialized;
}

int ShaderImpl< Target >::GetInterfaceSlot( ShaderStage stage,
    ShaderInterfaceType shaderInterfaceType, const char* pName ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pName );
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    if( this->pOnlineCompiledShader )
    {
        return static_cast< const OnlineCompiledShader* >(
            this->pOnlineCompiledShader )->GetInterfaceSlot( stage, shaderInterfaceType, pName );
    }

    NN_SDK_ASSERT_NOT_NULL( this->pReflection.ptr );

    static nn::util::BinTPtr< ResShaderReflectionStageData >
        const ResShaderReflectionData::* s_pResShaderReflectionStages[] =
    {
        &ResShaderReflectionData::pVertexReflection,
        &ResShaderReflectionData::pHullReflection,
        &ResShaderReflectionData::pDomainReflection,
        &ResShaderReflectionData::pGeometryReflection,
        &ResShaderReflectionData::pPixelReflection,
        &ResShaderReflectionData::pComputeReflection
    };

    static nn::util::BinTPtr< nn::util::ResDic >
        const ResShaderReflectionStageData::*s_pInterfaceDics[] =
    {
        &ResShaderReflectionStageData::pShaderInputDic,
        &ResShaderReflectionStageData::pShaderOutputDic,
        &ResShaderReflectionStageData::pSamplerDic,
        &ResShaderReflectionStageData::pConstantBufferDic,
        &ResShaderReflectionStageData::pUnorderedAccessBufferDic,
        &ResShaderReflectionStageData::pImageDic
    };

    static int32_t const ResShaderReflectionStageData::*s_pOffsets[] =
    {
        NULL,
        &ResShaderReflectionStageData::offsetShaderOutput,
        &ResShaderReflectionStageData::offsetSampler,
        &ResShaderReflectionStageData::offsetConstantBuffer,
        &ResShaderReflectionStageData::offsetUnorderedAccessBuffer,
        &ResShaderReflectionStageData::offsetImage
    };

    static nn::util::BinTPtr< nn::util::ResDic >
        const ResShaderReflectionStageData2::*s_pInterfaceDics2[] =
    {
        &ResShaderReflectionStageData2::pSeparateTextureDic,
        &ResShaderReflectionStageData2::pSeparateSamplerDic
    };

    static int32_t const ResShaderReflectionStageData2::*s_pOffsets2[] =
    {
        &ResShaderReflectionStageData2::offsetSeparateTexture,
        &ResShaderReflectionStageData2::offsetSeparateSampler
    };

    const ResShaderReflectionData* pResShaderReflection =
        static_cast< const ResShaderReflectionData* >( this->pReflection );
    if( const ResShaderReflectionStageData* pResShaderReflectionStage =
        ( pResShaderReflection->*s_pResShaderReflectionStages[ stage ] ).Get() )
    {
        const nn::util::ResDic* pResDic = NULL;
        int offset = 0;
        if( shaderInterfaceType < nn::gfx::ShaderInterfaceType_SeparateTexture )
        {
            pResDic = ( pResShaderReflectionStage->*s_pInterfaceDics[ shaderInterfaceType ] ).Get();
            offset = s_pOffsets[ shaderInterfaceType ] == NULL
                ? 0 : pResShaderReflectionStage->*s_pOffsets[ shaderInterfaceType ];
        }
        else if( const ResShaderReflectionStageData2* pResShaderReflectionStage2 =
            pResShaderReflectionStage->pReflectionStageData2.Get() )
        {
            int shaderInterfaceType2 = shaderInterfaceType - nn::gfx::ShaderInterfaceType_SeparateTexture;
            pResDic = ( pResShaderReflectionStage2->*s_pInterfaceDics2[ shaderInterfaceType2 ] ).Get();
            offset = pResShaderReflectionStage2->*s_pOffsets2[ shaderInterfaceType2 ];
        }
        if( pResDic )
        {
            int idxFound = pResDic->FindIndex( pName );
            if( idxFound >= 0 )
            {
                return pResShaderReflectionStage->pShaderSlotArray.Get()[ offset + idxFound ];
            }
        }
    }

    return -1;
}

void ShaderImpl< Target >::GetWorkGroupSize( int* pOutWorkGroupSizeX,
    int* pOutWorkGroupSizeY, int* pOutWorkGroupSizeZ ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutWorkGroupSizeX );
    NN_SDK_REQUIRES_NOT_NULL( pOutWorkGroupSizeY );
    NN_SDK_REQUIRES_NOT_NULL( pOutWorkGroupSizeZ );
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    if( this->pOnlineCompiledShader )
    {
        return static_cast< const OnlineCompiledShader* >(
            this->pOnlineCompiledShader )->GetWorkGroupSize(
            pOutWorkGroupSizeX, pOutWorkGroupSizeY, pOutWorkGroupSizeZ );
    }

    NN_SDK_ASSERT_NOT_NULL( this->pReflection.ptr );

    const ResShaderReflectionData* pResShaderReflection =
        static_cast< const ResShaderReflectionData* >( this->pReflection );
    const ResShaderReflectionStageData* pComputeReflection
        = pResShaderReflection->pComputeReflection.Get();
    NN_SDK_ASSERT_NOT_NULL( pComputeReflection );
    *pOutWorkGroupSizeX = pComputeReflection->computeWorkGroupSizeX;
    *pOutWorkGroupSizeY = pComputeReflection->computeWorkGroupSizeY;
    *pOutWorkGroupSizeZ = pComputeReflection->computeWorkGroupSizeZ;
}

}
}
}
