﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx/gfx_ShaderInfo.h>

#include <nn/gfx/detail/gfx_Shader-api.gx.2.h>

#include "gfx_GxHelper.h"

namespace nn {
namespace gfx {
namespace detail {

namespace {

template< typename TGx2Shader >
void InvalidateShaderPtr( const void* pShader )
{
    if( const TGx2Shader* pGx2Shader = static_cast< const TGx2Shader* >( pShader ) )
    {
        if ( pGx2Shader->shaderPtr )
        {
            NN_GFX_CALL_GX_FUNCTION( GX2Invalidate( GX2_INVALIDATE_CPU,
                pGx2Shader->shaderPtr, pGx2Shader->shaderSize ) );
        }
        else
        {
            GX2RInvalidateBuffer( &pGx2Shader->shaderProgram, GX2R_OPTION_NO_GPU_INVALIDATE );
        }
    }
}

template< typename TGx2Shader, typename TInterface,
    TInterface* (*pGx2GetLocationFunc)( const TGx2Shader* pShader, const char* pName ) >
int GetLocation( const void* pShader, const char* pName )
{
    const TInterface* pInterface = NN_GFX_CALL_GX_FUNCTION( pGx2GetLocationFunc(
        reinterpret_cast< const TGx2Shader* >( pShader ), pName ) );
    return pInterface ? static_cast< int >( pInterface->location ) : -1;
}

}

typedef ApiVariationGx2 Target;

size_t ShaderImpl< Target >::GetBinaryCodeAlignment( DeviceImpl< Target >* ) NN_NOEXCEPT
{
    return GX2_SHADER_ALIGNMENT;
}

ShaderImpl< Target >::ShaderImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

ShaderImpl< Target >::~ShaderImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

ShaderInitializeResult ShaderImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );

    this->pDevice = pDevice;
    this->flags = info.ToData()->flags;

    // Cafe only supports binary code
    if( info.GetCodeType() != ShaderCodeType_Binary )
    {
        return ShaderInitializeResult_InvalidFormat;
    }

    this->pVertexShader = info.GetShaderCodePtr( nn::gfx::ShaderStage_Vertex );
    this->pGeometryShader = info.GetShaderCodePtr( nn::gfx::ShaderStage_Geometry );
    this->pPixelShader = info.GetShaderCodePtr( nn::gfx::ShaderStage_Pixel );
    this->pComputeShader = info.GetShaderCodePtr( nn::gfx::ShaderStage_Compute );
    InvalidateShaderPtr< GX2VertexShader >( this->pVertexShader );
    InvalidateShaderPtr< GX2GeometryShader >( this->pGeometryShader );
    InvalidateShaderPtr< GX2PixelShader >( this->pPixelShader );
    InvalidateShaderPtr< GX2ComputeShader >( this->pComputeShader );

    this->state = State_Initialized;
    return ShaderInitializeResult_Success;
}

void ShaderImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    this->state = State_NotInitialized;
}

int ShaderImpl< Target >::GetInterfaceSlot( ShaderStage stage,
    ShaderInterfaceType shaderInterfaceType, const char* pName ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pName );
    NN_SDK_REQUIRES( this->state == State_Initialized );

    int idxStage = static_cast< int >( stage );
    int ret = GX2_UNIFORM_VAR_INVALID_OFFSET;

    detail::Ptr< const void > pShaders[] =
    {
        this->pVertexShader,
        NULL,
        NULL,
        this->pGeometryShader,
        this->pPixelShader,
        this->pComputeShader
    };
    NN_SDK_ASSERT( pShaders[ idxStage ] );

    switch( shaderInterfaceType )
    {
    case ShaderInterfaceType_Input:
        {
            NN_SDK_ASSERT( stage == ShaderStage_Vertex );
            ret = NN_GFX_CALL_GX_FUNCTION( GX2GetVertexAttribVarLocation( this->pVertexShader, pName ) );
        }
        break;
    case ShaderInterfaceType_ConstantBuffer:
    case ShaderInterfaceType_UnorderedAccessBuffer:
        {
            static int (*s_pGetConstantBufferLocationTable [])( const void*, const char* ) =
            {
                GetLocation< GX2VertexShader, GX2UniformBlock, GX2GetVertexUniformBlock >,
                NULL,
                NULL,
                GetLocation< GX2GeometryShader, GX2UniformBlock, GX2GetGeometryUniformBlock >,
                GetLocation< GX2PixelShader, GX2UniformBlock, GX2GetPixelUniformBlock >,
                GetLocation< GX2ComputeShader, GX2UniformBlock, GX2GetComputeUniformBlock >
            };
            ret = s_pGetConstantBufferLocationTable[ idxStage ]( pShaders[ idxStage ], pName );
        }
        break;
    case ShaderInterfaceType_Sampler:
        {
            static int (*s_pGetSamplerLocationTable[])( const void*, const char* ) =
            {
                GetLocation< GX2VertexShader, GX2SamplerVar, GX2GetVertexSamplerVar >,
                NULL,
                NULL,
                GetLocation< GX2GeometryShader, GX2SamplerVar, GX2GetGeometrySamplerVar >,
                GetLocation< GX2PixelShader, GX2SamplerVar, GX2GetPixelSamplerVar >,
                GetLocation< GX2ComputeShader, GX2SamplerVar, GX2GetComputeSamplerVar >
            };
            ret = s_pGetSamplerLocationTable[ idxStage ]( pShaders[ idxStage ], pName );
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
    return ret;
}

void ShaderImpl< Target >::GetWorkGroupSize( int* pOutWorkGroupSizeX,
    int* pOutWorkGroupSizeY, int* pOutWorkGroupSizeZ ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutWorkGroupSizeX );
    NN_SDK_REQUIRES_NOT_NULL( pOutWorkGroupSizeY );
    NN_SDK_REQUIRES_NOT_NULL( pOutWorkGroupSizeZ );
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    NN_SDK_REQUIRES( this->pComputeShader.ptr );

    const GX2ComputeShader* pComputeShader = this->pComputeShader;
    *pOutWorkGroupSizeX = pComputeShader->layout_size_x;
    *pOutWorkGroupSizeY = pComputeShader->layout_size_y;
    *pOutWorkGroupSizeZ = pComputeShader->layout_size_z;
}

}
}
}
