﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <memory>

#include <nn/util/util_BitArray.h>
#include <nn/util/util_Decompression.h>

#include <nn/gfx/gfx_ShaderInfo.h>

#include <nn/gfx/detail/gfx_Shader-api.gl.4.h>

#include "gfx_GlHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationGl4 Target;

namespace {

static GlHandle ShaderImpl< Target >::DataType::* const pStageProgram[] =
{
    &ShaderImpl< Target >::DataType::hVertexProgram,
    &ShaderImpl< Target >::DataType::hHullProgram,
    &ShaderImpl< Target >::DataType::hDomainProgram,
    &ShaderImpl< Target >::DataType::hGeometryProgram,
    &ShaderImpl< Target >::DataType::hPixelProgram,
    &ShaderImpl< Target >::DataType::hComputeProgram
};

void DetachAndDeleteShaders( GlHandle hProgram ) NN_NOEXCEPT
{
    GLsizei count;
    GLuint shader;
    NN_SDK_ASSERT( IsValid( hProgram ) );
    for( ; ; )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glGetAttachedShaders( hProgram, 1, &count, &shader ) );
        if( count == 0 )
        {
            break;
        }
        NN_SDK_ASSERT( IsValid( shader ) );
        NN_GFX_CALL_GL_FUNCTION( ::glDetachShader( hProgram, shader ) );
        NN_GFX_CALL_GL_FUNCTION( ::glDeleteShader( shader ) );
    }
    NN_GFX_GL_ASSERT();
}

void DeletePrograms( ShaderImpl< Target >* pThis ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pThis );
    for( int idxStage = 0; idxStage < static_cast< int >( ShaderStage_End ); ++idxStage )
    {
        if( IsValid( pThis->ToData()->*pStageProgram[ idxStage ] ) )
        {
            NN_GFX_CALL_GL_FUNCTION( ::glDeleteProgram(
                pThis->ToData()->*pStageProgram[ idxStage ] ) );
            NN_GFX_GL_ASSERT();
            pThis->ToData()->*pStageProgram[ idxStage ] = GlInvalidHandle;
        }
    }
}

template< GLenum ResourceType, bool ( *IsTargetType )( GLenum ) >
void AssignShaderSlots( GlHandle hProgram ) NN_NOEXCEPT
{
    // 他 API と挙動を揃えるためステージごとにバインディング範囲を分割
    // TODO: イメージ変数などはこれにより足りなくなりがちなので対策が必要かもしれない

    // 0 でない explicit binding は上書きしない
    const int maxBinding = 255;
    const int bitArrayLength = maxBinding + 1;
    Bit32 bitArrayMemory[ ( bitArrayLength + 31 ) >> 5 ] = {};
    nn::util::BitArray usedBindings( bitArrayMemory, sizeof( bitArrayMemory ), bitArrayLength );

    GLint resourceCount = 0;
    NN_GFX_CALL_GL_FUNCTION( ::glGetProgramInterfaceiv( hProgram,
        ResourceType, GL_ACTIVE_RESOURCES, &resourceCount ) );

    GLint binding;
    const GLenum bufferProps[] = { GL_BUFFER_BINDING };
    const GLenum uniformProps[] = { GL_TYPE, GL_LOCATION, GL_ARRAY_SIZE };
    const int propCount = NN_ARRAY_SIZE( uniformProps );
    GLint uniformValues[ propCount ];
    for( int idxResource = 0; idxResource < resourceCount; ++idxResource )
    {
        int arrayLength = 1;
        if( NN_STATIC_CONDITION( ResourceType == GL_UNIFORM ) )
        {
            NN_GFX_CALL_GL_FUNCTION( ::glGetProgramResourceiv( hProgram,
                ResourceType, idxResource, propCount, uniformProps, propCount, NULL, uniformValues ) );
            if( !IsTargetType( uniformValues[ 0 ] ) )
            {
                continue;
            }
            NN_GFX_CALL_GL_FUNCTION( ::glGetUniformiv( hProgram, uniformValues[ 1 ], &binding ) );
            arrayLength = uniformValues[ 2 ];
        }
        else
        {
            NN_GFX_CALL_GL_FUNCTION( ::glGetProgramResourceiv( hProgram,
                ResourceType, idxResource, 1, bufferProps, 1, NULL, &binding ) );
        }
        if( binding != 0 )
        {
            for( int endBinding = std::min NN_PREVENT_MACRO_FUNC ( binding + arrayLength, maxBinding );
                binding < endBinding && binding < maxBinding; ++binding )
            {
                usedBindings.set( binding, true );
            }
        }
    }

    GLint autoBinding = 0;
    for( int idxResource = 0; idxResource < resourceCount; ++idxResource )
    {
        int arrayLength = 1;
        if( NN_STATIC_CONDITION( ResourceType == GL_UNIFORM ) )
        {
            NN_GFX_CALL_GL_FUNCTION( ::glGetProgramResourceiv( hProgram,
                ResourceType, idxResource, propCount, uniformProps, propCount, NULL, uniformValues ) );
            if( !IsTargetType( uniformValues[ 0 ] ) )
            {
                continue;
            }
            NN_GFX_CALL_GL_FUNCTION( ::glGetUniformiv( hProgram, uniformValues[ 1 ], &binding ) );
            arrayLength = uniformValues[ 2 ];
        }
        else
        {
            NN_GFX_CALL_GL_FUNCTION( ::glGetProgramResourceiv( hProgram,
                ResourceType, idxResource, 1, bufferProps, 1, NULL, &binding ) );
        }

        if( binding == 0 )
        {
            // 要素を連続して割り当てられる場所を探索
            while( autoBinding + arrayLength < maxBinding )
            {
                int element = 0;
                for( ; element < arrayLength; ++element )
                {
                    if( usedBindings.test( autoBinding + element ) )
                    {
                        autoBinding += element + 1;
                        break;
                    }
                }
                if( element >= arrayLength )
                {
                    break;
                }
            }
            NN_SDK_ASSERT( autoBinding + arrayLength < maxBinding );

            if( NN_STATIC_CONDITION( ResourceType == GL_UNIFORM_BLOCK ) )
            {
                NN_GFX_CALL_GL_FUNCTION( ::glUniformBlockBinding( hProgram, idxResource, autoBinding ) );
            }
            else if( NN_STATIC_CONDITION( ResourceType == GL_SHADER_STORAGE_BLOCK ) )
            {
                NN_GFX_CALL_GL_FUNCTION( ::glShaderStorageBlockBinding( hProgram, idxResource, autoBinding ) );
            }
            else if( NN_STATIC_CONDITION( ResourceType == GL_UNIFORM ) )
            {
                for( int element = 0; element < arrayLength; ++element )
                {
                    NN_GFX_CALL_GL_FUNCTION( ::glProgramUniform1i( hProgram,
                        uniformValues[ 1 ] + element, autoBinding + element ) );
                }
            }
            autoBinding += arrayLength;
        }
    }
} // NOLINT

template< GLenum ResourceType, GLenum MaxBindings, bool ( *IsTargetType )( GLenum ) >
void AssignShaderSlots( GlHandle hProgram, nn::gfx::ShaderStage stage ) NN_NOEXCEPT
{
    // explicit binding を上書きしている
    int maxBindings;
    NN_GFX_CALL_GL_FUNCTION( ::glGetIntegerv( MaxBindings, &maxBindings ) );
    int bindingCountPerStage = maxBindings / ShaderStage_End;
    int bindingBase = bindingCountPerStage * stage;
    int resourceCount;
    NN_GFX_CALL_GL_FUNCTION( ::glGetProgramInterfaceiv( hProgram,
        ResourceType, GL_ACTIVE_RESOURCES, &resourceCount ) );
    int binding = bindingBase;
    for( int idxResource = 0; idxResource < resourceCount; ++idxResource )
    {
        if( NN_STATIC_CONDITION( ResourceType == GL_UNIFORM_BLOCK ) )
        {
            NN_GFX_CALL_GL_FUNCTION( ::glUniformBlockBinding( hProgram, idxResource, binding++ ) );
        }
        else if( NN_STATIC_CONDITION( ResourceType == GL_SHADER_STORAGE_BLOCK ) )
        {
            NN_GFX_CALL_GL_FUNCTION( ::glShaderStorageBlockBinding( hProgram, idxResource, binding++ ) );
        }
        else if( NN_STATIC_CONDITION( ResourceType == GL_UNIFORM ) )
        {
            const GLenum props[] = { GL_TYPE, GL_LOCATION, GL_ARRAY_SIZE };
            const int propCount = NN_ARRAY_SIZE( props );
            GLint values[ propCount ];
            NN_GFX_CALL_GL_FUNCTION( ::glGetProgramResourceiv( hProgram,
                ResourceType, idxResource, propCount, props, propCount, NULL, values ) );
            if( IsTargetType( values[ 0 ] ) )
            {
                for( int element = 0; element < values[ 2 ]; ++element )
                {
                    NN_GFX_CALL_GL_FUNCTION( ::glProgramUniform1i( hProgram, values[ 1 ] + element, binding++ ) );
                }
            }
        }
    }
    NN_SDK_ASSERT( binding <= bindingBase + bindingCountPerStage );
    NN_GFX_GL_ASSERT();
}

template< bool IsArray >
ShaderInitializeResult InitializeSourceShader(
    ShaderImpl< Target >* pThis, const ShaderInfo& info ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pThis );

    static const GLenum s_StageTable[] =
    {
        GL_VERTEX_SHADER,
        GL_TESS_CONTROL_SHADER,
        GL_TESS_EVALUATION_SHADER,
        GL_GEOMETRY_SHADER,
        GL_FRAGMENT_SHADER,
        GL_COMPUTE_SHADER
    };

    if( !info.IsSeparationEnabled() )
    {
        pThis->ToData()->hCombinedProgram = NN_GFX_CALL_GL_FUNCTION( ::glCreateProgram() );
    }

    for( int idxStage = 0; idxStage < ShaderStage_End; ++idxStage )
    {
        if( const void* pCode = info.GetShaderCodePtr( static_cast< ShaderStage >( idxStage ) ) )
        {
            GLuint hShader = NN_GFX_CALL_GL_FUNCTION( ::glCreateShader( s_StageTable[ idxStage ] ) );
            NN_GFX_GL_ASSERT();
            NN_SDK_ASSERT( IsValid( hShader ) );

            if( NN_STATIC_CONDITION( IsArray ) )
            {
                const SourceArrayCode* pSourceArrayCode = static_cast< const SourceArrayCode* >( pCode );
                const GLchar* pCodeArray[ SourceArrayCode::MaxCodeArrayLength ];
                int codeCount = static_cast< int >( pSourceArrayCode->codeArrayLength );
                NN_SDK_ASSERT( codeCount <= SourceArrayCode::MaxCodeArrayLength );
                for( int idxCode = 0; idxCode < codeCount; ++idxCode )
                {
                    pCodeArray[ idxCode ] = pSourceArrayCode->pCodePtrArray.ptr[ idxCode ];
                }
                NN_GFX_CALL_GL_FUNCTION( ::glShaderSource( hShader, codeCount, pCodeArray,
                    static_cast< const GLint* >( static_cast< const void* >( pSourceArrayCode->pCodeSizeArray ) ) ) );
            }
            else
            {
                const ShaderCode* pShaderCode = static_cast< const ShaderCode* >( pCode );
                int codeSize = static_cast< int >( pShaderCode->codeSize );
                const GLchar* pCodeString = static_cast< const GLchar* >( pShaderCode->pCode );
                NN_GFX_CALL_GL_FUNCTION( ::glShaderSource( hShader, 1, &pCodeString, &codeSize ) );
            }
            NN_GFX_CALL_GL_FUNCTION( ::glCompileShader( hShader ) );

            if( !Gl::CheckShaderStatus( hShader ) )
            {
                NN_GFX_CALL_GL_FUNCTION( ::glDeleteShader( hShader ) );
                if( !info.IsSeparationEnabled() )
                {
                    DetachAndDeleteShaders( pThis->ToData()->hCombinedProgram );
                }
                return ShaderInitializeResult_SetupFailed;
            }
            NN_GFX_GL_ASSERT();

            if( info.IsSeparationEnabled() )
            {
                pThis->ToData()->*pStageProgram[ idxStage ] = NN_GFX_CALL_GL_FUNCTION( ::glCreateProgram() );
                NN_GFX_CALL_GL_FUNCTION( ::glProgramParameteri( pThis->ToData(
                    )->*pStageProgram[ idxStage ], GL_PROGRAM_SEPARABLE, GL_TRUE ) );
                NN_GFX_CALL_GL_FUNCTION( ::glAttachShader(
                    pThis->ToData()->*pStageProgram[ idxStage ], hShader ) );
                NN_GFX_CALL_GL_FUNCTION( ::glLinkProgram( pThis->ToData()->*pStageProgram[ idxStage ] ) );
                DetachAndDeleteShaders( pThis->ToData()->*pStageProgram[ idxStage ] );
                if( !Gl::CheckProgramStatus( pThis->ToData()->*pStageProgram[ idxStage ] ) )
                {
                    return ShaderInitializeResult_SetupFailed;
                }
            }
            else
            {
                NN_GFX_CALL_GL_FUNCTION( ::glAttachShader( pThis->ToData()->hCombinedProgram, hShader ) );
            }
            NN_GFX_GL_ASSERT();
        }
    }

    if( !info.IsSeparationEnabled() )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glLinkProgram( pThis->ToData()->hCombinedProgram ) );
        DetachAndDeleteShaders( pThis->ToData()->hCombinedProgram );
        if( !Gl::CheckProgramStatus( pThis->ToData()->hCombinedProgram ) )
        {
            return ShaderInitializeResult_SetupFailed;
        }
    }
    NN_GFX_GL_ASSERT();

    return ShaderInitializeResult_Success;
}

template< ShaderCodeType CodeType >
ShaderInitializeResult InitializeShader( ShaderImpl< Target >* pThis, const ShaderInfo& info )
{
    NN_UNUSED( pThis );
    NN_UNUSED( info );
    return ShaderInitializeResult_InvalidType;
}

template<>
ShaderInitializeResult InitializeShader< ShaderCodeType_Binary >(
    ShaderImpl< Target >* pThis, const ShaderInfo& info )
{
    NN_SDK_ASSERT_NOT_NULL( pThis );

    struct FreeFunctor
    {
        void operator()( void* ptr )
        {
            if( ptr )
            {
                free( ptr );
            }
        }
    };

    if( !Gl::IsCompatibleBinaryFormat( info.GetBinaryFormat() ) )
    {
        return ShaderInitializeResult_InvalidFormat;
    }

    for( int idxStage = 0; idxStage < ( info.IsSeparationEnabled() ? ShaderStage_End : 1 ); ++idxStage )
    {
        const ShaderCode* pShaderCode = static_cast< const ShaderCode* >(
            info.GetShaderCodePtr( static_cast< ShaderStage >( idxStage ) ) );
        if( pShaderCode )
        {
            pThis->ToData()->*pStageProgram[ idxStage ] = NN_GFX_CALL_GL_FUNCTION( ::glCreateProgram() );
            NN_GFX_CALL_GL_FUNCTION( ::glProgramParameteri( pThis->ToData(
                )->*pStageProgram[ idxStage ], GL_PROGRAM_SEPARABLE, GL_TRUE ) );
            if( pShaderCode->decompressedCodeSize )
            {
                std::unique_ptr< GLchar[], FreeFunctor > pCode(
                    static_cast< GLchar* >( malloc( pShaderCode->decompressedCodeSize ) ) );
                std::unique_ptr< void, FreeFunctor > pWorkBuffer( malloc( nn::util::DecompressZlibWorkBufferSize ) );
                bool resultDecompress = nn::util::DecompressZlib( pCode.get(),
                    pShaderCode->decompressedCodeSize, pShaderCode->pCode,
                    pShaderCode->codeSize, pWorkBuffer.get(), nn::util::DecompressZlibWorkBufferSize );
                NN_SDK_ASSERT( resultDecompress );
                NN_UNUSED( resultDecompress );
                NN_GFX_CALL_GL_FUNCTION( ::glProgramBinary( pThis->ToData()->*pStageProgram[ idxStage ],
                    info.GetBinaryFormat(), pCode.get(), pShaderCode->decompressedCodeSize ) );
            }
            else
            {
                NN_GFX_CALL_GL_FUNCTION( ::glProgramBinary( pThis->ToData()->*pStageProgram[ idxStage ],
                    info.GetBinaryFormat(), pShaderCode->pCode, pShaderCode->codeSize ) );
            }
            if( !Gl::CheckProgramStatus( pThis->ToData()->*pStageProgram[ idxStage ] ) )
            {
                return ShaderInitializeResult_SetupFailed;
            }
        }
    }
    NN_GFX_GL_ASSERT();

    return ShaderInitializeResult_Success;
}

}

size_t ShaderImpl< Target >::GetBinaryCodeAlignment( DeviceImpl< Target >* ) NN_NOEXCEPT
{
    return 1;
}

ShaderImpl< Target >::ShaderImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

ShaderImpl< Target >::~ShaderImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

ShaderInitializeResult ShaderImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_ASSERT( info.IsSeparationEnabled() || ( info.GetShaderCodePtr(
        ShaderStage_Vertex ) || info.GetShaderCodePtr( ShaderStage_Compute ) ) );

    this->pGfxDevice = pDevice;
    this->flags = info.ToData()->flags;

    for( int idxStage = 0; idxStage < static_cast< int >( ShaderStage_End ); ++idxStage )
    {
        this->*pStageProgram[ idxStage ] = GlInvalidHandle;
    }

    GlDeviceActivator activator( pDevice );

    static ShaderInitializeResult ( * const s_InitializeFunction[] )(
        ShaderImpl< Target >*, const ShaderInfo& ) =
    {
        InitializeShader< ShaderCodeType_Binary >,
        InitializeShader< ShaderCodeType_Ir >,
        InitializeSourceShader< false >,
        InitializeSourceShader< true >
    };

    ShaderInitializeResult result = s_InitializeFunction[ info.GetCodeType() ]( this, info );

    if( result != ShaderInitializeResult_Success )
    {
        DeletePrograms( this );
        return result;
    }

    if( info.IsSeparationEnabled() )
    {
        for( int idxStage = 0; idxStage < ShaderStage_End; ++idxStage )
        {
            if( IsValid( this->*pStageProgram[ idxStage ] ) )
            {
                GlHandle hProgram = this->*pStageProgram[ idxStage ];
                ShaderStage stage = static_cast< ShaderStage >( idxStage );
                AssignShaderSlots< GL_UNIFORM_BLOCK, GL_MAX_UNIFORM_BUFFER_BINDINGS,
                    static_cast< bool( * )( GLenum ) >( NULL ) >( hProgram, stage );
                AssignShaderSlots< GL_SHADER_STORAGE_BLOCK,
                    GL_MAX_SHADER_STORAGE_BUFFER_BINDINGS,
                    static_cast< bool( *)( GLenum ) >( NULL ) >( hProgram, stage );
                AssignShaderSlots< GL_UNIFORM,  GL_MAX_COMBINED_TEXTURE_IMAGE_UNITS,
                    Gl::IsSamplerType >( hProgram, stage );
                AssignShaderSlots< GL_UNIFORM,
                    GL_MAX_IMAGE_UNITS, Gl::IsImageType >( hProgram, stage );
            }
        }
    }
    else
    {
        AssignShaderSlots< GL_UNIFORM_BLOCK,
            static_cast< bool( *)( GLenum ) >( NULL ) >( this->hCombinedProgram );
        AssignShaderSlots< GL_SHADER_STORAGE_BLOCK,
            static_cast< bool( *)( GLenum ) >( NULL ) >( this->hCombinedProgram );
        AssignShaderSlots< GL_UNIFORM, Gl::IsSamplerType >( this->hCombinedProgram );
        AssignShaderSlots< GL_UNIFORM, Gl::IsImageType >( this->hCombinedProgram );
    }

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
    return ShaderInitializeResult_Success;
}

void ShaderImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );

    GlDeviceActivator activator( pDevice );

    DeletePrograms( this );

    this->state = State_NotInitialized;
}

int ShaderImpl< Target >::GetInterfaceSlot( ShaderStage stage,
    ShaderInterfaceType shaderInterfaceType, const char* pName ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pName );
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    GlDeviceActivator activator( this->pGfxDevice.ptr );

    int ret = -1;

    detail::GlHandle hTargetProgram = flags.GetBit( Flag_SeparationEnable ) ?
        this->*pStageProgram[ static_cast< int >( stage ) ] : this->hCombinedProgram;

    switch( shaderInterfaceType )
    {
    case ShaderInterfaceType_Input:
        {
            ret = NN_GFX_CALL_GL_FUNCTION( ::glGetProgramResourceLocation(
                hTargetProgram, GL_PROGRAM_INPUT, pName ) );
        }
        break;
    case ShaderInterfaceType_Output:
        {
            ret = NN_GFX_CALL_GL_FUNCTION( ::glGetProgramResourceLocation(
                hTargetProgram, GL_PROGRAM_OUTPUT, pName ) );
        }
        break;
    case ShaderInterfaceType_ConstantBuffer:
        {
            GLuint blockIndex = NN_GFX_CALL_GL_FUNCTION( ::glGetProgramResourceIndex(
                hTargetProgram, GL_UNIFORM_BLOCK, pName ) );
            if( blockIndex == GL_INVALID_INDEX )
            {
                ret = -1;
            }
            else
            {
                GLenum propBinding = GL_BUFFER_BINDING;
                NN_GFX_CALL_GL_FUNCTION( ::glGetProgramResourceiv( hTargetProgram,
                    GL_UNIFORM_BLOCK, blockIndex, 1, &propBinding, 1, NULL, &ret ) );
            }
        }
        break;
    case ShaderInterfaceType_UnorderedAccessBuffer:
        {
            GLuint shaderStorageIndex = NN_GFX_CALL_GL_FUNCTION( ::glGetProgramResourceIndex(
                hTargetProgram, GL_SHADER_STORAGE_BLOCK, pName ) );
            if( shaderStorageIndex == GL_INVALID_INDEX )
            {
                ret = -1;
            }
            else
            {
                GLenum propBinding = GL_BUFFER_BINDING;
                NN_GFX_CALL_GL_FUNCTION( ::glGetProgramResourceiv( hTargetProgram,
                    GL_SHADER_STORAGE_BLOCK, shaderStorageIndex, 1, &propBinding, 1, NULL, &ret ) );
            }
        }
        break;
    case ShaderInterfaceType_Sampler:
    case ShaderInterfaceType_Image:
        {
            int location = NN_GFX_CALL_GL_FUNCTION( ::glGetProgramResourceLocation(
                hTargetProgram, GL_UNIFORM, pName ) );
            if( location != -1 )
            {
                NN_GFX_CALL_GL_FUNCTION( ::glGetUniformiv( hTargetProgram, location, &location ) );
            }
            ret = location;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    NN_GFX_GL_ASSERT();
    return ret;
}

void ShaderImpl< Target >::GetWorkGroupSize( int* pOutWorkGroupSizeX,
    int* pOutWorkGroupSizeY, int* pOutWorkGroupSizeZ ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutWorkGroupSizeX );
    NN_SDK_REQUIRES_NOT_NULL( pOutWorkGroupSizeY );
    NN_SDK_REQUIRES_NOT_NULL( pOutWorkGroupSizeZ );
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    GlDeviceActivator activator( this->pGfxDevice.ptr );

    GlHandle hProgram = flags.GetBit( Flag_SeparationEnable ) ?
        this->hComputeProgram : this->hCombinedProgram;

    GLint workGroupSize[ 3 ];
    NN_GFX_CALL_GL_FUNCTION( ::glGetProgramiv( hProgram,
        GL_COMPUTE_WORK_GROUP_SIZE, workGroupSize ) );
    NN_GFX_GL_ASSERT();

    *pOutWorkGroupSizeX = workGroupSize[ 0 ];
    *pOutWorkGroupSizeY = workGroupSize[ 1 ];
    *pOutWorkGroupSizeZ = workGroupSize[ 2 ];
}

}
}
}
