﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>

#include <nn/gfx/gfx_QueueInfo.h>

#include <nn/gfx/detail/gfx_Queue-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Device-api.nvn.8.h>
#include <nn/gfx/detail/gfx_CommandBuffer-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Sync-api.nvn.8.h>

#include "gfx_NvnHelper.h"

// 削除予定
#include <new>
#include <cstdlib>
#include <nn/util/util_BytePtr.h>
#include <nn/gfx/gfx_MemoryPoolInfo.h>
#include <nn/gfx/gfx_CommandBufferInfo.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.nvn.8.h>
#include <nn/gfx/detail/gfx_SwapChain-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Texture-api.nvn.8.h>

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationNvn8 Target;

namespace {

// 削除予定
class CopyToScanBufferHelper
{
private:
    static const size_t CommandMemorySize = 1024;
    static const size_t ControlMemorySize = 1024;

    MemoryPoolImpl< Target > m_MemoryPool;
    CommandBufferImpl< Target > m_CommandBuffer;

    void* m_pMemory;
    size_t m_MemorySize;

    void* m_pControlMemory;

public:
    static size_t GetRequiredMemorySize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
    {
        MemoryPoolInfo memoryPoolInfo;
        memoryPoolInfo.SetMemoryPoolProperty(
            MemoryPoolProperty_CpuUncached | MemoryPoolProperty_GpuCached );
        size_t alignment = MemoryPoolImpl< Target >::GetPoolMemoryAlignment( pDevice, memoryPoolInfo );
        size_t granularity = MemoryPoolImpl< Target >::GetPoolMemorySizeGranularity( pDevice, memoryPoolInfo );
        return alignment + nn::util::align_up( CommandMemorySize, granularity ) + ControlMemorySize;
    }

    CopyToScanBufferHelper() NN_NOEXCEPT
        : m_pMemory( NULL )
        , m_pControlMemory( NULL )
    {
    }

    void SetMemory( void* pMemory, size_t memorySize ) NN_NOEXCEPT
    {
        m_pMemory = pMemory;
        m_MemorySize = memorySize;
    }

    void Initialize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL( pDevice );
        NN_SDK_ASSERT( IsInitialized( *pDevice ) );
        NN_SDK_ASSERT_NOT_NULL( m_pMemory );
        NN_SDK_ASSERT_GREATER_EQUAL( m_MemorySize, GetRequiredMemorySize( pDevice ) );

        MemoryPoolInfo memoryPoolInfo;
        memoryPoolInfo.SetMemoryPoolProperty(
            MemoryPoolProperty_CpuUncached | MemoryPoolProperty_GpuCached );
        size_t alignment = MemoryPoolImpl< Target >::GetPoolMemoryAlignment( pDevice, memoryPoolInfo );
        size_t granularity = MemoryPoolImpl< Target >::GetPoolMemorySizeGranularity( pDevice, memoryPoolInfo );
        void* pPoolMemory = nn::util::BytePtr( m_pMemory ).AlignUp( alignment ).Get();
        size_t poolMemorySize = nn::util::align_up( CommandMemorySize, granularity );
        memoryPoolInfo.SetPoolMemory( pPoolMemory, poolMemorySize );
        m_MemoryPool.Initialize( pDevice, memoryPoolInfo );
        m_pControlMemory = nn::util::BytePtr( pPoolMemory, poolMemorySize ).Get();

        CommandBufferInfo commandBufferInfo;
        commandBufferInfo.SetCommandBufferType( CommandBufferType_Direct );
        commandBufferInfo.SetQueueCapability( QueueCapability_Copy );
        m_CommandBuffer.Initialize( pDevice, commandBufferInfo );
    }

    void Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL( pDevice );
        NN_SDK_ASSERT( IsInitialized( *pDevice ) );

        m_CommandBuffer.Finalize( pDevice );
        m_MemoryPool.Finalize( pDevice );

        m_pMemory = NULL;
    }

    void CopyToScanBuffer( QueueImpl< Target >* pQueue, SwapChainImpl< Target >* pSwapChain,
        const ColorTargetViewImpl< Target >* pColorTarget ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL( pSwapChain );
        NN_SDK_REQUIRES_NOT_NULL( pColorTarget );
        NN_SDK_REQUIRES( IsInitialized( *pSwapChain ) );
        NN_SDK_REQUIRES( IsInitialized( *pColorTarget ) );
        NN_SDK_ASSERT_NOT_NULL( pQueue );
        NN_SDK_ASSERT( IsInitialized( *pQueue ) );

        m_CommandBuffer.AddCommandMemory( &m_MemoryPool, 0, CommandMemorySize );
        m_CommandBuffer.AddControlMemory( m_pControlMemory, ControlMemorySize );
        m_CommandBuffer.Begin();
        {
            SwapChainImpl< Target >::DataType& swapChain = pSwapChain->ToData();
            NVNtexture* pScanBuffer = swapChain.scanBufferTextures[
                swapChain.currentScanBufferIndex ].pNvnTexture;
            NVNtexture* pSrcTexture = pColorTarget->ToData()->pNvnTexture;
            NVNtextureView view;
            NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetDefaults( &view ) );
            NVNcopyRegion srcRegion;
            srcRegion.xoffset = 0;
            srcRegion.yoffset = 0;
            srcRegion.zoffset = 0;
            NN_GFX_CALL_NVN_FUNCTION( srcRegion.width = nvnTextureGetWidth( pSrcTexture ) );
            NN_GFX_CALL_NVN_FUNCTION( srcRegion.height = nvnTextureGetHeight( pSrcTexture ) );
            srcRegion.depth = 1;
            NVNcopyRegion dstRegion;
            dstRegion.xoffset = 0;
            dstRegion.yoffset = 0;
            dstRegion.zoffset = 0;
            NN_GFX_CALL_NVN_FUNCTION( dstRegion.width = nvnTextureGetWidth( pScanBuffer ) );
            NN_GFX_CALL_NVN_FUNCTION( dstRegion.height = nvnTextureGetHeight( pScanBuffer ) );
            dstRegion.depth = 1;
            NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferCopyTextureToTexture(
                m_CommandBuffer.ToData()->pNvnCommandBuffer, pColorTarget->ToData()->pNvnTexture,
                &view, &srcRegion, pScanBuffer, &view, &dstRegion, NVN_COPY_FLAGS_LINEAR_FILTER_BIT ) );
        }
        m_CommandBuffer.End();

        pQueue->ExecuteCommand( &m_CommandBuffer, NULL );
    }
};

}

QueueImpl< Target >::QueueImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

QueueImpl< Target >::~QueueImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void QueueImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_STATIC_ASSERT( sizeof( this->nvnQueue ) == sizeof( NVNqueue ) );
    NN_UNUSED( info );

    this->pNnDevice = pDevice;

    // Use the pointer as convenience to the rest of the code.
    this->pNvnQueue = &this->nvnQueue[ 0 ];

    NVNqueueBuilder builder;
    NN_GFX_CALL_NVN_FUNCTION( nvnQueueBuilderSetDevice( &builder,
        pDevice->ToData()->pNvnDevice ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnQueueBuilderSetDefaults( &builder ) );
    if( ( info.GetCapability() & QueueCapability_Compute ) == 0 )
    {
        NN_GFX_CALL_NVN_FUNCTION( nvnQueueBuilderSetComputeMemorySize( &builder, 0 ) );
    }

    NVNboolean isQueueOK = NN_GFX_CALL_NVN_FUNCTION(
        nvnQueueInitialize( this->pNvnQueue, &builder ) );
    NN_SDK_ASSERT( isQueueOK );
    NN_UNUSED( isQueueOK );

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void QueueImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );
    NN_UNUSED( pDevice );

    NN_GFX_CALL_NVN_FUNCTION( nvnQueueFinalize( this->pNvnQueue ) );
    this->pNvnQueue = NULL;

    if( this->pImpl )
    {
        CopyToScanBufferHelper* pHelper = static_cast< CopyToScanBufferHelper* >( this->pImpl );
        pHelper->Finalize( pDevice );
        free( this->pImpl );
        this->pImpl = NULL;
    }

    this->state = State_NotInitialized;
}


void QueueImpl< Target >::ExecuteCommand( CommandBufferImpl< Target >* pCommandBuffer,
    FenceImpl< Target >* pFence ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pCommandBuffer );
    NN_SDK_REQUIRES( IsInitialized( *pCommandBuffer ) );
    NN_SDK_REQUIRES( pFence == NULL || IsInitialized( *pFence ) );
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnQueueSubmitCommands( this->pNvnQueue, 1, reinterpret_cast<
        const NVNcommandHandle* >( &pCommandBuffer->ToData()->hNvnCommandBuffer ) ) );

    if ( pFence )
    {
        // 過去にディスプレイフェンスに使ったオブジェクトの場合にポインターを貼り直す
        pFence->ToData()->pNvnSync = pFence->ToData()->nvnSync;
        NN_GFX_CALL_NVN_FUNCTION( nvnQueueFenceSync( this->pNvnQueue, pFence->ToData()->pNvnSync,
            NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, NVN_SYNC_FLAG_FLUSH_FOR_CPU_BIT ) );
    }
}

void QueueImpl< Target >::Flush() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnQueueFlush( this->pNvnQueue ) );
}

void QueueImpl< Target >::Sync() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnQueueFinish( this->pNvnQueue ) );
}

void QueueImpl< Target >::SetSemaphore( SemaphoreImpl< ApiVariationNvn8 >* pSemaphore ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pSemaphore );
    NN_SDK_REQUIRES( IsInitialized( *pSemaphore ) );
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnQueueFenceSync( this->pNvnQueue,
        pSemaphore->ToData()->pNvnSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, 0 ) );
}

void QueueImpl< Target >::SyncSemaphore( const SemaphoreImpl< ApiVariationNvn8 >* pSemaphore ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pSemaphore );
    NN_SDK_REQUIRES( IsInitialized( *pSemaphore ) );
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    NN_GFX_CALL_NVN_FUNCTION(
        nvnQueueWaitSync( this->pNvnQueue, pSemaphore->ToData()->pNvnSync ) );
}

void QueueImpl< Target >::CopyToScanBuffer( SwapChainImpl< Target >* pSwapChain,
    const ColorTargetViewImpl< Target >* pColorTarget ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pSwapChain );
    NN_SDK_REQUIRES_NOT_NULL( pColorTarget );
    NN_SDK_REQUIRES( IsInitialized( *pSwapChain ) );
    NN_SDK_REQUIRES( IsInitialized( *pColorTarget ) );
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    if( this->pImpl.ptr == NULL )
    {
        NN_SDK_ASSERT_NOT_NULL( this->pNnDevice.ptr );
        size_t memorySize = sizeof( CopyToScanBufferHelper ) +
            CopyToScanBufferHelper::GetRequiredMemorySize( this->pNnDevice );
        this->pImpl = malloc( memorySize );
        CopyToScanBufferHelper* pHelper = new( this->pImpl.ptr ) CopyToScanBufferHelper();

        pHelper->SetMemory( nn::util::BytePtr( this->pImpl,
            sizeof( CopyToScanBufferHelper ) ).Get(), memorySize - sizeof( CopyToScanBufferHelper ) );
        pHelper->Initialize( this->pNnDevice );
    }

    CopyToScanBufferHelper* pHelper = static_cast< CopyToScanBufferHelper* >( this->pImpl );
    return pHelper->CopyToScanBuffer( this, pSwapChain, pColorTarget );
}

void QueueImpl< Target >::Present( SwapChainImpl< Target >* pSwapChain,
    int presentInterval ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pSwapChain );
    NN_SDK_REQUIRES( IsInitialized( *pSwapChain ) );
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    SwapChainImpl< Target >::DataType& swapChain = pSwapChain->ToData();

    int oldPresentInterval = int();
    NN_GFX_CALL_NVN_FUNCTION( oldPresentInterval =
        nvnWindowGetPresentInterval( swapChain.pNvnWindow ) );
    if( oldPresentInterval != presentInterval )
    {
        NN_GFX_CALL_NVN_FUNCTION( nvnWindowSetPresentInterval(
            swapChain.pNvnWindow, presentInterval ) );
    }

    // 旧パス
    if( !swapChain.flag.GetBit( SwapChainImpl< Target >::DataType::Flag_Acquired ) &&
        swapChain.flag.GetBit( SwapChainImpl< Target >::DataType::Flag_IsFirstPresentation ) )
    {
        swapChain.flag.SetBit( SwapChainImpl< Target >::DataType::Flag_IsFirstPresentation, false );

        // Prime NVNwindow by acquiring the first texture
        int textureIndex = -1;
        NVNqueueAcquireTextureResult result = NN_GFX_CALL_NVN_FUNCTION(
            nvnQueueAcquireTexture( this->pNvnQueue, swapChain.pNvnWindow, &textureIndex ) );
        NN_SDK_ASSERT( result == NVN_QUEUE_ACQUIRE_TEXTURE_RESULT_SUCCESS );
        NN_UNUSED( result );

        swapChain.currentScanBufferIndex = textureIndex;
    }

    NN_GFX_CALL_NVN_FUNCTION( nvnQueuePresentTexture(
        this->pNvnQueue, swapChain.pNvnWindow, swapChain.currentScanBufferIndex ) );

    // 旧パス
    if( !swapChain.flag.GetBit( SwapChainImpl< Target >::DataType::Flag_Acquired ) )
    {
        int textureIndex = -1;
        NVNqueueAcquireTextureResult result = NN_GFX_CALL_NVN_FUNCTION(
            nvnQueueAcquireTexture( this->pNvnQueue, swapChain.pNvnWindow, &textureIndex ) );
        NN_SDK_ASSERT( result == NVN_QUEUE_ACQUIRE_TEXTURE_RESULT_SUCCESS );
        NN_UNUSED( result );
        swapChain.currentScanBufferIndex = textureIndex;
    }
}

}
}
}
