﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>

#include <nn/os.h>

#include <nn/gfx/gfx_QueueInfo.h>

#include <nn/gfx/detail/gfx_Queue-api.d3d.11.h>
#include <nn/gfx/detail/gfx_Device-api.d3d.11.h>
#include <nn/gfx/detail/gfx_CommandBuffer-api.d3d.11.h>
#include <nn/gfx/detail/gfx_Sync-api.d3d.11.h>
#include <nn/gfx/detail/gfx_SwapChain-api.d3d.11.h>
#include <nn/gfx/detail/gfx_Texture-api.d3d.11.h>

#include "gfx_CommonHelper.h"
#include "gfx_D3dHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationD3d11 Target;

template< bool IsDstBackBuffer, bool IsSrcSwapChain >
void CopyToScanbufferImpl( QueueImpl< Target >* pThis,
    SwapChainImpl< Target >* pSwapChain, const TextureImpl< Target >* pSrc ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pThis );
    NN_SDK_ASSERT_NOT_NULL( pSwapChain );
    NN_SDK_ASSERT_NOT_NULL( pSrc );

    NN_SDK_ASSERT( IsInitialized( *pSwapChain ) );
    NN_SDK_ASSERT( IsInitialized( *pSrc ) );
    NN_SDK_ASSERT( IsInitialized( *pThis ) );

    QueueImpl< Target >::DataType& data = pThis->ToData();

    SwapChainImpl< Target >::DataType& swapChain = pSwapChain->ToData();

    IDXGISwapChain* pD3dSwapChain = static_cast< IDXGISwapChain* >( swapChain.pSwapChain );
    ID3D11Texture2D* pDstResource;
    NN_GFX_CALL_D3D_FUNCTION( pD3dSwapChain->GetBuffer( 0, __uuidof( ID3D11Texture2D ), reinterpret_cast< void** >( &pDstResource ) ) );
    NN_SDK_ASSERT_NOT_NULL( pDstResource );

    ID3D11Texture2D* pSrcResource = static_cast< ID3D11Texture2D* >( pSrc->ToData()->pTexture );
    ID3D11DeviceContext* pD3dDeviceContext = static_cast< ID3D11DeviceContext* >( data.pGfxDevice->ToData()->renderingContext.hD3dDeviceContext );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeviceContext->CopyResource( pDstResource, pSrcResource ) );

    NN_GFX_CALL_D3D_FUNCTION( pDstResource->Release() );
}

template void CopyToScanbufferImpl< true, true >( QueueImpl< Target >*,
    SwapChainImpl< Target >*, const TextureImpl< Target >* );

QueueImpl< Target >::QueueImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

QueueImpl< Target >::~QueueImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

template< bool IsDstBackBuffer, bool IsSrcSwapChain >
void CopyToScanbufferImpl( QueueImpl< Target >* pThis,
    SwapChainImpl< Target >* pSwapChain, const TextureImpl< Target >* pSrc ) NN_NOEXCEPT;

void QueueImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_UNUSED( info );

    this->pGfxDevice = pDevice;

    DeviceImpl< Target >::DataType& data = pDevice->ToData();
    NN_SDK_ASSERT_NOT_NULL( data.renderingContext.hD3dDevice );
    NN_SDK_ASSERT_NOT_NULL( data.renderingContext.hD3dDeviceContext );
    NN_UNUSED( data );

    this->state = State_Initialized;
}

void QueueImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    this->pGfxDevice = NULL;

    this->state = State_NotInitialized;
}

void QueueImpl< Target >::Present( SwapChainImpl< Target >* pSwapChain,
    int presentInterval ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pSwapChain );
    NN_SDK_REQUIRES( IsInitialized( *pSwapChain ) );
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    SwapChainImpl< Target >::DataType& swapChain = pSwapChain->ToData();

    if( swapChain.flag.GetBit( SwapChainImpl< Target >::DataType::Flag_Acquired ) )
    {
        CopyToScanbufferImpl< true, true >( this, pSwapChain,
            nn::gfx::DataToAccessor( swapChain.scanBuffers[ swapChain.currentScanBufferIndex ] ) );
        Flush();
    }

    if( !::IsWindowVisible( static_cast< HWND >( swapChain.pWnd ) ) )
    {
        ::ShowWindow( static_cast< HWND >( swapChain.pWnd ), SW_SHOW );
    }

    IDXGISwapChain* pD3dSwapChain = static_cast< IDXGISwapChain* >( pSwapChain->ToData()->pSwapChain );
    NN_SDK_ASSERT_NOT_NULL( pD3dSwapChain );

    NN_GFX_CALL_D3D_FUNCTION( pD3dSwapChain->Present( presentInterval, 0 ) );

    if( ++swapChain.currentScanBufferIndex >= swapChain.scanBufferCount )
    {
        swapChain.currentScanBufferIndex = 0;
    }

    swapChain.flag.SetBit( SwapChainImpl< Target >::DataType::Flag_Acquired, false );
    swapChain.flag.SetBit( SwapChainImpl< Target >::DataType::Flag_Copied, false );
}

void QueueImpl< Target >::Flush() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    ID3D11DeviceContext* pD3dImmediateContext = static_cast< ID3D11DeviceContext* >( this->ToData()->pGfxDevice->ToData()->renderingContext.hD3dDeviceContext );

    NN_GFX_CALL_D3D_FUNCTION( pD3dImmediateContext->Flush() );
}

void QueueImpl< Target >::Sync() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );
}

void QueueImpl< Target >::ExecuteCommand( const CommandBufferImpl< Target >* pCommandBuffer,
    FenceImpl< Target >* pFence ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pCommandBuffer );
    NN_SDK_REQUIRES( pCommandBuffer->ToData()->state == CommandBufferImpl< Target >::DataType::State_Initialized );
    NN_SDK_REQUIRES( pCommandBuffer->ToData()->commandBufferType == CommandBufferType_Direct );
    NN_SDK_REQUIRES( pFence == NULL || IsInitialized( *pFence ) );
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    NN_SDK_REQUIRES( pCommandBuffer->ToData()->pCommandList );

    ID3D11DeviceContext* pD3dImmediateContext = static_cast< ID3D11DeviceContext* >( this->ToData()->pGfxDevice->ToData()->renderingContext.hD3dDeviceContext );
    ID3D11CommandList* pD3dCommandList = static_cast< ID3D11CommandList* >( pCommandBuffer->ToData()->pCommandList );

    NN_GFX_CALL_D3D_FUNCTION( pD3dImmediateContext->ExecuteCommandList( pD3dCommandList, TRUE ) );

    if( pFence )
    {
        pFence->ToData()->flags.SetBit( FenceImpl< Target >::DataType::Flag_Signaled, false );
        pFence->ToData()->pQueue = this;
    }
}

void QueueImpl< Target >::SetSemaphore( SemaphoreImpl< ApiVariationD3d11 >* pSemaphore ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pSemaphore );
    NN_SDK_REQUIRES( IsInitialized( *pSemaphore ) );
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    pSemaphore->ToData()->pQueue = this;
    pSemaphore->ToData()->flags.SetBit( SemaphoreImpl< Target >::DataType::Flag_Signaled, false );
}

void QueueImpl< Target >::SyncSemaphore( const SemaphoreImpl< ApiVariationD3d11 >* pSemaphore ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pSemaphore );
    NN_SDK_REQUIRES( IsInitialized( *pSemaphore ) );
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    NN_UNUSED( pSemaphore );
}

void QueueImpl< Target >::CopyToScanBuffer( SwapChainImpl< Target >* pSwapChain,
    const ColorTargetViewImpl< Target >* pColorTarget ) NN_NOEXCEPT
{
    NN_UNUSED( pSwapChain );
    NN_UNUSED( pColorTarget );

    // 本機能は未実装です。
    // 実装可能性は未調査です。
    NN_SDK_ASSERT( 0 );
}

}
}
}
