﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>

#include <nn/nn_Common.h>

#include <nn/util/util_BytePtr.h>

#include <nn/gfx/detail/gfx_Common-api.gl.h>

namespace nn {
namespace gfx {
namespace detail {

typedef void ( *GlCommandProcType )( const void* pParam, const GlCommandContext* pContext );

template< typename TParam >
void GlCommandProc( const void* pParam, const GlCommandContext* pContext ) NN_NOEXCEPT;

struct GlCommand
{
    union
    {
        GlCommandProcType pCommandProc;
        uint64_t commandProcPtr;
    };
    uint32_t paramSize;
    uint8_t param[ 1 ];
};

inline const void* ExecuteGlCommand( const GlCommand& command,
    const GlCommandContext* pContext ) NN_NOEXCEPT
{
    command.pCommandProc( command.param, pContext );
    return command.param + command.paramSize;
}

inline const void* ExecuteGlCommand( const void* pCommand,
    const GlCommandContext* pContext ) NN_NOEXCEPT
{
    return ExecuteGlCommand( *static_cast< const GlCommand* >( pCommand ), pContext );
}

// NULL 終端
inline void ExecuteGlCommands( const void* pCommands,
    const GlCommandContext* pContext ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pCommands );
    while( *static_cast< const uintptr_t* >( pCommands ) )
    {
        pCommands = ExecuteGlCommand( pCommands, pContext );
    }
}

// NULL 終端
inline void ExecuteGlCommandList( const void* pCommandList,
    const GlCommandContext* pContext ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pCommandList );
    for( ; ; )
    {
        if( *static_cast< const void* const* >( pCommandList ) == NULL )
        {
            pCommandList = nn::util::ConstBytePtr( pCommandList, sizeof( uint64_t ) ).Get();
            if( *static_cast< const void* const* >( pCommandList ) == NULL )
            {
                break;
            }
            pCommandList = *static_cast< const void* const* >( pCommandList );
            continue;
        }
        ExecuteGlCommands( *static_cast< const void* const* >( pCommandList ), pContext );
        pCommandList = nn::util::ConstBytePtr( pCommandList, sizeof( uint64_t ) ).Get();
    }
}

struct GlDispatchParam
{
    uint32_t groupCountX;
    uint32_t groupCountY;
    uint32_t groupCountZ;
};

struct GlDrawParam
{
    GlEnum mode;
    int32_t offset;
    int32_t count;
};

struct GlDrawInstancedParam
{
    GlDrawParam drawParam;
    int32_t primCount;
    uint32_t baseInstance;
};

struct GlDrawIndexedParam
{
    GlHandle hIndexBuffer;
    GlEnum mode;
    int32_t count;
    GlEnum type;
    int32_t offset;
    int32_t baseVertex;
};

struct GlDrawIndexedInstancedParam
{
    GlDrawIndexedParam drawIndexedParam;
    int32_t primCount;
    uint32_t baseInstance;
};

struct GlDispatchIndirectParam
{
    GlHandle hIndirectBuffer;
    uint32_t offset;
};

struct GlDrawIndirectParam
{
    GlHandle hIndirectBuffer;
    uint32_t offset;
    GlEnum mode;
};

struct GlDrawIndexedIndirectParam
{
    GlHandle hIndirectBuffer;
    uint32_t offset;
    GlEnum mode;
    GlEnum type;
    GlHandle hIndexBuffer;
};

struct GlSetRenderTargetsParam
{
    struct RenderTarget
    {
        GlHandle hColorTarget;
        int layer;
    };
    GlHandle hDepthStencil;
    GlEnum depthStencilAttachment;
    int32_t maxColorAttachments;
    int32_t colorTargetCount;
    RenderTarget colorTargets[ 1 ];
};

struct GlSetVertexBufferParam
{
    uint32_t bufferIndex;
    GlHandle hBuffer;
    uint32_t offset;
    int32_t stride;
};

struct GlSetShaderParam
{
    GlHandle hProgram;
};

struct GlSetSeparateShaderParam
{
    GlHandle hVertexProgram;
    GlHandle hHullProgram;
    GlHandle hDomainProgram;
    GlHandle hGeometryProgram;
    GlHandle hPixelProgram;
    GlHandle hComputeProgram;
    int32_t stageBits;
};

struct GlSetTextureParam
{
    uint32_t unit;
    GlHandle hTexture;
    uint32_t target;
};

struct GlSetSamplerParam
{
    uint32_t unit;
    GlHandle hSampler;
};

struct GlSetTextureAndSamplerParam
{
    uint32_t unit;
    GlHandle hTexture;
    uint32_t target;
    GlHandle hSampler;
};

struct GlSetImageParam
{
    uint32_t unit;
    GlHandle hTexture;
    uint32_t target;
};

struct GlSetBufferParam
{
    GlEnum target;
    uint32_t index;
    GlHandle hBuffer;
    uint32_t offset;
    uint32_t size;
};

struct GlCopyBufferParam
{
    GlHandle hSrcBuffer;
    GlHandle hDstBuffer;
    uint32_t srcOffset;
    uint32_t dstOffset;
    uint32_t size;
};

struct GlCopyImageParam
{
    GlHandle hSrcTexture;
    GlEnum srcTarget;
    int32_t srcLevel;
    int32_t srcX;
    int32_t srcY;
    int32_t srcZ;
    GlHandle hDstTexture;
    GlEnum dstTarget;
    int32_t dstLevel;
    int32_t dstX;
    int32_t dstY;
    int32_t dstZ;
    int32_t width;
    int32_t height;
    int32_t depth;
};

struct GlBufferImageCopyParam
{
    GlHandle hBuffer;
    int32_t bufferOffset;
    GlHandle hTexture;
    GlEnum target;
    int32_t level;
    int32_t isCompressed;
    GlEnum internalFormat;
    GlEnum pixelFormat;
    GlEnum pixelType;
    int32_t xOffset;
    int32_t yOffset;
    int32_t zOffset;
    int32_t width;
    int32_t height;
    int32_t depth;
    int32_t imageSize;
    int32_t rowLength;
    int32_t imageHeight;
};

struct GlCopyBufferToImageParam
{
    GlBufferImageCopyParam param;
};

struct GlCopyImageToBufferParam
{
    GlBufferImageCopyParam param;
};

struct GlBlitImageParam
{
    GlHandle hDstTexture;
    int32_t dstMipLevel;
    int32_t dstStartArrayIndex;
    int32_t dstX0;
    int32_t dstY0;
    int32_t dstX1;
    int32_t dstY1;
    GlHandle hSrcTexture;
    int32_t srcMipLevel;
    int32_t srcStartArrayIndex;
    int32_t srcX0;
    int32_t srcY0;
    int32_t srcX1;
    int32_t srcY1;
    int32_t arrayCount;
    int32_t mask;
    int32_t filter;
};

struct GlClearBufferParam
{
    GlHandle hBuffer;
    int32_t offset;
    uint32_t size;
    uint32_t value;
};

struct GlClearColorParam
{
    GlHandle hTexture;
    int32_t startLayer;
    int32_t endLayer;
    GlEnum target;
    union
    {
        float color[ 4 ];
        int32_t colori[ 4 ];
        uint32_t coloru[ 4 ];
    };
};

struct GlClearDepthStencilParam
{
    GlHandle hTexture;
    int32_t startLayer;
    int32_t endLayer;
    GlEnum attachment;
    uint32_t clearMode;
    float depth;
    int32_t stencil;
};

struct GlResolveParam
{
    GlHandle hDstTexture;
    GlHandle hSrcTexture;
    int32_t dstMipLevel;
    int32_t dstStartArrayIndex;
    int32_t startLayer;
    int32_t endLayer;
    int32_t dstWidth;
    int32_t dstHeight;
    int32_t srcWidth;
    int32_t srcHeight;
};

struct GlMemoryBarrierParam
{
    uint32_t barriers;
};

struct GlBeginQueryParam
{
    GlEnum target;
    int32_t idxQueryObject;
};

struct GlEndQueryParam
{
    GlHandle hBuffer;
    uint32_t bufferOffset;
    GlEnum target;
    int32_t idxQueryObject;
};

struct GlWriteTimestampParam
{
    GlHandle hBuffer;
    uint32_t bufferOffset;
};

struct GlSetDepthBoundsParam
{
    float minDepthBounds;
    float maxDepthBounds;
};

struct GlSetLineWidthParam
{
    float lineWidth;
};

struct GlSetViewportsParam
{
    int32_t first;
    int32_t count;
    // float[ count * 4 ] + double[ count * 2 ]
};

struct GlSetScissorsParam
{
    int32_t first;
    int32_t count;
    // int[ count * 4 ]
};

struct GlBufferSubDataParam
{
    GlHandle hBuffer;
    int32_t offset;
    uint32_t size;
    // char[ size ]
};

struct GlCallCommandListParam
{
    union
    {
        const void* pCommandList;
        uint64_t commandListPtr;
    };
};

struct GlCallbackParam
{
    union
    {
        void ( *pCallback )( const void* );
        uint64_t callbackPtr;
    };
    union
    {
        const void* pParam;
        uint64_t paramPtr;
    };
};

}
}
}
