﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>

#include <nn/nn_SdkAssert.h>

#include <nn/util/util_BytePtr.h>
#include <nn/util/util_BitFlagSet.h>

#include <nn/gfx/gfx_CommandBufferInfo.h>
#include <nn/gfx/gfx_TextureInfo.h>
#include <nn/gfx/gfx_RootSignatureInfo.h>
#include <nn/gfx/gfx_CommandBuffer.h>
#include <nn/gfx/gfx_StateInfo.h>

#include <nn/gfx/detail/gfx_Misc.h>
#include <nn/gfx/detail/gfx_CommandBuffer-api.nvn.8.h>
#include <nn/gfx/detail/gfx_State-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Texture-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Pipeline-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Device-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Buffer-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Sampler-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Shader-api.nvn.8.h>
#include <nn/gfx/detail/gfx_DescriptorPool-api.nvn.8.h>
#include <nn/gfx/detail/gfx_RootSignature-api.nvn.8.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.nvn.8.h>

#include "gfx_CommonHelper.h"
#include "gfx_NvnHelper.h"

#include <list>

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_DEPRECATED_DECLARATIONS

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationNvn8 Target;

namespace {

template< typename T >
T* ToPtr( const DescriptorSlot& slot )
{
    return reinterpret_cast< T* >( slot.ToData()->value );
}

#if defined( NN_SDK_BUILD_DEBUG ) || defined( NN_SDK_BUILD_DEVELOP )
bool IsRecording( const CommandBufferImpl< Target >& obj )
{
    // 相互運用時を考慮すると nvn から判定するほうが確実
    return NVN_TRUE == NN_GFX_CALL_NVN_FUNCTION(
        nvnCommandBufferIsRecording( obj.ToData()->pNvnCommandBuffer ) );
}
#endif

void GetNvnCopyRegion( int* pOffsetY, int* pHeight, int* pOffsetZ, int* pDepth,
    const TextureCopyRegion& region, NVNtextureTarget target ) NN_NOEXCEPT
{
    *pOffsetY = region.GetOffsetV();
    *pHeight = region.GetHeight();
    *pOffsetZ = region.GetOffsetW();
    *pDepth = region.GetDepth();
    switch( target )
    {
    case NVN_TEXTURE_TARGET_1D_ARRAY:
        {
            *pOffsetY = region.GetSubresource().GetArrayIndex();
            *pHeight = std::max NN_PREVENT_MACRO_FUNC ( region.GetArrayLength(), 1 );
        }
        break;
    case NVN_TEXTURE_TARGET_2D_ARRAY:
    case NVN_TEXTURE_TARGET_2D_MULTISAMPLE_ARRAY:
        {
            *pOffsetZ = region.GetSubresource().GetArrayIndex();
            *pDepth = std::max NN_PREVENT_MACRO_FUNC ( region.GetArrayLength(), 1 );
        }
        break;
    default:
        break;
    }
}

void GetNvnCopyStride( ptrdiff_t* pRowStride, ptrdiff_t* pImageStride,
    const BufferTextureCopyRegion& region, NVNtexture* pTexture ) NN_NOEXCEPT
{
    int rowStride = 0;
    int imageStride = 0;
    if( region.GetBufferImageWidth() != 0 || region.GetBufferImageHeight() != 0 )
    {
        NVNformat nvnFormat = NN_GFX_CALL_NVN_FUNCTION(
            nvnTextureGetFormat( pTexture ) );
        ImageFormat imageFormat = Nvn::GetGfxImageFormat( nvnFormat );
        ChannelFormat channelFormat = GetChannelFormat( imageFormat );
        rowStride = ( region.GetBufferImageWidth() == 0 ? region.GetTextureCopyRegion().GetWidth()
            : region.GetBufferImageWidth() ) * GetBytePerPixel( channelFormat );
        if( IsCompressedFormat( channelFormat ) )
        {
            rowStride /= GetBlockWidth( channelFormat ) * GetBlockHeight( channelFormat );
        }
    }
    if( region.GetBufferImageHeight() != 0 )
    {
        imageStride = rowStride * region.GetBufferImageHeight();
    }

    *pRowStride = rowStride;
    *pImageStride = imageStride;
}

void SetTextureAndSampler( CommandBufferImpl< Target >* pNnCb, ShaderStage stage, int slot,
    unsigned int nvnTextureID, unsigned int nvnSamplerID ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pNnCb );
    NN_SDK_ASSERT( slot >= 0 );
    NN_SDK_ASSERT( IsRecording( *pNnCb ) );

    const DeviceImpl< Target >* pNnDevice = pNnCb->ToData()->pNnDevice;
    NVNtextureHandle textureHandle = NN_GFX_CALL_NVN_FUNCTION(
        nvnDeviceGetTextureHandle( pNnDevice->ToData()->pNvnDevice, nvnTextureID, nvnSamplerID ) );
    NN_SDK_ASSERT( textureHandle );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferBindTexture( pNnCb->ToData()->pNvnCommandBuffer,
        Nvn::GetShaderStage( stage ), slot, textureHandle ) );
}

void
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_WIN32 )
    __stdcall
#endif
    CommandBufferMemoryCallbackProcedure( NVNcommandBuffer *pNvnCommandBuffer,
    NVNcommandBufferMemoryEvent event, size_t minSize, void *pCallbackData ) NN_NOEXCEPT
{
    CommandBufferImpl< Target >* pThis = static_cast< CommandBufferImpl< Target >* >( pCallbackData );
    NN_SDK_ASSERT_NOT_NULL( pThis );
    CommandBufferImpl< Target >::DataType& obj = pThis->ToData();
    NN_SDK_ASSERT( obj.pNvnCommandBuffer == pNvnCommandBuffer );
    NN_UNUSED( pNvnCommandBuffer );
    TCommandBuffer< Target >* pCommandBuffer = static_cast< TCommandBuffer< Target >* >( pThis );

    OutOfMemoryEventArg arg;
    arg.minRequiredSize = minSize;

    switch( event)
    {
    case NVN_COMMAND_BUFFER_MEMORY_EVENT_OUT_OF_COMMAND_MEMORY:
        {
            NN_SDK_ASSERT_NOT_NULL( obj.pOutOfCommandMemoryCallback.ptr );
            ( *reinterpret_cast< CommandBufferImpl< Target >::OutOfMemoryEventCallback >(
                obj.pOutOfCommandMemoryCallback.ptr ) )( pCommandBuffer, arg );
        }
        break;
    case NVN_COMMAND_BUFFER_MEMORY_EVENT_OUT_OF_CONTROL_MEMORY:
        {
            NN_SDK_ASSERT_NOT_NULL( obj.pOutOfControlMemoryCallback.ptr );
            ( *reinterpret_cast< CommandBufferImpl< Target >::OutOfMemoryEventCallback >(
                obj.pOutOfControlMemoryCallback.ptr ) )( pCommandBuffer, arg );
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

}

size_t CommandBufferImpl< Target >::GetCommandMemoryAlignment( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    int nvnAlignment;
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
        NVN_DEVICE_INFO_COMMAND_BUFFER_COMMAND_ALIGNMENT, &nvnAlignment ) );

    return nvnAlignment;
}

size_t CommandBufferImpl< Target >::GetControlMemoryAlignment( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    int nvnAlignment;
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
        NVN_DEVICE_INFO_COMMAND_BUFFER_CONTROL_ALIGNMENT, &nvnAlignment ) );

    return nvnAlignment;
}

CommandBufferImpl< Target >::CommandBufferImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

CommandBufferImpl< Target >::~CommandBufferImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void CommandBufferImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_UNUSED( info );
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( state == State_NotInitialized );

    this->pNnDevice = pDevice;
    NVNdevice* pNvnDevice = pDevice->ToData()->pNvnDevice;

    this->pNvnCommandBuffer = &this->nvnCommandBuffer;
    NN_GFX_CALL_NVN_FUNCTION(
        nvnCommandBufferInitialize( this->pNvnCommandBuffer, pNvnDevice ) );

    this->pOutOfCommandMemoryCallback = NULL;
    this->pOutOfControlMemoryCallback = NULL;

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetMemoryCallback(
        this->pNvnCommandBuffer, CommandBufferMemoryCallbackProcedure ) );
    NN_GFX_CALL_NVN_FUNCTION(
        nvnCommandBufferSetMemoryCallbackData( this->pNvnCommandBuffer, this ) );

    this->hNvnCommandBuffer = 0;

    this->flags.SetBit( Flag_ConservativeRasterSupported,
        pDevice->ToData()->supportedFeatures.GetBit( NvnDeviceFeature_SupportConservativeRaster ) );

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void CommandBufferImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_UNUSED( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );

    if( this->hNvnCommandBuffer )
    {
        NN_GFX_CALL_NVN_FUNCTION( nvnDeviceFinalizeCommandHandle(
            this->pNnDevice->ToData()->pNvnDevice, this->hNvnCommandBuffer ) );
        this->hNvnCommandBuffer = 0;
    }

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferFinalize( this->pNvnCommandBuffer ) );

    this->state = State_NotInitialized;
}

void CommandBufferImpl< Target >::AddCommandMemory( MemoryPoolImpl< Target >* pMemoryPool,
    ptrdiff_t memoryPoolOffset, size_t memorySize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pMemoryPool );
    NN_SDK_REQUIRES( IsInitialized( *pMemoryPool ) );
    //NN_SDK_REQUIRES( memoryPoolOffset + memorySize <= pMemoryPool->ToData()->memorySize );
    NN_SDK_REQUIRES( this->state != State_NotInitialized );
    NN_SDK_ASSERT_ALIGNED( memoryPoolOffset, GetCommandMemoryAlignment( this->pNnDevice ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferAddCommandMemory( this->pNvnCommandBuffer,
        pMemoryPool->ToData()->pNvnMemoryPool, memoryPoolOffset, memorySize) );
}

void CommandBufferImpl< Target >::AddControlMemory( void* pMemory, size_t memorySize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pMemory );
    NN_SDK_REQUIRES( this->state != State_NotInitialized );
    NN_SDK_ASSERT_ALIGNED( pMemory, GetControlMemoryAlignment( this->pNnDevice ) );

    NN_GFX_CALL_NVN_FUNCTION(
        nvnCommandBufferAddControlMemory( this->pNvnCommandBuffer, pMemory, memorySize ) );
}

void CommandBufferImpl< Target >::SetOutOfCommandMemoryEventCallback(
    OutOfMemoryEventCallback pCallback ) NN_NOEXCEPT
{
    this->pOutOfCommandMemoryCallback.ptr = reinterpret_cast< void (*)() >( pCallback );
}

void CommandBufferImpl< Target >::SetOutOfControlMemoryEventCallback(
    OutOfMemoryEventCallback pCallback ) NN_NOEXCEPT
{
    this->pOutOfControlMemoryCallback.ptr = reinterpret_cast< void (*)() >( pCallback );
}

void CommandBufferImpl< Target >::Reset() NN_NOEXCEPT
{
    // do nothing.
}

void CommandBufferImpl< Target >::Begin() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );

    if( this->hNvnCommandBuffer )
    {
        NN_GFX_CALL_NVN_FUNCTION( nvnDeviceFinalizeCommandHandle(
            this->pNnDevice->ToData()->pNvnDevice, this->hNvnCommandBuffer ) );
        this->hNvnCommandBuffer = 0;
    }

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferBeginRecording( this->pNvnCommandBuffer ) );

    this->state = State_Begun;
}

void CommandBufferImpl< Target >::End() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    // コマンドメモリー未使用のコマンドバッファーになることがある
    this->hNvnCommandBuffer = NN_GFX_CALL_NVN_FUNCTION(
        nvnCommandBufferEndRecording( this->pNvnCommandBuffer ) );

    this->state = State_Initialized;
}

void CommandBufferImpl< Target >::Dispatch( int groupCountX, int groupCountY, int groupCountZ ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferDispatchCompute(
        this->pNvnCommandBuffer, groupCountX, groupCountY, groupCountZ ) );
}

void CommandBufferImpl< Target >::Draw( PrimitiveTopology primitiveTopology,
    int vertexCount, int vertexOffset ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferDrawArrays( this->pNvnCommandBuffer,
        Nvn::GetDrawPrimitive( primitiveTopology ), vertexOffset, vertexCount ) );
}

void CommandBufferImpl< Target >::Draw( PrimitiveTopology primitiveTopology,
    int vertexCountPerInstance, int vertexOffset, int instanceCount, int baseInstance ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );

    NN_GFX_CALL_NVN_FUNCTION(
        nvnCommandBufferDrawArraysInstanced( this->pNvnCommandBuffer,
            Nvn::GetDrawPrimitive( primitiveTopology ),
            vertexOffset, vertexCountPerInstance, baseInstance, instanceCount ) );
}

void CommandBufferImpl< Target >::DrawIndexed( PrimitiveTopology primitiveTopology,
    IndexFormat indexFormat, const GpuAddress& indexBufferAddress, int indexCount, int baseVertex ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );
    NN_SDK_ASSERT_NOT_EQUAL( Nvn::GetBufferAddress( indexBufferAddress ), 0 );

    // take into account the offset in the index buffer view
    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferDrawElementsBaseVertex( this->pNvnCommandBuffer,
        Nvn::GetDrawPrimitive( primitiveTopology ),
        Nvn::GetIndexFormat( indexFormat ),
        indexCount,
        Nvn::GetBufferAddress( indexBufferAddress ),
        baseVertex ) );
}

void CommandBufferImpl< Target >::DrawIndexed( PrimitiveTopology primitiveTopology, IndexFormat indexFormat,
    const GpuAddress& indexBufferAddress, int indexCountPerInstance, int baseVertex, int instanceCount, int baseInstance ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );
    NN_SDK_ASSERT_NOT_EQUAL( Nvn::GetBufferAddress( indexBufferAddress ), 0 );

    // take into account the offset in the index buffer view
    NN_GFX_CALL_NVN_FUNCTION(
        nvnCommandBufferDrawElementsInstanced( this->pNvnCommandBuffer,
            Nvn::GetDrawPrimitive( primitiveTopology ),
            Nvn::GetIndexFormat( indexFormat ),
            indexCountPerInstance,
            Nvn::GetBufferAddress( indexBufferAddress ),
            baseVertex, baseInstance, instanceCount ) );
}

void CommandBufferImpl< Target >::DispatchIndirect( const GpuAddress& indirectBufferAddress ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );
    NN_SDK_ASSERT_NOT_EQUAL( Nvn::GetBufferAddress( indirectBufferAddress ), 0 );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferDispatchComputeIndirect(
        this->pNvnCommandBuffer, Nvn::GetBufferAddress( indirectBufferAddress ) ) );
}

void CommandBufferImpl< Target >::DrawIndirect(
    PrimitiveTopology primitiveTopology, const GpuAddress& indirectBufferAddress ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );
    NN_SDK_ASSERT_NOT_EQUAL( Nvn::GetBufferAddress( indirectBufferAddress ), 0 );

    NN_GFX_CALL_NVN_FUNCTION(
        nvnCommandBufferDrawArraysIndirect( this->pNvnCommandBuffer,
            Nvn::GetDrawPrimitive( primitiveTopology ),
            Nvn::GetBufferAddress( indirectBufferAddress ) ) );
}

void CommandBufferImpl< Target >::DrawIndexedIndirect( PrimitiveTopology primitiveTopology,
    IndexFormat indexFormat, const GpuAddress& indexBufferAddress, const GpuAddress& indirectBufferAddress ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );
    NN_SDK_ASSERT_NOT_EQUAL( Nvn::GetBufferAddress( indexBufferAddress ), 0 );
    NN_SDK_ASSERT_NOT_EQUAL( Nvn::GetBufferAddress( indirectBufferAddress ), 0 );

    NN_GFX_CALL_NVN_FUNCTION(
        nvnCommandBufferDrawElementsIndirect( this->pNvnCommandBuffer,
            Nvn::GetDrawPrimitive( primitiveTopology ),
            Nvn::GetIndexFormat( indexFormat ),
            Nvn::GetBufferAddress( indexBufferAddress ),
            Nvn::GetBufferAddress( indirectBufferAddress ) ) );
}

void CommandBufferImpl< Target >::SetPipeline( const PipelineImpl< Target >* pPipeline ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pPipeline );
    NN_SDK_REQUIRES( IsInitialized( *pPipeline ) );
    NN_SDK_REQUIRES_NOT_NULL( this->pNnDevice.ptr );
    NN_SDK_REQUIRES( IsInitialized( *this->pNnDevice.ptr ) );
    NN_SDK_REQUIRES( IsRecording( *this ) );

    const PipelineImpl< Target >::DataType& pipeline = pPipeline->ToData();

    if ( pPipeline->ToData()->nnPipelineType == PipelineImplData< Target >::PipelineType_Graphics )
    {
        SetRasterizerState( nn::gfx::DataToAccessor( pipeline.nnRasterizerState ) );
        SetBlendState( nn::gfx::DataToAccessor( pipeline.nnBlendState ) );
        SetDepthStencilState( nn::gfx::DataToAccessor( pipeline.nnDepthStencilState ) );
        SetVertexState( nn::gfx::DataToAccessor( pipeline.nnVertexState ) );
        if( pipeline.flags.GetBit( PipelineImpl< Target >::DataType::Flag_HasTessellationState ) )
        {
            SetTessellationState( nn::gfx::DataToAccessor( pipeline.nnTessellationState ) );
        }
    }
    SetShader( pipeline.pShader, ShaderStageBit_All );
}

void CommandBufferImpl< Target >::SetRenderTargets( int colorTargetCount,
    const ColorTargetViewImpl< Target >* const * ppColorTargets,
    const DepthStencilViewImpl< Target >* pDepthStencil ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( colorTargetCount < 1 || ppColorTargets );
    NN_SDK_REQUIRES( IsRecording( *this ) );
    const int MaxRenderTarget = 8;
    NN_SDK_REQUIRES( colorTargetCount <= MaxRenderTarget );

    NVNtexture* pNvnColorTargets[ MaxRenderTarget ] = {};
    NVNtextureView* pNvnColorTargetViews[ MaxRenderTarget ] = {};

    // Sanity check that we don't start rendering to nowhere.
    NN_SDK_ASSERT( colorTargetCount || pDepthStencil != NULL );

    for ( int idxTarget = 0; idxTarget < colorTargetCount; ++idxTarget )
    {
        if( ppColorTargets[ idxTarget ] )
        {
            NN_SDK_REQUIRES( IsInitialized( *( ppColorTargets[ idxTarget ] ) ) );
            NN_SDK_ASSERT_NOT_NULL( ppColorTargets[ idxTarget ]->ToData()->pNvnTexture.ptr );
            pNvnColorTargets[ idxTarget ] = ppColorTargets[ idxTarget ]->ToData()->pNvnTexture;
            pNvnColorTargetViews[ idxTarget ] = ppColorTargets[ idxTarget ]->ToData()->pNvnTextureView;
        }
    }

    NVNtexture* pDepthTarget = NULL;
    NVNtextureView* pDepthTargetView = NULL;
    if ( pDepthStencil )
    {
        NN_SDK_ASSERT_NOT_NULL( pDepthStencil->ToData()->pNvnTexture.ptr );
        pDepthTarget = pDepthStencil->ToData()->pNvnTexture;
        pDepthTargetView = pDepthStencil->ToData()->pNvnTextureView;
    }
    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetRenderTargets( this->pNvnCommandBuffer,
        colorTargetCount, pNvnColorTargets, pNvnColorTargetViews, pDepthTarget, pDepthTargetView ) );
}

void CommandBufferImpl< Target >::SetVertexBuffer( int bufferIndex,
    const GpuAddress& vertexBuffer, ptrdiff_t stride, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );
    NN_SDK_ASSERT_NOT_EQUAL( Nvn::GetBufferAddress( vertexBuffer ), 0 );
    NN_UNUSED( stride ); // This is specified in the VertexStateImpl

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferBindVertexBuffer( this->pNvnCommandBuffer,
        bufferIndex, Nvn::GetBufferAddress( vertexBuffer ), size ) );
}

void CommandBufferImpl< Target >::SetViewportScissorState(
    const ViewportScissorStateImpl< Target >* pViewportScissor ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pViewportScissor );
    NN_SDK_REQUIRES( IsInitialized( *pViewportScissor ) );
    NN_SDK_REQUIRES( IsRecording( *this ) );

    const ViewportScissorStateImpl< Target >::DataType& data = pViewportScissor->ToData();

    NN_GFX_CALL_NVN_FUNCTION(
        nvnCommandBufferSetDepthRange( this->pNvnCommandBuffer,
            static_cast< float >( data.depthRange[ 0 ] ),
            static_cast< float >( data.depthRange[ 1 ] ) ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetViewports(
        this->pNvnCommandBuffer, 0, 1, data.viewport ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetScissors(
        this->pNvnCommandBuffer, 0, 1, data.scissor ) );

    int viewportCount = pViewportScissor->ToData()->viewportCount;
    if ( viewportCount > 1 )
    {
        int extraViewportCount = viewportCount - 1;

        nn::util::BytePtr ptr( pViewportScissor->ToData()->pWorkMemory.ptr );

        float* viewportArray = ptr.Get< float >();
        float* depthRangeArray = ptr.Advance( sizeof( float ) * 4 * extraViewportCount ).Get< float >();
        int32_t* scissorArray = ptr.Advance( sizeof( float ) * 2 * extraViewportCount ).Get< int32_t >();

        NN_GFX_CALL_NVN_FUNCTION(
            nvnCommandBufferSetViewports( this->pNvnCommandBuffer, 1,
                viewportCount - 1, viewportArray ) );

        NN_GFX_CALL_NVN_FUNCTION(
            nvnCommandBufferSetDepthRanges( this->pNvnCommandBuffer, 1,
                viewportCount - 1, depthRangeArray ) );

        NN_GFX_CALL_NVN_FUNCTION(
            nvnCommandBufferSetScissors( this->pNvnCommandBuffer, 1,
                viewportCount - 1, scissorArray ) );
    }
}

void CommandBufferImpl< Target >::CopyBuffer( BufferImpl< Target >* pDstBuffer, ptrdiff_t dstOffset,
    const BufferImpl< Target >* pSrcBuffer, ptrdiff_t srcOffset, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstBuffer );
    NN_SDK_REQUIRES_NOT_NULL( pSrcBuffer );
    NN_SDK_REQUIRES( IsInitialized( *pDstBuffer ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcBuffer ) );
    NN_SDK_REQUIRES( IsRecording( *this ) );

    NVNbufferAddress src = NN_GFX_CALL_NVN_FUNCTION(
        nvnBufferGetAddress( pSrcBuffer->ToData()->pNvnBuffer ) ) + srcOffset;
    NVNbufferAddress dst = NN_GFX_CALL_NVN_FUNCTION(
        nvnBufferGetAddress( pDstBuffer->ToData()->pNvnBuffer ) ) + dstOffset;
    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferCopyBufferToBuffer(
        this->pNvnCommandBuffer, src, dst, size, 0 ) );
}

void CommandBufferImpl< Target >::CopyImage( TextureImpl< Target >* pDstTexture,
    const TextureSubresource& dstSubresource, int dstOffsetU, int dstOffsetV, int dstOffsetW,
    const TextureImpl< Target >* pSrcTexture, const TextureCopyRegion& srcCopyRegion ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstTexture );
    NN_SDK_REQUIRES_NOT_NULL( pSrcTexture );
    NN_SDK_REQUIRES( IsInitialized( *pDstTexture ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcTexture ) );
    NN_SDK_REQUIRES( IsRecording( *this ) );
    NN_SDK_ASSERT_NOT_NULL( pSrcTexture->ToData()->pNvnTexture.ptr );
    NN_SDK_ASSERT_NOT_NULL( pDstTexture->ToData()->pNvnTexture.ptr );

    NVNtextureTarget target = NN_GFX_CALL_NVN_FUNCTION(
        nvnTextureGetTarget( pSrcTexture->ToData()->pNvnTexture ) );

    int srcV;
    int srcHeight;
    int srcW;
    int srcDepth;
    GetNvnCopyRegion( &srcV, &srcHeight, &srcW, &srcDepth, srcCopyRegion, target );

    NVNcopyRegion srcRegion;
    srcRegion.xoffset = srcCopyRegion.GetOffsetU();
    srcRegion.yoffset = srcV;
    srcRegion.zoffset = srcW;
    srcRegion.width = srcCopyRegion.GetWidth();
    srcRegion.height = srcHeight;
    srcRegion.depth = srcDepth;
    NVNcopyRegion dstRegion;
    dstRegion.xoffset = dstOffsetU;
    dstRegion.yoffset = target == NVN_TEXTURE_TARGET_1D_ARRAY ? dstSubresource.GetArrayIndex() : dstOffsetV;
    dstRegion.zoffset = ( target == NVN_TEXTURE_TARGET_2D_ARRAY ||
        target == NVN_TEXTURE_TARGET_2D_MULTISAMPLE_ARRAY ) ? dstSubresource.GetArrayIndex() : dstOffsetW;
    dstRegion.width = srcRegion.width;
    dstRegion.height = srcRegion.height;
    dstRegion.depth = srcRegion.depth;

    NVNtextureView srcView;
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetDefaults( &srcView ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetLevels(
        &srcView, srcCopyRegion.GetSubresource().GetMipLevel(), 1 ) );
    NVNtextureView dstView;
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetDefaults( &dstView ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetLevels(
        &dstView, dstSubresource.GetMipLevel(), 1 ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferCopyTextureToTexture( this->pNvnCommandBuffer,
        pSrcTexture->ToData()->pNvnTexture, &srcView, &srcRegion,
        pDstTexture->ToData()->pNvnTexture, &dstView, &dstRegion, NVN_COPY_FLAGS_NONE ) );
}

void CommandBufferImpl< Target >::CopyBufferToImage( TextureImpl< Target >* pDstTexture,
    const BufferImpl< Target >* pSrcBuffer, const BufferTextureCopyRegion& copyRegion ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstTexture );
    NN_SDK_REQUIRES_NOT_NULL( pSrcBuffer );
    NN_SDK_REQUIRES( IsInitialized( *pDstTexture ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcBuffer ) );
    NN_SDK_REQUIRES( IsRecording( *this ) );

    NVNtextureTarget target = NN_GFX_CALL_NVN_FUNCTION(
        nvnTextureGetTarget( pDstTexture->ToData()->pNvnTexture ) );

    const TextureCopyRegion& dstRegion = copyRegion.GetTextureCopyRegion();

    int offsetY;
    int height;
    int offsetZ;
    int depth;
    GetNvnCopyRegion( &offsetY, &height, &offsetZ, &depth, dstRegion, target );

    NVNcopyRegion region;
    region.xoffset = copyRegion.GetTextureCopyRegion().GetOffsetU();
    region.yoffset = offsetY;
    region.zoffset = offsetZ;
    region.width = copyRegion.GetTextureCopyRegion().GetWidth();
    region.height = height;
    region.depth = depth;

    NVNtextureView view;
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetDefaults( &view ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetLevels(
        &view, copyRegion.GetTextureCopyRegion().GetSubresource().GetMipLevel(), 1 ) );

    NVNbufferAddress bufferAddress = NN_GFX_CALL_NVN_FUNCTION(
        nvnBufferGetAddress( pSrcBuffer->ToData()->pNvnBuffer ) ) + copyRegion.GetBufferOffset();

    ptrdiff_t rowStride;
    ptrdiff_t imageStride;
    GetNvnCopyStride( &rowStride, &imageStride, copyRegion, pDstTexture->ToData()->pNvnTexture );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetCopyRowStride(
        this->pNvnCommandBuffer, rowStride ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetCopyImageStride(
        this->pNvnCommandBuffer, imageStride ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferCopyBufferToTexture( this->pNvnCommandBuffer,
        bufferAddress, pDstTexture->ToData()->pNvnTexture, &view, &region, NVN_COPY_FLAGS_NONE ) );
}

void CommandBufferImpl< Target >::CopyImageToBuffer( BufferImpl< Target >* pDstBuffer,
    const TextureImpl< Target >* pSrcTexture, const BufferTextureCopyRegion& copyRegion ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstBuffer );
    NN_SDK_REQUIRES_NOT_NULL( pSrcTexture );
    NN_SDK_REQUIRES( IsInitialized( *pDstBuffer ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcTexture ) );
    NN_SDK_REQUIRES( IsRecording( *this ) );

    NVNtextureTarget target = NN_GFX_CALL_NVN_FUNCTION(
        nvnTextureGetTarget( pSrcTexture->ToData()->pNvnTexture ) );

    const TextureCopyRegion& srcRegion = copyRegion.GetTextureCopyRegion();

    int offsetY;
    int height;
    int offsetZ;
    int depth;
    GetNvnCopyRegion( &offsetY, &height, &offsetZ, &depth, srcRegion, target );

    NVNcopyRegion region;
    region.xoffset = srcRegion.GetOffsetU();
    region.yoffset = offsetY;
    region.zoffset = offsetZ;
    region.width = srcRegion.GetWidth();
    region.height = height;
    region.depth = depth;

    NVNtextureView view;
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetDefaults( &view ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetLevels(
        &view, srcRegion.GetSubresource().GetMipLevel(), 1 ) );

    NVNbufferAddress bufferAddress = NN_GFX_CALL_NVN_FUNCTION(
        nvnBufferGetAddress( pDstBuffer->ToData()->pNvnBuffer ) ) + copyRegion.GetBufferOffset();

    ptrdiff_t rowStride;
    ptrdiff_t imageStride;
    GetNvnCopyStride( &rowStride, &imageStride, copyRegion, pSrcTexture->ToData()->pNvnTexture );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetCopyRowStride(
        this->pNvnCommandBuffer, rowStride ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetCopyImageStride(
        this->pNvnCommandBuffer, imageStride ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferCopyTextureToBuffer( this->pNvnCommandBuffer,
        pSrcTexture->ToData()->pNvnTexture, &view, &region, bufferAddress, NVN_COPY_FLAGS_NONE ) );
}

void CommandBufferImpl< Target >::CopyBufferToImage( TextureImpl< Target >* pDstTexture,
    const TextureCopyRegion& dstRegion, const BufferImpl< Target >* pSrcBuffer, ptrdiff_t srcOffset ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstTexture );
    NN_SDK_REQUIRES_NOT_NULL( pSrcBuffer );
    NN_SDK_REQUIRES( IsInitialized( *pDstTexture ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcBuffer ) );
    NN_SDK_REQUIRES( IsRecording( *this ) );

    NVNtextureTarget target = NN_GFX_CALL_NVN_FUNCTION(
        nvnTextureGetTarget( pDstTexture->ToData()->pNvnTexture ) );

    int offsetY;
    int height;
    int offsetZ;
    int depth;
    GetNvnCopyRegion( &offsetY, &height, &offsetZ, &depth, dstRegion, target );

    NVNcopyRegion region;
    region.xoffset = dstRegion.GetOffsetU();
    region.yoffset = offsetY;
    region.zoffset = offsetZ;
    region.width = dstRegion.GetWidth();
    region.height = height;
    region.depth = depth;

    NVNtextureView view;
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetDefaults( &view ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetLevels(
        &view, dstRegion.GetSubresource().GetMipLevel(), 1 ) );

    NVNbufferAddress bufferAddress = NN_GFX_CALL_NVN_FUNCTION(
        nvnBufferGetAddress( pSrcBuffer->ToData()->pNvnBuffer ) ) + srcOffset;

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetCopyRowStride( this->pNvnCommandBuffer, 0 ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetCopyImageStride( this->pNvnCommandBuffer, 0 ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferCopyBufferToTexture( this->pNvnCommandBuffer,
        bufferAddress, pDstTexture->ToData()->pNvnTexture, &view, &region, NVN_COPY_FLAGS_NONE ) );
}

void CommandBufferImpl< Target >::CopyImageToBuffer( BufferImpl< Target >* pDstBuffer,
    ptrdiff_t dstOffset, const TextureImpl< Target >* pSrcTexture, const TextureCopyRegion& srcRegion ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstBuffer );
    NN_SDK_REQUIRES_NOT_NULL( pSrcTexture );
    NN_SDK_REQUIRES( IsInitialized( *pDstBuffer ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcTexture ) );
    NN_SDK_REQUIRES( IsRecording( *this ) );

    NVNtextureTarget target = NN_GFX_CALL_NVN_FUNCTION(
        nvnTextureGetTarget( pSrcTexture->ToData()->pNvnTexture ) );

    int offsetY;
    int height;
    int offsetZ;
    int depth;
    GetNvnCopyRegion( &offsetY, &height, &offsetZ, &depth, srcRegion, target );

    NVNcopyRegion region;
    region.xoffset = srcRegion.GetOffsetU();
    region.yoffset = offsetY;
    region.zoffset = offsetZ;
    region.width = srcRegion.GetWidth();
    region.height = height;
    region.depth = depth;

    NVNtextureView view;
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetDefaults( &view ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetLevels(
        &view, srcRegion.GetSubresource().GetMipLevel(), 1 ) );

    NVNbufferAddress bufferAddress = NN_GFX_CALL_NVN_FUNCTION(
        nvnBufferGetAddress( pDstBuffer->ToData()->pNvnBuffer ) ) + dstOffset;

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetCopyRowStride( this->pNvnCommandBuffer, 0 ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetCopyImageStride( this->pNvnCommandBuffer, 0 ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferCopyTextureToBuffer( this->pNvnCommandBuffer,
        pSrcTexture->ToData()->pNvnTexture, &view, &region, bufferAddress, NVN_COPY_FLAGS_NONE ) );
}

void CommandBufferImpl< Target >::BlitImage( TextureImpl< Target >* pDstTexture,
    const TextureCopyRegion& dstCopyRegion, const TextureImpl< Target >* pSrcTexture,
    const TextureCopyRegion& srcCopyRegion, int copyFlags ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstTexture );
    NN_SDK_REQUIRES_NOT_NULL( pSrcTexture );
    NN_SDK_REQUIRES( IsInitialized( *pDstTexture ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcTexture ) );
    NN_SDK_REQUIRES( IsRecording( *this ) );
    NN_SDK_ASSERT_NOT_NULL( pDstTexture->ToData()->pNvnTexture.ptr );
    NN_SDK_ASSERT_NOT_NULL( pSrcTexture->ToData()->pNvnTexture.ptr );

    NVNtextureTarget dstTarget = NN_GFX_CALL_NVN_FUNCTION(
        nvnTextureGetTarget( pDstTexture->ToData()->pNvnTexture ) );

    int dstV;
    int dstHeight;
    int dstW;
    int dstDepth;
    GetNvnCopyRegion( &dstV, &dstHeight, &dstW, &dstDepth, dstCopyRegion, dstTarget );

    NVNtextureTarget srcTarget = NN_GFX_CALL_NVN_FUNCTION(
        nvnTextureGetTarget( pSrcTexture->ToData()->pNvnTexture ) );

    int srcV;
    int srcHeight;
    int srcW;
    int srcDepth;
    GetNvnCopyRegion( &srcV, &srcHeight, &srcW, &srcDepth, srcCopyRegion, srcTarget );

    NVNcopyRegion dstRegion;
    dstRegion.xoffset = dstCopyRegion.GetOffsetU();
    dstRegion.yoffset = dstV;
    dstRegion.zoffset = dstW;
    dstRegion.width = dstCopyRegion.GetWidth();
    dstRegion.height = dstHeight;
    dstRegion.depth = dstDepth;

    NVNcopyRegion srcRegion;
    srcRegion.xoffset = srcCopyRegion.GetOffsetU();
    srcRegion.yoffset = srcV;
    srcRegion.zoffset = srcW;
    srcRegion.width = srcCopyRegion.GetWidth();
    srcRegion.height = srcHeight;
    srcRegion.depth = srcDepth;

    NVNtextureView dstView;
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetDefaults( &dstView ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetLevels( &dstView,
        dstCopyRegion.GetSubresource().GetMipLevel(), 1 ) );

    NVNtextureView srcView;
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetDefaults( &srcView ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetLevels( &srcView,
        srcCopyRegion.GetSubresource().GetMipLevel(), 1 ) );

    int nvnCopyFlags = 0;
    nvnCopyFlags |= ( copyFlags & ImageCopyFlag_LinearFilter )
        ? NVN_COPY_FLAGS_LINEAR_FILTER_BIT : 0;
    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferCopyTextureToTexture(
        this->ToData()->pNvnCommandBuffer, pSrcTexture->ToData()->pNvnTexture, &srcView,
        &srcRegion, pDstTexture->ToData()->pNvnTexture, &dstView, &dstRegion, nvnCopyFlags ) );
}

void CommandBufferImpl< Target >::ClearBuffer( BufferImpl< Target >* pBuffer,
    ptrdiff_t offset, size_t size, uint32_t value ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );
    NN_SDK_REQUIRES_NOT_NULL( pBuffer );
    NN_SDK_REQUIRES( offset >= 0 );
    NN_SDK_REQUIRES( nn::util::is_aligned( offset, 4 ) );
    NN_SDK_REQUIRES( nn::util::is_aligned( size, 4 ) );

    NVNbufferAddress bufferAddress = NVNbufferAddress();
    NN_GFX_CALL_NVN_FUNCTION( bufferAddress = nvnBufferGetAddress( pBuffer->ToData()->pNvnBuffer ) );
    bufferAddress += offset;

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferClearBuffer(
        this->ToData()->pNvnCommandBuffer, bufferAddress, size, value ) );
}

void CommandBufferImpl< Target >::ClearColor( ColorTargetViewImpl< Target >* pColorTarget,
    float r, float g, float b, float a, const TextureArrayRange* pArrayRange ) NN_NOEXCEPT
{
    ClearColorValue clearColor = { { r, g, b, a } };
    return ClearColorTarget( pColorTarget, clearColor, pArrayRange );
}

void CommandBufferImpl< Target >::ClearColorTarget( ColorTargetViewImpl< Target >* pColorTarget,
    const ClearColorValue& clearColor, const TextureArrayRange* pArrayRange ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pColorTarget );
    NN_SDK_REQUIRES( IsInitialized( *pColorTarget ) );
    NN_SDK_REQUIRES( IsRecording( *this ) );
    NN_UNUSED( pArrayRange );

    const NVNtexture* const pNvnTexture = pColorTarget->ToData()->pNvnTexture;
    const NVNtextureView* const pNvnTextureView = pColorTarget->ToData()->pNvnTextureView;
    NN_SDK_ASSERT_NOT_NULL( pNvnTexture );

    NVNcopyRegion region = {};

    int level = 0;
    NVNtextureTarget target;
    bool hasViewLayer = false;
    int viewLayerCount = 0;
    if( pNvnTextureView )
    {
        int levelCount;
        NVNboolean result;
        NN_UNUSED( result );
        NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewGetLevels(
            pNvnTextureView, &level, &levelCount ) );
        NN_GFX_CALL_NVN_FUNCTION( result = nvnTextureViewGetTarget( pNvnTextureView, &target ) );
        NN_SDK_ASSERT_EQUAL( result, NVN_TRUE );
        int minLayer;
        NN_GFX_CALL_NVN_FUNCTION( hasViewLayer = ( nvnTextureViewGetLayers(
            pNvnTextureView, &minLayer, &viewLayerCount ) == NVN_TRUE ) );
    }
    else
    {
        NN_GFX_CALL_NVN_FUNCTION( target = nvnTextureGetTarget( pNvnTexture ) );
    }
    NN_GFX_CALL_NVN_FUNCTION( region.width = nvnTextureGetWidth( pNvnTexture ) );
    NN_GFX_CALL_NVN_FUNCTION( region.height = nvnTextureGetHeight( pNvnTexture ) );
    NN_GFX_CALL_NVN_FUNCTION( region.depth = nvnTextureGetDepth( pNvnTexture ) );

    region.width = std::max NN_PREVENT_MACRO_FUNC( region.width >> level, 1 );
    if( target != NVN_TEXTURE_TARGET_1D && target != NVN_TEXTURE_TARGET_1D_ARRAY )
    {
        region.height = std::max NN_PREVENT_MACRO_FUNC( region.height >> level, 1 );
    }
    if( target == NVN_TEXTURE_TARGET_3D )
    {
        region.depth = std::max NN_PREVENT_MACRO_FUNC( region.depth >> level, 1 );
        if( hasViewLayer )
        {
            region.depth = viewLayerCount;
        }
    }

    if( pArrayRange )
    {
        if( target == NVN_TEXTURE_TARGET_1D_ARRAY )
        {
            region.yoffset = pArrayRange->GetBaseArrayIndex();
            region.height = pArrayRange->GetArrayLength();
        }
        else if( target == NVN_TEXTURE_TARGET_2D_ARRAY
            || target == NVN_TEXTURE_TARGET_2D_MULTISAMPLE_ARRAY )
        {
            region.zoffset = pArrayRange->GetBaseArrayIndex();
            region.depth = pArrayRange->GetArrayLength();
        }
    }
    else if( hasViewLayer )
    {
        if( target == NVN_TEXTURE_TARGET_1D_ARRAY )
        {
            region.height = viewLayerCount;
        }
        else if( target == NVN_TEXTURE_TARGET_2D_ARRAY
            || target == NVN_TEXTURE_TARGET_2D_MULTISAMPLE_ARRAY )
        {
            region.depth = viewLayerCount;
        }
    }

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferClearTexture( this->pNvnCommandBuffer,
        pNvnTexture, pNvnTextureView, &region, clearColor.valueFloat, NVN_CLEAR_COLOR_MASK_RGBA ) );
}

void CommandBufferImpl< Target >::ClearDepthStencil( DepthStencilViewImpl< Target >* pDepthStencil,
    float depth, int stencil, DepthStencilClearMode clearMode, const TextureArrayRange* pArrayRange ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDepthStencil );
    NN_SDK_REQUIRES( IsInitialized( *pDepthStencil ) );
    NN_SDK_REQUIRES( IsRecording( *this ) );
    NN_UNUSED( pArrayRange );

    const NVNtexture* const pNvnTexture = pDepthStencil->ToData()->pNvnTexture;
    const NVNtextureView* const pNvnTextureView = pDepthStencil->ToData()->pNvnTextureView;
    NN_SDK_ASSERT_NOT_NULL( pNvnTexture );

    // レンダリングターゲットを切り替えている
    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetScissor(
        this->pNvnCommandBuffer, 0, 0, 0x7FFFFFFF, 0x7FFFFFFF ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetRenderTargets(
        this->pNvnCommandBuffer, 0, NULL, NULL, pNvnTexture, pNvnTextureView ) );

    switch ( clearMode )
    {
        case DepthStencilClearMode_Depth:
        {
            NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferClearDepthStencil(
                this->pNvnCommandBuffer, depth, NVN_TRUE, stencil, 0 ) );
        }
        break;

        case DepthStencilClearMode_Stencil:
        {
            NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferClearDepthStencil(
                this->pNvnCommandBuffer, depth, NVN_FALSE, stencil, -1 ) );
        }
        break;

        case DepthStencilClearMode_DepthStencil:
        {
            NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferClearDepthStencil(
                this->pNvnCommandBuffer, depth, NVN_TRUE, stencil, -1 ) );
        }
        break;

        default: NN_UNEXPECTED_DEFAULT;
    }
}

void CommandBufferImpl< Target >::Resolve( TextureImpl< Target >* pDstTexture, int dstMipLevel,
    int dstStartArrayIndex, const ColorTargetViewImpl< Target >* pSrcColorTarget,
    const TextureArrayRange* pSrcArrayRange ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstTexture );
    NN_SDK_REQUIRES_NOT_NULL( pSrcColorTarget );
    NN_SDK_REQUIRES( IsInitialized( *pDstTexture ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcColorTarget ) );
    NN_SDK_REQUIRES( IsRecording( *this ) );

    NN_UNUSED( dstMipLevel );
    NN_UNUSED( dstStartArrayIndex );
    NN_UNUSED( pSrcArrayRange );

    const NVNtexture* pSrcNvnTexture = pSrcColorTarget->ToData()->pNvnTexture;
    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferDownsample(
        this->pNvnCommandBuffer, pSrcNvnTexture, pDstTexture->ToData()->pNvnTexture));
}

void CommandBufferImpl< Target >::FlushMemory( int gpuAccessFlags ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );

    int barrier = 0;
    barrier |= ( gpuAccessFlags & ( GpuAccess_ColorBuffer | GpuAccess_DepthStencil
        | GpuAccess_UnorderedAccessBuffer | GpuAccess_QueryBuffer | GpuAccess_Image ) ) ?
        NVN_BARRIER_ORDER_PRIMITIVES_BIT : 0;
    if( barrier )
    {
        NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferBarrier( this->pNvnCommandBuffer, barrier ) );
    }
}

void CommandBufferImpl< Target >::InvalidateMemory( int gpuAccessFlags ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );

    int barrier = 0;
    // 意味的には Flush 側だが IndirectData に限定できるのでこちら側に書いている
    barrier |= ( gpuAccessFlags & GpuAccess_IndirectBuffer ) ?
        NVN_BARRIER_ORDER_INDIRECT_DATA_BIT : 0;
    barrier |= ( gpuAccessFlags & ( GpuAccess_Texture | GpuAccess_Image ) ) ?
        NVN_BARRIER_INVALIDATE_TEXTURE_BIT : 0;
    barrier |= ( gpuAccessFlags & ( GpuAccess_ShaderCode | GpuAccess_ConstantBuffer
        | GpuAccess_UnorderedAccessBuffer ) ) ?  NVN_BARRIER_INVALIDATE_SHADER_BIT : 0;
    barrier |= ( gpuAccessFlags & GpuAccess_Descriptor ) ?
        NVN_BARRIER_INVALIDATE_TEXTURE_DESCRIPTOR_BIT : 0;
    if( barrier )
    {
        NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferBarrier( this->pNvnCommandBuffer, barrier ) );
    }
}

void CommandBufferImpl< Target >::CallCommandBuffer(
    const CommandBufferImpl< Target >* pNestedCommandBuffer ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );

    NVNcommandHandle nvnCommandHandle = pNestedCommandBuffer->ToData()->hNvnCommandBuffer;

    NN_GFX_CALL_NVN_FUNCTION(
        nvnCommandBufferCallCommands(
            this->pNvnCommandBuffer,
            1,
            &nvnCommandHandle
            ) );
}

void CommandBufferImpl< Target >::CopyCommandBuffer(
    const CommandBufferImpl< Target >* pNestedCommandBuffer ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );

    NVNcommandHandle nvnCommandHandle = pNestedCommandBuffer->ToData()->hNvnCommandBuffer;

    NN_GFX_CALL_NVN_FUNCTION(
        nvnCommandBufferCopyCommands(
            this->pNvnCommandBuffer,
            1,
            &nvnCommandHandle
            ) );
}

void CommandBufferImpl< Target >::SetBufferStateTransition(
    BufferImpl< Target >*, int oldState, int oldStageBits, int newState, int newStageBits ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );
    NN_UNUSED( oldStageBits );

    int barrier = 0;

    if( ( oldState | newState ) & ( BufferState_UnorderedAccessBuffer | BufferState_QueryBuffer ) )
    {
        barrier |= ( newStageBits & ( PipelineStageBit_VertexInput | PipelineStageBit_VertexShader
            | PipelineStageBit_HullShader | PipelineStageBit_DomainShader |
            PipelineStageBit_GeometryShader | PipelineStageBit_ComputeShader ) ) ?
            NVN_BARRIER_ORDER_PRIMITIVES_BIT : 0;
        barrier |= ( newStageBits & ( PipelineStageBit_PixelShader | PipelineStageBit_RenderTarget ) ) ?
            NVN_BARRIER_ORDER_FRAGMENTS_BIT : 0;
        barrier |= ( newState & ( BufferState_IndirectArgument ) ) ?
            NVN_BARRIER_ORDER_INDIRECT_DATA_BIT : 0;
    }

    barrier |= ( newState & ( BufferState_ConstantBuffer | BufferState_UnorderedAccessBuffer ) ) ?
        NVN_BARRIER_INVALIDATE_SHADER_BIT : 0;

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferBarrier( this->pNvnCommandBuffer, barrier ) );
}

void CommandBufferImpl< Target >::SetTextureStateTransition( TextureImpl< Target >*,
    const TextureSubresourceRange*, int oldState, int oldStageBits, int newState, int newStageBits ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );
    NN_UNUSED( oldStageBits );

    int barrier = 0;

    if( ( oldState | newState ) & ( TextureState_ColorTarget |
        TextureState_DepthWrite | TextureState_ShaderWrite | TextureState_Clear ) )
    {
        barrier |= ( newStageBits & ( PipelineStageBit_VertexInput | PipelineStageBit_VertexShader
            | PipelineStageBit_HullShader | PipelineStageBit_DomainShader |
            PipelineStageBit_GeometryShader | PipelineStageBit_ComputeShader ) ) ?
            NVN_BARRIER_ORDER_PRIMITIVES_BIT : 0;
        barrier |= ( newStageBits & ( PipelineStageBit_PixelShader | PipelineStageBit_RenderTarget ) ) ?
            NVN_BARRIER_ORDER_FRAGMENTS_BIT : 0;
        // アーリィ深度テスト
        barrier |= ( newState & ( TextureState_DepthRead | TextureState_DepthWrite | TextureState_Clear ) ) ?
            NVN_BARRIER_ORDER_PRIMITIVES_BIT : 0;
    }

    barrier |= ( newState & TextureState_ShaderRead ) ?
        NVN_BARRIER_INVALIDATE_TEXTURE_BIT : 0;

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferBarrier( this->pNvnCommandBuffer, barrier ) );
}

void CommandBufferImpl< Target >::SetDescriptorPool(
    const DescriptorPoolImpl< Target >* pDescriptorPool ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );
    NN_SDK_REQUIRES_NOT_NULL( pDescriptorPool );
    NN_SDK_REQUIRES( IsInitialized( *pDescriptorPool ) );

    switch ( pDescriptorPool->ToData()->descriptorPoolType )
    {
    case DescriptorPoolType_TextureView:
        {
            NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetTexturePool(
                this->pNvnCommandBuffer, pDescriptorPool->ToData()->pDescriptorPool ) );
        }
        break;
    case DescriptorPoolType_Sampler:
        {
            NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetSamplerPool(
                this->pNvnCommandBuffer, pDescriptorPool->ToData()->pDescriptorPool ) );
        }
        break;
    default:
        break;
    }
}

void CommandBufferImpl< Target >::SetRootSignature(
    PipelineType pipelineType, RootSignatureImpl< Target >* pRootSignature ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );
    NN_SDK_REQUIRES( !pRootSignature || pRootSignature->ToData()->pWorkMemory );

    NN_UNUSED( pipelineType );

    this->pGfxRootSignature = pRootSignature;
}

void CommandBufferImpl< Target >::SetRootBufferDescriptorTable( PipelineType pipelineType,
    int indexDescriptorTable, const DescriptorSlot& startBufferDescriptorSlot ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );

    return nn::gfx::detail::SetRootBufferDescriptorTable( this, pipelineType, indexDescriptorTable,
        startBufferDescriptorSlot, DescriptorPoolImpl< Target >::GetDescriptorSlotIncrementSize(
        this->pNnDevice, DescriptorPoolType_BufferView ) );
}

void CommandBufferImpl< Target >::SetRootTextureAndSamplerDescriptorTable(
    PipelineType pipelineType, int indexDescriptorTable, const DescriptorSlot& startTextureDescriptorSlot,
    const DescriptorSlot& startSamplerDescriptorSlot ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );

    return nn::gfx::detail::SetRootTextureAndSamplerDescriptorTable( this, pipelineType,
        indexDescriptorTable, startTextureDescriptorSlot, startSamplerDescriptorSlot,
        DescriptorPoolImpl< Target >::GetDescriptorSlotIncrementSize( this->pNnDevice, DescriptorPoolType_TextureView ),
        DescriptorPoolImpl< Target >::GetDescriptorSlotIncrementSize( this->pNnDevice, DescriptorPoolType_Sampler ) );
}

void CommandBufferImpl< Target >::SetRootConstantBuffer( PipelineType pipelineType,
    int indexDynamicDescriptor, const GpuAddress& constantBufferAddress, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );

    return nn::gfx::detail::SetRootConstantBuffer( this,
        pipelineType, indexDynamicDescriptor, constantBufferAddress, size );
}

void CommandBufferImpl< Target >::SetRootUnorderedAccessBuffer( PipelineType pipelineType,
    int indexDynamicDescriptor, const GpuAddress& unorderedAccessBufferAddress, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );

    return nn::gfx::detail::SetRootUnorderedAccessBuffer( this,
        pipelineType, indexDynamicDescriptor, unorderedAccessBufferAddress, size );
}

void CommandBufferImpl< Target >::SetRootTextureAndSampler( PipelineType pipelineType, int indexDynamicDescriptor,
    const TextureViewImpl< Target >* pTextureView, const SamplerImpl< Target >* pSampler ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );

    return nn::gfx::detail::SetRootTextureAndSampler( this,
        pipelineType, indexDynamicDescriptor, pTextureView, pSampler );
}

void CommandBufferImpl< Target >::BeginQuery( QueryTarget target ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );
    NN_SDK_REQUIRES_NOT_EQUAL( target, QueryTarget_Timestamp );

    if( target != QueryTarget_ComputeShaderInvocations )
    {
        NVNcounterType counterType = Nvn::GetCounterType( target );
        NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferResetCounter(
            this->ToData()->pNvnCommandBuffer, counterType ) );
    }
}

void CommandBufferImpl< Target >::EndQuery(
    const GpuAddress& dstBufferAddress, QueryTarget target ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );
    NN_SDK_REQUIRES_NOT_EQUAL( target, QueryTarget_Timestamp );
    NN_SDK_ASSERT_NOT_EQUAL( Nvn::GetBufferAddress( dstBufferAddress ), 0 );

    if( target != QueryTarget_ComputeShaderInvocations )
    {
        NVNcounterType counterType = Nvn::GetCounterType( target );
        NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferReportCounter(
            this->pNvnCommandBuffer, counterType, Nvn::GetBufferAddress( dstBufferAddress ) ) );
    }
}

void CommandBufferImpl< Target >::WriteTimestamp( const GpuAddress& dstBufferAddress ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );
    NN_SDK_ASSERT_NOT_EQUAL( Nvn::GetBufferAddress( dstBufferAddress ), 0 );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferReportCounter( this->pNvnCommandBuffer,
        NVN_COUNTER_TYPE_TIMESTAMP, Nvn::GetBufferAddress( dstBufferAddress ) ) );
}

void CommandBufferImpl< Target >::SetDepthBounds( float minDepthBounds, float maxDepthBounds ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetDepthBounds(
        this->pNvnCommandBuffer, NVN_TRUE, minDepthBounds, maxDepthBounds ) );
}

void CommandBufferImpl< Target >::SetLineWidth( float lineWidth ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetLineWidth(
        this->pNvnCommandBuffer, lineWidth ) );
}

void CommandBufferImpl< Target >::SetViewports( int firstViewport,
    int viewportCount, const ViewportStateInfo* pViewports ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );

    const int maxViewports = 16;
    float viewports[ maxViewports * 4 ];
    float depthRanges[ maxViewports * 2 ];
    float* pViewport = viewports;
    float* pDepthRange = depthRanges;
    for( int idxViewport = 0; idxViewport < viewportCount; ++idxViewport )
    {
        const ViewportStateInfo& viewport = pViewports[ idxViewport ];
        *pViewport++ = viewport.GetOriginX();
        *pViewport++ = viewport.GetOriginY();
        *pViewport++ = viewport.GetWidth();
        *pViewport++ = viewport.GetHeight();
        *pDepthRange++ = viewport.GetMinDepth();
        *pDepthRange++ = viewport.GetMaxDepth();
    }
    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetViewports(
        this->pNvnCommandBuffer, firstViewport, viewportCount, viewports ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetDepthRanges(
        this->pNvnCommandBuffer, firstViewport, viewportCount, depthRanges ) );
}

void CommandBufferImpl< Target >::SetScissors( int firstScissor,
    int scissorCount, const ScissorStateInfo* pScissors ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );

    const int maxScissors = 16;
    int scissors[ maxScissors * 4 ];
    int* pScissor = scissors;
    for( int idxScissor = 0; idxScissor < scissorCount; ++idxScissor )
    {
        const ScissorStateInfo& scissor = pScissors[ idxScissor ];
        *pScissor++ = scissor.GetOriginX();
        *pScissor++ = scissor.GetOriginY();
        *pScissor++ = scissor.GetWidth();
        *pScissor++ = scissor.GetHeight();
    }
    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetScissors(
        this->pNvnCommandBuffer, firstScissor, scissorCount, scissors ) );
}

void CommandBufferImpl< Target >::UpdateBuffer( const GpuAddress& dstBufferAddress,
    size_t bufferSize, ptrdiff_t dstOffset, size_t dataSize, const void* pData ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( dstOffset >= 0 );
    NN_SDK_REQUIRES( dstOffset + dataSize <= bufferSize );
    NN_SDK_REQUIRES( pData != NULL || dataSize == 0 );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferUpdateUniformBuffer( this->pNvnCommandBuffer,
        Nvn::GetBufferAddress( dstBufferAddress ), bufferSize, dstOffset, dataSize, pData ) );
}

// 以下はルートシグネチャを使わない場合

void CommandBufferImpl< Target >::SetConstantBuffer( int slot, ShaderStage stage,
    const DescriptorSlot& constantBufferDescriptor ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );
    NN_SDK_ASSERT( slot >= 0 );

    nn::util::ConstBytePtr pDescriptor(ToPtr< void* >(constantBufferDescriptor));
    NVNbufferAddress address = *pDescriptor.Get< NVNbufferAddress >();
    size_t size = *pDescriptor.Advance( sizeof( NVNbufferAddress ) ).Get< size_t >();
    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferBindUniformBuffer(
        this->pNvnCommandBuffer, Nvn::GetShaderStage( stage ), slot, address, size ) );
}

void CommandBufferImpl< Target >::SetUnorderedAccessBuffer( int slot, ShaderStage stage,
    const DescriptorSlot& unorderedAccessBufferDescriptor ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );
    NN_SDK_ASSERT( slot >= 0 );

    nn::util::ConstBytePtr pDescriptor( ToPtr< void* >( unorderedAccessBufferDescriptor ) );
    NVNbufferAddress address = *pDescriptor.Get< NVNbufferAddress >();
    size_t size = *pDescriptor.Advance( sizeof( NVNbufferAddress ) ).Get< size_t >();
    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferBindStorageBuffer(
        this->pNvnCommandBuffer, Nvn::GetShaderStage( stage ), slot, address, size ) );
}

void CommandBufferImpl< Target >::SetTextureAndSampler( int slot, ShaderStage stage,
    const DescriptorSlot& textureDescriptor, const DescriptorSlot& samplerDescriptor ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );

    nn::gfx::detail::SetTextureAndSampler( this, stage, slot,
        static_cast< unsigned int >( textureDescriptor.ToData()->value ),
        static_cast< unsigned int >( samplerDescriptor.ToData()->value ) );
}

void CommandBufferImpl< Target >::SetTexture( int slot, ShaderStage stage,
    const DescriptorSlot& textureDescriptor ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );
    NN_SDK_ASSERT( slot >= 0 );

    NVNtextureHandle textureHandle = NVNtextureHandle();
    NN_GFX_CALL_NVN_FUNCTION( textureHandle = nvnDeviceGetTexelFetchHandle(
        this->pNnDevice->ToData()->pNvnDevice, static_cast<int>( textureDescriptor.ToData()->value ) ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferBindTexture(
        this->pNvnCommandBuffer, Nvn::GetShaderStage( stage ), slot, textureHandle ) );
}

void CommandBufferImpl< Target >::SetImage( int slot, ShaderStage stage,
    const DescriptorSlot& imageDescriptor ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );
    NN_SDK_ASSERT( slot >= 0 );

    NVNimageHandle imageHandle = NVNimageHandle();
    NN_GFX_CALL_NVN_FUNCTION( imageHandle = nvnDeviceGetImageHandle(
        this->pNnDevice->ToData()->pNvnDevice, static_cast< int >( imageDescriptor.ToData()->value ) ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferBindImage(
        this->pNvnCommandBuffer, Nvn::GetShaderStage( stage ), slot, imageHandle ) );
}

// 以下はデスクリプタープールを使わない場合

void CommandBufferImpl< Target >::SetConstantBuffer( int slot, ShaderStage stage,
    const GpuAddress& constantBufferAddress, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );
    NN_SDK_ASSERT( slot >= 0 );
    NN_SDK_ASSERT_NOT_EQUAL( Nvn::GetBufferAddress( constantBufferAddress ), 0 );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferBindUniformBuffer( this->pNvnCommandBuffer,
        Nvn::GetShaderStage( stage ), slot, Nvn::GetBufferAddress( constantBufferAddress ), size ) );
}

void CommandBufferImpl< Target >::SetUnorderedAccessBuffer( int slot, ShaderStage stage,
    const GpuAddress& unorderedAccessBufferAddress, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRecording( *this ) );
    NN_SDK_ASSERT( slot >= 0 );
    NN_SDK_ASSERT_NOT_EQUAL( Nvn::GetBufferAddress( unorderedAccessBufferAddress ), 0 );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferBindStorageBuffer(
        this->pNvnCommandBuffer, Nvn::GetShaderStage( stage ), slot,
        Nvn::GetBufferAddress( unorderedAccessBufferAddress ), size ) );
}

void CommandBufferImpl< Target >::SetTextureAndSampler( int, ShaderStage,
    const TextureViewImpl< Target >*, const SamplerImpl< Target >* ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( 0 ); // nvn 版ではデスクリプタプールの使用が必須
}

void CommandBufferImpl< Target >::SetImage(
    int, ShaderStage, const TextureViewImpl< Target >* ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( 0 ); // nvn 版ではデスクリプタープールの使用が必須
}

// 以下はパイプラインを使わない場合

void CommandBufferImpl< Target >::SetShader( const ShaderImpl< Target >* pShader, int stageBits ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pShader == NULL || pShader->ToData()->state ==
        ShaderImpl< Target >::DataType::State_Initialized );
    NN_SDK_REQUIRES( IsRecording( *this ) );

    NVNprogram* pProgram = NULL;
    int shaderStageBits = 0;
    if( pShader )
    {
        pProgram = pShader->ToData()->pNvnProgram;
    }
    if( pShader && !pShader->ToData()->flags.GetBit(
        ShaderImpl< Target >::DataType::Flag_SeparationEnable ) )
    {
        shaderStageBits = pShader->ToData()->nvnShaderStageBits;
    }
    else
    {
        shaderStageBits = Nvn::GetShaderStageBits( stageBits );
    }

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferBindProgram(
        this->pNvnCommandBuffer, pProgram, shaderStageBits ) );
}

void CommandBufferImpl< Target >::SetRasterizerState(
    const RasterizerStateImpl< Target >* pRasterizerState ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pRasterizerState );
    NN_SDK_REQUIRES( IsInitialized( *pRasterizerState ) );
    NN_SDK_REQUIRES( IsRecording( *this ) );

    const RasterizerStateImpl< Target >::DataType& rasterizerState = pRasterizerState->ToData();

    // Set polygon mode information
    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferBindPolygonState( this->pNvnCommandBuffer,
        reinterpret_cast< const NVNpolygonState* >( &rasterizerState.nvnPolygonState ) ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetPolygonOffsetClamp(
        this->pNvnCommandBuffer, rasterizerState.nvnSlopeScaledDepthBias,
        rasterizerState.nvnDepthBias, rasterizerState.nvnDepthBiasClamp ) );

    // Set multisample information
    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferBindMultisampleState( this->pNvnCommandBuffer,
        reinterpret_cast< const NVNmultisampleState* >( &rasterizerState.nvnMultisampleState ) ) );

    if ( rasterizerState.flags.GetBit( RasterizerStateImplData< Target >::Flag_MultisampleEnabled ) )
    {
        NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetSampleMask(
            this->pNvnCommandBuffer, rasterizerState.nvnSampleMask ) );
    }

    // This behaves like GL_DEPTH_CLAMP so we must flip the logic.
    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetDepthClamp( this->pNvnCommandBuffer,
        !rasterizerState.flags.GetBit( RasterizerStateImplData< Target >::Flag_DepthClipEnabled ) ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetRasterizerDiscard( this->pNvnCommandBuffer,
        !rasterizerState.flags.GetBit( RasterizerStateImplData< Target >::Flag_RasterEnabled ) ) );

    if( this->flags.GetBit( Flag_ConservativeRasterSupported ) )
    {
        NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetConservativeRasterEnable( this->pNvnCommandBuffer,
            rasterizerState.flags.GetBit( RasterizerStateImplData< Target >::Flag_ConservativeRasterEnabled ) ) );
    }
}

void CommandBufferImpl< Target >::SetBlendState(
    const BlendStateImpl< Target >* pBlendState ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pBlendState );
    NN_SDK_REQUIRES( IsInitialized( *pBlendState ) );
    NN_SDK_REQUIRES( IsRecording( *this ) );

    const BlendStateImpl< Target >::DataType& blendState = pBlendState->ToData();

    const NVNblendState* pBlendStates = blendState.pNvnBlendStateData;
    for( int idxTarget = 0, targetCount = static_cast< int >( blendState.targetCount );
        idxTarget < targetCount; ++idxTarget )
    {
        NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferBindBlendState(
            this->pNvnCommandBuffer, pBlendStates + idxTarget ) );
    }

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferBindChannelMaskState( this->pNvnCommandBuffer,
        reinterpret_cast< const NVNchannelMaskState* >( &blendState.nvnChannelMaskState) ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferBindColorState( this->pNvnCommandBuffer,
        reinterpret_cast< const NVNcolorState* >( &blendState.nvnColorState ) ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetBlendColor(
        this->pNvnCommandBuffer, blendState.nvnBlendConstant ) );
}

void CommandBufferImpl< Target >::SetDepthStencilState(
    const DepthStencilStateImpl< Target >* pDepthStencilState ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDepthStencilState );
    NN_SDK_REQUIRES( IsInitialized( *pDepthStencilState ) );
    NN_SDK_REQUIRES( IsRecording( *this ) );

    const DepthStencilStateImpl< Target >::DataType& depthStencilState = pDepthStencilState->ToData();

     NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferBindDepthStencilState( this->pNvnCommandBuffer,
         reinterpret_cast< const NVNdepthStencilState* >( &depthStencilState.nvnDepthStencilState ) ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetStencilValueMask( this->pNvnCommandBuffer,
        NVN_FACE_FRONT_AND_BACK, depthStencilState.nvnStencilValueMask ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetStencilMask( this->pNvnCommandBuffer,
        NVN_FACE_FRONT_AND_BACK, depthStencilState.nvnStencilMask ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetStencilRef( this->pNvnCommandBuffer,
        NVN_FACE_BACK, depthStencilState.nvnStencilBackRef ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetStencilRef( this->pNvnCommandBuffer,
        NVN_FACE_FRONT, depthStencilState.nvnStencilFrontRef ) );

    if( !depthStencilState.flag.GetBit( DepthStencilStateImpl< Target >::DataType::Flag_DepthBoundsTestEnable ) )
    {
        NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetDepthBounds(
            this->pNvnCommandBuffer, NVN_FALSE, 0.0f, 1.0f ) );
    }
}

void CommandBufferImpl< Target >::SetVertexState(
    const VertexStateImpl< Target >* pVertexState ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pVertexState );
    NN_SDK_REQUIRES( IsInitialized( *pVertexState ) );
    NN_SDK_REQUIRES( IsRecording( *this ) );

    const VertexStateImpl< Target >::DataType& vertexState = pVertexState->ToData();
    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferBindVertexAttribState(
        this->pNvnCommandBuffer, vertexState.vertexAttributeStateCount, vertexState.pNvnVertexAttribState ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferBindVertexStreamState(
        this->pNvnCommandBuffer, vertexState.vertexStreamStateCount, vertexState.pNvnVertexStreamState ) );
}

void CommandBufferImpl< Target >::SetTessellationState(
    const TessellationStateImpl< Target >* pTessellationState ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pTessellationState );
    NN_SDK_REQUIRES( IsInitialized( *pTessellationState ) );
    NN_SDK_REQUIRES( IsRecording( *this ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnCommandBufferSetPatchSize(
        this->pNvnCommandBuffer, pTessellationState->ToData()->patchSize ) );
}

}
}
}

NN_PRAGMA_POP_WARNINGS
