﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>

#include <nn/nn_SdkAssert.h>

#include <nn/util/util_BitUtil.h>
#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_CommandBufferInfo.h>
#include <nn/gfx/gfx_TextureInfo.h>
#include <nn/gfx/gfx_GpuAddress.h>
#include <nn/gfx/gfx_RootSignatureInfo.h>
#include <nn/gfx/gfx_CommandBuffer.h>

#include <nn/gfx/detail/gfx_Misc.h>
#include <nn/gfx/detail/gfx_CommandBuffer-api.gx.2.h>
#include <nn/gfx/detail/gfx_State-api.gx.2.h>
#include <nn/gfx/detail/gfx_Texture-api.gx.2.h>
#include <nn/gfx/detail/gfx_Pipeline-api.gx.2.h>
#include <nn/gfx/detail/gfx_Device-api.gx.2.h>
#include <nn/gfx/detail/gfx_Buffer-api.gx.2.h>
#include <nn/gfx/detail/gfx_Sampler-api.gx.2.h>
#include <nn/gfx/detail/gfx_Shader-api.gx.2.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.gx.2.h>
#include <nn/gfx/detail/gfx_RootSignature-api.gx.2.h>
#include <nn/gfx/detail/gfx_DescriptorPool-api.gx.2.h>

#include "gfx_CommonHelper.h"
#include "gfx_GxHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationGx2 Target;

namespace {

const GX2PrimitiveType g_PrimitiveTable[] =
{
    GX2_PRIMITIVE_POINTS,                  // PrimitiveTopology_PointList
    GX2_PRIMITIVE_LINES,                   // PrimitiveTopology_LineList
    GX2_PRIMITIVE_LINE_STRIP,              // PrimitiveTopology_LineStrip
    GX2_PRIMITIVE_TRIANGLES,               // PrimitiveTopology_TriangleList
    GX2_PRIMITIVE_TRIANGLE_STRIP,          // PrimitiveTopology_TriangleStrip
    GX2_PRIMITIVE_LINES_ADJACENCY,         // PrimitiveTopology_LineListAdjacency
    GX2_PRIMITIVE_LINE_STRIP_ADJACENCY,    // PrimitiveTopology_LineStripAdjacency
    GX2_PRIMITIVE_TRIANGLES_ADJACENCY,     // PrimitiveTopology_TriangleListAdjacency
    GX2_PRIMITIVE_TRIANGLE_STRIP_ADJACENCY // PrimitiveTopology_TriangleStripAdjacency
};

const GX2IndexFormat g_IndexFormatTable[] =
{
    GX2_INDEX_FORMAT_U16, // IndexFormat_Uint8 - asserts on GX2
    GX2_INDEX_FORMAT_U16, // IndexFormat_Uint16
    GX2_INDEX_FORMAT_U32, // IndexFormat_Uint32
};

template< typename T >
T* ToPtr( uint64_t handle )
{
    return reinterpret_cast< T* >( static_cast< uintptr_t >( handle ) );
}

bool HasEnoughControlMemory( const CommandBufferImpl< Target >* pObj )
{
    NN_SDK_ASSERT( pObj );
    const CommandBufferImpl< Target >::DataType& obj = pObj->ToData();
    return obj.pHeadControlMemory && obj.pControlMemory && obj.controlMemorySize >=
        nn::util::ConstBytePtr( obj.pHeadControlMemory.ptr ).Distance( obj.pControlMemory ) + sizeof( DisplayList ) * 2;
}

void CheckControlMemory( CommandBufferImpl< Target >* pObj )
{
    NN_SDK_ASSERT( pObj );
    CommandBufferImpl< Target >::DataType& obj = pObj->ToData();
    if( !HasEnoughControlMemory( pObj ) )
    {
        Bit8 oldState = obj.state;
        obj.state = CommandBufferImpl< Target >::DataType::State_Callback;
        NN_SDK_ASSERT_NOT_NULL( obj.pOutOfControlMemoryCallback );
        OutOfMemoryEventArg arg;
        arg.minRequiredSize = obj.controlMemorySize - nn::util::ConstBytePtr(
            obj.pHeadControlMemory.ptr ).Distance( obj.pControlMemory ) + sizeof( DisplayList ) * 2;
        ( *reinterpret_cast< CommandBufferImpl< Target >::OutOfMemoryEventCallback >(
            obj.pOutOfControlMemoryCallback.ptr ) )( static_cast< TCommandBuffer< Target >* >( pObj ), arg );
        obj.state = oldState;
        NN_SDK_ASSERT( HasEnoughControlMemory( pObj ) );
    }
}

void SetTextureAndSamplerWorker( int slot, ShaderStage stage,
    const GX2Texture* pGx2Texture, const GX2Sampler* pGx2Sampler )
{
    NN_SDK_ASSERT( pGx2Texture );
    NN_SDK_ASSERT( pGx2Sampler );

    switch( stage )
    {
    case ShaderStage_Vertex:
        {
            NN_GFX_CALL_GX_FUNCTION( GX2SetVertexTexture( pGx2Texture, slot ) );
            NN_GFX_CALL_GX_FUNCTION( GX2SetVertexSampler( pGx2Sampler, slot ) );
        }
        break;
    case ShaderStage_Geometry:
        {
            NN_GFX_CALL_GX_FUNCTION( GX2SetGeometryTexture( pGx2Texture, slot ) );
            NN_GFX_CALL_GX_FUNCTION( GX2SetGeometrySampler( pGx2Sampler, slot ) );
        }
        break;
    case ShaderStage_Pixel:
        {
            NN_GFX_CALL_GX_FUNCTION( GX2SetPixelTexture( pGx2Texture, slot ) );
            NN_GFX_CALL_GX_FUNCTION( GX2SetPixelSampler( pGx2Sampler, slot ) );
        }
        break;
    case ShaderStage_Compute:
        {
            NN_GFX_CALL_GX_FUNCTION( GX2SetComputeTexture( pGx2Texture, slot ) );
            NN_GFX_CALL_GX_FUNCTION( GX2SetComputeSampler( pGx2Sampler, slot ) );
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void SetupBufferSurface( GX2Surface* pDst, void* pBufferData,
    const TextureImpl< Target >* pTexture, int mipLevel )
{
    NN_SDK_ASSERT_NOT_NULL( pBufferData );

    const GX2Surface* pTextureSurface =
        static_cast< const GX2Surface* >( pTexture->ToData()->pGx2Surface );

    *pDst = *pTextureSurface;
    pDst->width = std::max NN_PREVENT_MACRO_FUNC ( 1, static_cast< int >( pDst->width >> mipLevel ) );
    pDst->height = std::max NN_PREVENT_MACRO_FUNC ( 1, static_cast< int >( pDst->height >> mipLevel ) );
    pDst->tileMode = GX2_TILE_MODE_LINEAR_ALIGNED;
    pDst->numMips = 1;
    pDst->aa = GX2_AA_MODE_1X;
    pDst->imagePtr = pBufferData;
    GX2CalcSurfaceSizeAndAlignment( pDst );

    // GX2 only supports aligned linear buffers
    NN_SDK_ASSERT( pTextureSurface->width == pDst->pitch );
    NN_SDK_ASSERT( nn::util::BytePtr( pBufferData ).IsAligned( pDst->alignment ) );
}

};

size_t CommandBufferImpl< Target >::GetCommandMemoryAlignment( DeviceImpl< Target >* ) NN_NOEXCEPT
{
    return GX2_DISPLAY_LIST_ALIGNMENT;
}

size_t CommandBufferImpl< Target >::GetControlMemoryAlignment( DeviceImpl< Target >* ) NN_NOEXCEPT
{
    return 1;
}

CommandBufferImpl< Target >::CommandBufferImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

CommandBufferImpl< Target >::~CommandBufferImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void CommandBufferImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_ASSERT( pDevice );

    this->commandBufferType = static_cast< Bit8 >( info.GetCommandBufferType() );
    this->queueCapability = static_cast< Bit16 >( info.GetQueueCapability() );

    Reset();

    this->pOutOfCommandMemoryCallback = NULL;
    this->pOutOfControlMemoryCallback = NULL;

    this->pDevice = pDevice;
    this->state = State_Initialized;
}

void CommandBufferImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( pDevice );
    NN_UNUSED( pDevice );

    this->pDevice = NULL;
    this->state = State_NotInitialized;
}

void CommandBufferImpl< Target >::AddCommandMemory( MemoryPoolImpl< Target >* pMemoryPool,
    ptrdiff_t memoryPoolOffset, size_t memorySize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pMemoryPool );
    NN_SDK_REQUIRES( IsInitialized( *pMemoryPool ) );
    NN_SDK_REQUIRES( memoryPoolOffset + memorySize <= pMemoryPool->ToData()->memorySize );
    NN_SDK_ASSERT( pMemoryPool->ToData()->pMemory );
    NN_SDK_REQUIRES( this->state != State_NotInitialized );

    void* pMemory = nn::util::BytePtr( pMemoryPool->ToData()->pMemory, memoryPoolOffset ).Get();
    NN_SDK_ASSERT( nn::util::is_aligned( reinterpret_cast< uintptr_t >( pMemory ), GX2_DISPLAY_LIST_ALIGNMENT ) );
    // The WG needs this memory to be out of the cache before we start any command buffers.
    NN_GFX_CALL_GX_FUNCTION( GX2Invalidate( GX2_INVALIDATE_CPU, pMemory, memorySize ) );

    CheckControlMemory( this );

    DisplayList* pDisplayList = static_cast< DisplayList* >( this->pControlMemory );
    pDisplayList->bufferSize = static_cast< uint32_t >( memorySize );
    pDisplayList->usedSize = 0;
    pDisplayList->pBuffer = pMemory;
    this->pCommandMemory = pDisplayList;
    this->pControlMemory = pDisplayList + 1;
}

void CommandBufferImpl< Target >::AddControlMemory( void* pMemory, size_t memorySize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pMemory );
    NN_SDK_REQUIRES( this->state != State_NotInitialized );
    NN_SDK_ASSERT( memorySize >= sizeof( DisplayList ) * 2 );

    if( this->pControlMemory )
    {
        NN_SDK_ASSERT( this->pHeadControlMemory );
        NN_SDK_ASSERT( static_cast< ptrdiff_t >( this->controlMemorySize ) >=
            nn::util::ConstBytePtr( this->pHeadControlMemory.ptr ).Distance(
            nn::util::ConstBytePtr( this->pControlMemory.ptr, sizeof( DisplayList ) ).Get() ) );
        DisplayList* pDisplayList = static_cast< DisplayList* >( this->pControlMemory );
        pDisplayList->bufferSize = 0;
        pDisplayList->pNextDisplayList = pMemory;
    }

    this->controlMemorySize = memorySize;
    this->pHeadControlMemory = pMemory;
    this->pControlMemory = pMemory;
}

void CommandBufferImpl< Target >::SetOutOfCommandMemoryEventCallback(
    OutOfMemoryEventCallback pCallback ) NN_NOEXCEPT
{
    this->pOutOfCommandMemoryCallback.ptr = reinterpret_cast< void (*)() >( pCallback );
}

void CommandBufferImpl< Target >::SetOutOfControlMemoryEventCallback(
    OutOfMemoryEventCallback pCallback ) NN_NOEXCEPT
{
    this->pOutOfControlMemoryCallback.ptr = reinterpret_cast< void (*)() >( pCallback );
}


void CommandBufferImpl< Target >::Reset() NN_NOEXCEPT
{
    this->pCommandMemory = NULL;
    this->controlMemorySize = sizeof( this->defaultControlMemory );
    this->pHeadControlMemory = this->defaultControlMemory;
    this->pControlMemory = this->defaultControlMemory;
    this->pCommandList = NULL;
}

void CommandBufferImpl< Target >::Begin() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );

    CheckControlMemory( this );

    if( this->pCommandMemory )
    {
        DisplayList* pLastDisplayList = static_cast< DisplayList* >( this->pCommandMemory );
        void* pNextStart = nn::util::BytePtr( pLastDisplayList->pBuffer,
            pLastDisplayList->usedSize ).AlignUp( GX2_DISPLAY_LIST_ALIGNMENT ).Get();
        ptrdiff_t nextSize = static_cast< ptrdiff_t >( pLastDisplayList->bufferSize ) -
            nn::util::ConstBytePtr( pLastDisplayList->pBuffer.ptr ).Distance( pNextStart );
        if( nextSize > 0 )
        {
            DisplayList* pDisplayList = static_cast< DisplayList* >( this->pControlMemory );
            pDisplayList->bufferSize = static_cast< uint32_t >( nextSize );
            pDisplayList->usedSize = 0;
            pDisplayList->pBuffer = pNextStart;
            this->pCommandMemory = pDisplayList;
            this->pControlMemory = pDisplayList + 1;
        }
        else
        {
            this->pCommandMemory = NULL;
        }
    }

    NN_SDK_REQUIRES_NOT_NULL( this->pCommandMemory );

    this->pCommandList = this->pCommandMemory.ptr;

    int coreId = static_cast< int >( OSGetCoreId() );
    NN_SDK_ASSERT( coreId < Gx::MaxCoreCount );
    Gx::g_pCurrentCommandBuffer[ coreId ] = this;

    DisplayList* pDisplayList = static_cast< DisplayList* >( this->pCommandMemory );
    NN_GFX_CALL_GX_FUNCTION( GX2BeginDisplayList( pDisplayList->pBuffer, pDisplayList->bufferSize ) );

    this->state = State_Begun;
}

void CommandBufferImpl< Target >::End() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    DisplayList* pDisplayList = static_cast< DisplayList* >( this->pCommandMemory );
    pDisplayList->usedSize = NN_GFX_CALL_GX_FUNCTION( GX2EndDisplayList( pDisplayList->pBuffer ) );

    pDisplayList = static_cast< DisplayList* >( this->pControlMemory );
    pDisplayList->bufferSize= 0;
    pDisplayList->pNextDisplayList = NULL;

    this->state = State_Initialized;
}

void CommandBufferImpl< Target >::Dispatch( int, int, int ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    NN_SDK_ASSERT( 0 );
}

void CommandBufferImpl< Target >::Draw( PrimitiveTopology primitiveTopology,
    int vertexCount, int vertexOffset ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    NN_GFX_CALL_GX_FUNCTION( GX2DrawEx2( g_PrimitiveTable[
        primitiveTopology ], vertexCount, vertexOffset, 1, 0 ) );
}

void CommandBufferImpl< Target >::Draw( PrimitiveTopology primitiveTopology,
    int vertexCountPerInstance, int vertexOffset, int instanceCount, int baseInstance ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    NN_GFX_CALL_GX_FUNCTION( GX2DrawEx2( g_PrimitiveTable[ primitiveTopology ],
        vertexCountPerInstance, vertexOffset, instanceCount, baseInstance ) );
}

void CommandBufferImpl< Target >::DrawIndexed( PrimitiveTopology primitiveTopology,
    IndexFormat indexFormat, const GpuAddress& indexBufferAddress, int indexCount, int baseVertex ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( indexFormat != IndexFormat_Uint8 );

    NN_GFX_CALL_GX_FUNCTION( GX2DrawIndexedEx2( g_PrimitiveTable[ primitiveTopology ], indexCount,
        g_IndexFormatTable[ indexFormat ], ToPtr< const void >( indexBufferAddress.ToData()->value ), baseVertex, 1, 0 ) );
}

void CommandBufferImpl< Target >::DrawIndexed( PrimitiveTopology primitiveTopology,
    IndexFormat indexFormat, const GpuAddress& indexBufferAddress, int indexCountPerInstance,
    int baseVertex, int instanceCount, int baseInstance ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    NN_GFX_CALL_GX_FUNCTION( GX2DrawIndexedEx2( g_PrimitiveTable[ primitiveTopology ],
        indexCountPerInstance, g_IndexFormatTable[ indexFormat ],
        ToPtr< const void >( indexBufferAddress.ToData()->value ), baseVertex, instanceCount, baseInstance ) );
}

void CommandBufferImpl< Target >::DispatchIndirect( const GpuAddress& indirectBuffer ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    // NOTE: Unlike OpenGL, Cafe does not support the GPU producing the values in the indirect buffer without
    // synchronizing with the CPU.
    GX2DispatchParams* pDispatchParams = ToPtr< GX2DispatchParams >( indirectBuffer.ToData()->value );
    NN_GFX_CALL_GX_FUNCTION( GX2Invalidate(
        GX2_INVALIDATE_ATTRIB_BUFFER, pDispatchParams, sizeof( GX2DispatchParams ) ) );
    NN_GFX_CALL_GX_FUNCTION( GX2DispatchCompute( pDispatchParams ) );
}

void CommandBufferImpl< Target >::DrawIndirect( PrimitiveTopology, const GpuAddress& ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( NULL ); // This is not supported in Cafe
}

void CommandBufferImpl< Target >::DrawIndexedIndirect( PrimitiveTopology,
    IndexFormat, const GpuAddress&, const GpuAddress& ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( NULL ); // This is not supported in Cafe
}

void CommandBufferImpl< Target >::SetPipeline( const PipelineImpl< Target >* pPipeline ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pPipeline );
    NN_SDK_REQUIRES( IsInitialized( *pPipeline ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( pPipeline );

    const PipelineImpl< Target >::DataType& pipeline = pPipeline->ToData();

    if( pPipeline->ToData()->pipelineType == PipelineImplData< Target >::PipelineType_Graphics )
    {
        SetRasterizerState( &pipeline.rasterizerState );
        SetBlendState( &pipeline.blendState );
        SetDepthStencilState( &pipeline.depthStencilState );
        SetVertexState( &pipeline.vertexState );
    }

    SetShader( pipeline.pShader, ShaderStageBit_All );
}

void CommandBufferImpl< Target >::SetRenderTargets( int colorTargetCount,
    const ColorTargetViewImpl< Target >* const* ppColorTargets,
    const DepthStencilViewImpl< Target >* pDepthStencil ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( colorTargetCount < 1 || ppColorTargets );
    NN_SDK_REQUIRES( this->state == State_Begun );

    for( int idx = 0; idx < colorTargetCount; idx++ )
    {
        NN_SDK_REQUIRES( IsInitialized( *( ppColorTargets[ idx ] ) ) );
        const GX2ColorBuffer* pColorBuffer = static_cast<
            const GX2ColorBuffer *>( ppColorTargets[idx]->ToData()->pGx2ColorBuffer );
        NN_GFX_CALL_GX_FUNCTION( GX2SetColorBuffer(
            pColorBuffer, static_cast< GX2RenderTarget >( GX2_RENDER_TARGET_0 + idx ) ) );
    }

    if( pDepthStencil )
    {
        NN_SDK_REQUIRES( IsInitialized( *pDepthStencil ) );

        GX2DepthBuffer* pDepthBuffer = const_cast< GX2DepthBuffer* >(
            static_cast< const GX2DepthBuffer *>( pDepthStencil->ToData()->pGx2DepthBuffer ) );
        NN_GFX_CALL_GX_FUNCTION( GX2SetDepthBuffer( pDepthBuffer ) );

        if( !colorTargetCount )
        {
            NN_GFX_CALL_GX_FUNCTION( GX2SetAAMode( pDepthBuffer->surface.aa ) );
        }
    }
}

void CommandBufferImpl< Target >::SetVertexBuffer( int bufferIndex,
    const GpuAddress& vertexBuffer, size_t stride, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( bufferIndex >= 0 && bufferIndex < GX2_MAX_ATTRIB_BUFFERS );

    NN_GFX_CALL_GX_FUNCTION( GX2SetAttribBuffer(
        bufferIndex, size, stride, ToPtr< const void >( vertexBuffer.ToData()->value ) ) );
}

void CommandBufferImpl< Target >::SetViewportScissorState(
    const ViewportScissorStateImpl< Target >* pViewportScissor ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pViewportScissor );
    NN_SDK_REQUIRES( IsInitialized( *pViewportScissor ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    const ViewportScissorStateImplData< Target >& param = pViewportScissor->ToData();

    // There is only 1 viewport/scissor in GX2
    NN_GFX_CALL_GX_FUNCTION( GX2SetViewportReg( static_cast<
        const GX2ViewportReg* >( static_cast< const void* >( param.viewportRegister ) ) ) );
    NN_GFX_CALL_GX_FUNCTION( GX2SetScissorReg( static_cast<
        const GX2ScissorReg* >( static_cast< const void* >( param.scissorRegister ) ) ) );
}

void CommandBufferImpl< Target >::CopyBuffer( BufferImpl< Target >* pDstBuffer, ptrdiff_t dstOffset,
    const BufferImpl< Target >* pSrcBuffer, ptrdiff_t srcOffset, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pDstBuffer );
    NN_SDK_REQUIRES( pSrcBuffer );
    NN_SDK_REQUIRES( IsInitialized( *pDstBuffer ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcBuffer ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( NULL ); // Written but not (yet) tested

    GX2Surface srcSurface;
    GX2Surface dstSurface;
    srcSurface.dim = dstSurface.dim = GX2_SURFACE_DIM_1D;
    srcSurface.height = srcSurface.depth = srcSurface.numMips = dstSurface.height = dstSurface.depth = dstSurface.numMips = 0;
    srcSurface.width = pSrcBuffer->ToData()->size;
    dstSurface.width = pDstBuffer->ToData()->size;
    srcSurface.format = dstSurface.format = GX2_SURFACE_FORMAT_TC_R8_UINT;
    srcSurface.aa = dstSurface.aa = GX2_AA_MODE_1X;
    srcSurface.imagePtr = pSrcBuffer->ToData()->pBuffer;
    dstSurface.imagePtr = pDstBuffer->ToData()->pBuffer;

    GX2UTRect srcRect;
    GX2UTRect dstRect;
    srcRect.bottom = srcRect.top = dstRect.bottom = dstRect.top = 0;
    srcRect.left = srcRect.right = srcOffset;
    dstRect.left = dstRect.right = dstOffset;
    srcRect.right += size;
    dstRect.right += size;

    NN_GFX_CALL_GX_FUNCTION( GX2UTSetCopyState( GX2_TRUE ) );
    NN_GFX_CALL_GX_FUNCTION( GX2UTCopySurfaceRectOp(
        &srcSurface, 0, 0, &srcRect, &dstSurface, 0, 0, &dstRect, NULL, 0 ) );
    NN_GFX_CALL_GX_FUNCTION( GX2UTSetCopyState( GX2_FALSE ) );
}

void CommandBufferImpl< Target >::CopyImage( TextureImpl< Target >* pDstTexture,
    const TextureSubresource& dstSubresource, int dstOffsetU, int dstOffsetV, int dstOffsetW,
    const TextureImpl< Target >* pSrcTexture, const TextureCopyRegion& srcCopyRegion ) NN_NOEXCEPT
{
    // TODO: 配列対応

    NN_SDK_REQUIRES( pDstTexture );
    NN_SDK_REQUIRES( pSrcTexture );
    NN_SDK_REQUIRES( IsInitialized( *pDstTexture ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcTexture ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    const GX2Surface* pSrcSurface = static_cast< const GX2Surface* >( pSrcTexture->ToData()->pGx2Surface );
    GX2Surface* pDstSurface = static_cast< GX2Surface* >( pDstTexture->ToData()->pGx2Surface );

    GX2UTRect srcRect;
    GX2UTRect dstRect;
    srcRect.left = srcRect.right = srcCopyRegion.GetOffsetU();
    srcRect.right += srcCopyRegion.GetWidth();
    srcRect.top = srcRect.bottom = srcCopyRegion.GetOffsetV();
    srcRect.bottom += srcCopyRegion.GetHeight();
    dstRect.left = dstRect.right = dstOffsetU;
    dstRect.right += srcCopyRegion.GetWidth();
    dstRect.top = dstRect.bottom = dstOffsetV;
    dstRect.bottom += srcCopyRegion.GetHeight();

    s32 dstW = Gx::IsArrayTarget(pDstSurface->dim) ?
        dstSubresource.GetArrayIndex() : dstOffsetW;
    s32 srcW = Gx::IsArrayTarget( pSrcSurface->dim ) ?
        srcCopyRegion.GetSubresource().GetArrayIndex() : srcCopyRegion.GetOffsetW();

    NN_GFX_CALL_GX_FUNCTION( GX2UTSetCopyState( GX2_TRUE ) );
    NN_GFX_CALL_GX_FUNCTION( GX2UTCopySurfaceRectOp( pSrcSurface,
                                                     srcCopyRegion.GetSubresource().GetMipLevel(),
                                                     srcW,
                                                     &srcRect,
                                                     pDstSurface,
                                                     dstSubresource.GetMipLevel(),
                                                     dstW,
                                                     &dstRect,
                                                     NULL,          // FIXME, need to copy AA surfaces?
                                                     0 ) );
    NN_GFX_CALL_GX_FUNCTION( GX2UTSetCopyState( GX2_FALSE ) );
}

void CommandBufferImpl< Target >::CopyBufferToImage( TextureImpl< Target >* pDstTexture,
    const BufferImpl< Target >* pSrcBuffer, const BufferTextureCopyRegion& copyRegion ) NN_NOEXCEPT
{
    // TODO: stride
    NN_SDK_REQUIRES_NOT_NULL( pDstTexture );
    NN_SDK_REQUIRES_NOT_NULL( pSrcBuffer );
    NN_SDK_REQUIRES( IsInitialized( *pDstTexture ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcBuffer ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    GX2Surface* pDstSurface = static_cast< GX2Surface* >( pDstTexture->ToData()->pGx2Surface );

    const TextureCopyRegion& dstRegion = copyRegion.GetTextureCopyRegion();

    nn::util::BytePtr pSrcBufferData( pSrcBuffer->ToData()->pBuffer, copyRegion.GetBufferOffset() );

    GX2Surface srcSurface;
    SetupBufferSurface( &srcSurface, pSrcBufferData.Get(), pDstTexture,
        dstRegion.GetSubresource().GetMipLevel() );

    GX2UTRect srcRect;
    srcRect.left = 0;
    srcRect.right = srcSurface.width;
    srcRect.top = 0;
    srcRect.bottom = srcSurface.height;

    GX2UTRect dstRect;
    dstRect.left = dstRegion.GetOffsetU();
    dstRect.right = dstRect.left + dstRegion.GetWidth();
    dstRect.top = dstRegion.GetOffsetV();
    dstRect.bottom = dstRect.top + dstRegion.GetHeight();

    s32 dstW = Gx::IsArrayTarget( pDstSurface->dim ) ?
        dstRegion.GetSubresource().GetArrayIndex() : dstRegion.GetOffsetW();

    NN_GFX_CALL_GX_FUNCTION( GX2UTSetCopyState( GX2_TRUE ) );
    NN_GFX_CALL_GX_FUNCTION( GX2UTCopySurfaceRectOp( &srcSurface, 0, 0, &srcRect,
        pDstSurface, dstRegion.GetSubresource().GetMipLevel(), dstW, &dstRect, NULL, 0 ) );
    NN_GFX_CALL_GX_FUNCTION( GX2UTSetCopyState( GX2_FALSE ) );
}

void CommandBufferImpl< Target >::CopyImageToBuffer( BufferImpl< Target >* pDstBuffer,
    const TextureImpl< Target >* pSrcTexture, const BufferTextureCopyRegion& copyRegion ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pDstBuffer );
    NN_SDK_ASSERT_NOT_NULL( pSrcTexture );
    NN_SDK_REQUIRES( IsInitialized( *pDstBuffer ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcTexture ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    const GX2Surface* pSrcSurface = static_cast< const GX2Surface* >(
        pSrcTexture->ToData()->pGx2Surface );

    const TextureCopyRegion& srcRegion = copyRegion.GetTextureCopyRegion();

    nn::util::BytePtr pDstBufferData( pDstBuffer->ToData()->pBuffer, copyRegion.GetBufferOffset() );

    GX2Surface dstSurface;
    SetupBufferSurface( &dstSurface, pDstBufferData.Get(), pSrcTexture,
        srcRegion.GetSubresource().GetMipLevel() );

    GX2UTRect dstRect;
    dstRect.left = 0;
    dstRect.right = dstSurface.width;
    dstRect.top = 0;
    dstRect.bottom = dstSurface.height;

    GX2UTRect srcRect;
    srcRect.left = srcRegion.GetOffsetU();
    srcRect.right = srcRect.left + srcRegion.GetWidth();
    srcRect.top = srcRegion.GetOffsetV();
    srcRect.bottom = srcRect.top + srcRegion.GetHeight();

    s32 srcW = Gx::IsArrayTarget( pSrcSurface->dim ) ?
        srcRegion.GetSubresource().GetArrayIndex() : srcRegion.GetOffsetW();

    NN_GFX_CALL_GX_FUNCTION( GX2UTSetCopyState( GX2_TRUE ) );
    NN_GFX_CALL_GX_FUNCTION( GX2UTCopySurfaceRectOp( pSrcSurface,
        srcRegion.GetSubresource().GetMipLevel(), srcW, &srcRect, &dstSurface, 0, 0, &dstRect, NULL, 0 ) );
    NN_GFX_CALL_GX_FUNCTION( GX2UTSetCopyState( GX2_FALSE ) );
}

void CommandBufferImpl< Target >::CopyBufferToImage( TextureImpl< Target >* pDstTexture,
    const TextureCopyRegion& dstRegion, const BufferImpl< Target >* pSrcBuffer,ptrdiff_t srcOffset ) NN_NOEXCEPT
{
    nn::gfx::BufferTextureCopyRegion copyRegion;
    copyRegion.EditTextureCopyRegion() = dstRegion;
    copyRegion.SetBufferOffset( srcOffset );
    copyRegion.SetBufferRowLength( 0 );
    copyRegion.SetBufferImageHeight( 0 );
    return CopyBufferToImage( pDstTexture, pSrcBuffer, copyRegion );
}

void CommandBufferImpl< Target >::CopyImageToBuffer( BufferImpl< Target >* pDstBuffer,
    ptrdiff_t dstOffset, const TextureImpl< Target >* pSrcTexture, const TextureCopyRegion& srcRegion ) NN_NOEXCEPT
{
    nn::gfx::BufferTextureCopyRegion copyRegion;
    copyRegion.EditTextureCopyRegion() = srcRegion;
    copyRegion.SetBufferOffset( dstOffset );
    copyRegion.SetBufferRowLength( 0 );
    copyRegion.SetBufferImageHeight( 0 );
    return CopyImageToBuffer( pDstBuffer, pSrcTexture, copyRegion );
}

void CommandBufferImpl< Target >::BlitImage( TextureImpl< Target >*,
    const TextureCopyRegion&, const TextureImpl< Target >*, const TextureCopyRegion&, int  ) NN_NOEXCEPT
{
    // TODO
}

void CommandBufferImpl< Target >::ClearBuffer(
    BufferImpl< Target >*, ptrdiff_t, size_t, uint32_t ) NN_NOEXCEPT
{
    // TODO
}

void CommandBufferImpl< Target >::ClearColor( ColorTargetViewImpl< Target >* pColorTarget,
    float red, float green, float blue, float alpha, const TextureArrayRange* pArrayRange ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pColorTarget );
    NN_SDK_REQUIRES( IsInitialized( *pColorTarget ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    GX2ColorBuffer* pColorBuffer = static_cast< GX2ColorBuffer*>( pColorTarget->ToData()->pGx2ColorBuffer );

    NN_GFX_CALL_GX_FUNCTION( GX2UTSetClearState( GX2_TRUE ) );
    NN_GFX_CALL_GX_FUNCTION( GX2UTClearOp( pColorBuffer,
        NULL, red, green, blue, alpha, 0.0f, 0, GX2_CLEAR_NONE, NULL ) );
    NN_GFX_CALL_GX_FUNCTION( GX2UTSetClearState( GX2_FALSE ) );
}

void CommandBufferImpl< Target >::ClearColorTarget( ColorTargetViewImpl< Target >*,
    const ClearColorValue&, const TextureArrayRange* ) NN_NOEXCEPT
{
    // TODO
}

void CommandBufferImpl< Target >::ClearDepthStencil( DepthStencilViewImpl< Target >* pDepthStencil,
    float depth, int stencil, DepthStencilClearMode clearMode, const TextureArrayRange* pArrayRange ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pDepthStencil );
    NN_SDK_REQUIRES( IsInitialized( *pDepthStencil ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    GX2DepthBuffer* pDepthBuffer = static_cast< GX2DepthBuffer* >( pDepthStencil->ToData()->pGx2DepthBuffer );

    NN_GFX_CALL_GX_FUNCTION( GX2UTSetClearState( GX2_TRUE ) );
    NN_GFX_CALL_GX_FUNCTION( GX2UTClearOp( NULL, pDepthBuffer, 0.0f, 0.0f, 0.0f, 0.0f, depth, stencil, GX2_CLEAR_BOTH_D_S_REG, NULL ) );
    NN_GFX_CALL_GX_FUNCTION( GX2UTSetClearState( GX2_FALSE ) );
}

void CommandBufferImpl< Target >::Resolve( TextureImpl< Target >* pDstTexture, int dstMipLevel, int dstStartArrayIndex,
    const ColorTargetViewImpl< Target >* pSrcColorTarget, const TextureArrayRange* pSrcArrayRange ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pDstTexture );
    NN_SDK_REQUIRES( pSrcColorTarget );
    NN_SDK_REQUIRES( IsInitialized( *pDstTexture ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcColorTarget ) );

    const GX2ColorBuffer* pColorBuffer = static_cast< const GX2ColorBuffer* >( pSrcColorTarget->ToData()->pGx2ColorBuffer );
    GX2Surface* pDstSurface = static_cast< GX2Surface* >( pDstTexture->ToData()->pGx2Surface );

    NN_GFX_CALL_GX_FUNCTION( GX2UTSetResolveAAColorState( GX2_TRUE ) );
    NN_GFX_CALL_GX_FUNCTION( GX2UTResolveAAColorBufferOp( pColorBuffer, pDstSurface, dstMipLevel, dstStartArrayIndex ) );
    NN_GFX_CALL_GX_FUNCTION( GX2UTSetResolveAAColorState( GX2_FALSE ) );
}

void CommandBufferImpl< Target >::FlushMemory( int gpuAccessFlags ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    nn::Bit32 invalidationType = 0;

    if( gpuAccessFlags & GpuAccess_UnorderedAccessBuffer )
    {
        invalidationType |= GX2_INVALIDATE_EXPORT_BUFFER;
    }

    if( gpuAccessFlags & GpuAccess_ColorBuffer )
    {
        invalidationType |= GX2_INVALIDATE_COLOR_BUFFER;
    }

    if( gpuAccessFlags & GpuAccess_DepthStencil )
    {
        invalidationType |= GX2_INVALIDATE_DEPTH_BUFFER;
    }

    NN_GFX_CALL_GX_FUNCTION( GX2Invalidate( static_cast< GX2InvalidateType >( invalidationType ),
        0, ~0 ) );
}

void CommandBufferImpl< Target >::InvalidateMemory( int gpuAccessFlags ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    nn::Bit32 invalidationType = 0;

    if( gpuAccessFlags & GpuAccess_VertexBuffer )
    {
        invalidationType |= GX2_INVALIDATE_ATTRIB_BUFFER;
    }

    if( gpuAccessFlags & GpuAccess_IndexBuffer )
    {
        // Nothing to do
    }

    if( gpuAccessFlags & GpuAccess_ConstantBuffer )
    {
        invalidationType |= GX2_INVALIDATE_UNIFORM_BLOCK;
    }

    if( gpuAccessFlags & GpuAccess_Texture )
    {
        invalidationType |= GX2_INVALIDATE_TEXTURE;
    }

    if( gpuAccessFlags & GpuAccess_UnorderedAccessBuffer )
    {
        invalidationType |= GX2_INVALIDATE_EXPORT_BUFFER;
    }

    if( gpuAccessFlags & GpuAccess_ColorBuffer )
    {
        invalidationType |= GX2_INVALIDATE_COLOR_BUFFER;
    }

    if( gpuAccessFlags & GpuAccess_DepthStencil )
    {
        invalidationType |= GX2_INVALIDATE_DEPTH_BUFFER;
    }

    if( gpuAccessFlags & GpuAccess_IndirectBuffer )
    {
        // Sometimes used by GX2DispatchParams
        invalidationType |= GX2_INVALIDATE_ATTRIB_BUFFER;
    }

    NN_GFX_CALL_GX_FUNCTION( GX2Invalidate( static_cast< GX2InvalidateType >( invalidationType ),
        0, ~0 ) );
}

void CommandBufferImpl< Target >::CallCommandBuffer( const CommandBufferImpl< Target >* pNestedCommandBuffer )
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( pNestedCommandBuffer );

    const CommandBufferImpl< Target >::DataType& nestedCommandBuffer = pNestedCommandBuffer->ToData();
    NN_SDK_REQUIRES( nestedCommandBuffer.commandBufferType == CommandBufferType_Nested );
    NN_SDK_REQUIRES( nestedCommandBuffer.state == State_Initialized );

    if( nestedCommandBuffer.pCommandList == NULL )
    {
        return;
    }

    for( const DisplayList* pDisplayList = static_cast< const DisplayList* >(
        nestedCommandBuffer.pCommandList ); pDisplayList; )
    {
        if( pDisplayList->bufferSize == 0 )
        {
            pDisplayList = static_cast< const DisplayList* >( pDisplayList->pNextDisplayList );
        }
        else
        {
            NN_GFX_CALL_GX_FUNCTION( GX2CallDisplayList( pDisplayList->pBuffer, pDisplayList->usedSize ) );
            ++pDisplayList;
        }
    }
}

void CommandBufferImpl< Target >::CopyCommandBuffer( const CommandBufferImpl< Target >* pNestedCommandBuffer )
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( pNestedCommandBuffer );

    const CommandBufferImpl< Target >::DataType& nestedCommandBuffer = pNestedCommandBuffer->ToData();
    NN_SDK_REQUIRES( nestedCommandBuffer.commandBufferType == CommandBufferType_Nested );
    NN_SDK_REQUIRES( nestedCommandBuffer.state == State_Initialized );

    for( const DisplayList* pDisplayList = static_cast< const DisplayList* >(
        nestedCommandBuffer.pCommandList ); pDisplayList; )
    {
        if( pDisplayList->bufferSize == 0 )
        {
            pDisplayList = static_cast< const DisplayList* >( pDisplayList->pNextDisplayList );
        }
        else
        {
            NN_GFX_CALL_GX_FUNCTION( GX2CopyDisplayList( pDisplayList->pBuffer, pDisplayList->usedSize ) );
            ++pDisplayList;
        }
    }
}

void CommandBufferImpl< Target >::SetBufferStateTransition(
    BufferImpl< Target >* pBuffer, int oldState, int, int newState, int ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( pBuffer );

    Bit32 invalidateType = 0;
    if( newState )
    {
        if( oldState & BufferState_UnorderedAccessBuffer )
        {
            invalidateType |= GX2_INVALIDATE_EXPORT_BUFFER;
        }
    }

    if( newState & BufferState_VertexBuffer )
    {
        invalidateType |= GX2_INVALIDATE_ATTRIB_BUFFER;
    }
    if( newState & BufferState_ConstantBuffer )
    {
        invalidateType |= GX2_INVALIDATE_UNIFORM_BLOCK;
    }
    if( newState & BufferState_UnorderedAccessBuffer )
    {
        invalidateType |= GX2_INVALIDATE_EXPORT_BUFFER;
    }
    if( newState & BufferState_IndirectArgument )
    {
        invalidateType |= GX2_INVALIDATE_ATTRIB_BUFFER;
    }

    NN_GFX_CALL_GX_FUNCTION( GX2Invalidate( static_cast< GX2InvalidateType >(
        invalidateType ), pBuffer->ToData()->pBuffer, pBuffer->ToData()->size ) );
}

void CommandBufferImpl< Target >::SetTextureStateTransition( TextureImpl< Target >* pTexture,
    const TextureSubresourceRange* pRange, int oldState, int, int newState, int ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( pTexture );

    Bit32 invalidateType = 0;
    if( newState )
    {
        if( oldState & TextureState_ColorTarget )
        {
            invalidateType |= GX2_INVALIDATE_COLOR_BUFFER;
        }
        if( oldState & TextureState_DepthWrite )
        {
            invalidateType |= GX2_INVALIDATE_DEPTH_BUFFER;
        }
    }

    if( newState & TextureState_ShaderRead )
    {
        invalidateType |= GX2_INVALIDATE_TEXTURE;
    }

    GX2Surface* pGx2Surface = static_cast< GX2Surface* >( pTexture->ToData()->pGx2Surface );

    // TODO 配列
    int start = pRange ? pRange->GetMipRange().GetMinMipLevel() : 0;
    int end = pRange ? start + pRange->GetMipRange().GetMipCount() : pGx2Surface->mipSize + 1;
    if( start <= 0 && end >= 1 )
    {
        NN_GFX_CALL_GX_FUNCTION( GX2Invalidate( static_cast< GX2InvalidateType >(
            invalidateType ), pGx2Surface->imagePtr, pGx2Surface->imageSize ) );
    }
    void* pStart = nn::util::BytePtr( pGx2Surface->mipPtr, start < 2 ? 0 : pGx2Surface->mipOffset[ start - 1 ] ).Get();
    void* pEnd = nn::util::BytePtr( pGx2Surface->mipPtr, end < 2 ? 0 : pGx2Surface->mipOffset[ end - 1 ] ).Get();
    NN_GFX_CALL_GX_FUNCTION( GX2Invalidate( static_cast< GX2InvalidateType >( invalidateType ),
        pStart, nn::util::BytePtr( pStart ).Distance( pEnd ) ) );
}

void CommandBufferImpl< Target >::SetDescriptorPool( const DescriptorPoolImpl< Target >* ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
}

void CommandBufferImpl< Target >::SetRootSignature( PipelineType,
    RootSignatureImpl< Target >* pRootSignature ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( !pRootSignature || pRootSignature->ToData()->pWorkMemory );
    this->pGfxRootSignature = pRootSignature;
}

void CommandBufferImpl< Target >::SetRootBufferDescriptorTable( PipelineType pipelineType,
    int indexDescriptorTable, const DescriptorSlot& startBufferDescriptorSlot ) NN_NOEXCEPT
{
    return nn::gfx::detail::SetRootBufferDescriptorTable( this, pipelineType, indexDescriptorTable,
        startBufferDescriptorSlot, DescriptorPoolImpl< Target >::GetDescriptorSlotIncrementSize(
        this->pDevice, DescriptorPoolType_BufferView ) );
}

void CommandBufferImpl< Target >::SetRootTextureAndSamplerDescriptorTable(
    PipelineType pipelineType, int indexDescriptorTable, const DescriptorSlot& startTextureDescriptorSlot,
    const DescriptorSlot& startSamplerDescriptorSlot ) NN_NOEXCEPT
{
    return nn::gfx::detail::SetRootTextureAndSamplerDescriptorTable( this,
        pipelineType, indexDescriptorTable, startTextureDescriptorSlot, startSamplerDescriptorSlot,
        DescriptorPoolImpl< Target >::GetDescriptorSlotIncrementSize( this->pDevice, DescriptorPoolType_TextureView ),
        DescriptorPoolImpl< Target >::GetDescriptorSlotIncrementSize( this->pDevice, DescriptorPoolType_Sampler ) );
}

void CommandBufferImpl< Target >::SetRootConstantBuffer( PipelineType pipelineType,
    int indexDynamicDescriptor, const GpuAddress& constantBufferAddress, size_t size ) NN_NOEXCEPT
{
    return nn::gfx::detail::SetRootConstantBuffer( this,
        pipelineType, indexDynamicDescriptor, constantBufferAddress, size );
}

void CommandBufferImpl< Target >::SetRootUnorderedAccessBuffer( PipelineType pipelineType,
    int indexDynamicDescriptor, const GpuAddress& unorderedAccessBufferAddress, size_t size ) NN_NOEXCEPT
{
    return nn::gfx::detail::SetRootUnorderedAccessBuffer( this,
        pipelineType, indexDynamicDescriptor, unorderedAccessBufferAddress, size );
}

void CommandBufferImpl< Target >::SetRootTextureAndSampler( PipelineType pipelineType, int indexDynamicDescriptor,
    const TextureViewImpl< Target >* pTextureView, const SamplerImpl< Target >* pSampler ) NN_NOEXCEPT
{
    return nn::gfx::detail::SetRootTextureAndSampler( this,
        pipelineType, indexDynamicDescriptor, pTextureView, pSampler );
}

void CommandBufferImpl< Target >::BeginQuery( QueryTarget ) NN_NOEXCEPT
{
    // TODO
}

void CommandBufferImpl< Target >::EndQuery( const GpuAddress&, QueryTarget ) NN_NOEXCEPT
{
    // TODO
}

void CommandBufferImpl< Target >::WriteTimestamp( const GpuAddress& ) NN_NOEXCEPT
{
    // TODO
}

void CommandBufferImpl< Target >::SetDepthBounds( float, float ) NN_NOEXCEPT
{
}

void CommandBufferImpl< Target >::SetLineWidth( float lineWidth ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_GFX_CALL_GX_FUNCTION( GX2SetLineWidth( lineWidth ) );
}

void CommandBufferImpl< Target >::SetViewports( int,
    int, const ViewportStateInfo* ) NN_NOEXCEPT
{
    // TODO
}

void CommandBufferImpl< Target >::SetScissors( int,
    int, const ScissorStateInfo* ) NN_NOEXCEPT
{
    // TODO
}

void CommandBufferImpl< Target >::SetConstantBuffer( int slot, ShaderStage stage,
    const DescriptorSlot& constantBufferDescriptor ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    nn::util::ConstBytePtr pDescriptor( ToPtr< const void >( constantBufferDescriptor.ToData()->value ) );
    const void* pBuffer = *pDescriptor.Get< const void* >();
    uint32_t size = *pDescriptor.Advance( sizeof( void* ) ).Get< uint32_t >();
    switch( stage )
    {
    case ShaderStage_Vertex:
        {
            NN_GFX_CALL_GX_FUNCTION( GX2SetVertexUniformBlock( slot, size, pBuffer ) );
        }
        break;
    case ShaderStage_Geometry:
        {
            NN_GFX_CALL_GX_FUNCTION( GX2SetGeometryUniformBlock( slot, size, pBuffer ) );
        }
        break;
    case ShaderStage_Pixel:
        {
            NN_GFX_CALL_GX_FUNCTION( GX2SetPixelUniformBlock( slot, size, pBuffer ) );
        }
        break;
    case ShaderStage_Compute:
        {
            NN_GFX_CALL_GX_FUNCTION( GX2SetComputeUniformBlock( slot, size, pBuffer ) );
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void CommandBufferImpl< Target >::SetUnorderedAccessBuffer( int slot, ShaderStage stage,
    const DescriptorSlot& unorderedAccessBufferDescriptor ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_UNUSED( slot );
    NN_UNUSED( stage );
    nn::util::BytePtr pDescriptor( ToPtr< void >( unorderedAccessBufferDescriptor.ToData()->value ) );
    NN_SDK_ASSERT( pDescriptor.Get() );
    void* pBuffer = *pDescriptor.Get< void* >();
    uint32_t size = *pDescriptor.Advance( sizeof( void* ) ).Get< uint32_t >();
    NN_GFX_CALL_GX_FUNCTION( GX2SetShaderExportBuffer( pBuffer, size ) );
}

void CommandBufferImpl< Target >::SetTextureAndSampler( int slot, ShaderStage stage,
    const DescriptorSlot& textureDescriptor, const DescriptorSlot& samplerDescriptor ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( slot >= 0 );
    const GX2Texture** ppGx2Texture = ToPtr< const GX2Texture* >( textureDescriptor.ToData()->value );
    const GX2Sampler** ppGx2Sampler = ToPtr< const GX2Sampler* >( samplerDescriptor.ToData()->value );
    NN_SDK_ASSERT( ppGx2Texture );
    NN_SDK_ASSERT( ppGx2Sampler );
    SetTextureAndSamplerWorker( slot, stage, *ppGx2Texture, *ppGx2Sampler );
}

void CommandBufferImpl< Target >::SetTexture( int, ShaderStage, const DescriptorSlot& ) NN_NOEXCEPT
{
    // TODO
}

void CommandBufferImpl< Target >::SetImage( int, ShaderStage, const DescriptorSlot& ) NN_NOEXCEPT
{
    // TODO
}

void CommandBufferImpl< Target >::SetConstantBuffer(
    int slot, ShaderStage stage, const GpuAddress& constantBufferAddress, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( slot >= 0 && slot < GX2_MAX_UNIFORM_BLOCKS );

    const void* pBuffer = ToPtr< const void >( constantBufferAddress.ToData()->value );
    switch( stage )
    {
    case ShaderStage_Vertex:
        {
            NN_GFX_CALL_GX_FUNCTION( GX2SetVertexUniformBlock( slot, size, pBuffer ) );
        }
        break;
    case ShaderStage_Geometry:
        {
            NN_GFX_CALL_GX_FUNCTION( GX2SetGeometryUniformBlock( slot, size, pBuffer ) );
        }
        break;
    case ShaderStage_Pixel:
        {
            NN_GFX_CALL_GX_FUNCTION( GX2SetPixelUniformBlock( slot, size, pBuffer ) );
        }
        break;
    case ShaderStage_Compute:
        {
            NN_GFX_CALL_GX_FUNCTION( GX2SetComputeUniformBlock( slot, size, pBuffer ) );
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void CommandBufferImpl< Target >::SetUnorderedAccessBuffer( int slot, ShaderStage stage,
    const GpuAddress& unorderedAccessBufferAddress, size_t size ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( !slot ); // NOTE: There's only one SSBO in Cafe
    NN_UNUSED( slot );
    NN_UNUSED( stage );

    void* pBuffer = ToPtr< void >( unorderedAccessBufferAddress.ToData()->value );
    NN_GFX_CALL_GX_FUNCTION( GX2SetShaderExportBuffer( pBuffer, size ) );
}

void CommandBufferImpl< Target >::SetTextureAndSampler( int slot, ShaderStage stage,
    const TextureViewImpl< Target >* pTextureView, const SamplerImpl< Target >* pSampler ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pTextureView );
    NN_SDK_REQUIRES( IsInitialized( *pTextureView ) );
    NN_SDK_REQUIRES( pSampler );
    NN_SDK_REQUIRES( IsInitialized( *pSampler ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    const GX2Texture* pGx2Texture = static_cast< const GX2Texture * >( pTextureView->ToData()->pGx2Texture );
    const GX2Sampler* pGx2Sampler = static_cast< const GX2Sampler * >( pSampler->ToData()->pGx2Sampler );

    SetTextureAndSamplerWorker( slot, stage, pGx2Texture, pGx2Sampler );
}

void CommandBufferImpl< Target >::SetImage( int, ShaderStage, const TextureViewImpl< Target >* ) NN_NOEXCEPT
{
    // TODO
}

void CommandBufferImpl< Target >::SetShader( const ShaderImpl< Target >* pShader, int stageBits ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pShader == NULL || pShader->ToData()->state ==
        ShaderImpl< Target >::DataType::State_Initialized );
    NN_SDK_REQUIRES( this->state == State_Begun );

    stageBits = pShader->ToData()->flags.GetBit( ShaderImpl<
        Target >::DataType::Flag_SeparationEnable ) ? stageBits : ShaderStageBit_All;

    if( pShader )
    {
        if( stageBits & ShaderStageBit_Vertex )
        {
            if( const GX2VertexShader* pVertexShader = pShader->ToData()->pVertexShader )
            {
                GX2Invalidate( GX2_INVALIDATE_SHADER, pVertexShader->shaderPtr, pVertexShader->shaderSize );
                NN_GFX_CALL_GX_FUNCTION( GX2SetVertexShader( pVertexShader ) );
            }
        }
        if( stageBits & ShaderStageBit_Geometry )
        {
            if( const GX2GeometryShader* pGeometryShader = pShader->ToData()->pGeometryShader )
            {
                GX2Invalidate( GX2_INVALIDATE_SHADER, pGeometryShader->shaderPtr, pGeometryShader->shaderSize );
                GX2Invalidate( GX2_INVALIDATE_SHADER, pGeometryShader->copyShaderPtr, pGeometryShader->copyShaderSize );
                NN_GFX_CALL_GX_FUNCTION( GX2SetGeometryShader( pGeometryShader ) );
            }
        }
        if( stageBits & ShaderStageBit_Pixel )
        {
            if( const GX2PixelShader* pPixelShader = pShader->ToData()->pPixelShader )
            {
                GX2Invalidate( GX2_INVALIDATE_SHADER, pPixelShader->shaderPtr, pPixelShader->shaderSize );
                NN_GFX_CALL_GX_FUNCTION( GX2SetPixelShader( pPixelShader ) );
            }
        }
        if( stageBits & ShaderStageBit_Compute )
        {
            if( const GX2ComputeShader* pComputeShader = pShader->ToData()->pComputeShader )
            {
                GX2Invalidate( GX2_INVALIDATE_SHADER, pComputeShader->shaderPtr, pComputeShader->shaderSize );
                NN_GFX_CALL_GX_FUNCTION( GX2SetComputeShader( pComputeShader ) );
            }
        }
    }
}

void CommandBufferImpl< Target >::SetRasterizerState(
    const RasterizerStateImpl< Target >* pRasterizerState ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pRasterizerState );
    NN_SDK_REQUIRES( IsInitialized( *pRasterizerState ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    const RasterizerStateImplData< Target >& param = pRasterizerState->ToData();

    NN_GFX_CALL_GX_FUNCTION( GX2SetPolygonControlReg(
        reinterpret_cast< const GX2PolygonControlReg* >( &param.polygonControlRegister ) ) );
    NN_GFX_CALL_GX_FUNCTION( GX2SetPolygonOffsetReg(
        reinterpret_cast< const GX2PolygonOffsetReg* >( &param.polygonOffsetRegister ) ) );
    NN_GFX_CALL_GX_FUNCTION( GX2SetAAMaskReg(
        reinterpret_cast< const GX2AAMaskReg* >( &param.aaMaskRegister ) ) );
    NN_GFX_CALL_GX_FUNCTION( GX2SetAAMode( static_cast< GX2AAMode >( param.aaModeRegister ) ) );
    NN_GFX_CALL_GX_FUNCTION( GX2SetRasterizerClipControl(
        static_cast< GX2Boolean >( param.rasterizerEnable ),
        static_cast< GX2Boolean >( param.zClipEnable ) ) );
}

void CommandBufferImpl< Target >::SetBlendState(
    const BlendStateImpl< Target >* pBlendState ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pBlendState );
    NN_SDK_REQUIRES( IsInitialized( *pBlendState ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    const BlendStateImplData< Target >& param = pBlendState->ToData();

    NN_GFX_CALL_GX_FUNCTION( GX2SetColorControlReg(
        reinterpret_cast< const GX2ColorControlReg* >( &param.colorControlRegister ) ) );
    for( int idxTarget = 0; idxTarget < param.blendTargetCount; idxTarget++ )
    {
        NN_GFX_CALL_GX_FUNCTION( GX2SetBlendControlReg(
            reinterpret_cast< const GX2BlendControlReg* >( &param.pTargetArray[ idxTarget ].blendControlRegister ) ) );
    }
    NN_GFX_CALL_GX_FUNCTION( GX2SetBlendConstantColorReg(
        reinterpret_cast< const GX2BlendConstantColorReg* >( &param.blendConstantColorRegister ) ) );
    NN_GFX_CALL_GX_FUNCTION( GX2SetAlphaToMaskReg(
        reinterpret_cast< const GX2AlphaToMaskReg* >( &param.alphaToMaskRegister ) ) );
    NN_GFX_CALL_GX_FUNCTION( GX2SetTargetChannelMasksReg(
        reinterpret_cast< const GX2TargetChannelMaskReg* >( &param.targetChannelMaskRegister ) ) );
}

void CommandBufferImpl< Target >::SetDepthStencilState(
    const DepthStencilStateImpl< Target >* pDepthStencilState ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pDepthStencilState );
    NN_SDK_REQUIRES( IsInitialized( *pDepthStencilState ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    const DepthStencilStateImplData< Target >& param = pDepthStencilState->ToData();

    NN_GFX_CALL_GX_FUNCTION( GX2SetDepthStencilControlReg(
        reinterpret_cast< const GX2DepthStencilControlReg* >( &param.depthStencilControlRegister) ) );
    NN_GFX_CALL_GX_FUNCTION( GX2SetStencilMaskReg(
        reinterpret_cast< const GX2StencilMaskReg* >( &param.stencilMaskRegister[ 0 ] ) ) );
}

void CommandBufferImpl< Target >::SetVertexState(
    const VertexStateImpl< Target >* pVertexState ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pVertexState );
    NN_SDK_REQUIRES( IsInitialized( *pVertexState ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    const VertexStateImplData< Target >& param = pVertexState->ToData();
    const GX2FetchShader* pFetchShader = reinterpret_cast< const GX2FetchShader* >( &param.fetchShader );

    NN_GFX_CALL_GX_FUNCTION( GX2Invalidate( GX2_INVALIDATE_SHADER, pFetchShader->shaderPtr, pFetchShader->shaderSize ) );
    NN_GFX_CALL_GX_FUNCTION( GX2SetFetchShader( pFetchShader ) );
}

void CommandBufferImpl< Target >::SetTessellationState(
    const TessellationStateImpl< Target >* pTessellationState ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pTessellationState );
    NN_SDK_REQUIRES( IsInitialized( *pTessellationState ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
}

}
}
}
