﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>

#include <nn/nn_SdkAssert.h>

#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_CommandBufferInfo.h>
#include <nn/gfx/gfx_TextureInfo.h>
#include <nn/gfx/gfx_RootSignatureInfo.h>
#include <nn/gfx/gfx_GpuAddress.h>
#include <nn/gfx/gfx_DescriptorSlot.h>
#include <nn/gfx/gfx_CommandBuffer.h>
#include <nn/gfx/gfx_StateInfo.h>

#include <nn/gfx/detail/gfx_Misc.h>
#include <nn/gfx/detail/gfx_CommandBuffer-api.gl.4.h>
#include <nn/gfx/detail/gfx_State-api.gl.4.h>
#include <nn/gfx/detail/gfx_Texture-api.gl.4.h>
#include <nn/gfx/detail/gfx_Pipeline-api.gl.4.h>
#include <nn/gfx/detail/gfx_Device-api.gl.4.h>
#include <nn/gfx/detail/gfx_Buffer-api.gl.4.h>
#include <nn/gfx/detail/gfx_Sampler-api.gl.4.h>
#include <nn/gfx/detail/gfx_Shader-api.gl.4.h>
#include <nn/gfx/detail/gfx_DescriptorPool-api.gl.4.h>
#include <nn/gfx/detail/gfx_RootSignature-api.gl.4.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.gl.4.h>

#include "gfx_CommonHelper.h"
#include "gfx_GlHelper.h"
#include "gfx_GlCommand.h"

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_DEPRECATED_DECLARATIONS

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationGl4 Target;

namespace {

static const GLenum PrimitiveModeTable[] =
{
    GL_POINTS,
    GL_LINES,
    GL_LINE_STRIP,
    GL_TRIANGLES,
    GL_TRIANGLE_STRIP,
    GL_LINES_ADJACENCY,
    GL_LINE_STRIP_ADJACENCY,
    GL_TRIANGLES_ADJACENCY,
    GL_TRIANGLE_STRIP_ADJACENCY,
    GL_PATCHES
};
NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( PrimitiveModeTable ) == PrimitiveTopology_End );

static const GLenum IndexFormatTable[] =
{
    GL_UNSIGNED_BYTE,
    GL_UNSIGNED_SHORT,
    GL_UNSIGNED_INT
};
NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( IndexFormatTable ) == IndexFormat_End );

template< typename T >
T* ToPtr( const DescriptorSlot& slot )
{
    return reinterpret_cast< T* >( slot.ToData()->value );
}

bool HasEnoughWorkingMemory( const CommandBufferImpl< Target >* pObj )
{
    NN_SDK_ASSERT_NOT_NULL( pObj );
    const CommandBufferImpl< Target >::DataType& obj = pObj->ToData();
    return obj.pHeadControlMemory && obj.pControlMemory && obj.controlMemorySize >=
        nn::util::ConstBytePtr( obj.pHeadControlMemory.ptr ).Distance( obj.pControlMemory ) + sizeof( uint64_t ) * 3;
}

bool HasEnoughCommandMemory( const CommandBufferImpl< Target >* pObj, uint32_t paramSize )
{
    NN_SDK_ASSERT_NOT_NULL( pObj );
    const CommandBufferImpl< Target >::DataType& obj = pObj->ToData();
    return obj.pHeadCommandMemory && obj.pCommandMemory &&
        obj.commandMemorySize >= nn::util::ConstBytePtr( obj.pHeadCommandMemory.ptr ).Distance(
        obj.pCommandMemory ) + offsetof( GlCommand, param ) + paramSize + sizeof( uint64_t );
}

void CheckWorkingMemory( CommandBufferImpl< Target >* pObj )
{
    NN_SDK_ASSERT_NOT_NULL( pObj );
    CommandBufferImpl< Target >::DataType& obj = pObj->ToData();
    if( !HasEnoughWorkingMemory( pObj ) )
    {
        Bit8 oldState = obj.state;
        obj.state = CommandBufferImpl< Target >::DataType::State_Callback;
        NN_SDK_ASSERT_NOT_NULL( obj.pOutOfControlMemoryCallback.ptr );
        OutOfMemoryEventArg arg;
        arg.minRequiredSize = obj.controlMemorySize - nn::util::ConstBytePtr(
            obj.pHeadControlMemory.ptr ).Distance( obj.pControlMemory ) + sizeof( uint64_t ) * 3;
        ( *reinterpret_cast< CommandBufferImpl< Target >::OutOfMemoryEventCallback >(
            obj.pOutOfControlMemoryCallback.ptr ) )( static_cast< TCommandBuffer< Target >* >( pObj ), arg );
        obj.state = oldState;
        NN_SDK_ASSERT( HasEnoughWorkingMemory( pObj ) );
    }
}

void CheckCommandMemory( CommandBufferImpl< Target >* pObj, uint32_t paramSize )
{
    NN_SDK_ASSERT_NOT_NULL( pObj );
    CommandBufferImpl< Target >::DataType& obj = pObj->ToData();
    if( !HasEnoughCommandMemory( pObj, paramSize ) )
    {
        Bit8 oldState = obj.state;
        obj.state = CommandBufferImpl< Target >::DataType::State_Callback;
        NN_SDK_ASSERT_NOT_NULL( obj.pOutOfCommandMemoryCallback.ptr );
        OutOfMemoryEventArg arg;
        arg.minRequiredSize = offsetof( GlCommand, param ) + paramSize + sizeof( uint64_t ) +
            ( ( obj.pHeadCommandMemory.ptr && obj.pCommandMemory.ptr ) ? obj.commandMemorySize -
            nn::util::ConstBytePtr( obj.pHeadCommandMemory.ptr ).Distance( obj.pCommandMemory ) : 0 );
        ( *reinterpret_cast< CommandBufferImpl< Target >::OutOfMemoryEventCallback >(
            obj.pOutOfCommandMemoryCallback.ptr ) )( static_cast< TCommandBuffer< Target >* >( pObj ), arg );
        obj.state = oldState;
        NN_SDK_ASSERT( HasEnoughCommandMemory( pObj, paramSize ) );
    }
}

template< typename TParam >
TParam* WriteGlCommand( CommandBufferImpl< Target >* pObj,
    uint32_t paramSize = sizeof( TParam ) ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pObj );
    NN_SDK_ASSERT( pObj->ToData()->state == CommandBufferImpl< Target >::DataType::State_Begun );

    uint32_t alignedParamSize = nn::util::align_up( paramSize, 8 );
    CheckCommandMemory( pObj, alignedParamSize );

    GlCommand* pCommand = static_cast< GlCommand* >( pObj->ToData()->pCommandMemory );
    pCommand->pCommandProc = GlCommandProc< TParam >;
    pCommand->paramSize = alignedParamSize;
    pObj->ToData()->pCommandMemory = pCommand->param + alignedParamSize;

    return static_cast< TParam* >( static_cast< void* >( pCommand->param ) );
}

template< GlEnum BindTarget >
void SetBuffer( CommandBufferImpl< Target >* pObj, int slot,
    GlHandle handle, uint32_t offset, uint32_t size ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pObj );
    NN_SDK_ASSERT( slot >= 0 );
    NN_SDK_ASSERT( pObj->ToData()->state == CommandBufferImpl< Target >::DataType::State_Begun );

    NN_SDK_ASSERT( IsValid( handle ) );

    GlSetBufferParam& param = *WriteGlCommand< GlSetBufferParam >( pObj );
    {
        param.target = BindTarget;
        param.index = slot;
        param.hBuffer = handle;
        param.offset = offset;
        param.size = size;
    };
}

void GetGlCopyRegion( int* pOffsetY, int* pHeight, int* pOffsetZ, int* pDepth,
    const TextureCopyRegion& region, GlEnum target ) NN_NOEXCEPT
{
    *pOffsetY = region.GetOffsetV();
    *pHeight = region.GetHeight();
    *pOffsetZ = region.GetOffsetW();
    *pDepth = 1;

    switch( target )
    {
    case GL_TEXTURE_1D:
        {
            *pHeight = 1;
        }
        break;
    case GL_TEXTURE_1D_ARRAY:
        {
            *pOffsetY = region.GetSubresource().GetArrayIndex();
            *pHeight = std::max NN_PREVENT_MACRO_FUNC ( region.GetArrayLength(), 1 );
        }
        break;
    case GL_TEXTURE_2D_ARRAY:
    case GL_TEXTURE_2D_MULTISAMPLE_ARRAY:
        {
            *pOffsetZ = region.GetSubresource().GetArrayIndex();
            *pDepth = std::max NN_PREVENT_MACRO_FUNC ( region.GetArrayLength(), 1 );
        }
        break;
    case GL_TEXTURE_3D:
        {
            *pDepth = region.GetDepth();
        }
        break;
    default:
        break;
    }
}

void SetBufferImageCopyParam( GlBufferImageCopyParam* pDst, const BufferImpl< Target >* pBuffer,
    ptrdiff_t bufferOffset, const TextureImpl< Target >* pTexture, const TextureCopyRegion& region,
    int bufferRowLength, int bufferImageHeight ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pDst );
    NN_SDK_ASSERT_NOT_NULL( pBuffer );
    NN_SDK_ASSERT_NOT_NULL( pTexture );

    ImageFormat imageFormat = static_cast< ImageFormat >( pTexture->ToData()->imageFormat );
    ChannelFormat channelFormat = GetChannelFormat( imageFormat );
    GlTextureFormat format = Gl::GetTextureFormat( imageFormat );

    int offsetY;
    int height;
    int offsetZ;
    int depth;
    GetGlCopyRegion( &offsetY, &height, &offsetZ, &depth, region, pTexture->ToData()->target );

    pDst->hBuffer = pBuffer->ToData()->hBuffer;
    pDst->bufferOffset = static_cast< int32_t >( bufferOffset );
    pDst->hTexture = pTexture->ToData()->hTexture;
    pDst->target = pTexture->ToData()->target;
    pDst->level = region.GetSubresource().GetMipLevel();
    pDst->isCompressed = IsCompressedFormat( channelFormat );
    pDst->internalFormat = format.imageFormat;
    pDst->pixelFormat = format.pixelFormat;
    pDst->pixelType = format.pixelType;
    pDst->xOffset = region.GetOffsetU();
    pDst->yOffset = offsetY;
    pDst->zOffset = offsetZ;
    pDst->width = region.GetWidth();
    pDst->height = height;
    pDst->depth = depth;
    pDst->imageSize = static_cast< int >( CalculateImageSize(
        channelFormat, region.GetWidth(), region.GetHeight(), region.GetDepth() ) );
    pDst->rowLength = bufferRowLength;
    pDst->imageHeight = bufferImageHeight;
}

}

size_t CommandBufferImpl< Target >::GetCommandMemoryAlignment( DeviceImpl< Target >* ) NN_NOEXCEPT
{
    return 1;
}

size_t CommandBufferImpl< Target >::GetControlMemoryAlignment( DeviceImpl< Target >* ) NN_NOEXCEPT
{
    return 1;
}

CommandBufferImpl< Target >::CommandBufferImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

CommandBufferImpl< Target >::~CommandBufferImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void CommandBufferImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );

    this->pGfxDevice = pDevice;
    this->commandBufferType = static_cast< Bit8 >( info.GetCommandBufferType() );
    this->queueCapability = static_cast< Bit16 >( info.GetQueueCapability() );

    Reset();

    this->pOutOfCommandMemoryCallback = NULL;
    this->pOutOfControlMemoryCallback = NULL;

    this->state = State_Initialized;
}

void CommandBufferImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    Reset();

    this->pGfxDevice = NULL;
    this->state = State_NotInitialized;
}

void CommandBufferImpl< Target >::AddCommandMemory( MemoryPoolImpl< Target >* pMemoryPool,
    ptrdiff_t memoryPoolOffset, size_t memorySize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pMemoryPool );
    NN_SDK_REQUIRES( IsInitialized( *pMemoryPool ) );
    NN_SDK_REQUIRES( memoryPoolOffset + memorySize <= pMemoryPool->ToData()->memorySize );
    NN_SDK_ASSERT( pMemoryPool->ToData()->pMemory );
    NN_SDK_REQUIRES( this->state != State_NotInitialized );

    void* pMemory = nn::util::BytePtr( pMemoryPool->ToData()->pMemory, memoryPoolOffset ).Get();
    if( this->pCommandMemory )
    {
        NN_SDK_ASSERT_NOT_NULL( this->pHeadCommandMemory.ptr );
        NN_SDK_ASSERT( static_cast< ptrdiff_t >( this->commandMemorySize ) >=
            nn::util::ConstBytePtr( this->pHeadCommandMemory.ptr ).Distance(
            nn::util::BytePtr( this->pCommandMemory.ptr, sizeof( uint64_t ) ).Get() ) );
        *static_cast< uint64_t* >( this->pCommandMemory ) = 0;
    }
    this->commandMemorySize = static_cast< uint32_t >( memorySize );
    this->pHeadCommandMemory = pMemory;
    this->pCommandMemory = pMemory;

    CheckWorkingMemory( this );

    *static_cast< void** >( this->pControlMemory ) = this->pCommandMemory;
    this->pControlMemory = nn::util::BytePtr( this->pControlMemory, sizeof( uint64_t ) ).Get();
}

void CommandBufferImpl< Target >::AddControlMemory( void* pMemory, size_t memorySize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pMemory );
    NN_SDK_REQUIRES( this->state != State_NotInitialized );
    NN_SDK_ASSERT( memorySize >= sizeof( uint64_t ) * 3 );

    if( ( this->state == State_Begun || this->state == State_Callback ) &&
        this->pControlMemory && this->pCommandList )
    {
        NN_SDK_ASSERT_NOT_NULL( this->pHeadControlMemory.ptr );
        NN_SDK_ASSERT( static_cast< ptrdiff_t >( this->controlMemorySize ) >=
            nn::util::ConstBytePtr( this->pHeadControlMemory.ptr ).Distance(
            nn::util::ConstBytePtr( this->pControlMemory.ptr, sizeof( uint64_t ) * 2 ).Get() ) );
        *static_cast< uint64_t* >( this->pControlMemory ) = 0;
        *nn::util::BytePtr( this->pControlMemory, sizeof( uint64_t ) ).Get< void* >() = pMemory;
    }

    this->controlMemorySize = static_cast< uint32_t >( memorySize );
    this->pHeadControlMemory = pMemory;
    this->pControlMemory = pMemory;
}

void CommandBufferImpl< Target >::SetOutOfCommandMemoryEventCallback(
    OutOfMemoryEventCallback pCallback ) NN_NOEXCEPT
{
    this->pOutOfCommandMemoryCallback.ptr = reinterpret_cast< void (*)() >( pCallback );
}

void CommandBufferImpl< Target >::SetOutOfControlMemoryEventCallback(
    OutOfMemoryEventCallback pCallback ) NN_NOEXCEPT
{
    this->pOutOfControlMemoryCallback.ptr = reinterpret_cast< void (*)() >( pCallback );
}

void CommandBufferImpl< Target >::Reset() NN_NOEXCEPT
{
    this->commandMemorySize = 0;
    this->pHeadCommandMemory = NULL;
    this->pCommandMemory = NULL;
    this->controlMemorySize = sizeof( this->defaultControlMemory );
    this->pHeadControlMemory = this->defaultControlMemory;
    this->pControlMemory = this->defaultControlMemory;
    this->pCommandList = NULL;
}

void CommandBufferImpl< Target >::Begin() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_REQUIRES_NOT_NULL( this->pGfxDevice.ptr );

    this->pCommandList = NULL;

    CheckWorkingMemory( this );

    if( this->pCommandMemory && HasEnoughCommandMemory( this, 0 ) )
    {
        *static_cast< void** >( this->pControlMemory ) = this->pCommandMemory;
        this->pCommandList = this->pControlMemory.ptr;
        this->pControlMemory = nn::util::BytePtr( this->pControlMemory, sizeof( uint64_t ) ).Get();
    }
    else
    {
        this->pCommandMemory = NULL;
        this->pCommandList = this->pControlMemory.ptr;
    }

    this->state = State_Begun;
}

void CommandBufferImpl< Target >::End() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    if( this->pCommandMemory )
    {
        *static_cast< uint64_t* >( this->pCommandMemory ) = 0;
        this->pCommandMemory = nn::util::BytePtr( this->pCommandMemory, sizeof( uint64_t ) ).Get();
        NN_SDK_ASSERT( static_cast< ptrdiff_t >( this->commandMemorySize ) >=
            nn::util::ConstBytePtr( this->pHeadCommandMemory.ptr ).Distance( this->pCommandMemory ) );
    }

    *static_cast< uint64_t* >( this->pControlMemory ) = 0;
    this->pControlMemory = nn::util::BytePtr( this->pControlMemory, sizeof( uint64_t ) ).Get();
    *static_cast< uint64_t* >( this->pControlMemory ) = 0;
    this->pControlMemory = nn::util::BytePtr( this->pControlMemory, sizeof( uint64_t ) ).Get();
    NN_SDK_ASSERT( static_cast< ptrdiff_t >( this->controlMemorySize ) >=
        nn::util::ConstBytePtr( this->pHeadControlMemory.ptr ).Distance( this->pControlMemory ) );

    this->state = State_Initialized;
}

void CommandBufferImpl< Target >::Dispatch(
    int groupCountX, int groupCountY, int groupCountZ ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    GlDispatchParam& param = *WriteGlCommand< GlDispatchParam >( this );
    {
        param.groupCountX = groupCountX;
        param.groupCountY = groupCountY;
        param.groupCountZ = groupCountZ;
    };
}

void CommandBufferImpl< Target >::Draw(
    PrimitiveTopology primitiveTopology, int vertexCount, int vertexOffset ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    GlDrawParam& param = *WriteGlCommand< GlDrawParam >( this );
    {
        param.mode = PrimitiveModeTable[ primitiveTopology ];
        param.offset = vertexOffset;
        param.count = vertexCount;
    };
}

void CommandBufferImpl< Target >::Draw( PrimitiveTopology primitiveTopology,
    int vertexCountPerInstance, int vertexOffset, int instanceCount, int baseInstance ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    GlDrawInstancedParam& param = *WriteGlCommand< GlDrawInstancedParam >( this );
    {
        param.drawParam.mode = PrimitiveModeTable[ primitiveTopology ];
        param.drawParam.offset = vertexOffset;
        param.drawParam.count = vertexCountPerInstance;
        param.primCount = instanceCount;
        param.baseInstance = baseInstance;
    };
}

void CommandBufferImpl< Target >::DrawIndexed( PrimitiveTopology primitiveTopology, IndexFormat indexFormat,
    const GpuAddress& indexBufferAddress, int indexCount, int baseVertex ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    NN_SDK_ASSERT( IsValid( static_cast< GlHandle >( indexBufferAddress.ToData()->impl ) ) );

    GlDrawIndexedParam& param = *WriteGlCommand< GlDrawIndexedParam >( this );
    {
        param.hIndexBuffer = static_cast< GlHandle >( indexBufferAddress.ToData()->impl );
        param.mode = PrimitiveModeTable[ primitiveTopology ];
        param.count = indexCount;
        param.type = IndexFormatTable[ indexFormat ];
        param.offset = static_cast< int >( indexBufferAddress.ToData()->value );
        param.baseVertex = baseVertex;
    }
}

void CommandBufferImpl< Target >::DrawIndexed( PrimitiveTopology primitiveTopology, IndexFormat indexFormat,
    const GpuAddress& indexBufferAddress, int indexCountPerInstance, int baseVertex, int instanceCount, int baseInstance ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    NN_SDK_ASSERT( IsValid( static_cast< GlHandle >( indexBufferAddress.ToData()->impl ) ) );

    GlDrawIndexedInstancedParam& param = *WriteGlCommand< GlDrawIndexedInstancedParam >( this );
    {
        param.drawIndexedParam.hIndexBuffer = static_cast< GlHandle >( indexBufferAddress.ToData()->impl );
        param.drawIndexedParam.mode = PrimitiveModeTable[ primitiveTopology ];
        param.drawIndexedParam.count = indexCountPerInstance;
        param.drawIndexedParam.type = IndexFormatTable[ indexFormat ];
        param.drawIndexedParam.offset = static_cast< int >( indexBufferAddress.ToData()->value );
        param.drawIndexedParam.baseVertex = baseVertex;
        param.primCount = instanceCount;
        param.baseInstance = baseInstance;
    };
}

void CommandBufferImpl< Target >::DispatchIndirect( const GpuAddress& indirectBufferAddress ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    NN_SDK_ASSERT( IsValid( static_cast< GlHandle >( indirectBufferAddress.ToData()->impl ) ) );

    GlDispatchIndirectParam& param = *WriteGlCommand< GlDispatchIndirectParam >( this );
    {
        param.hIndirectBuffer = static_cast< GlHandle >( indirectBufferAddress.ToData()->impl );
        param.offset = static_cast< uint32_t >( indirectBufferAddress.ToData()->value );
    }
}

void CommandBufferImpl< Target >::DrawIndirect(
    PrimitiveTopology primitiveTopology, const GpuAddress& indirectBufferAddress ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    NN_SDK_ASSERT( IsValid( static_cast< GlHandle >( indirectBufferAddress.ToData()->impl ) ) );

    GlDrawIndirectParam& param = *WriteGlCommand< GlDrawIndirectParam >( this );
    {
        param.hIndirectBuffer = static_cast< GlHandle >( indirectBufferAddress.ToData()->impl );
        param.offset = static_cast< uint32_t >( indirectBufferAddress.ToData()->value );
        param.mode = PrimitiveModeTable[ primitiveTopology ];
    }
}

void CommandBufferImpl< Target >::DrawIndexedIndirect( PrimitiveTopology primitiveTopology,
    IndexFormat indexFormat, const GpuAddress& indexBufferAddress, const GpuAddress& indirectBufferAddress ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    NN_SDK_ASSERT( IsValid( static_cast< GlHandle >( indexBufferAddress.ToData()->impl ) ) );
    NN_SDK_ASSERT( indexBufferAddress.ToData()->value == 0 );
    NN_SDK_ASSERT( IsValid( static_cast< GlHandle >( indirectBufferAddress.ToData()->impl ) ) );

    GlDrawIndexedIndirectParam& param = *WriteGlCommand< GlDrawIndexedIndirectParam >( this );
    {
        param.hIndirectBuffer = static_cast< GlHandle >( indirectBufferAddress.ToData()->impl );
        param.offset = static_cast< uint32_t >( indirectBufferAddress.ToData()->value );
        param.mode = PrimitiveModeTable[ primitiveTopology ];
        param.type = IndexFormatTable[ indexFormat ];
        param.hIndexBuffer = static_cast< GlHandle >( indexBufferAddress.ToData()->impl );
    }
}

void CommandBufferImpl< Target >::SetPipeline( const PipelineImpl< Target >* pPipeline ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pPipeline );
    NN_SDK_REQUIRES( IsInitialized( *pPipeline ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    const PipelineImpl< Target >::DataType& pipeline = pPipeline->ToData();

    if( pipeline.pipelineType == PipelineType_Graphics )
    {
        SetRasterizerState( nn::gfx::DataToAccessor( pipeline.rasterizerState ) );
        SetBlendState( nn::gfx::DataToAccessor( pipeline.blendState ) );
        SetDepthStencilState( nn::gfx::DataToAccessor( pipeline.depthStencilState ) );
        SetVertexState( nn::gfx::DataToAccessor( pipeline.vertexState ) );
        if( pipeline.flags.GetBit( PipelineImpl< Target >::DataType::Flag_HasTessellationState ) )
        {
            SetTessellationState( nn::gfx::DataToAccessor( pipeline.tessellationState ) );
        }
    }
    SetShader( pipeline.pShader, ShaderStageBit_All );
}

void CommandBufferImpl< Target >::SetRenderTargets( int colorTargetCount,
    const ColorTargetViewImpl< Target >* const * ppColorTargets,
    const DepthStencilViewImpl< Target >* pDepthStencil ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( colorTargetCount < 1 || ppColorTargets );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( colorTargetCount <=
        static_cast< int >( this->pGfxDevice->ToData()->maxColorAttachments ) );

    GlSetRenderTargetsParam& param = *WriteGlCommand< GlSetRenderTargetsParam >( this,
        sizeof( GlSetRenderTargetsParam ) + ( colorTargetCount - 1 ) * sizeof( GlSetRenderTargetsParam::RenderTarget ) );
    {
        param.hDepthStencil = GlInvalidHandle;
        if( pDepthStencil )
        {
            NN_SDK_REQUIRES( IsInitialized( *pDepthStencil ) );
            param.hDepthStencil = pDepthStencil->ToData()->hTexture;
            param.depthStencilAttachment = pDepthStencil->ToData()->internalFormat == GL_DEPTH24_STENCIL8
                || pDepthStencil->ToData()->internalFormat == GL_DEPTH32F_STENCIL8 ?
                GL_DEPTH_STENCIL_ATTACHMENT : GL_DEPTH_ATTACHMENT;
        }
        param.colorTargetCount = colorTargetCount;
        for( int idxTarget = 0; idxTarget < colorTargetCount; ++idxTarget )
        {
            GlSetRenderTargetsParam::RenderTarget& target = param.colorTargets[ idxTarget ];
            target.hColorTarget = GlInvalidHandle;
            if( ppColorTargets[ idxTarget ] )
            {
                NN_SDK_REQUIRES( IsInitialized( *( ppColorTargets[ idxTarget ] ) ) );
                target.hColorTarget = ppColorTargets[ idxTarget ]->ToData()->hTexture;
                target.layer = ppColorTargets[ idxTarget ]->ToData()->target == GL_TEXTURE_3D ?
                    ppColorTargets[ idxTarget ]->ToData()->layer : -1;
            }
        }

        // Keep track of the maximum color attachments so we can detach them later.
        param.maxColorAttachments = this->pGfxDevice->ToData()->maxColorAttachments;
    }
}

void CommandBufferImpl< Target >::SetVertexBuffer( int bufferIndex,
    const GpuAddress& vertexBufferAddress, ptrdiff_t stride, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_UNUSED( size );

    NN_SDK_ASSERT( IsValid( static_cast< GlHandle >( vertexBufferAddress.ToData()->impl ) ) );

    GlSetVertexBufferParam& param = *WriteGlCommand< GlSetVertexBufferParam >( this );
    {
        param.bufferIndex = bufferIndex;
        param.hBuffer = static_cast< GlHandle >( vertexBufferAddress.ToData()->impl );
        param.offset = static_cast< uint32_t >( vertexBufferAddress.ToData()->value );
        param.stride = static_cast< int32_t >( stride );
    };
}

void CommandBufferImpl< Target >::SetViewportScissorState(
    const ViewportScissorStateImpl< Target >* pViewportScissor ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pViewportScissor );
    NN_SDK_REQUIRES( IsInitialized( *pViewportScissor ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    const ViewportScissorStateImpl< Target >::DataType& data = pViewportScissor->ToData();
    NN_SDK_ASSERT( data.viewportCount > 0 );
    int staticSize = offsetof( ViewportScissorStateImplData< Target >, pWorkMemory );
    int dynamicSize = ( sizeof( float ) * 4 + sizeof( double ) * 2 + sizeof( int32_t ) * 4 ) * ( data.viewportCount - 1 );

    ViewportScissorStateImplData< Target >& param = *WriteGlCommand<
        ViewportScissorStateImplData< Target > >( this, staticSize + dynamicSize );
    {
        memcpy( &param, &data, staticSize );
        memcpy( &param.pWorkMemory, data.pWorkMemory, dynamicSize );
    }
}

void CommandBufferImpl< Target >::CopyBuffer( BufferImpl< Target >* pDstBuffer, ptrdiff_t dstOffset,
    const BufferImpl< Target >* pSrcBuffer, ptrdiff_t srcOffset, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstBuffer );
    NN_SDK_REQUIRES_NOT_NULL( pSrcBuffer );
    NN_SDK_REQUIRES( IsInitialized( *pDstBuffer ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcBuffer ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( IsValid( pDstBuffer->ToData()->hBuffer ) );
    NN_SDK_ASSERT( IsValid( pSrcBuffer->ToData()->hBuffer ) );

    GlCopyBufferParam& param = *WriteGlCommand< GlCopyBufferParam >( this );
    {
        param.hSrcBuffer = pSrcBuffer->ToData()->hBuffer;
        param.hDstBuffer = pDstBuffer->ToData()->hBuffer;
        param.srcOffset = static_cast< uint32_t >( srcOffset );
        param.dstOffset = static_cast< uint32_t >( dstOffset );
        param.size = static_cast< uint32_t >( size );
    }
}

void CommandBufferImpl< Target >::CopyImage( TextureImpl< Target >* pDstTexture,
    const TextureSubresource& dstSubresource, int dstOffsetU, int dstOffsetV, int dstOffsetW,
    const TextureImpl< Target >* pSrcTexture, const TextureCopyRegion& srcCopyRegion ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstTexture );
    NN_SDK_REQUIRES_NOT_NULL( pSrcTexture );
    NN_SDK_REQUIRES( IsInitialized( *pDstTexture ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcTexture ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( IsValid( pDstTexture->ToData()->hTexture ) );
    NN_SDK_ASSERT( IsValid( pSrcTexture->ToData()->hTexture ) );

    GLint dstV = pDstTexture->ToData()->target == GL_TEXTURE_1D_ARRAY
        ? dstSubresource.GetArrayIndex() : dstOffsetV;
    GLint dstW = ( pDstTexture->ToData()->target == GL_TEXTURE_2D_ARRAY ||
        pDstTexture->ToData()->target == GL_TEXTURE_2D_MULTISAMPLE_ARRAY )
        ? dstSubresource.GetArrayIndex() : dstOffsetW;

    int srcV;
    int srcHeight;
    int srcW;
    int srcDepth;
    GetGlCopyRegion( &srcV, &srcHeight, &srcW, &srcDepth, srcCopyRegion, pSrcTexture->ToData()->target );

    GlCopyImageParam& param = *WriteGlCommand< GlCopyImageParam >( this );
    {
        param.hSrcTexture = pSrcTexture->ToData()->hTexture;
        param.srcTarget = pSrcTexture->ToData()->target;
        param.srcLevel = srcCopyRegion.GetSubresource().GetMipLevel();
        param.srcX = srcCopyRegion.GetOffsetU();
        param.srcY = srcV;
        param.srcZ = srcW;
        param.hDstTexture = pDstTexture->ToData()->hTexture;
        param.dstTarget = pDstTexture->ToData()->target;
        param.dstLevel = dstSubresource.GetMipLevel();
        param.dstX = dstOffsetU;
        param.dstY = dstV;
        param.dstZ = dstW;
        param.width = srcCopyRegion.GetWidth();
        param.height = srcHeight;
        param.depth = srcDepth;
    }
}

void CommandBufferImpl< Target >::CopyBufferToImage( TextureImpl< Target >* pDstTexture,
     const BufferImpl< Target >* pSrcBuffer, const BufferTextureCopyRegion& region ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstTexture );
    NN_SDK_REQUIRES_NOT_NULL( pSrcBuffer );
    NN_SDK_REQUIRES( IsInitialized( *pDstTexture ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcBuffer ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( IsValid( pDstTexture->ToData()->hTexture ) );
    NN_SDK_ASSERT( IsValid( pSrcBuffer->ToData()->hBuffer ) );

    GlCopyBufferToImageParam& param = *WriteGlCommand< GlCopyBufferToImageParam >( this );
    {
        SetBufferImageCopyParam( &param.param, pSrcBuffer, region.GetBufferOffset(), pDstTexture,
            region.GetTextureCopyRegion(), region.GetBufferImageWidth(), region.GetBufferImageHeight() );
    }
}

void CommandBufferImpl< Target >::CopyImageToBuffer( BufferImpl< Target >* pDstBuffer,
    const TextureImpl< Target >* pSrcTexture, const BufferTextureCopyRegion& region ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstBuffer );
    NN_SDK_REQUIRES_NOT_NULL( pSrcTexture );
    NN_SDK_REQUIRES( IsInitialized( *pDstBuffer ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcTexture ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( IsValid( pDstBuffer->ToData()->hBuffer ) );
    NN_SDK_ASSERT( IsValid( pSrcTexture->ToData()->hTexture ) );

    GlCopyImageToBufferParam& param = *WriteGlCommand< GlCopyImageToBufferParam >( this );
    {
        SetBufferImageCopyParam( &param.param, pDstBuffer, region.GetBufferOffset(), pSrcTexture,
            region.GetTextureCopyRegion(), region.GetBufferImageWidth(), region.GetBufferImageHeight() );
    }
}

void CommandBufferImpl< Target >::CopyBufferToImage( TextureImpl< Target >* pDstTexture,
    const TextureCopyRegion& dstRegion, const BufferImpl< Target >* pSrcBuffer, ptrdiff_t srcOffset ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstTexture );
    NN_SDK_REQUIRES_NOT_NULL( pSrcBuffer );
    NN_SDK_REQUIRES( IsInitialized( *pDstTexture ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcBuffer ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( IsValid( pDstTexture->ToData()->hTexture ) );
    NN_SDK_ASSERT( IsValid( pSrcBuffer->ToData()->hBuffer ) );

    GlCopyBufferToImageParam& param = *WriteGlCommand< GlCopyBufferToImageParam >( this );
    {
        SetBufferImageCopyParam( &param.param, pSrcBuffer, srcOffset, pDstTexture, dstRegion, 0, 0 );
    }
}

void CommandBufferImpl< Target >::CopyImageToBuffer( BufferImpl< Target >* pDstBuffer,
    ptrdiff_t dstOffset, const TextureImpl< Target >* pSrcTexture, const TextureCopyRegion& srcRegion ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstBuffer );
    NN_SDK_REQUIRES_NOT_NULL( pSrcTexture );
    NN_SDK_REQUIRES( IsInitialized( *pDstBuffer ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcTexture ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( IsValid( pDstBuffer->ToData()->hBuffer ) );
    NN_SDK_ASSERT( IsValid( pSrcTexture->ToData()->hTexture ) );

    GlCopyImageToBufferParam& param = *WriteGlCommand< GlCopyImageToBufferParam >( this );
    {
        SetBufferImageCopyParam( &param.param, pDstBuffer, dstOffset, pSrcTexture, srcRegion, 0, 0 );
    }
}

void CommandBufferImpl< Target >::BlitImage( TextureImpl< Target >* pDstTexture,
    const TextureCopyRegion& dstRegion, const TextureImpl< Target >* pSrcTexture,
    const TextureCopyRegion& srcRegion, int copyFlags ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstTexture );
    NN_SDK_REQUIRES_NOT_NULL( pSrcTexture );
    NN_SDK_REQUIRES( IsInitialized( *pDstTexture ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcTexture ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( IsValid( pDstTexture->ToData()->hTexture ) );
    NN_SDK_ASSERT( IsValid( pSrcTexture->ToData()->hTexture ) );

    int dstV;
    int dstHeight;
    int dstW;
    int dstDepth;
    GetGlCopyRegion( &dstV, &dstHeight, &dstW, &dstDepth, dstRegion, pDstTexture->ToData()->target );

    int srcV;
    int srcHeight;
    int srcW;
    int srcDepth;
    GetGlCopyRegion( &srcV, &srcHeight, &srcW, &srcDepth, srcRegion, pSrcTexture->ToData()->target );

    GlBlitImageParam& param = *WriteGlCommand< GlBlitImageParam >( this );
    {
        param.hDstTexture = pDstTexture->ToData()->hTexture;
        param.dstMipLevel = dstRegion.GetSubresource().GetMipLevel();
        param.dstStartArrayIndex = Gl::IsLayeredTarget( pDstTexture->ToData()->target ) ? dstW : -1;
        param.dstX0 = dstRegion.GetOffsetU();
        param.dstY0 = dstV;
        param.dstX1 = dstRegion.GetOffsetU() + dstRegion.GetWidth();
        param.dstY1 = dstV + dstHeight;
        param.hSrcTexture = pSrcTexture->ToData()->hTexture;
        param.srcMipLevel = srcRegion.GetSubresource().GetMipLevel();
        param.srcStartArrayIndex = Gl::IsLayeredTarget( pSrcTexture->ToData()->target ) ? srcW : -1;
        param.srcX0 = srcRegion.GetOffsetU();
        param.srcY0 = srcV;
        param.srcX1 = srcRegion.GetOffsetU() + srcRegion.GetWidth();
        param.srcY1 = srcV + srcHeight;
        param.arrayCount = std::min NN_PREVENT_MACRO_FUNC ( dstDepth, srcDepth );
        param.mask = GL_COLOR_BUFFER_BIT;
        if( GetTypeFormat( static_cast< ImageFormat >(
            pDstTexture->ToData()->imageFormat ) ) == TypeFormat_DepthStencil )
        {
            NN_SDK_ASSERT_EQUAL( GetTypeFormat( static_cast< ImageFormat >(
                pSrcTexture->ToData()->imageFormat ) ), TypeFormat_DepthStencil );
            param.mask = GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT;
        }
        param.filter = ( copyFlags & ImageCopyFlag_LinearFilter ) ? GL_LINEAR : GL_NEAREST;
    }
}

void CommandBufferImpl< Target >::ClearBuffer( BufferImpl< Target >* pBuffer,
    ptrdiff_t offset, size_t size, uint32_t value ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES_NOT_NULL( pBuffer );
    NN_SDK_REQUIRES( offset >= 0 );
    NN_SDK_REQUIRES( nn::util::is_aligned( offset, 4 ) );
    NN_SDK_REQUIRES( nn::util::is_aligned( size, 4 ) );

    GlClearBufferParam& param = *WriteGlCommand< GlClearBufferParam >( this );
    {
        param.hBuffer = pBuffer->ToData()->hBuffer;
        param.offset = static_cast< int32_t >( offset );
        param.size = static_cast<uint32_t>( size );
        param.value = value;
    }
}

void CommandBufferImpl< Target >::ClearColor( ColorTargetViewImpl< Target >* pColorTarget,
    float red, float green, float blue, float alpha, const TextureArrayRange* pArrayRange ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pColorTarget );
    NN_SDK_REQUIRES( IsInitialized( *pColorTarget ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( IsValid( pColorTarget->ToData()->hTexture ) );

    ColorTargetViewImpl< Target >::DataType& colorTargetData = pColorTarget->ToData();

    int startLayer = -1;
    int endLayer = 0;
    if( pArrayRange )
    {
        startLayer = static_cast< int >( pArrayRange->GetBaseArrayIndex() );
        endLayer = startLayer + static_cast< int >( pArrayRange->GetArrayLength() );
    }
    else if( Gl::IsLayeredTarget( colorTargetData.target ) )
    {
        GlDeviceActivator activator( this->pGfxDevice.ptr );
        startLayer = 0;
        NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glGetTextureParameteriv )(
            colorTargetData.hTexture, colorTargetData.target,
            GL_TEXTURE_VIEW_NUM_LAYERS, &endLayer ) );
        if( colorTargetData.target == GL_TEXTURE_3D )
        {
            startLayer = pColorTarget->ToData()->layer;
            endLayer += startLayer;
        }
        NN_GFX_GL_ASSERT();
    }

    GlClearColorParam& param = *WriteGlCommand< GlClearColorParam >( this );
    {
        param.hTexture = pColorTarget->ToData()->hTexture;
        param.startLayer = startLayer;
        param.endLayer = endLayer;
        param.target = pColorTarget->ToData()->target;
        param.color[ 0 ] = red;
        param.color[ 1 ] = green;
        param.color[ 2 ] = blue;
        param.color[ 3 ] = alpha;
    }
}

void CommandBufferImpl< Target >::ClearColorTarget( ColorTargetViewImpl< Target >* pColorTarget,
    const ClearColorValue& clearColor, const TextureArrayRange* pArrayRange ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pColorTarget );
    NN_SDK_REQUIRES( IsInitialized( *pColorTarget ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( IsValid( pColorTarget->ToData()->hTexture ) );

    ColorTargetViewImpl< Target >::DataType& colorTargetData = pColorTarget->ToData();

    int startLayer = -1;
    int endLayer = 0;
    if( pArrayRange )
    {
        startLayer = static_cast< int >( pArrayRange->GetBaseArrayIndex() );
        endLayer = startLayer + static_cast< int >( pArrayRange->GetArrayLength() );
    }
    else if( Gl::IsLayeredTarget( colorTargetData.target ) )
    {
        GlDeviceActivator activator( this->pGfxDevice.ptr );
        startLayer = 0;
        NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glGetTextureParameteriv )(
            colorTargetData.hTexture, colorTargetData.target,
            GL_TEXTURE_VIEW_NUM_LAYERS, &endLayer ) );
        if( colorTargetData.target == GL_TEXTURE_3D )
        {
            startLayer = pColorTarget->ToData()->layer;
            endLayer += startLayer;
        }
        NN_GFX_GL_ASSERT();
    }

    GlClearColorParam& param = *WriteGlCommand< GlClearColorParam >( this );
    {
        param.hTexture = pColorTarget->ToData()->hTexture;
        param.startLayer = startLayer;
        param.endLayer = endLayer;
        param.target = pColorTarget->ToData()->target;
        param.color[ 0 ] = clearColor.valueFloat[ 0 ];
        param.color[ 1 ] = clearColor.valueFloat[ 1 ];
        param.color[ 2 ] = clearColor.valueFloat[ 2 ];
        param.color[ 3 ] = clearColor.valueFloat[ 3 ];
    }
}

void CommandBufferImpl< Target >::ClearDepthStencil( DepthStencilViewImpl< Target >* pDepthStencil,
    float depth, int stencil, DepthStencilClearMode clearMode, const TextureArrayRange* pArrayRange ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDepthStencil );
    NN_SDK_REQUIRES( IsInitialized( *pDepthStencil ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( IsValid( pDepthStencil->ToData()->hTexture ) );

    DepthStencilViewImpl< Target >::DataType& depthStencilData = pDepthStencil->ToData();

    GLenum attachment = depthStencilData.internalFormat == GL_DEPTH24_STENCIL8
        || depthStencilData.internalFormat == GL_DEPTH32F_STENCIL8 ?
        GL_DEPTH_STENCIL_ATTACHMENT : GL_DEPTH_ATTACHMENT;

    int startLayer = -1;
    int endLayer = 0;
    if( pArrayRange )
    {
        startLayer = static_cast< int >( pArrayRange->GetBaseArrayIndex() );
        endLayer = startLayer + static_cast< int >( pArrayRange->GetArrayLength() );
    }
    else if( Gl::IsLayeredTarget( depthStencilData.target ) )
    {
        GlDeviceActivator activator( this->pGfxDevice.ptr );
        startLayer = 0;
        NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glGetTextureParameteriv )(
            depthStencilData.hTexture, depthStencilData.target,
            GL_TEXTURE_VIEW_NUM_LAYERS, &endLayer ) );
        NN_GFX_GL_ASSERT();
    }

    GlClearDepthStencilParam& param = *WriteGlCommand< GlClearDepthStencilParam >( this );
    {
        param.hTexture = depthStencilData.hTexture;
        param.startLayer = startLayer;
        param.endLayer = endLayer;
        param.attachment = attachment;
        param.clearMode = clearMode;
        param.depth = depth;
        param.stencil = stencil;
    }
}

void CommandBufferImpl< Target >::Resolve( TextureImpl< Target >* pDstTexture, int dstMipLevel, int dstStartArrayIndex,
    const ColorTargetViewImpl< Target >* pSrcColorTarget, const TextureArrayRange* pSrcArrayRange ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstTexture );
    NN_SDK_REQUIRES_NOT_NULL( pSrcColorTarget );
    NN_SDK_REQUIRES( IsInitialized( *pDstTexture ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcColorTarget ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    GlDeviceActivator activator( this->pGfxDevice.ptr );

    const ColorTargetViewImpl< Target >::DataType& srcTextureData = pSrcColorTarget->ToData();
    TextureImpl< Target >::DataType& dstTextureData = pDstTexture->ToData();

    int startLayer = -1;
    int endLayer = 0;
    if( pSrcArrayRange )
    {
        startLayer = static_cast< int >( pSrcArrayRange->GetBaseArrayIndex() );
        endLayer = static_cast< int >( pSrcArrayRange->GetArrayLength() );
    }
    else if( Gl::IsLayeredTarget( srcTextureData.target ) )
    {
        startLayer = 0;
        NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glGetTextureParameteriv )(
            srcTextureData.hTexture, srcTextureData.target,
            GL_TEXTURE_MAX_LEVEL, &endLayer ) );
    }

    GLint dstWidth;
    GLint dstHeight;
    GLint srcWidth;
    GLint srcHeight;
    NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glGetTextureLevelParameteriv )(
        dstTextureData.hTexture, dstTextureData.target, dstMipLevel, GL_TEXTURE_WIDTH, &dstWidth ) );
    NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glGetTextureLevelParameteriv )(
        dstTextureData.hTexture, dstTextureData.target, dstMipLevel, GL_TEXTURE_HEIGHT, &dstHeight ) );
    NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glGetTextureLevelParameteriv )(
        srcTextureData.hTexture, srcTextureData.target,
        dstMipLevel, GL_TEXTURE_WIDTH, &srcWidth ) );
    NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glGetTextureLevelParameteriv )(
        srcTextureData.hTexture, srcTextureData.target,
        dstMipLevel, GL_TEXTURE_HEIGHT, &srcHeight ) );
    NN_GFX_GL_ASSERT();

    GlResolveParam& param = *WriteGlCommand< GlResolveParam >( this );
    {
        param.hDstTexture = dstTextureData.hTexture;
        param.hSrcTexture = srcTextureData.hTexture;
        param.dstMipLevel = dstMipLevel;
        param.dstStartArrayIndex = Gl::IsLayeredTarget( dstTextureData.target ) ?
            dstStartArrayIndex : startLayer - endLayer;
        param.startLayer = startLayer;
        param.endLayer = endLayer;
        param.dstWidth = dstWidth;
        param.dstHeight = dstHeight;
        param.srcWidth = srcWidth;
        param.srcHeight = srcHeight;
    }
}

void CommandBufferImpl< Target >::FlushMemory( int gpuAccessFlags ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    GLbitfield barriers = 0;
    if( gpuAccessFlags & ( GpuAccess_VertexBuffer | GpuAccess_IndexBuffer
        | GpuAccess_ConstantBuffer | GpuAccess_UnorderedAccessBuffer | GpuAccess_QueryBuffer ) )
    {
        barriers |= GL_BUFFER_UPDATE_BARRIER_BIT | GL_CLIENT_MAPPED_BUFFER_BARRIER_BIT;
    }
    if( gpuAccessFlags & ( GpuAccess_ColorBuffer | GpuAccess_DepthStencil | GpuAccess_Image ) )
    {
        barriers |= GL_TEXTURE_UPDATE_BARRIER_BIT | GL_PIXEL_BUFFER_BARRIER_BIT;
    }

    GlMemoryBarrierParam& param = *WriteGlCommand< GlMemoryBarrierParam >( this );
    {
        param.barriers = barriers;
    }
}

void CommandBufferImpl< Target >::InvalidateMemory( int gpuAccessFlags ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    GLbitfield barriers = 0;
    if( gpuAccessFlags &  GpuAccess_VertexBuffer )
    {
        barriers |= GL_VERTEX_ATTRIB_ARRAY_BARRIER_BIT;
    }
    if( gpuAccessFlags & GpuAccess_IndexBuffer )
    {
        barriers |= GL_ELEMENT_ARRAY_BARRIER_BIT;
    }
    if( gpuAccessFlags & GpuAccess_ConstantBuffer )
    {
        barriers |= GL_UNIFORM_BARRIER_BIT;
    }
    if( gpuAccessFlags & GpuAccess_Texture )
    {
        barriers |= GL_TEXTURE_FETCH_BARRIER_BIT;
    }
    if( gpuAccessFlags & GpuAccess_UnorderedAccessBuffer )
    {
        barriers |= GL_SHADER_STORAGE_BARRIER_BIT;
    }
    if( gpuAccessFlags & ( GpuAccess_ColorBuffer | GpuAccess_DepthStencil ) )
    {
        barriers |= GL_FRAMEBUFFER_BARRIER_BIT;
    }
    if( gpuAccessFlags & GpuAccess_IndirectBuffer )
    {
        barriers |= GL_COMMAND_BARRIER_BIT;
    }
    if( gpuAccessFlags & ( GpuAccess_Read | GpuAccess_Write ) )
    {
        barriers |= GL_PIXEL_BUFFER_BARRIER_BIT;
    }
    if( gpuAccessFlags & GpuAccess_Image )
    {
        barriers |= GL_SHADER_IMAGE_ACCESS_BARRIER_BIT;
    }

    GlMemoryBarrierParam& param = *WriteGlCommand< GlMemoryBarrierParam >( this );
    {
        param.barriers = barriers;
    }
}

void CommandBufferImpl< Target >::CallCommandBuffer( const CommandBufferImpl< Target >* pNestedCommandBuffer ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( pNestedCommandBuffer );

    const CommandBufferImpl< Target >::DataType& nestedCommandBuffer = pNestedCommandBuffer->ToData();
    NN_SDK_REQUIRES( nestedCommandBuffer.commandBufferType == CommandBufferType_Nested );
    NN_SDK_REQUIRES( nestedCommandBuffer.state == State_Initialized );

    if( nestedCommandBuffer.pCommandList.ptr == NULL )
    {
        return;
    }

    GlCallCommandListParam& param = *WriteGlCommand< GlCallCommandListParam >( this );
    {
        param.pCommandList = nestedCommandBuffer.pCommandList;
    }
}

void CommandBufferImpl< Target >::CopyCommandBuffer( const CommandBufferImpl< Target >* pNestedCommandBuffer ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( pNestedCommandBuffer );

    const CommandBufferImpl< Target >::DataType& nestedCommandBuffer = pNestedCommandBuffer->ToData();
    NN_SDK_REQUIRES( nestedCommandBuffer.commandBufferType == CommandBufferType_Nested );
    NN_SDK_REQUIRES( nestedCommandBuffer.state == State_Initialized );

    if( nestedCommandBuffer.pCommandList.ptr == NULL )
    {
        return;
    }

    const void* pCurrentCommandList = nestedCommandBuffer.pCommandList;
    for( ; ; )
    {
        if( *static_cast< const void* const* >( pCurrentCommandList ) == NULL )
        {
            pCurrentCommandList = nn::util::ConstBytePtr( pCurrentCommandList, sizeof( uint64_t ) ).Get();
            if( *static_cast< const void* const* >( pCurrentCommandList ) == NULL )
            {
                break;
            }
            pCurrentCommandList = *static_cast< const void* const* >( pCurrentCommandList );
        }
        const void* pCommandStart = *static_cast< const void* const* >( pCurrentCommandList );
        const void* pCurrentCommand = pCommandStart;
        while( *static_cast< const uintptr_t* >( pCurrentCommand ) )
        {
            const GlCommand* pCommand = static_cast< const GlCommand* >( pCurrentCommand );
            pCurrentCommand = pCommand->param + pCommand->paramSize;
        }
        size_t commandSize = nn::util::ConstBytePtr( pCommandStart ).Distance( pCurrentCommand );
        CheckCommandMemory( this, static_cast< uint32_t >( commandSize ) );
        memcpy( this->pCommandMemory, pCommandStart, commandSize );
        this->pCommandMemory = nn::util::BytePtr( this->pCommandMemory,
            static_cast< ptrdiff_t >( commandSize ) ).Get();

        pCurrentCommandList = nn::util::ConstBytePtr( pCurrentCommandList, sizeof( uint64_t ) ).Get();
    }
}

void CommandBufferImpl< Target >::SetBufferStateTransition(
    BufferImpl< Target >* pBuffer, int oldState, int, int newState, int ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES_NOT_NULL( pBuffer );
    NN_UNUSED( pBuffer );

    GLbitfield barriers = 0;
    if( oldState & ( BufferState_DataTransfer |
        BufferState_UnorderedAccessBuffer | BufferState_QueryBuffer ) )
    {
        if( newState & BufferState_VertexBuffer )
        {
            barriers |= GL_VERTEX_ATTRIB_ARRAY_BARRIER_BIT;
        }
        if( newState & BufferState_IndexBuffer )
        {
            barriers |= GL_ELEMENT_ARRAY_BARRIER_BIT;
        }
        if( newState & BufferState_ConstantBuffer )
        {
            barriers |= GL_UNIFORM_BARRIER_BIT;
        }
        if( newState & BufferState_UnorderedAccessBuffer )
        {
            barriers |= GL_SHADER_STORAGE_BARRIER_BIT;
        }
        if( newState & BufferState_IndirectArgument )
        {
            barriers |= GL_COMMAND_BARRIER_BIT;
        }
    }

    if( newState & ( BufferState_CopySource | BufferState_CopyDestination ) )
    {
        barriers |= GL_PIXEL_BUFFER_BARRIER_BIT;
    }
    if( newState & BufferState_DataTransfer )
    {
        barriers |= GL_BUFFER_UPDATE_BARRIER_BIT | GL_CLIENT_MAPPED_BUFFER_BARRIER_BIT;
    }

    if( barriers )
    {
        GlMemoryBarrierParam& param = *WriteGlCommand< GlMemoryBarrierParam >( this );
        {
            param.barriers = barriers;
        }
    }
}

void CommandBufferImpl< Target >::SetTextureStateTransition( TextureImpl< Target >* pTexture,
    const TextureSubresourceRange*, int oldState, int, int newState, int ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES_NOT_NULL( pTexture );
    NN_UNUSED( pTexture );

    GLbitfield barriers = 0;
    if( oldState & ( TextureState_DataTransfer | TextureState_ShaderWrite | TextureState_ColorTarget |
        TextureState_DepthWrite | TextureState_Clear | TextureState_ResolveDestination ) )
    {
        if( newState & TextureState_ShaderRead )
        {
            barriers |= GL_TEXTURE_FETCH_BARRIER_BIT | GL_SHADER_IMAGE_ACCESS_BARRIER_BIT;
        }
        if( newState & ( TextureState_ColorTarget | TextureState_DepthRead | TextureState_DepthWrite ) )
        {
            barriers |= GL_FRAMEBUFFER_BARRIER_BIT;
        }
        if( newState & TextureState_ShaderWrite )
        {
            barriers |= GL_SHADER_IMAGE_ACCESS_BARRIER_BIT;
        }
    }

    if( newState & TextureState_DataTransfer )
    {
        barriers |= GL_TEXTURE_UPDATE_BARRIER_BIT | GL_PIXEL_BUFFER_BARRIER_BIT;
    }

    if( barriers )
    {
        GlMemoryBarrierParam& param = *WriteGlCommand< GlMemoryBarrierParam >( this );
        {
            param.barriers = barriers;
        }
    }
}

void CommandBufferImpl< Target >::SetDescriptorPool( const DescriptorPoolImpl< Target >* ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
}

void CommandBufferImpl< Target >::SetRootSignature( PipelineType,
    RootSignatureImpl< Target >* pRootSignatureValue ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( !pRootSignatureValue || pRootSignatureValue->ToData()->pWorkMemory );
    this->pGfxRootSignature = pRootSignatureValue;
}

void CommandBufferImpl< Target >::SetRootBufferDescriptorTable( PipelineType pipelineType,
    int indexDescriptorTable, const DescriptorSlot& startBufferDescriptorSlot ) NN_NOEXCEPT
{
    return nn::gfx::detail::SetRootBufferDescriptorTable( this, pipelineType, indexDescriptorTable,
        startBufferDescriptorSlot, DescriptorPoolImpl< Target >::GetDescriptorSlotIncrementSize(
        this->pGfxDevice, DescriptorPoolType_BufferView ) );
}

void CommandBufferImpl< Target >::SetRootTextureAndSamplerDescriptorTable(
    PipelineType pipelineType, int indexDescriptorTable, const DescriptorSlot& startTextureDescriptorSlot,
    const DescriptorSlot& startSamplerDescriptorSlot ) NN_NOEXCEPT
{
    return nn::gfx::detail::SetRootTextureAndSamplerDescriptorTable( this, pipelineType,
        indexDescriptorTable, startTextureDescriptorSlot, startSamplerDescriptorSlot,
        DescriptorPoolImpl< Target >::GetDescriptorSlotIncrementSize( this->pGfxDevice, DescriptorPoolType_TextureView ),
        DescriptorPoolImpl< Target >::GetDescriptorSlotIncrementSize( this->pGfxDevice, DescriptorPoolType_Sampler ) );
}

void CommandBufferImpl< Target >::SetRootConstantBuffer( PipelineType pipelineType,
    int indexDynamicDescriptor, const GpuAddress& constantBufferAddress, size_t size ) NN_NOEXCEPT
{
    return nn::gfx::detail::SetRootConstantBuffer( this,
        pipelineType, indexDynamicDescriptor, constantBufferAddress, size );
}

void CommandBufferImpl< Target >::SetRootUnorderedAccessBuffer( PipelineType pipelineType,
    int indexDynamicDescriptor, const GpuAddress& unorderedAccessBufferAddress, size_t size ) NN_NOEXCEPT
{
    return nn::gfx::detail::SetRootUnorderedAccessBuffer( this,
        pipelineType, indexDynamicDescriptor, unorderedAccessBufferAddress, size );
}

void CommandBufferImpl< Target >::SetRootTextureAndSampler( PipelineType pipelineType, int indexDynamicDescriptor,
    const TextureViewImpl< Target >* pTextureView, const SamplerImpl< Target >* pSampler ) NN_NOEXCEPT
{
    return nn::gfx::detail::SetRootTextureAndSampler( this,
        pipelineType, indexDynamicDescriptor, pTextureView, pSampler );
}

void CommandBufferImpl< Target >::BeginQuery( QueryTarget target ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES_NOT_EQUAL( target, QueryTarget_Timestamp );

    GLenum glTarget = Gl::GetQueryTarget( target );
    if( glTarget != GL_INVALID_ENUM )
    {
        GlBeginQueryParam& param = *WriteGlCommand< GlBeginQueryParam >( this );
        {
            param.target = glTarget;
            param.idxQueryObject = target;
        }
    }
}

void CommandBufferImpl< Target >::EndQuery( const GpuAddress& dstBufferAddress,
    QueryTarget target ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES_NOT_EQUAL( target, QueryTarget_Timestamp );
    NN_SDK_ASSERT( IsValid( static_cast< GlHandle >( dstBufferAddress.ToData()->impl ) ) );

    GLenum glTarget = Gl::GetQueryTarget( target );
    if( glTarget != GL_INVALID_ENUM )
    {
        GlEndQueryParam& param = *WriteGlCommand< GlEndQueryParam >( this );
        {
            param.hBuffer = static_cast< GlHandle >( dstBufferAddress.ToData()->impl );
            param.bufferOffset = static_cast< uint32_t >( dstBufferAddress.ToData()->value );
            param.target = glTarget;
            param.idxQueryObject = target;
        }
    }
}

void CommandBufferImpl< Target >::WriteTimestamp( const GpuAddress& dstBufferAddress ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    GlWriteTimestampParam& param = *WriteGlCommand< GlWriteTimestampParam >( this );
    {
        param.hBuffer = static_cast< GlHandle >( dstBufferAddress.ToData()->impl );
        param.bufferOffset = static_cast< uint32_t >( dstBufferAddress.ToData()->value );
    }
}

void CommandBufferImpl< Target >::SetDepthBounds( float minDepthBounds, float maxDepthBounds ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    GlSetDepthBoundsParam& param = *WriteGlCommand< GlSetDepthBoundsParam >( this );
    {
        param.minDepthBounds = minDepthBounds;
        param.maxDepthBounds = maxDepthBounds;
    }
}

void CommandBufferImpl< Target >::SetLineWidth( float lineWidth ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    GlSetLineWidthParam& param = *WriteGlCommand< GlSetLineWidthParam >( this );
    {
        param.lineWidth = lineWidth;
    }
}

void CommandBufferImpl< Target >::SetViewports( int firstViewport,
    int viewportCount, const ViewportStateInfo* pViewports ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    GlSetViewportsParam& param = *WriteGlCommand< GlSetViewportsParam >( this,
        sizeof( GlSetViewportsParam ) + sizeof( float ) * 4 * viewportCount
        + sizeof( double ) * 2 * viewportCount );
    {
        param.first = firstViewport;
        param.count = viewportCount;
        float* pDstViewports = nn::util::BytePtr( &param, sizeof( GlSetViewportsParam ) ).Get< float >();
        double* pDstDepthRanges = nn::util::BytePtr( &param, sizeof( GlSetViewportsParam )
            + sizeof( float ) * 4 * viewportCount ).Get< double >();
        for( int idxViewport = 0; idxViewport < viewportCount; ++idxViewport )
        {
            const ViewportStateInfo& srcViewport = pViewports[ idxViewport ];
            *pDstViewports++ = srcViewport.GetOriginX();
            *pDstViewports++ = srcViewport.GetOriginY();
            *pDstViewports++ = srcViewport.GetWidth();
            *pDstViewports++ = srcViewport.GetHeight();
            *pDstDepthRanges++ = srcViewport.GetMinDepth();
            *pDstDepthRanges++ = srcViewport.GetMaxDepth();
        }
    }
}

void CommandBufferImpl< Target >::SetScissors( int firstScissor,
    int scissorCount, const ScissorStateInfo* pScissors ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    GlSetScissorsParam& param = *WriteGlCommand< GlSetScissorsParam >( this,
        sizeof( GlSetScissorsParam ) + sizeof( int32_t ) * 4 * scissorCount );
    {
        param.first = firstScissor;
        param.count = scissorCount;
        int32_t* pDstScissors = nn::util::BytePtr( &param, sizeof( GlSetScissorsParam ) ).Get< int32_t >();
        for( int idxScissor = 0; idxScissor < scissorCount; ++idxScissor )
        {
            const ScissorStateInfo& srcScissor = pScissors[ idxScissor ];
            *pDstScissors++ = srcScissor.GetOriginX();
            *pDstScissors++ = srcScissor.GetOriginY();
            *pDstScissors++ = srcScissor.GetWidth();
            *pDstScissors++ = srcScissor.GetHeight();
        }
    }
}

void CommandBufferImpl< Target >::UpdateBuffer( const GpuAddress& dstBufferAddress, size_t bufferSize,
    ptrdiff_t dstOffset, size_t dataSize, const void* pData ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( dstOffset >= 0 );
    NN_SDK_REQUIRES( dstOffset + dataSize <= bufferSize );
    NN_SDK_REQUIRES( pData != NULL || dataSize == 0 );
    NN_UNUSED( bufferSize );

    GlBufferSubDataParam& param = *WriteGlCommand< GlBufferSubDataParam >(
        this, static_cast< uint32_t >( sizeof( GlBufferSubDataParam ) + dataSize ) );
    {
        param.hBuffer = static_cast< uint32_t >( dstBufferAddress.ToData()->impl );
        param.offset = static_cast< int32_t >( dstBufferAddress.ToData()->value + dstOffset );
        param.size = static_cast< uint32_t >( dataSize );
        memcpy( nn::util::BytePtr( &param, sizeof( GlBufferSubDataParam ) ).Get(), pData, dataSize );
    }
}

// 以下はルートシグネチャを使わない場合

void CommandBufferImpl< Target >::SetConstantBuffer( int slot,
    ShaderStage, const DescriptorSlot& constantBufferDescriptor ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    const uint32_t* pDescriptor = ToPtr< const uint32_t >( constantBufferDescriptor );
    SetBuffer< GL_UNIFORM_BUFFER >( this, slot, pDescriptor[ 0 ], pDescriptor[ 1 ], pDescriptor[ 2 ] );
}

void CommandBufferImpl< Target >::SetUnorderedAccessBuffer( int slot,
    ShaderStage, const DescriptorSlot& unorderedAccessBufferDescriptor ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    const uint32_t* pDescriptor = ToPtr< const uint32_t >( unorderedAccessBufferDescriptor );
    SetBuffer< GL_SHADER_STORAGE_BUFFER >( this, slot, pDescriptor[ 0 ], pDescriptor[ 1 ], pDescriptor[ 2 ] );
}

void CommandBufferImpl< Target >::SetTextureAndSampler( int slot, ShaderStage,
    const DescriptorSlot& textureDescriptor, const DescriptorSlot& samplerDescriptor ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( slot >= 0 );
    nn::util::ConstBytePtr pTexture( reinterpret_cast< const void* >( textureDescriptor.ToData()->value ) );
    GlSetTextureAndSamplerParam& param = *WriteGlCommand< GlSetTextureAndSamplerParam >( this );
    {
        param.unit = slot;
        param.hTexture = *pTexture.Get< GlHandle >();
        param.target = *pTexture.Advance( sizeof( GlHandle ) ).Get< GlEnum >();
        param.hSampler = *reinterpret_cast< const GlHandle* >( samplerDescriptor.ToData()->value );
    }
}

void CommandBufferImpl< Target >::SetTexture( int slot, ShaderStage,
    const DescriptorSlot& textureDescriptor ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( slot >= 0 );
    nn::util::ConstBytePtr pTexture( reinterpret_cast< const void* >( textureDescriptor.ToData()->value ) );
    GlSetTextureParam& param = *WriteGlCommand< GlSetTextureParam >( this );
    {
        param.unit = slot;
        param.hTexture = *pTexture.Get< GlHandle >();
        param.target = *pTexture.Advance( sizeof( GlHandle ) ).Get< GlEnum >();
    }
}

void CommandBufferImpl< Target >::SetImage( int slot, ShaderStage,
    const DescriptorSlot& imageDescriptor ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( slot >= 0 );
    nn::util::ConstBytePtr pTexture( reinterpret_cast< const void* >( imageDescriptor.ToData()->value ) );
    GlSetImageParam& param = *WriteGlCommand< GlSetImageParam >( this );
    {
        param.unit = slot;
        param.hTexture = *pTexture.Get< GlHandle >();
        param.target = *pTexture.Advance( sizeof( GlHandle ) ).Get< GlEnum >();
    }
}

// 以下はデスクリプタープールを使わない場合

void CommandBufferImpl< Target >::SetConstantBuffer( int slot, ShaderStage,
    const GpuAddress& constantBufferAddress, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( IsValid( static_cast< GlHandle >( constantBufferAddress.ToData()->impl ) ) );
    NN_SDK_ASSERT( nn::util::is_aligned( constantBufferAddress.ToData()->value,
        this->pGfxDevice->ToData()->alignmentConstantBuffer ) );
    return SetBuffer< GL_UNIFORM_BUFFER >( this, slot,
        static_cast< GlHandle >( constantBufferAddress.ToData()->impl ),
        static_cast< uint32_t >( constantBufferAddress.ToData()->value ),
        static_cast< uint32_t >( size ) );
}

void CommandBufferImpl< Target >::SetUnorderedAccessBuffer( int slot, ShaderStage,
    const GpuAddress& unorderedAccessBufferAddress, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( IsValid( static_cast< GlHandle >( unorderedAccessBufferAddress.ToData()->impl ) ) );
    NN_SDK_ASSERT( nn::util::is_aligned( unorderedAccessBufferAddress.ToData()->value,
        this->pGfxDevice->ToData()->alignmentUnorderedAccessBuffer ) );
    return SetBuffer< GL_SHADER_STORAGE_BUFFER >( this, slot,
        static_cast< GlHandle >( unorderedAccessBufferAddress.ToData()->impl ),
        static_cast< uint32_t >( unorderedAccessBufferAddress.ToData()->value ),
        static_cast< uint32_t >( size ) );
}

void CommandBufferImpl< Target >::SetTextureAndSampler( int slot, ShaderStage stage,
    const TextureViewImpl< Target >* pTextureView, const SamplerImpl< Target >* pSampler ) NN_NOEXCEPT
{
    NN_UNUSED( stage );
    NN_SDK_REQUIRES_NOT_NULL( pTextureView );
    NN_SDK_REQUIRES( IsInitialized( *pTextureView ) );
    NN_SDK_ASSERT( pTextureView->ToData()->hTexture );
    NN_SDK_REQUIRES_NOT_NULL( pSampler );
    NN_SDK_REQUIRES( IsInitialized( *pSampler ) );
    NN_SDK_ASSERT( pSampler->ToData()->hSampler );
    NN_SDK_ASSERT( slot >= 0 );
    NN_SDK_ASSERT( this->state == State_Begun );

    GlSetTextureAndSamplerParam& param = *WriteGlCommand< GlSetTextureAndSamplerParam >( this );
    {
        param.unit = slot;
        param.hTexture = pTextureView->ToData()->hTexture;
        param.target = pTextureView->ToData()->target;
        param.hSampler = pSampler->ToData()->hSampler;
    }
}

void CommandBufferImpl< Target >::SetImage(
    int slot, ShaderStage, const TextureViewImpl< Target >* pImage ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pImage );
    NN_SDK_REQUIRES( IsInitialized( *pImage ) );
    NN_SDK_ASSERT( pImage->ToData()->hTexture );
    NN_SDK_ASSERT( slot >= 0 );
    NN_SDK_ASSERT( this->state == State_Begun );

    GlSetImageParam& param = *WriteGlCommand< GlSetImageParam >( this );
    {
        param.unit = slot;
        param.hTexture = pImage->ToData()->hTexture;
        param.target = pImage->ToData()->target;
    }
}

// 以下はパイプラインを使わない場合
void CommandBufferImpl< Target >::SetShader( const ShaderImpl< Target >* pShader, int stageBits ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pShader == NULL || pShader->ToData()->state ==
        ShaderImpl< Target >::DataType::State_Initialized );
    NN_SDK_REQUIRES( this->state == State_Begun );

    if( pShader )
    {
        if( pShader->ToData()->flags.GetBit( ShaderImpl< Target >::DataType::Flag_SeparationEnable ) )
        {
            GlSetSeparateShaderParam& param = *WriteGlCommand< GlSetSeparateShaderParam >( this );
            {
                param.hVertexProgram = pShader->ToData()->hVertexProgram;
                param.hHullProgram = pShader->ToData()->hHullProgram;
                param.hDomainProgram = pShader->ToData()->hDomainProgram;
                param.hGeometryProgram = pShader->ToData()->hGeometryProgram;
                param.hPixelProgram = pShader->ToData()->hPixelProgram;
                param.hComputeProgram = pShader->ToData()->hComputeProgram;
                param.stageBits = stageBits;
            }
        }
        else
        {
            GlSetShaderParam& param = *WriteGlCommand< GlSetShaderParam >( this );
            {
                param.hProgram = pShader->ToData()->hCombinedProgram;
            }
        }
    }
    else
    {
        GlSetSeparateShaderParam& param = *WriteGlCommand< GlSetSeparateShaderParam >( this );
        {
            param.hVertexProgram = GlInvalidHandle;
            param.hHullProgram = GlInvalidHandle;
            param.hDomainProgram = GlInvalidHandle;
            param.hGeometryProgram = GlInvalidHandle;
            param.hPixelProgram = GlInvalidHandle;
            param.hComputeProgram = GlInvalidHandle;
            param.stageBits = stageBits;
        }
    }
}

void CommandBufferImpl< Target >::SetRasterizerState(
    const RasterizerStateImpl< Target >* pRasterizerState ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pRasterizerState );
    NN_SDK_REQUIRES( IsInitialized( *pRasterizerState ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    RasterizerStateImplData< Target >& param = *WriteGlCommand<
        RasterizerStateImplData< Target > >( this );
    param = pRasterizerState->ToData();
}

void CommandBufferImpl< Target >::SetBlendState(
    const BlendStateImpl< Target >* pBlendState ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pBlendState );
    NN_SDK_REQUIRES( IsInitialized( *pBlendState ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    int staticSize = offsetof( BlendStateImplData< Target >, pTargetArray );
    int dynamicSize = sizeof( BlendStateImplData< Target >::BlendTargetState ) * pBlendState->ToData()->blendTargetCount;

    const BlendStateImplData< Target >& src = pBlendState->ToData();
    BlendStateImplData< Target >& param = *WriteGlCommand<
        BlendStateImplData< Target > >( this, staticSize + dynamicSize );
    {
        memcpy( &param, &src, staticSize );
        memcpy( &param.pTargetArray, src.pTargetArray, dynamicSize );
    }
}

void CommandBufferImpl< Target >::SetDepthStencilState(
    const DepthStencilStateImpl< Target >* pDepthStencilState ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDepthStencilState );
    NN_SDK_REQUIRES( IsInitialized( *pDepthStencilState ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    DepthStencilStateImplData< Target >& param = *WriteGlCommand<
        DepthStencilStateImplData< Target > >( this );
    param = pDepthStencilState->ToData();
}

void CommandBufferImpl< Target >::SetVertexState(
    const VertexStateImpl< Target >* pVertexState ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pVertexState );
    NN_SDK_REQUIRES( IsInitialized( *pVertexState ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    int staticSize = offsetof( VertexStateImplData< Target >, pWorkMemory );
    int dynamicSize = sizeof( VertexStateImplData< Target >::AttributeState ) *
        pVertexState->ToData()->attributeCount + sizeof( uint32_t ) * pVertexState->ToData()->bufferCount;

    const VertexStateImplData< Target >& src = pVertexState->ToData();
    VertexStateImplData< Target >& param = *WriteGlCommand<
        VertexStateImplData< Target > >( this, staticSize + dynamicSize );
    {
        memcpy( &param, &src, staticSize );
        memcpy( &param.pWorkMemory, src.pWorkMemory, dynamicSize );
    }
}

void CommandBufferImpl< Target >::SetTessellationState(
    const TessellationStateImpl< Target >* pTessellationState ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pTessellationState );
    NN_SDK_REQUIRES( IsInitialized( *pTessellationState ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    TessellationStateImplData< Target >& param = *WriteGlCommand<
        TessellationStateImplData< Target > >( this );
    param = pTessellationState->ToData();
}

// GL4 専用インターフェイス
void CommandBufferImpl< Target >::Gl4SetUserCommand(
    Gl4UserCommandCallbackType pCallback, const void* pParam ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pCallback );
    NN_SDK_REQUIRES( this->state == State_Begun );

    GlCallbackParam& param = *WriteGlCommand< GlCallbackParam >( this );
    {
        param.pCallback = pCallback;
        param.pParam = pParam;
    }
}

void CommandBufferImpl< Target >::Gl4SetUserCommandDynamic(
    Gl4UserCommandCallbackType pCallback, const void* pParam, size_t paramSize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pCallback );
    NN_SDK_REQUIRES( this->state == State_Begun );

    GlCallbackParam& param = *WriteGlCommand< GlCallbackParam >( this,
        static_cast< uint32_t >( sizeof( GlCallbackParam ) + paramSize ) );
    {
        void* pDstParam = nn::util::BytePtr( &param, sizeof( GlCallbackParam ) ).Get();
        param.pCallback = pCallback;
        param.pParam = pDstParam;
        memcpy( pDstParam, pParam, paramSize );
    }
}

}
}
}

NN_PRAGMA_POP_WARNINGS
