﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>
#include <memory>

#include <nn/nn_SdkAssert.h>

#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_CommandBufferInfo.h>
#include <nn/gfx/gfx_TextureInfo.h>
#include <nn/gfx/gfx_RootSignatureInfo.h>
#include <nn/gfx/gfx_GpuAddress.h>
#include <nn/gfx/gfx_DescriptorSlot.h>
#include <nn/gfx/gfx_CommandBuffer.h>
#include <nn/gfx/gfx_StateInfo.h>

#include <nn/gfx/detail/gfx_Misc.h>
#include <nn/gfx/detail/gfx_CommandBuffer-api.d3d.11.h>
#include <nn/gfx/detail/gfx_State-api.d3d.11.h>
#include <nn/gfx/detail/gfx_Texture-api.d3d.11.h>
#include <nn/gfx/detail/gfx_Pipeline-api.d3d.11.h>
#include <nn/gfx/detail/gfx_Device-api.d3d.11.h>
#include <nn/gfx/detail/gfx_Buffer-api.d3d.11.h>
#include <nn/gfx/detail/gfx_Sampler-api.d3d.11.h>
#include <nn/gfx/detail/gfx_Shader-api.d3d.11.h>
#include <nn/gfx/detail/gfx_DescriptorPool-api.d3d.11.h>
#include <nn/gfx/detail/gfx_RootSignature-api.d3d.11.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.d3d.11.h>

#include "gfx_CommonHelper.h"
#include "gfx_D3dHelper.h"

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_DEPRECATED_DECLARATIONS

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationD3d11 Target;

namespace {

template< typename T >
T* ToPtr( const DescriptorSlot& slot )
{
    return reinterpret_cast< T* >( slot.ToData()->value );
}

void GetTextureCopyRegion( D3D11_BOX* pOutBox,
    ID3D11Resource* pResource, const TextureCopyRegion& region ) NN_NOEXCEPT
{
    int offsetX = region.GetOffsetU();
    int offsetY = region.GetOffsetV();
    int offsetZ = region.GetOffsetW();
    int width = region.GetWidth();
    int height = region.GetHeight();
    int depth = 1;

    switch( D3d::GetImageDimension( pResource ) )
    {
    case ImageDimension_1d:
        {
            height = 1;
        }
        break;
    case ImageDimension_1dArray:
        {
            offsetY = region.GetSubresource().GetArrayIndex();
            height = std::max NN_PREVENT_MACRO_FUNC ( region.GetArrayLength(), 1 );
        }
        break;
    case ImageDimension_2dArray:
    case ImageDimension_2dMultisampleArray:
        {
            offsetZ = region.GetSubresource().GetArrayIndex();
            depth = std::max NN_PREVENT_MACRO_FUNC ( region.GetArrayLength(), 1 );
        }
        break;
    case ImageDimension_3d:
        {
            depth = region.GetDepth();
        }
        break;
    default:
        break;
    }

    pOutBox->left = offsetX;
    pOutBox->right = offsetX + width;
    pOutBox->top = offsetY;
    pOutBox->bottom = offsetY + height;
    pOutBox->front = offsetZ;
    pOutBox->back = offsetZ + depth;
}

}

size_t CommandBufferImpl< Target >::GetCommandMemoryAlignment( DeviceImpl< Target >* ) NN_NOEXCEPT
{
    return 1;
}

size_t CommandBufferImpl< Target >::GetControlMemoryAlignment( DeviceImpl< Target >* ) NN_NOEXCEPT
{
    return 1;
}

CommandBufferImpl< Target >::CommandBufferImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

CommandBufferImpl< Target >::~CommandBufferImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void CommandBufferImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );

    this->pGfxDevice = pDevice;
    this->commandBufferType = static_cast< Bit8 >( info.GetCommandBufferType() );
    this->queueCapability = static_cast< Bit16 >( info.GetQueueCapability() );

    Reset();

    this->pOutOfCommandMemoryCallback = NULL;
    this->pOutOfControlMemoryCallback = NULL;
    this->patchControlPointCount = 0;
    this->sampleMask = 0xffffffff;

    DeviceImpl< Target >::DataType &data = pDevice->ToData();
    ID3D11DeviceContext* pD3dDeferredContext;
    ID3D11Device* pD3dDevice = static_cast< ID3D11Device* >( data.renderingContext.hD3dDevice );

    HRESULT hResult = NN_GFX_CALL_D3D_FUNCTION( pD3dDevice->CreateDeferredContext( 0, &pD3dDeferredContext ) );

    NN_SDK_ASSERT( SUCCEEDED( hResult ) );
    NN_UNUSED( hResult );
    NN_SDK_ASSERT_NOT_NULL( pD3dDeferredContext );

    this->pDeferredContext = pD3dDeferredContext;
    this->pCommandList = NULL;

    this->state = State_Initialized;
}

void CommandBufferImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    if ( this->pCommandList )
    {
        ID3D11CommandList* pD3dCommandList = static_cast< ID3D11CommandList* >( this->pCommandList );
        NN_GFX_CALL_D3D_FUNCTION( pD3dCommandList->Release() );
        this->pCommandList = NULL;
    }

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );
    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->Release() );
    this->pDeferredContext = NULL;

    Reset();

    this->pGfxDevice = NULL;
    this->state = State_NotInitialized;
}

void CommandBufferImpl< Target >::AddCommandMemory( MemoryPoolImpl< Target >* pMemoryPool,
    ptrdiff_t memoryPoolOffset, size_t memorySize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pMemoryPool );
    NN_SDK_REQUIRES( IsInitialized( *pMemoryPool ) );
    NN_SDK_REQUIRES( memoryPoolOffset + memorySize <= pMemoryPool->ToData()->memorySize );
    NN_SDK_ASSERT( pMemoryPool->ToData()->pMemory );
    NN_SDK_REQUIRES( this->state != State_NotInitialized );

    // d3d11ではコマンドメモリは必要としません。
    // 念のためメモリアドレスとサイズは保持します。
    void* pMemory = nn::util::BytePtr( pMemoryPool->ToData()->pMemory, memoryPoolOffset ).Get();
    this->commandMemorySize = static_cast< uint32_t >( memorySize );
    this->pHeadCommandMemory = pMemory;
}

void CommandBufferImpl< Target >::AddControlMemory( void* pMemory, size_t memorySize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pMemory );
    NN_SDK_REQUIRES( this->state != State_NotInitialized );

    // d3d11ではコントロールメモリは必要としません。
    // 念のためメモリアドレスとサイズは保持します。
    this->controlMemorySize = static_cast< uint32_t >( memorySize );
    this->pHeadControlMemory = pMemory;
}

void CommandBufferImpl< Target >::SetOutOfCommandMemoryEventCallback(
    OutOfMemoryEventCallback pCallback ) NN_NOEXCEPT
{
    this->pOutOfCommandMemoryCallback.ptr = reinterpret_cast< void (*)() >( pCallback );
}

void CommandBufferImpl< Target >::SetOutOfControlMemoryEventCallback(
    OutOfMemoryEventCallback pCallback ) NN_NOEXCEPT
{
    this->pOutOfControlMemoryCallback.ptr = reinterpret_cast< void (*)() >( pCallback );
}

void CommandBufferImpl< Target >::Reset() NN_NOEXCEPT
{
    this->commandMemorySize = 0;
    this->pHeadCommandMemory = NULL;
    this->controlMemorySize = sizeof( this->defaultControlMemory );
    this->pHeadControlMemory = this->defaultControlMemory;
}

void CommandBufferImpl< Target >::Begin() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_REQUIRES_NOT_NULL( this->pGfxDevice.ptr );

    if ( ID3D11CommandList* pD3dCommandList = static_cast< ID3D11CommandList* >( this->pCommandList ) )
    {
        NN_GFX_CALL_D3D_FUNCTION( pD3dCommandList->Release() );
        this->pCommandList = NULL;
    }

    this->state = State_Begun;
}

void CommandBufferImpl< Target >::End() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    ID3D11CommandList* pD3dCommandList;
    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext*  >( this->pDeferredContext );
    HRESULT hResult = NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->FinishCommandList( TRUE, &pD3dCommandList ) );
    NN_SDK_ASSERT( SUCCEEDED( hResult ) );
    NN_UNUSED( hResult );

    this->pCommandList = pD3dCommandList;

    this->state = State_Initialized;
}

void CommandBufferImpl< Target >::Dispatch(
    int groupCountX, int groupCountY, int groupCountZ ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->Dispatch(
        static_cast< UINT >( groupCountX ), static_cast< UINT >( groupCountY ), static_cast< UINT >( groupCountZ ) ) );
}

void CommandBufferImpl< Target >::Draw(
    PrimitiveTopology primitiveTopology, int vertexCount, int vertexOffset ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->IASetPrimitiveTopology( D3d::GetPrimitiveTopology( primitiveTopology, this->patchControlPointCount ) ) );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->Draw( static_cast< UINT >( vertexCount ), static_cast< UINT >( vertexOffset ) ) );

}

void CommandBufferImpl< Target >::Draw( PrimitiveTopology primitiveTopology,
    int vertexCountPerInstance, int vertexOffset, int instanceCount, int baseInstance ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->IASetPrimitiveTopology( D3d::GetPrimitiveTopology( primitiveTopology, this->patchControlPointCount ) ) );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->DrawInstanced( static_cast< UINT >( vertexCountPerInstance ),
        static_cast< UINT >( instanceCount ), static_cast< UINT >( vertexOffset ), static_cast< UINT >( baseInstance ) ) );
}

void CommandBufferImpl< Target >::DrawIndexed( PrimitiveTopology primitiveTopology, IndexFormat indexFormat,
    const GpuAddress& indexBufferAddress, int indexCount, int baseVertex ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( indexBufferAddress.ToData()->impl );

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );
    ID3D11Buffer* pD3dIndexBuffer = D3d::GetBufferResource( indexBufferAddress.ToData()->impl );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->IASetPrimitiveTopology( D3d::GetPrimitiveTopology( primitiveTopology, this->patchControlPointCount ) ) );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->IASetIndexBuffer( pD3dIndexBuffer,
        D3d::GetIndexFormat( indexFormat ).dxgiFormat, static_cast< UINT >( indexBufferAddress.ToData()->value ) ) );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->DrawIndexed( static_cast< UINT >( indexCount ), 0, baseVertex ) );
}

void CommandBufferImpl< Target >::DrawIndexed( PrimitiveTopology primitiveTopology, IndexFormat indexFormat,
    const GpuAddress& indexBufferAddress, int indexCountPerInstance, int baseVertex, int instanceCount, int baseInstance ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( indexBufferAddress.ToData()->impl );

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );
    ID3D11Buffer* pD3dIndexBuffer = D3d::GetBufferResource( indexBufferAddress.ToData()->impl );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->IASetPrimitiveTopology( D3d::GetPrimitiveTopology( primitiveTopology, this->patchControlPointCount ) ) );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->IASetIndexBuffer( pD3dIndexBuffer,
        D3d::GetIndexFormat( indexFormat ).dxgiFormat, static_cast< UINT >( indexBufferAddress.ToData()->value ) ) );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->DrawIndexedInstanced( static_cast< UINT >( indexCountPerInstance ),
        static_cast< UINT >( instanceCount ), 0, baseVertex, static_cast< UINT >( baseInstance ) ) );
}

void CommandBufferImpl< Target >::DispatchIndirect( const GpuAddress& indirectBufferAddress ) NN_NOEXCEPT
{
    NN_UNUSED( indirectBufferAddress );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( indirectBufferAddress.ToData()->impl );

    ID3D11Buffer* pD3dIndirectBuffer = D3d::GetBufferResource( indirectBufferAddress.ToData()->impl );
    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->DispatchIndirect( pD3dIndirectBuffer, 0 ) );
}

void CommandBufferImpl< Target >::DrawIndirect(
    PrimitiveTopology primitiveTopology, const GpuAddress& indirectBufferAddress ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( indirectBufferAddress.ToData()->impl );

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );
    ID3D11Buffer* pD3dIndirectBuffer = D3d::GetBufferResource( indirectBufferAddress.ToData()->impl );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->IASetPrimitiveTopology( D3d::GetPrimitiveTopology( primitiveTopology, this->patchControlPointCount ) ) );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->DrawInstancedIndirect( pD3dIndirectBuffer, 0 ) );
}

void CommandBufferImpl< Target >::DrawIndexedIndirect( PrimitiveTopology primitiveTopology,
    IndexFormat indexFormat, const GpuAddress& indexBufferAddress, const GpuAddress& indirectBufferAddress ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( indexBufferAddress.ToData()->impl );
    NN_SDK_ASSERT( indirectBufferAddress.ToData()->impl );

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );
    ID3D11Buffer* pD3dIndexBuffer = D3d::GetBufferResource( indexBufferAddress.ToData()->impl );
    ID3D11Buffer* pD3dIndirectBuffer = D3d::GetBufferResource( indirectBufferAddress.ToData()->impl );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->IASetPrimitiveTopology( D3d::GetPrimitiveTopology( primitiveTopology, this->patchControlPointCount ) ) );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->IASetIndexBuffer( pD3dIndexBuffer,
        D3d::GetIndexFormat( indexFormat ).dxgiFormat, static_cast< UINT >( indexBufferAddress.ToData()->value ) ) );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->DrawIndexedInstancedIndirect( pD3dIndirectBuffer, 0 ) );
}

void CommandBufferImpl< Target >::SetPipeline( const PipelineImpl< Target >* pPipeline ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pPipeline );
    NN_SDK_REQUIRES( IsInitialized( *pPipeline ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    const PipelineImpl< Target >::DataType& pipeline = pPipeline->ToData();

    if( pipeline.pipelineType == PipelineType_Graphics )
    {
        SetRasterizerState( nn::gfx::DataToAccessor( pipeline.rasterizerState ) );
        SetBlendState( nn::gfx::DataToAccessor( pipeline.blendState ) );
        SetDepthStencilState( nn::gfx::DataToAccessor( pipeline.depthStencilState ) );
        SetVertexState( nn::gfx::DataToAccessor( pipeline.vertexState ) );
        if( pipeline.flags.GetBit( PipelineImpl< Target >::DataType::Flag_HasTessellationState ) )
        {
            SetTessellationState( nn::gfx::DataToAccessor( pipeline.tessellationState ) );
        }
        SetShader( pipeline.pShader, ShaderStageBit_All );
    }
    else if ( pipeline.pipelineType == PipelineType_Compute )
    {
        SetShader( pipeline.pShader, ShaderStageBit_Compute );
    }
}

void CommandBufferImpl< Target >::SetRenderTargets( int colorTargetCount,
    const ColorTargetViewImpl< Target >* const * ppColorTargets,
    const DepthStencilViewImpl< Target >* pDepthStencil ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( colorTargetCount < 1 || ppColorTargets );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( colorTargetCount <=
        static_cast< int >( this->pGfxDevice->ToData()->maxColorAttachments ) );

    ID3D11DepthStencilView* pD3dDepthStencilView = NULL;
    if( pDepthStencil )
    {
        NN_SDK_REQUIRES( IsInitialized( *pDepthStencil ) );
        pD3dDepthStencilView = static_cast< ID3D11DepthStencilView* >( pDepthStencil->ToData()->pDepthStencilView );
    }

    ID3D11RenderTargetView* pD3dRenderTargetViews[ D3D11_SIMULTANEOUS_RENDER_TARGET_COUNT ];

    for( int idxTarget = 0; idxTarget < colorTargetCount; ++idxTarget )
    {
        NN_SDK_REQUIRES_NOT_NULL( ppColorTargets[ idxTarget ] );
        NN_SDK_REQUIRES( IsInitialized( *( ppColorTargets[ idxTarget ] ) ) );
        pD3dRenderTargetViews[ idxTarget ] = static_cast< ID3D11RenderTargetView* >( ppColorTargets[ idxTarget ]->ToData()->pRenderTargetView );
    }

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->OMSetRenderTargets( colorTargetCount, pD3dRenderTargetViews, pD3dDepthStencilView ) );
}

void CommandBufferImpl< Target >::SetVertexBuffer( int bufferIndex,
    const GpuAddress& vertexBufferAddress, ptrdiff_t stride, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_UNUSED( size );

// gfx仕様外ですが、NULLアドレスの場合にはVertexBufferのデタッチとして扱います。
//    NN_SDK_ASSERT( IsD3dHandleValid( reinterpret_cast< D3dHandle >( vertexBufferAddress.ToData()->impl ) ) );

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );

    ID3D11Buffer* pD3dBuffer = D3d::GetBufferResource( vertexBufferAddress.ToData()->impl );
    UINT strideArray[ 1 ] = { static_cast< UINT >( stride ) };
    UINT offsetArray[ 1 ] = { static_cast< UINT >( vertexBufferAddress.ToData()->value ) };

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->IASetVertexBuffers( bufferIndex, 1, &pD3dBuffer, strideArray, offsetArray ) );
}

void CommandBufferImpl< Target >::SetViewportScissorState(
    const ViewportScissorStateImpl< Target >* pViewportScissor ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pViewportScissor );
    NN_SDK_REQUIRES( IsInitialized( *pViewportScissor ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    const ViewportScissorStateImpl< Target >::DataType &data = pViewportScissor->ToData();
    NN_SDK_ASSERT( data.viewportCount > 0 && data.viewportCount <= D3D11_VIEWPORT_AND_SCISSORRECT_MAX_INDEX );

    D3D11_VIEWPORT pViewportSettings[ D3D11_VIEWPORT_AND_SCISSORRECT_MAX_INDEX ];

    pViewportSettings[ 0 ].TopLeftX = data.viewport[ 0 ];
    pViewportSettings[ 0 ].TopLeftY = data.viewport[ 1 ];
    pViewportSettings[ 0 ].Width = data.viewport[ 2 ];
    pViewportSettings[ 0 ].Height = data.viewport[ 3 ];
    pViewportSettings[ 0 ].MinDepth = data.viewport[ 4 ];
    pViewportSettings[ 0 ].MaxDepth = data.viewport[ 5 ];

    int extraViewportCount = data.viewportCount - 1;
    nn::util::BytePtr ptr( data.pWorkMemory.ptr );
    D3D11_VIEWPORT* viewportArray = ptr.Get< D3D11_VIEWPORT >();
    D3D11_RECT* scissorArray = ptr.Advance( sizeof( D3D11_VIEWPORT ) * extraViewportCount ).Get< D3D11_RECT >();

    for ( int viewportIndex = 1; viewportIndex < data.viewportCount; ++viewportIndex )
    {
        int extraViewportIndex = viewportIndex - 1;
        pViewportSettings[ viewportIndex ] = viewportArray[ extraViewportIndex ];
    }

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->RSSetViewports( data.viewportCount, pViewportSettings ) );

    D3D11_RECT pScissorSettings[ D3D11_VIEWPORT_AND_SCISSORRECT_MAX_INDEX ];

    pScissorSettings[ 0 ].left = data.scissor[ 0 ];
    pScissorSettings[ 0 ].top = data.scissor[ 1 ];
    pScissorSettings[ 0 ].right = data.scissor[ 2 ];
    pScissorSettings[ 0 ].bottom = data.scissor[ 3 ];

    if ( data.flag.GetBit( data.Flag_ScissorEnable ) )
    {
        for ( int scissorIndex = 1; scissorIndex < data.viewportCount; ++scissorIndex )
        {
            int extraScissorIndex = scissorIndex - 1;
            pScissorSettings[ scissorIndex ] = scissorArray[ extraScissorIndex ];
        }
    }

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->RSSetScissorRects( data.viewportCount, pScissorSettings ) );
}

void CommandBufferImpl< Target >::CopyBuffer( BufferImpl< Target >* pDstBuffer, ptrdiff_t dstOffset,
    const BufferImpl< Target >* pSrcBuffer, ptrdiff_t srcOffset, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstBuffer );
    NN_SDK_REQUIRES_NOT_NULL( pSrcBuffer );
    NN_SDK_REQUIRES( IsInitialized( *pDstBuffer ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcBuffer ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( IsD3dHandleValid( pDstBuffer->ToData()->pGpuBuffer ) );
    NN_SDK_ASSERT( IsD3dHandleValid( pSrcBuffer->ToData()->pGpuBuffer ) );

    ID3D11Resource* pDstResource = static_cast< ID3D11Resource* >( pDstBuffer->ToData()->pGpuBuffer );
    ID3D11Resource* pSrcResource = static_cast< ID3D11Resource* >( pSrcBuffer->ToData()->pGpuBuffer );

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );

    D3D11_BOX srcBox;
    srcBox.left = static_cast< UINT >( srcOffset );
    srcBox.right = static_cast< UINT >( size - srcOffset );
    srcBox.top = srcBox.front = 0;
    srcBox.bottom = srcBox.back = 1;

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->CopySubresourceRegion(
        pDstResource, 0, static_cast< UINT >( dstOffset ), 0, 0, pSrcResource, 0, srcOffset == 0 ? NULL : &srcBox ) );
}

void CommandBufferImpl< Target >::CopyImage( TextureImpl< Target >* pDstTexture,
    const TextureSubresource& dstSubresource, int dstOffsetU, int dstOffsetV, int dstOffsetW,
    const TextureImpl< Target >* pSrcTexture, const TextureCopyRegion& srcCopyRegion ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstTexture );
    NN_SDK_REQUIRES_NOT_NULL( pSrcTexture );
    NN_SDK_REQUIRES( IsInitialized( *pDstTexture ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcTexture ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( IsD3dHandleValid( pDstTexture->ToData()->pTexture ) );
    NN_SDK_ASSERT( IsD3dHandleValid( pSrcTexture->ToData()->pTexture ) );

    ID3D11Resource* pDstResource = static_cast< ID3D11Resource* >( pDstTexture->ToData()->pTexture );
    ID3D11Resource* pSrcResource = static_cast< ID3D11Resource* >( pSrcTexture->ToData()->pTexture );
    UINT dstX = dstOffsetU;
    UINT dstY = D3d::GetImageDimension( pDstResource ) == ImageDimension_1dArray
        ? dstSubresource.GetArrayIndex() : dstOffsetV;
    UINT dstZ = ( D3d::GetImageDimension( pDstResource ) == ImageDimension_2dArray ||
        D3d::GetImageDimension( pDstResource ) == ImageDimension_2dMultisampleArray )
        ? dstSubresource.GetArrayIndex() : dstOffsetW;
    UINT dstSubresourceIndex = D3D11CalcSubresource( dstSubresource.GetMipLevel(),
        dstSubresource.GetArrayIndex(), D3d::GetTextureMipLevels( pDstResource ) );

    D3D11_BOX srcBox;
    GetTextureCopyRegion( &srcBox, pSrcResource, srcCopyRegion );

    UINT srcSubresourceIndex = D3D11CalcSubresource( srcCopyRegion.GetSubresource().GetMipLevel(),
        srcCopyRegion.GetSubresource().GetArrayIndex(), D3d::GetTextureMipLevels( pSrcResource ) );

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->CopySubresourceRegion(
        pDstResource, dstSubresourceIndex, dstX, dstY, dstZ, pSrcResource, srcSubresourceIndex, &srcBox ) );
}

void CommandBufferImpl< Target >::CopyBufferToImage( TextureImpl< Target >* pDstTexture,
     const BufferImpl< Target >* pSrcBuffer, const BufferTextureCopyRegion& region ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstTexture );
    NN_SDK_REQUIRES_NOT_NULL( pSrcBuffer );
    NN_SDK_REQUIRES( IsInitialized( *pDstTexture ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcBuffer ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_UNUSED( region );

    NN_SDK_ASSERT( IsD3dHandleValid( pDstTexture->ToData()->pTexture ) );
    NN_SDK_ASSERT( IsD3dHandleValid( pSrcBuffer->ToData()->pGpuBuffer ) );

    NN_SDK_ASSERT( region.GetBufferOffset() == 0 );
    NN_SDK_ASSERT( region.GetTextureCopyRegion().GetOffsetU() == 0 );
    NN_SDK_ASSERT( region.GetTextureCopyRegion().GetOffsetV() == 0 );
    NN_SDK_ASSERT( region.GetTextureCopyRegion().GetOffsetW() == 0 );

    // Mapしての全体コピーで仮実装
    ID3D11Resource* pDstResource = static_cast< ID3D11Resource* >( pDstTexture->ToData()->pTexture );
    ID3D11Resource* pSrcGpuResource = static_cast< ID3D11Resource* >( pSrcBuffer->ToData()->pGpuBuffer );
    ID3D11Resource* pSrcCpuResource = static_cast< ID3D11Resource* >( pSrcBuffer->ToData()->pCpuBuffer );

    int width, height, depth;
    D3d::GetTextureSize( &width, &height, &depth, pDstResource );
    NN_SDK_ASSERT( region.GetTextureCopyRegion().GetWidth() == width );
    NN_SDK_ASSERT( region.GetTextureCopyRegion().GetHeight() == height );
    NN_SDK_ASSERT( region.GetTextureCopyRegion().GetDepth() == depth );

    ID3D11DeviceContext* deviceContext = static_cast< ID3D11DeviceContext* >( this->pGfxDevice->ToData()->renderingContext.hD3dDeviceContext );
    NN_GFX_CALL_D3D_FUNCTION( deviceContext->CopyResource( pSrcCpuResource, pSrcGpuResource ) );

    D3D11_MAPPED_SUBRESOURCE mappedResource;
    HRESULT hResult = NN_GFX_CALL_D3D_FUNCTION( deviceContext->Map( pSrcCpuResource, 0, D3D11_MAP_READ, 0, &mappedResource ) );
    NN_SDK_ASSERT( SUCCEEDED( hResult ) );
    NN_UNUSED( hResult );
    int bytePerPixel = GetBytePerPixel( GetChannelFormat( static_cast< ImageFormat >( pDstTexture->ToData()->imageFormat ) ) );
    NN_GFX_CALL_D3D_FUNCTION( deviceContext->UpdateSubresource(
        pDstResource, 0, NULL, static_cast< uint8_t* >( mappedResource.pData ),
        width * bytePerPixel,
        width * height * bytePerPixel ) );

    NN_GFX_CALL_D3D_FUNCTION( deviceContext->Unmap( pSrcCpuResource, 0 ) );
}

void CommandBufferImpl< Target >::CopyImageToBuffer( BufferImpl< Target >* pDstBuffer,
    const TextureImpl< Target >* pSrcTexture, const BufferTextureCopyRegion& region ) NN_NOEXCEPT
{
    NN_UNUSED( pDstBuffer );
    NN_UNUSED( pSrcTexture );
    NN_UNUSED( region );

    // 本機能は未実装です。
    // D3D11における実装は可能です。（制限が付く可能性あり）
    // CopyResourceもしくは、UpdateSubresourceによる実装になります。
    NN_SDK_ASSERT( 0 );
}

void CommandBufferImpl< Target >::CopyBufferToImage( TextureImpl< Target >* pDstTexture,
    const TextureCopyRegion& dstRegion, const BufferImpl< Target >* pSrcBuffer, ptrdiff_t srcOffset ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstTexture );
    NN_SDK_REQUIRES_NOT_NULL( pSrcBuffer );
    NN_SDK_REQUIRES( IsInitialized( *pDstTexture ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcBuffer ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( IsD3dHandleValid( pDstTexture->ToData()->pTexture ) );
    NN_SDK_ASSERT( IsD3dHandleValid( pSrcBuffer->ToData()->pGpuBuffer ) );
    NN_SDK_ASSERT( IsD3dHandleValid( pSrcBuffer->ToData()->pCpuBuffer ) );
    NN_UNUSED( dstRegion );

    NN_SDK_ASSERT( srcOffset == 0 );
    NN_SDK_ASSERT( dstRegion.GetOffsetU() == 0 );
    NN_SDK_ASSERT( dstRegion.GetOffsetV() == 0 );
    NN_SDK_ASSERT( dstRegion.GetOffsetW() == 0 );

    // Mapしての全体コピーで仮実装
    ID3D11Resource* pDstResource = static_cast< ID3D11Resource* >( pDstTexture->ToData()->pTexture );
    ID3D11Resource* pSrcGpuResource = static_cast< ID3D11Resource* >( pSrcBuffer->ToData()->pGpuBuffer );
    ID3D11Resource* pSrcCpuResource = static_cast< ID3D11Resource* >( pSrcBuffer->ToData()->pCpuBuffer );

    int width, height, depth;
    D3d::GetTextureSize( &width, &height, &depth, pDstResource );
    NN_SDK_ASSERT( dstRegion.GetWidth() == width );
    NN_SDK_ASSERT( dstRegion.GetHeight() == height );
    NN_SDK_ASSERT( dstRegion.GetDepth() == depth );

    ID3D11DeviceContext* deviceContext = static_cast< ID3D11DeviceContext* >( this->pGfxDevice->ToData()->renderingContext.hD3dDeviceContext );
    NN_GFX_CALL_D3D_FUNCTION( deviceContext->CopyResource( pSrcCpuResource, pSrcGpuResource ) );

    D3D11_MAPPED_SUBRESOURCE mappedResource;
    HRESULT hResult = NN_GFX_CALL_D3D_FUNCTION( deviceContext->Map( pSrcCpuResource, 0, D3D11_MAP_READ, 0, &mappedResource ) );
    NN_SDK_ASSERT( SUCCEEDED( hResult ) );
    NN_UNUSED( hResult );
    int bytePerPixel = GetBytePerPixel( GetChannelFormat( static_cast< ImageFormat >( pDstTexture->ToData()->imageFormat ) ) );
    NN_GFX_CALL_D3D_FUNCTION( deviceContext->UpdateSubresource(
        pDstResource, 0, NULL, static_cast< uint8_t* >( mappedResource.pData ) + srcOffset, width * bytePerPixel, width * height * bytePerPixel ) );

    NN_GFX_CALL_D3D_FUNCTION( deviceContext->Unmap( pSrcCpuResource, 0 ) );
}

void CommandBufferImpl< Target >::CopyImageToBuffer( BufferImpl< Target >* pDstBuffer,
    ptrdiff_t dstOffset, const TextureImpl< Target >* pSrcTexture, const TextureCopyRegion& srcRegion ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstBuffer );
    NN_SDK_REQUIRES_NOT_NULL( pSrcTexture );
    NN_SDK_REQUIRES( IsInitialized( *pDstBuffer ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcTexture ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( IsD3dHandleValid( pDstBuffer->ToData()->pGpuBuffer ) );
    NN_SDK_ASSERT( IsD3dHandleValid( pSrcTexture->ToData()->pTexture ) );
    NN_UNUSED( dstOffset );

    NN_SDK_ASSERT( dstOffset == 0 );
    NN_SDK_ASSERT( srcRegion.GetOffsetU() == 0 );
    NN_SDK_ASSERT( srcRegion.GetOffsetV() == 0 );
    NN_SDK_ASSERT( srcRegion.GetOffsetW() == 0 );

    // Mapしての全体コピーで仮実装

    ID3D11DeviceContext* deviceContext = static_cast< ID3D11DeviceContext* >( this->pGfxDevice->ToData()->renderingContext.hD3dDeviceContext );

    // TextureをStagingのResourceへコピーします。
    ID3D11Resource* pStagingTexture = static_cast< ID3D11Resource* >( pSrcTexture->ToData()->pStagingTexture );
    ID3D11Resource* pTexture = static_cast< ID3D11Resource* >( pSrcTexture->ToData()->pTexture );

    int width, height, depth;
    D3d::GetTextureSize( &width, &height, &depth, pTexture );
    NN_SDK_ASSERT( srcRegion.GetWidth() == width );
    NN_SDK_ASSERT( srcRegion.GetHeight() == height );
    NN_SDK_ASSERT( srcRegion.GetDepth() == depth );

    NN_GFX_CALL_D3D_FUNCTION( deviceContext->CopyResource( pStagingTexture, pTexture ) );

    // Staging ResourceをMapします。
    D3D11_MAPPED_SUBRESOURCE mappedResource;

    HRESULT hResult = NN_GFX_CALL_D3D_FUNCTION( deviceContext->Map( pStagingTexture, 0, D3D11_MAP_READ, 0, &mappedResource ) );
    NN_SDK_ASSERT( SUCCEEDED( hResult ) );
    NN_UNUSED( hResult );

    // Bufferへコピーします。
    ID3D11Resource* pBuffer = static_cast< ID3D11Resource* >( pDstBuffer->ToData()->pGpuBuffer );
    UINT pixelSize = static_cast< UINT >( D3d::GetImageSize( static_cast< ImageFormat >( pSrcTexture->ToData()->imageFormat ), 1, 1, 1 ) );
    NN_GFX_CALL_D3D_FUNCTION( deviceContext->UpdateSubresource( pBuffer, 0, NULL, mappedResource.pData,
        srcRegion.GetWidth() * pixelSize, srcRegion.GetWidth() * srcRegion.GetHeight() * pixelSize ) );

    NN_GFX_CALL_D3D_FUNCTION( deviceContext->Unmap( pStagingTexture, 0 ) );
}

void CommandBufferImpl< Target >::BlitImage( TextureImpl< Target >* pDstTexture,
    const TextureCopyRegion& dstRegion, const TextureImpl< Target >* pSrcTexture,
    const TextureCopyRegion& srcRegion, int copyFlags ) NN_NOEXCEPT
{
    NN_UNUSED( pDstTexture );
    NN_UNUSED( dstRegion );
    NN_UNUSED( pSrcTexture );
    NN_UNUSED( srcRegion );
    NN_UNUSED( copyFlags );

    // 本機能は未実装です。
    // D3D11における実装は可能です。（制限が付く可能性あり）
    // BlitのD3D11コマンドはありません。描画によるBlitが必要です。
    // つまり、描画不可のフォーマットへはBlitできません。
    NN_SDK_ASSERT( 0 );
}

void CommandBufferImpl< Target >::ClearBuffer( BufferImpl< Target >* pBuffer,
    ptrdiff_t offset, size_t size, uint32_t value ) NN_NOEXCEPT
{
    NN_UNUSED( pBuffer );
    NN_UNUSED( offset );
    NN_UNUSED( size );
    NN_UNUSED( value );

    // 本機能は未実装です。
    // D3D11における実装は可能です。（制限が付く可能性あり）
    // バッファクリアのD3D11コマンドはありません。（UnorderedAccessViewのみあります。）
    // Mapしてクリアする、もしくはCopyResourceによるクリアが必要です。
    NN_SDK_ASSERT( 0 );
}

void CommandBufferImpl< Target >::ClearColor( ColorTargetViewImpl< Target >* pColorTarget,
    float red, float green, float blue, float alpha, const TextureArrayRange* pArrayRange ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pColorTarget );
    NN_SDK_REQUIRES( IsInitialized( *pColorTarget ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( IsD3dHandleValid( pColorTarget->ToData()->pRenderTargetView ) );
    NN_UNUSED( pArrayRange );

    ColorTargetViewImpl< Target >::DataType& colorTargetData = pColorTarget->ToData();

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );
    ID3D11RenderTargetView* pRenderTargetView = static_cast< ID3D11RenderTargetView* >( colorTargetData.pRenderTargetView );
    const FLOAT clearColor[4] = { red, green, blue, alpha };

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->ClearRenderTargetView( pRenderTargetView, clearColor ) );
}

void CommandBufferImpl< Target >::ClearColorTarget( ColorTargetViewImpl< Target >* pColorTarget,
    const ClearColorValue& clearColor, const TextureArrayRange* pArrayRange ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pColorTarget );
    NN_SDK_REQUIRES( IsInitialized( *pColorTarget ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( IsD3dHandleValid( pColorTarget->ToData()->pRenderTargetView ) );
    NN_UNUSED( pArrayRange );

    ColorTargetViewImpl< Target >::DataType& colorTargetData = pColorTarget->ToData();

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );
    ID3D11RenderTargetView* pRenderTargetView = static_cast< ID3D11RenderTargetView* >( colorTargetData.pRenderTargetView );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->ClearRenderTargetView( pRenderTargetView, clearColor.valueFloat ) );
}

void CommandBufferImpl< Target >::ClearDepthStencil( DepthStencilViewImpl< Target >* pDepthStencil,
    float depth, int stencil, DepthStencilClearMode clearMode, const TextureArrayRange* pArrayRange ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDepthStencil );
    NN_SDK_REQUIRES( IsInitialized( *pDepthStencil ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( IsD3dHandleValid( pDepthStencil->ToData()->pDepthStencilView ) );
    NN_UNUSED( pArrayRange );

    DepthStencilViewImpl< Target >::DataType& depthStencilData = pDepthStencil->ToData();

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );
    ID3D11DepthStencilView* pDepthStencilView = static_cast< ID3D11DepthStencilView* >( depthStencilData.pDepthStencilView );

    UINT clearFlags = static_cast< UINT >( D3d::GetDepthStencilClearMode( clearMode ) );
    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->ClearDepthStencilView( pDepthStencilView, clearFlags, depth, static_cast< UINT8 >( stencil ) ) );
}

void CommandBufferImpl< Target >::Resolve( TextureImpl< Target >* pDstTexture, int dstMipLevel, int dstStartArrayIndex,
    const ColorTargetViewImpl< Target >* pSrcColorTarget, const TextureArrayRange* pSrcArrayRange ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstTexture );
    NN_SDK_REQUIRES_NOT_NULL( pSrcColorTarget );
    NN_SDK_REQUIRES( IsInitialized( *pDstTexture ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcColorTarget ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    ID3D11Resource* pSrcResource;
    static_cast< ID3D11RenderTargetView* >( pSrcColorTarget->ToData()->pRenderTargetView )->GetResource( &pSrcResource );

    ID3D11Resource* pDstResource = static_cast< ID3D11Resource* >( pDstTexture->ToData()->pTexture );

    DXGI_FORMAT format = D3d::GetTextureFormat( static_cast< ImageFormat >( pDstTexture->ToData()->imageFormat ) ).dxgiFormat;

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );

    int arrayLength = 1;
    int srcBaseArrayIndex = 0;
    if ( pSrcArrayRange )
    {
        arrayLength = pSrcArrayRange->GetArrayLength();
        srcBaseArrayIndex = pSrcArrayRange->GetBaseArrayIndex();
    }

    for ( int idxArray = 0; idxArray < arrayLength; ++idxArray )
    {
        UINT dstSubresource = D3D11CalcSubresource( dstMipLevel, dstStartArrayIndex + idxArray, D3d::GetTextureMipLevels( pDstResource ) );
        UINT srcSubresource = D3D11CalcSubresource( 0, srcBaseArrayIndex + idxArray, D3d::GetTextureMipLevels( pSrcResource ) );

        NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->ResolveSubresource(
            pDstResource, dstSubresource, pSrcResource, srcSubresource, format ) );
    }
}

void CommandBufferImpl< Target >::FlushMemory( int gpuAccessFlags ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_UNUSED( gpuAccessFlags );
}

void CommandBufferImpl< Target >::InvalidateMemory( int gpuAccessFlags ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_UNUSED( gpuAccessFlags );
}

void CommandBufferImpl< Target >::CallCommandBuffer( const CommandBufferImpl< Target >* pNestedCommandBuffer ) NN_NOEXCEPT
{
    NN_UNUSED( pNestedCommandBuffer );

    // 本機能は未実装です。
    // DeferredContextの機能に、CallCommandBuffer相当の機能はありません。
    // また、コンテキストのステートは継承されません。
    // そのため、DeferredContextによる機能の実現は難しいと考えられます。
    NN_SDK_ASSERT( 0 );
}

void CommandBufferImpl< Target >::CopyCommandBuffer( const CommandBufferImpl< Target >* pNestedCommandBuffer ) NN_NOEXCEPT
{
    NN_UNUSED( pNestedCommandBuffer );

    // 本機能は未実装です。
    // DeferredContextの機能に、CopyCommandBuffer相当の機能はありません。
    // また、コンテキストのステートは継承されません。
    // そのため、DeferredContextによる機能の実現は難しいと考えられます。
    NN_SDK_ASSERT( 0 );
}

void CommandBufferImpl< Target >::SetBufferStateTransition(
    BufferImpl< Target >* pBuffer, int, int, int, int ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES_NOT_NULL( pBuffer );
    NN_UNUSED( pBuffer );
}

void CommandBufferImpl< Target >::SetTextureStateTransition( TextureImpl< Target >* pTexture,
    const TextureSubresourceRange*, int, int, int, int ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES_NOT_NULL( pTexture );
    NN_UNUSED( pTexture );
}

void CommandBufferImpl< Target >::SetDescriptorPool( const DescriptorPoolImpl< Target >* ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
}

void CommandBufferImpl< Target >::SetRootSignature( PipelineType,
    RootSignatureImpl< Target >* pRootSignatureValue ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( !pRootSignatureValue || pRootSignatureValue->ToData()->pWorkMemory );
    this->pGfxRootSignature = pRootSignatureValue;
}

void CommandBufferImpl< Target >::SetRootBufferDescriptorTable( PipelineType pipelineType,
    int indexDescriptorTable, const DescriptorSlot& startBufferDescriptorSlot ) NN_NOEXCEPT
{
    return nn::gfx::detail::SetRootBufferDescriptorTable( this, pipelineType, indexDescriptorTable,
        startBufferDescriptorSlot, DescriptorPoolImpl< Target >::GetDescriptorSlotIncrementSize(
        this->pGfxDevice, DescriptorPoolType_BufferView ) );
}

void CommandBufferImpl< Target >::SetRootTextureAndSamplerDescriptorTable(
    PipelineType pipelineType, int indexDescriptorTable, const DescriptorSlot& startTextureDescriptorSlot,
    const DescriptorSlot& startSamplerDescriptorSlot ) NN_NOEXCEPT
{
    return nn::gfx::detail::SetRootTextureAndSamplerDescriptorTable( this, pipelineType,
        indexDescriptorTable, startTextureDescriptorSlot, startSamplerDescriptorSlot,
        DescriptorPoolImpl< Target >::GetDescriptorSlotIncrementSize( this->pGfxDevice, DescriptorPoolType_TextureView ),
        DescriptorPoolImpl< Target >::GetDescriptorSlotIncrementSize( this->pGfxDevice, DescriptorPoolType_Sampler ) );
}

void CommandBufferImpl< Target >::SetRootConstantBuffer( PipelineType pipelineType,
    int indexDynamicDescriptor, const GpuAddress& constantBufferAddress, size_t size ) NN_NOEXCEPT
{
    return nn::gfx::detail::SetRootConstantBuffer( this,
        pipelineType, indexDynamicDescriptor, constantBufferAddress, size );
}

void CommandBufferImpl< Target >::SetRootUnorderedAccessBuffer( PipelineType pipelineType,
    int indexDynamicDescriptor, const GpuAddress& unorderedAccessBufferAddress, size_t size ) NN_NOEXCEPT
{
    return nn::gfx::detail::SetRootUnorderedAccessBuffer( this,
        pipelineType, indexDynamicDescriptor, unorderedAccessBufferAddress, size );
}

void CommandBufferImpl< Target >::SetRootTextureAndSampler( PipelineType pipelineType, int indexDynamicDescriptor,
    const TextureViewImpl< Target >* pTextureView, const SamplerImpl< Target >* pSampler ) NN_NOEXCEPT
{
    return nn::gfx::detail::SetRootTextureAndSampler( this,
        pipelineType, indexDynamicDescriptor, pTextureView, pSampler );
}

void CommandBufferImpl< Target >::BeginQuery( QueryTarget target ) NN_NOEXCEPT
{
    NN_UNUSED( target );

    // 本機能は未実装です。
    // 実装可能性は未調査です。
    NN_SDK_ASSERT( 0 );
}

void CommandBufferImpl< Target >::EndQuery( const GpuAddress& dstBufferAddress,
    QueryTarget target ) NN_NOEXCEPT
{
    NN_UNUSED( dstBufferAddress );
    NN_UNUSED( target );

    // 本機能は未実装です。
    // 実装可能性は未調査です。
    NN_SDK_ASSERT( 0 );
}

void CommandBufferImpl< Target >::WriteTimestamp( const GpuAddress& dstBufferAddress ) NN_NOEXCEPT
{
    NN_UNUSED( dstBufferAddress );

    // 本機能は未実装です。
    // 実装可能性は未調査です。
    NN_SDK_ASSERT( 0 );
}

void CommandBufferImpl< Target >::SetDepthBounds( float minDepthBounds, float maxDepthBounds ) NN_NOEXCEPT
{
    NN_UNUSED( minDepthBounds );
    NN_UNUSED( maxDepthBounds );

    // 本機能は未実装です。
    // D3D11において該当機能はなく、実装はできません。
    NN_SDK_ASSERT( 0 );
}

void CommandBufferImpl< Target >::SetLineWidth( float lineWidth ) NN_NOEXCEPT
{
    NN_UNUSED( lineWidth );

    // 本機能は未実装です。
    // D3D11において該当機能はなく、実装はできません。
    // ジオメトリシェーダによるユーザ実装が必要です。
    // D3DXの線描がの機能に、線幅指定が可能ですが、同機能は2D描画空間での単色描画のみです。
    NN_SDK_ASSERT( 0 );
}

void CommandBufferImpl< Target >::SetViewports( int firstViewport,
    int viewportCount, const ViewportStateInfo* pViewports ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( firstViewport == 0 );
    NN_SDK_ASSERT( viewportCount > 0 && viewportCount <= D3D11_VIEWPORT_AND_SCISSORRECT_MAX_INDEX );
    NN_UNUSED( firstViewport );

    D3D11_VIEWPORT pD3dViewports[ D3D11_VIEWPORT_AND_SCISSORRECT_MAX_INDEX ];

    for( int idxViewport = 0; idxViewport < viewportCount; ++idxViewport )
    {
        const ViewportStateInfo& srcViewport = pViewports[ idxViewport ];
        pD3dViewports[ idxViewport ].TopLeftX = srcViewport.GetOriginX();
        pD3dViewports[ idxViewport ].TopLeftY = srcViewport.GetOriginY();
        pD3dViewports[ idxViewport ].Width = srcViewport.GetWidth();
        pD3dViewports[ idxViewport ].Height = srcViewport.GetHeight();
        pD3dViewports[ idxViewport ].MinDepth = srcViewport.GetMinDepth();
        pD3dViewports[ idxViewport ].MaxDepth = srcViewport.GetMaxDepth();
    }

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->RSSetViewports( viewportCount, pD3dViewports ) );
}

void CommandBufferImpl< Target >::SetScissors( int firstScissor,
    int scissorCount, const ScissorStateInfo* pScissors ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( firstScissor == 0 );
    NN_SDK_ASSERT( scissorCount > 0 && scissorCount <= D3D11_VIEWPORT_AND_SCISSORRECT_MAX_INDEX );
    NN_UNUSED( firstScissor );

    D3D11_RECT pD3dScissors[ D3D11_VIEWPORT_AND_SCISSORRECT_MAX_INDEX ];

    for( int idxScissor = 0; idxScissor < scissorCount; ++idxScissor )
    {
        const ScissorStateInfo& srcScissor = pScissors[ idxScissor ];
        pD3dScissors[ idxScissor ].left = srcScissor.GetOriginX();
        pD3dScissors[ idxScissor ].top = srcScissor.GetOriginY();
        pD3dScissors[ idxScissor ].bottom = srcScissor.GetOriginY() + srcScissor.GetHeight();
        pD3dScissors[ idxScissor ].right = srcScissor.GetOriginX() + srcScissor.GetWidth();
    }

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->RSSetScissorRects( scissorCount, pD3dScissors ) );
}

// 以下はルートシグネチャを使わない場合

void CommandBufferImpl< Target >::SetConstantBuffer( int slot,
    ShaderStage stage, const DescriptorSlot& constantBufferDescriptor ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    const uint64_t* pDescriptor = ToPtr< const uint64_t >( constantBufferDescriptor );

    NN_SDK_ASSERT( pDescriptor[ 0 ] == 0 ); // offset未対応

    ID3D11Buffer* pD3dConstBuffer = reinterpret_cast< ID3D11Buffer* >( pDescriptor[ 1 ] );

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );

    switch ( stage )
    {
    case ShaderStage_Vertex:
        {
            NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->VSSetConstantBuffers( slot, 1, &pD3dConstBuffer ) );
        }
        break;
    case ShaderStage_Pixel:
        {
            NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->PSSetConstantBuffers( slot, 1, &pD3dConstBuffer ) );
        }
        break;
    case ShaderStage_Geometry:
        {
            NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->GSSetConstantBuffers( slot, 1, &pD3dConstBuffer ) );
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void CommandBufferImpl< Target >::SetUnorderedAccessBuffer( int slot,
    ShaderStage stage, const DescriptorSlot& unorderedAccessBufferDescriptor ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );

    const uint64_t* pDescriptor = ToPtr< const uint64_t >( unorderedAccessBufferDescriptor );

    NN_SDK_ASSERT( pDescriptor[ 0 ] == 0 ); // offset未対応

    ID3D11UnorderedAccessView* pD3dUnorderedAccessView = reinterpret_cast< ID3D11UnorderedAccessView* >( pDescriptor[ 2 ] );

    switch ( stage )
    {
    case ShaderStage_Compute:
        {
            ID3D11UnorderedAccessView* ppUAViewnullptr[1] = { NULL };
            NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->CSSetUnorderedAccessViews(
                slot, 1, pD3dUnorderedAccessView != NULL ? &pD3dUnorderedAccessView : ppUAViewnullptr, NULL ) );
        }
        break;
    case ShaderStage_Vertex:
    case ShaderStage_Pixel:
    case ShaderStage_Hull:
    case ShaderStage_Domain:
    case ShaderStage_Geometry:
    default: NN_UNEXPECTED_DEFAULT;
    }

    return;
}

void CommandBufferImpl< Target >::SetTextureAndSampler( int slot, ShaderStage stage,
    const DescriptorSlot& textureDescriptor, const DescriptorSlot& samplerDescriptor ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( slot >= 0 );
    const uint64_t* pTextureDescriptor = ToPtr< const uint64_t >( textureDescriptor );
    ID3D11ShaderResourceView* pD3dShaderResourceView =
        textureDescriptor.IsValid() ? reinterpret_cast< ID3D11ShaderResourceView* >( pTextureDescriptor[ 0 ] ) : NULL;

    const uint64_t* pSamplerDescriptor = ToPtr< const uint64_t >( samplerDescriptor );
    ID3D11SamplerState* pD3dSamplerState =
        samplerDescriptor.IsValid() ? reinterpret_cast< ID3D11SamplerState* >( pSamplerDescriptor[ 0 ] ) : NULL;

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );

    switch ( stage )
    {
    case ShaderStage_Vertex:
        {
            NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->VSSetShaderResources( slot, 1, &pD3dShaderResourceView ) );
            NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->VSSetSamplers( slot, 1, &pD3dSamplerState ) );
        }
        break;
    case ShaderStage_Pixel:
        {
            NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->PSSetShaderResources( slot, 1, &pD3dShaderResourceView ) );
            NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->PSSetSamplers( slot, 1, &pD3dSamplerState ) );
        }
        break;
    case ShaderStage_Geometry:
        {
            NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->GSSetShaderResources( slot, 1, &pD3dShaderResourceView ) );
            NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->GSSetSamplers( slot, 1, &pD3dSamplerState ) );
        }
        break;
    case ShaderStage_Hull:
        {
            NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->HSSetShaderResources( slot, 1, &pD3dShaderResourceView ) );
            NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->HSSetSamplers( slot, 1, &pD3dSamplerState ) );
        }
        break;
    case ShaderStage_Domain:
        {
            NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->DSSetShaderResources( slot, 1, &pD3dShaderResourceView ) );
            NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->DSSetSamplers( slot, 1, &pD3dSamplerState ) );
        }
        break;
    case ShaderStage_Compute:
        {
            NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->CSSetShaderResources( slot, 1, &pD3dShaderResourceView ) );
            NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->CSSetSamplers( slot, 1, &pD3dSamplerState ) );
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void CommandBufferImpl< Target >::SetTexture( int slot, ShaderStage stage,
    const DescriptorSlot& textureDescriptor ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( slot >= 0 );
    const uint64_t* pTextureDescriptor = ToPtr< const uint64_t >( textureDescriptor );
    ID3D11ShaderResourceView* pD3dShaderResourceView = reinterpret_cast< ID3D11ShaderResourceView* >( pTextureDescriptor[ 0 ] );
    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );

    switch ( stage )
    {
    case ShaderStage_Vertex:
        {
            NN_GFX_CALL_D3D_FUNCTION(
                pD3dDeferredContext->VSSetShaderResources( slot, 1, &pD3dShaderResourceView ) );
        }
        break;
    case ShaderStage_Pixel:
        {
            NN_GFX_CALL_D3D_FUNCTION(
                pD3dDeferredContext->PSSetShaderResources( slot, 1, &pD3dShaderResourceView ) );
        }
        break;
    case ShaderStage_Geometry:
        {
            NN_GFX_CALL_D3D_FUNCTION(
                pD3dDeferredContext->GSSetShaderResources( slot, 1, &pD3dShaderResourceView ) );
        }
        break;
    case ShaderStage_Hull:
        {
            NN_GFX_CALL_D3D_FUNCTION(
                pD3dDeferredContext->HSSetShaderResources( slot, 1, &pD3dShaderResourceView ) );
        }
        break;
    case ShaderStage_Domain:
        {
            NN_GFX_CALL_D3D_FUNCTION(
                pD3dDeferredContext->DSSetShaderResources( slot, 1, &pD3dShaderResourceView ) );
        }
        break;
    case ShaderStage_Compute:
        {
            NN_GFX_CALL_D3D_FUNCTION(
                pD3dDeferredContext->CSSetShaderResources( slot, 1, &pD3dShaderResourceView ) );
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void CommandBufferImpl< Target >::SetImage( int slot, ShaderStage,
    const DescriptorSlot& imageDescriptor ) NN_NOEXCEPT
{
    NN_UNUSED( slot );
    NN_UNUSED( imageDescriptor );

    // 本機能は未実装です。
    // 実装可能性は未調査です。
    NN_SDK_ASSERT( 0 );
}

// 以下はデスクリプタプールを使わない場合

void CommandBufferImpl< Target >::SetConstantBuffer( int slot, ShaderStage stage,
    const GpuAddress& constantBufferAddress, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( constantBufferAddress.ToData()->impl );
    NN_UNUSED( size );
    NN_SDK_ASSERT( constantBufferAddress.ToData()->value == 0 ); // offset未対応。Windows8からVSSetConstantBuffers1によりoffset対応が可能な見込み。

    ID3D11Buffer* pD3dConstBuffer = D3d::GetBufferResource( constantBufferAddress.ToData()->impl );
    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );

    switch ( stage )
    {
    case ShaderStage_Vertex:
        {
            NN_GFX_CALL_D3D_FUNCTION(
                pD3dDeferredContext->VSSetConstantBuffers( slot, 1, &pD3dConstBuffer ) );
        }
        break;
    case ShaderStage_Hull:
        {
            NN_GFX_CALL_D3D_FUNCTION(
                pD3dDeferredContext->HSSetConstantBuffers( slot, 1, &pD3dConstBuffer ) );
        }
        break;
    case ShaderStage_Domain:
        {
            NN_GFX_CALL_D3D_FUNCTION(
                pD3dDeferredContext->DSSetConstantBuffers( slot, 1, &pD3dConstBuffer ) );
        }
        break;
    case ShaderStage_Pixel:
        {
            NN_GFX_CALL_D3D_FUNCTION(
                pD3dDeferredContext->PSSetConstantBuffers( slot, 1, &pD3dConstBuffer ) );
        }
        break;
    case ShaderStage_Geometry:
        {
            NN_GFX_CALL_D3D_FUNCTION(
                pD3dDeferredContext->GSSetConstantBuffers( slot, 1, &pD3dConstBuffer ) );
        }
        break;
    case ShaderStage_Compute:
        {
            NN_GFX_CALL_D3D_FUNCTION(
                pD3dDeferredContext->CSSetConstantBuffers( slot, 1, &pD3dConstBuffer ) );
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void CommandBufferImpl< Target >::SetUnorderedAccessBuffer( int slot, ShaderStage stage,
    const GpuAddress& unorderedAccessBufferAddress, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( unorderedAccessBufferAddress.ToData()->value == 0 ); // offset未対応
    NN_UNUSED( size );

// alignmentの制約はないものと見られるため一先ずASSERTは無効化。
//    NN_SDK_ASSERT( nn::util::is_aligned( unorderedAccessBufferAddress.ToData()->value,
//        this->pGfxDevice->ToData()->alignmentUnorderedAccessBuffer ) );

// Bufferのデタッチ機能が必要なため、gfxの仕様にはないがNULLのGpuAddressの指定をデタッチ機能とする。
//    NN_SDK_ASSERT( IsD3dHandleValid( reinterpret_cast< D3dHandle >( unorderedAccessBufferAddress.ToData()->value ) ) );

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >(
        this->pDeferredContext );
    ID3D11UnorderedAccessView* pD3dUnorderedAccessView = D3d::GetBufferResourceView(
        unorderedAccessBufferAddress.ToData()->impl );

    switch ( stage )
    {
    case ShaderStage_Compute:
        {
            NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->CSSetUnorderedAccessViews(
                slot, 1, &pD3dUnorderedAccessView, NULL ) );
        }
        break;
    case ShaderStage_Vertex:
    case ShaderStage_Pixel:
    case ShaderStage_Hull:
    case ShaderStage_Domain:
    case ShaderStage_Geometry:
    default: NN_UNEXPECTED_DEFAULT;
    }

    return;
}

void CommandBufferImpl< Target >::SetTextureAndSampler( int slot, ShaderStage stage,
    const TextureViewImpl< Target >* pTextureView, const SamplerImpl< Target >* pSampler ) NN_NOEXCEPT
{
    NN_UNUSED( stage );
    NN_SDK_REQUIRES_NOT_NULL( pTextureView );
    NN_SDK_REQUIRES( IsInitialized( *pTextureView ) );
    NN_SDK_ASSERT( IsD3dHandleValid( pTextureView->ToData()->pShaderResourceView ) );
    NN_SDK_REQUIRES_NOT_NULL( pSampler );
    NN_SDK_REQUIRES( IsInitialized( *pSampler ) );
    NN_SDK_ASSERT( IsD3dHandleValid( pSampler->ToData()->pSampler ) );
    NN_SDK_ASSERT( slot >= 0 );
    NN_SDK_ASSERT( this->state == State_Begun );

    ID3D11DeviceContext* pD3dDeferredContext =
        static_cast< ID3D11DeviceContext* >( this->pDeferredContext );
    ID3D11ShaderResourceView* pD3dShaderResourceView =
        static_cast< ID3D11ShaderResourceView* >( pTextureView->ToData()->pShaderResourceView );
    ID3D11SamplerState* pD3dSamplerState =
        static_cast< ID3D11SamplerState* >( pSampler->ToData()->pSampler );

    switch ( stage )
    {
    case ShaderStage_Vertex:
        {
            NN_GFX_CALL_D3D_FUNCTION(
                pD3dDeferredContext->VSSetShaderResources( slot, 1, &pD3dShaderResourceView ) );
            NN_GFX_CALL_D3D_FUNCTION(
                pD3dDeferredContext->VSSetSamplers( slot, 1, &pD3dSamplerState ) );
        }
        break;
    case ShaderStage_Pixel:
        {
            NN_GFX_CALL_D3D_FUNCTION(
                pD3dDeferredContext->PSSetShaderResources( slot, 1, &pD3dShaderResourceView ) );
            NN_GFX_CALL_D3D_FUNCTION(
                pD3dDeferredContext->PSSetSamplers( slot, 1, &pD3dSamplerState ) );
        }
        break;
    case ShaderStage_Geometry:
        {
            NN_GFX_CALL_D3D_FUNCTION(
                pD3dDeferredContext->GSSetShaderResources( slot, 1, &pD3dShaderResourceView ) );
            NN_GFX_CALL_D3D_FUNCTION(
                pD3dDeferredContext->GSSetSamplers( slot, 1, &pD3dSamplerState ) );
        }
        break;
    case ShaderStage_Hull:
        {
            NN_GFX_CALL_D3D_FUNCTION(
                pD3dDeferredContext->HSSetShaderResources( slot, 1, &pD3dShaderResourceView ) );
            NN_GFX_CALL_D3D_FUNCTION(
                pD3dDeferredContext->HSSetSamplers( slot, 1, &pD3dSamplerState ) );
        }
        break;
    case ShaderStage_Domain:
        {
            NN_GFX_CALL_D3D_FUNCTION(
                pD3dDeferredContext->DSSetShaderResources( slot, 1, &pD3dShaderResourceView ) );
            NN_GFX_CALL_D3D_FUNCTION(
                pD3dDeferredContext->DSSetSamplers( slot, 1, &pD3dSamplerState ) );
        }
        break;
    case ShaderStage_Compute:
        {
            NN_GFX_CALL_D3D_FUNCTION(
                pD3dDeferredContext->CSSetShaderResources( slot, 1, &pD3dShaderResourceView ) );
            NN_GFX_CALL_D3D_FUNCTION(
                pD3dDeferredContext->CSSetSamplers( slot, 1, &pD3dSamplerState ) );
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void CommandBufferImpl< Target >::SetImage(
    int slot, ShaderStage, const TextureViewImpl< Target >* pImage ) NN_NOEXCEPT
{
    NN_UNUSED( slot );
    NN_UNUSED( pImage );

    // 本機能は未実装です。
    // 実装可能性は未調査です。
    NN_SDK_ASSERT( 0 );
}

// 以下はパイプラインを使わない場合
void CommandBufferImpl< Target >::SetShader( const ShaderImpl< Target >* pShader, int stageBits ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pShader == NULL || pShader->ToData()->state ==
        ShaderImpl< Target >::DataType::State_Initialized );
    NN_SDK_REQUIRES( this->state == State_Begun );

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );

    if ( stageBits & ShaderStageBit_Vertex )
    {
        NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->VSSetShader(
            pShader ? static_cast< ID3D11VertexShader* >(
            pShader->ToData()->pShaderProgram[ ShaderStage_Vertex ] ) : NULL, NULL, 0 ) );
    }
    if ( stageBits & ShaderStageBit_Hull )
    {
        NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->HSSetShader(
            pShader ? static_cast< ID3D11HullShader* >(
            pShader->ToData()->pShaderProgram[ ShaderStage_Hull ] ) : NULL, NULL, 0 ) );
    }
    if ( stageBits & ShaderStageBit_Domain )
    {
        NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->DSSetShader(
            pShader ? static_cast< ID3D11DomainShader* >(
            pShader->ToData()->pShaderProgram[ ShaderStage_Domain ] ) : NULL, NULL, 0 ) );
    }
    if ( stageBits & ShaderStageBit_Geometry )
    {
        NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->GSSetShader(
            pShader ? static_cast< ID3D11GeometryShader* >(
            pShader->ToData()->pShaderProgram[ ShaderStage_Geometry ] ) : NULL, NULL, 0 ) );
    }
    if ( stageBits & ShaderStageBit_Pixel )
    {
        NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->PSSetShader(
            pShader ? static_cast< ID3D11PixelShader* >(
                pShader->ToData()->pShaderProgram[ ShaderStage_Pixel ] ) : NULL, NULL, 0 ) );
    }
    if ( stageBits & ShaderStageBit_Compute )
    {
        NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->CSSetShader(
            pShader ? static_cast< ID3D11ComputeShader* >(
                pShader->ToData()->pShaderProgram[ ShaderStage_Compute ] ) : NULL, NULL, 0 ) );
    }
}

void CommandBufferImpl< Target >::SetRasterizerState(
    const RasterizerStateImpl< Target >* pRasterizerState ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pRasterizerState );
    NN_SDK_REQUIRES( IsInitialized( *pRasterizerState ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( IsD3dHandleValid( pRasterizerState->ToData()->pRasterizerState ) );

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );
    ID3D11RasterizerState* pD3dRasterizerState = static_cast< ID3D11RasterizerState* >( pRasterizerState->ToData()->pRasterizerState );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->RSSetState( pD3dRasterizerState ) );

    this->sampleMask = pRasterizerState->ToData()->sampleMask;
}

void CommandBufferImpl< Target >::SetBlendState(
    const BlendStateImpl< Target >* pBlendState ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pBlendState );
    NN_SDK_REQUIRES( IsInitialized( *pBlendState ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( IsD3dHandleValid( pBlendState->ToData()->pBlendState ) );

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );
    ID3D11BlendState* pD3dBlendState = static_cast< ID3D11BlendState* >( pBlendState->ToData()->pBlendState );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->OMSetBlendState( pD3dBlendState, pBlendState->ToData()->blendColor,  this->sampleMask ) );
}

void CommandBufferImpl< Target >::SetDepthStencilState(
    const DepthStencilStateImpl< Target >* pDepthStencilState ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDepthStencilState );
    NN_SDK_REQUIRES( IsInitialized( *pDepthStencilState ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( IsD3dHandleValid( pDepthStencilState->ToData()->pDepthStencilState ) );

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );
    ID3D11DepthStencilState* pD3dDepthStencilState = static_cast< ID3D11DepthStencilState* >( pDepthStencilState->ToData()->pDepthStencilState );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->OMSetDepthStencilState( pD3dDepthStencilState, pDepthStencilState->ToData()->stencilRef ) );
}

void CommandBufferImpl< Target >::SetVertexState(
    const VertexStateImpl< Target >* pVertexState ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pVertexState );
    NN_SDK_REQUIRES( IsInitialized( *pVertexState ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( IsD3dHandleValid( pVertexState->ToData()->pInputLayout ) );

    ID3D11DeviceContext* pD3dDeferredContext = static_cast< ID3D11DeviceContext* >( this->pDeferredContext );
    ID3D11InputLayout* pInputLayout = static_cast< ID3D11InputLayout* >( pVertexState->ToData()->pInputLayout );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeferredContext->IASetInputLayout( pInputLayout ) );
}

void CommandBufferImpl< Target >::SetTessellationState(
    const TessellationStateImpl< Target >* pTessellationState ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pTessellationState );
    NN_SDK_REQUIRES( IsInitialized( *pTessellationState ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    this->patchControlPointCount = pTessellationState->ToData()->patchControlPointCount;
}

}
}
}

NN_PRAGMA_POP_WARNINGS
