﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_BitUtil.h>
#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_BufferInfo.h>
#include <nn/gfx/gfx_GpuAddress.h>

#include <nn/gfx/detail/gfx_Buffer-api.gx.2.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.gx.2.h>
#include <nn/gfx/detail/gfx_Misc.h>

#include "gfx_GxHelper.h"


namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationGx2 Target;

size_t BufferImpl< Target >::GetBufferAlignment( DeviceImpl< Target >*, const InfoType& info ) NN_NOEXCEPT
{
    // 多い順
    if( info.GetGpuAccessFlags() & GpuAccess_ConstantBuffer )
    {
        return GX2_UNIFORM_BLOCK_ALIGNMENT;
    }
    if( info.GetGpuAccessFlags() & GpuAccess_UnorderedAccessBuffer )
    {
        return GX2_EXPORT_BUFFER_ALIGNMENT;
    }
    if( info.GetGpuAccessFlags() & GpuAccess_IndirectBuffer )
    {
        return GX2_DISPATCH_PARAMS_ALIGNMENT;
    }
    if( info.GetGpuAccessFlags() & GpuAccess_VertexBuffer )
    {
        return GX2_VERTEX_BUFFER_ALIGNMENT;
    }
    if( info.GetGpuAccessFlags() & GpuAccess_IndexBuffer )
    {
        return GX2_INDEX_BUFFER_ALIGNMENT;
    }
    return 1;
}

BufferImpl< Target >::BufferImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

BufferImpl< Target >::~BufferImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void BufferImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info,
    MemoryPoolImpl< Target >* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_REQUIRES( pMemoryPool->ToData()->memorySize >= memoryPoolOffset + memoryPoolSize );
    NN_SDK_REQUIRES( memoryPoolSize >= info.GetSize() );
    NN_UNUSED( memoryPoolSize );

    this->pBuffer = nn::util::BytePtr( pMemoryPool->ToData()->pMemory, memoryPoolOffset ).Get();
    this->size = info.GetSize();
    int gpuAccessFlags = info.GetGpuAccessFlags();

    // Check alignments where possible
    NN_SDK_REQUIRES( nn::util::is_aligned( this->pBuffer.value, GetBufferAlignment( pDevice, info ) ) );

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void BufferImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );

    this->pBuffer = NULL;
    this->state = State_NotInitialized;
}

void* BufferImpl< Target >::Map() const NN_NOEXCEPT
{
    return this->pBuffer;
}

void BufferImpl< Target >::Unmap() const NN_NOEXCEPT
{
}

void BufferImpl< Target >::FlushMappedRange( ptrdiff_t offset, size_t size ) const NN_NOEXCEPT
{
    ::DCFlushRange( nn::util::BytePtr( this->pBuffer, offset ).Get(), size );
}

void BufferImpl< Target >::InvalidateMappedRange( ptrdiff_t offset, size_t size ) const NN_NOEXCEPT
{
    ::DCInvalidateRange( nn::util::BytePtr( this->pBuffer, offset ).Get(), size );
}

void BufferImpl< Target >::GetGpuAddress( GpuAddress* pOutGpuAddress ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );

    pOutGpuAddress->ToData()->value = reinterpret_cast< uintptr_t >( this->pBuffer.ptr );
}

BufferTextureViewImpl< Target >::BufferTextureViewImpl() NN_NOEXCEPT
{
}

BufferTextureViewImpl< Target >::~BufferTextureViewImpl() NN_NOEXCEPT
{
}

void BufferTextureViewImpl< Target >::Initialize(
    DeviceImpl< Target >*, const BufferTextureViewInfo& ) NN_NOEXCEPT
{
}

void BufferTextureViewImpl< Target >::Finalize( DeviceImpl< Target >* ) NN_NOEXCEPT
{
}

}
}
}
