﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_Variation-api.d3d.h>
#include <nn/gfx/gfx_BufferInfo.h>
#include <nn/gfx/gfx_GpuAddress.h>

#include <nn/gfx/detail/gfx_Misc.h>
#include <nn/gfx/detail/gfx_Buffer-api.d3d.11.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.d3d.11.h>
#include <nn/gfx/detail/gfx_Device-api.d3d.11.h>

#include "gfx_D3dHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationD3d11 Target;

size_t BufferImpl< Target >::GetBufferAlignment(
    DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_UNUSED( pDevice );
    NN_UNUSED( info );

    return 1;
}

BufferImpl< Target >::BufferImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

BufferImpl< Target >::~BufferImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void BufferImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info,
    MemoryPoolImpl< Target >* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_REQUIRES( !pMemoryPool || memoryPoolOffset >= 0 );
    NN_SDK_ASSERT( !pMemoryPool || pMemoryPool->ToData()->pMemory );
    NN_SDK_REQUIRES( !pMemoryPool || memoryPoolSize >= info.GetSize() );
    NN_UNUSED( memoryPoolSize );

    NN_SDK_ASSERT( !IsD3dHandleValid( this->pGpuBuffer ) );
    NN_SDK_ASSERT( !IsD3dHandleValid( this->pCpuBuffer ) );

    MemoryPoolImpl< Target >::DataType* pMemoryPoolData = pMemoryPool->ToData();
    NN_SDK_ASSERT( !pMemoryPoolData || pMemoryPoolData->memorySize >= memoryPoolOffset + info.GetSize() );

    int memoryPoolProperty = pMemoryPoolData ? pMemoryPoolData->memoryPoolProperty
        : ( MemoryPoolProperty_CpuCached | MemoryPoolProperty_GpuCached );

    D3D11_BUFFER_DESC desc;
    ZeroMemory( &desc, sizeof(desc) );
    desc.ByteWidth = static_cast< UINT >( info.GetSize() );
    desc.Usage = D3D11_USAGE_DEFAULT;

    int gpuAccessFlags = info.GetGpuAccessFlags();
    desc.BindFlags = D3d::GetBindFlag( gpuAccessFlags );

    if ( GpuAccess_UnorderedAccessBuffer & gpuAccessFlags )
    {
        desc.MiscFlags |= D3D11_RESOURCE_MISC_BUFFER_ALLOW_RAW_VIEWS;
    }
    if ( GpuAccess_ConstantBuffer & gpuAccessFlags )
    {
        desc.ByteWidth = nn::util::align_up( desc.ByteWidth, 16 );
    }

    if ( desc.BindFlags == 0 )
    {
        // 用途が定められていないバッファは、ステージングバッファとしておきます。
        desc.CPUAccessFlags = D3D11_CPU_ACCESS_READ | D3D11_CPU_ACCESS_WRITE;
        desc.Usage = D3D11_USAGE_STAGING;
    }

    void* pData = pMemoryPoolData ? nn::util::BytePtr(
        pMemoryPoolData->pMemory, memoryPoolOffset ).Get() : NULL;
    D3D11_SUBRESOURCE_DATA initialData;
    initialData.pSysMem = pData;

    DeviceImpl< Target >::DataType &deviceData = pDevice->ToData();
    ID3D11Device* pD3dDevice = static_cast< ID3D11Device* >( deviceData.renderingContext.hD3dDevice );

    ID3D11Buffer* pD3dGpuBuffer;
    HRESULT hResult = NN_GFX_CALL_D3D_FUNCTION(
        pD3dDevice->CreateBuffer( &desc, pData ? &initialData : NULL, &pD3dGpuBuffer ) );
    NN_SDK_ASSERT( SUCCEEDED( hResult ) );
    NN_SDK_ASSERT( IsD3dHandleValid( pD3dGpuBuffer ) );
    this->pGpuBuffer = pD3dGpuBuffer;

    this->pBufferView = NULL;
    if ( gpuAccessFlags & GpuAccess_UnorderedAccessBuffer )
    {
        D3D11_UNORDERED_ACCESS_VIEW_DESC uavDesc;
        ZeroMemory( &uavDesc, sizeof( uavDesc ) );
        uavDesc.ViewDimension = D3D11_UAV_DIMENSION_BUFFER;
        uavDesc.Format = DXGI_FORMAT_R32_TYPELESS;
        uavDesc.Buffer.FirstElement = 0;
        uavDesc.Buffer.Flags = D3D11_BUFFER_UAV_FLAG_RAW;
        uavDesc.Buffer.NumElements = static_cast< UINT >( info.GetSize() / 4 );

        ID3D11UnorderedAccessView* pD3dResourceView;
        hResult = NN_GFX_CALL_D3D_FUNCTION( pD3dDevice->CreateUnorderedAccessView( pD3dGpuBuffer, &uavDesc, &pD3dResourceView ) );
        NN_SDK_ASSERT( SUCCEEDED( hResult ) );
        NN_SDK_ASSERT( IsD3dHandleValid( pD3dResourceView ) );

        this->pBufferView = pD3dResourceView;
    }

    D3D11_BUFFER_DESC stagingBufferDesc;
    ZeroMemory( &stagingBufferDesc, sizeof( stagingBufferDesc ) );
    pD3dGpuBuffer->GetDesc( &stagingBufferDesc );
    stagingBufferDesc.CPUAccessFlags = D3D11_CPU_ACCESS_READ | D3D11_CPU_ACCESS_WRITE;
    stagingBufferDesc.Usage = D3D11_USAGE_STAGING;
    stagingBufferDesc.BindFlags = 0;
    stagingBufferDesc.MiscFlags = 0;

    ID3D11Buffer* pD3dCpuBuffer;
    hResult = NN_GFX_CALL_D3D_FUNCTION( pD3dDevice->CreateBuffer( &stagingBufferDesc, pData ? &initialData : NULL, &pD3dCpuBuffer ) );
    NN_SDK_ASSERT( SUCCEEDED( hResult ) );
    NN_SDK_ASSERT( IsD3dHandleValid( pD3dCpuBuffer ) );
    this->pCpuBuffer = pD3dCpuBuffer;

    this->flags.SetBit( Flag_Mappable, false );
    if( !( memoryPoolProperty & MemoryPoolProperty_CpuInvisible ) )
    {
        ID3D11DeviceContext* pD3dDeviceContext = static_cast< ID3D11DeviceContext* >( deviceData.renderingContext.hD3dDeviceContext );
        D3D11_MAPPED_SUBRESOURCE mappedSubResource;
        hResult = pD3dDeviceContext->Map( pD3dCpuBuffer, 0, D3D11_MAP_READ_WRITE, 0, &mappedSubResource );
        NN_SDK_ASSERT( SUCCEEDED( hResult ) );

        this->flags.SetBit( Flag_Mappable, true );
        pD3dDeviceContext->Unmap( pD3dCpuBuffer, 0 );
    }

    this->pGfxDevice = pDevice;
    this->gpuAccessFlag = info.GetGpuAccessFlags();
    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
} //NOLINT(impl/function_size)

void BufferImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );
    NN_UNUSED( pDevice );

    if( this->pGpuBuffer )
    {
        ID3D11Buffer* pD3dBuffer = static_cast< ID3D11Buffer* >( this->pGpuBuffer );
        NN_GFX_CALL_D3D_FUNCTION( pD3dBuffer->Release() );
        this->pGpuBuffer = NULL;
    }
    if( this->pCpuBuffer )
    {
        ID3D11Buffer* pD3dBuffer = static_cast< ID3D11Buffer* >( this->pCpuBuffer );
        NN_GFX_CALL_D3D_FUNCTION( pD3dBuffer->Release() );
        this->pCpuBuffer = NULL;
    }

    if ( this->pBufferView )
    {
        switch ( this->gpuAccessFlag )
        {
        case GpuAccess_UnorderedAccessBuffer:
            {
                ID3D11UnorderedAccessView* pD3dResourceView = static_cast< ID3D11UnorderedAccessView* >( this->pBufferView );
                NN_GFX_CALL_D3D_FUNCTION( pD3dResourceView->Release() );
            }
            break;
        default:
            break;
        }
        this->pBufferView = NULL;
    }

    this->pGfxDevice = NULL;
    this->flags.SetBit( Flag_Mappable, false );

    this->state = State_NotInitialized;
}

void* BufferImpl< Target >::Map() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( this->flags.GetBit( Flag_Mappable ) );

    // Mapする前に、GPUバッファとCPUバッファを同期しておきます。(GPUバッファをCPUバッファにコピー）
    DeviceImpl< Target >::DataType &data = this->pGfxDevice->ToData();
    ID3D11DeviceContext* pD3dDeviceContext = static_cast< ID3D11DeviceContext* >( data.renderingContext.hD3dDeviceContext );

    ID3D11Buffer* pD3dGpuBuffer = static_cast< ID3D11Buffer* >( this->pGpuBuffer );
    ID3D11Buffer* pD3dCpuBuffer = static_cast< ID3D11Buffer* >( this->pCpuBuffer );

    // GPUバッファをCPUバッファへコピーします。
    NN_GFX_CALL_D3D_FUNCTION( pD3dDeviceContext->CopyResource( pD3dCpuBuffer, pD3dGpuBuffer ) );

    D3D11_MAPPED_SUBRESOURCE mappedSubResource;
    HRESULT hResult = NN_GFX_CALL_D3D_FUNCTION( pD3dDeviceContext->Map( pD3dCpuBuffer, 0, D3D11_MAP_READ_WRITE, 0, &mappedSubResource ) );

    NN_SDK_ASSERT( SUCCEEDED( hResult ) );
    NN_UNUSED( hResult );

    return mappedSubResource.pData;
}

void BufferImpl< Target >::Unmap() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( this->flags.GetBit( Flag_Mappable ) );

    // Unmapする前に、GPUバッファとCPUバッファを同期しておきます。(CPUバッファをGPUバッファにコピー）
    DeviceImpl< Target >::DataType &data = this->pGfxDevice->ToData();
    ID3D11DeviceContext* pD3dDeviceContext = static_cast< ID3D11DeviceContext* >( data.renderingContext.hD3dDeviceContext );

    ID3D11Buffer* pD3dGpuBuffer = static_cast< ID3D11Buffer* >( this->pGpuBuffer );
    ID3D11Buffer* pD3dCpuBuffer = static_cast< ID3D11Buffer* >( this->pCpuBuffer );

    // CPUバッファをGPUバッファへコピーします。
    NN_GFX_CALL_D3D_FUNCTION( pD3dDeviceContext->CopyResource( pD3dGpuBuffer, pD3dCpuBuffer ) );

    NN_GFX_CALL_D3D_FUNCTION( pD3dDeviceContext->Unmap( pD3dCpuBuffer, 0 ) );

}

void BufferImpl< Target >::FlushMappedRange( ptrdiff_t offset, size_t size ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    NN_UNUSED( offset );
    NN_UNUSED( size );
}

void BufferImpl< Target >::InvalidateMappedRange( ptrdiff_t, size_t ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );
}

void BufferImpl< Target >::GetGpuAddress( GpuAddress* pOutGpuAddress ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    NN_SDK_REQUIRES( IsD3dHandleValid( this->pGpuBuffer ) );

    pOutGpuAddress->ToData()->value = 0;
    pOutGpuAddress->ToData()->impl = reinterpret_cast< uint64_t >( this );
}

size_t BufferTextureViewImpl< Target >::GetOffsetAlignment(
    DeviceImpl< Target >* pDevice, const InfoType& ) NN_NOEXCEPT
{
    NN_UNUSED( pDevice );

    return 1;
}

BufferTextureViewImpl< Target >::BufferTextureViewImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

BufferTextureViewImpl< Target >::~BufferTextureViewImpl() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void BufferTextureViewImpl< Target >::Initialize(
    DeviceImpl< Target >* pDevice, const BufferTextureViewInfo& info ) NN_NOEXCEPT
{
    NN_UNUSED( pDevice );
    NN_UNUSED( info );

    // 本機能は未実装です。
    // 実装可能性は未調査です。
    NN_SDK_ASSERT( 0 );
}

void BufferTextureViewImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_UNUSED( pDevice );

    // 本機能は未実装です。
    // 実装可能性は未調査です。
    NN_SDK_ASSERT( 0 );
}

}
}
}
