﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ens/detail/core/task/ens_TaskGetAuthToken.h>
#include <nn/ens/detail/core/ens_HttpRequestHeader.h>
#include <nn/ens/detail/util/ens_MessagePackReader.h>
#include <nn/ens/detail/util/ens_MessagePackWriter.h>
#include <nn/ens/detail/util/ens_ResponseStructureReader.h>

namespace nn { namespace ens { namespace detail { namespace core { namespace task {

TaskGetAuthToken::TaskGetAuthToken() NN_NOEXCEPT
    : m_pOutAuthToken(nullptr)
    , m_Credential(InvalidCredential)
{
}

TaskGetAuthToken::~TaskGetAuthToken() NN_NOEXCEPT
{
}

void TaskGetAuthToken::SetParameter(AuthToken* pOutAuthToken, const Credential& credential) NN_NOEXCEPT
{
    m_pOutAuthToken = pOutAuthToken;
    m_Credential = credential;
}

nn::Result TaskGetAuthToken::Main() NN_NOEXCEPT
{
    NN_RESULT_DO(StepGet());

    NN_RESULT_SUCCESS;
}

nn::Result TaskGetAuthToken::StepGet() NN_NOEXCEPT
{
    NN_DETAIL_ENS_SCOPED_ACQUIRE_HTTP_CONNECTION(pCurl);

    char url[256] = {};
    detail::core::MakeUrl(url, sizeof (url), "api", "/api/v1/auth_token");

    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_CUSTOMREQUEST, "POST"));
    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_URL, url));
    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_POSTFIELDS, ""));

    detail::core::HttpRequestHeader header;

    // 入力パラメータへのアクセス
    {
        NN_UTIL_LOCK_GUARD(GetParameterLock());

        NN_RESULT_THROW_UNLESS(IsAsyncContextRegistered(), ResultInvalidState());

        NN_RESULT_DO(header.Add("Authorization", "Bearer %.*s",
            static_cast<int>(nn::account::NetworkServiceAccountIdTokenLengthMax), GetNetworkServiceAccountIdToken()));
        NN_RESULT_DO(header.Add("Content-Type", "application/x-msgpack"));

        detail::util::MessagePackWriter writer;

        writer.SetBuffer(GetRequestBuffer(), GetRequestBufferSize());

        NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteMap(2));
        {
            NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteKey("id"));
            NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteValue(m_Credential.userId.value));

            NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteKey("password"));
            NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteValue(m_Credential._password));
        }

        NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_POSTFIELDS, writer.GetBuffer()));
        NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_POSTFIELDSIZE, writer.GetWrittenSize()));
    }

    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_HTTPHEADER, header.Get()));

    NN_RESULT_DO(Perform());

    NN_DETAIL_ENS_DUMP_RESPONSE(detail::util::MessagePackReader, GetResponse(), GetResponseSize());

    detail::util::ResponseStructureReader<2> reader;

    nn::time::PosixTime expireAt = {};

    reader.Add("$.token",
        m_pOutAuthToken->value, sizeof (m_pOutAuthToken->value));
    reader.Add("$.expire_at",
        &expireAt.value);

    NN_RESULT_THROW_UNLESS(reader.Read<detail::util::MessagePackReader>(GetResponse(), GetResponseSize()), ResultInvalidResponse());

    NN_RESULT_SUCCESS;
}

}}}}}
