﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ens/detail/core/ens_ServiceThread.h>
#include <nn/ens/detail/core/ens_AuthenticationTask.h>
#include <nn/ens/detail/core/ens_AuthTokenCacheManager.h>
#include <nn/ens/detail/core/ens_HttpConnectionProvider.h>
#include <nn/ens/detail/core/ens_HttpTask.h>
#include <nn/ens/detail/core/ens_TaskManager.h>

namespace nn { namespace ens { namespace detail { namespace core {

namespace
{
    struct ExecutionContext : public nn::util::IntrusiveListBaseNode<ExecutionContext>
    {
        nn::os::ThreadType* pThread;
        Task* pRunningTask;
    };
}

namespace
{
    nn::os::SdkMutex g_MutexForExecution;
    nn::os::SdkMutex g_MutexForAuthTokenCacheManager;

    TaskManager g_TaskManager;

    nn::util::IntrusiveList<ExecutionContext,
        nn::util::IntrusiveListBaseNodeTraits<ExecutionContext>> g_ExecutionContextList;

    AuthTokenCacheManager<8> g_AuthTokenCacheManager;
}

namespace
{
    void RegisterExecutionContext(ExecutionContext* pContext) NN_NOEXCEPT
    {
        pContext->pThread = nn::os::GetCurrentThread();
        pContext->pRunningTask = nullptr;

        NN_UTIL_LOCK_GUARD(g_MutexForExecution);

        g_ExecutionContextList.push_back(*pContext);
    }

    void UnregisterExecutionContext(ExecutionContext* pContext) NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(g_MutexForExecution);

        for (auto it = g_ExecutionContextList.begin(); it != g_ExecutionContextList.end(); ++it)
        {
            if (&(*it) == pContext)
            {
                g_ExecutionContextList.erase(it);
                break;
            }
        }
    }

    void GetAuthTokenCache(AuthenticationTask* pTask) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(pTask->GetCredential().IsValid());

        AuthToken token = {};

        NN_UTIL_LOCK_GUARD(g_MutexForAuthTokenCacheManager);

        if (g_AuthTokenCacheManager.Get(&token, pTask->GetCredential().userId))
        {
            pTask->SetAuthToken(token);
        }
    }

    void UpdateAuthTokenCache(AuthenticationTask* pTask) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(pTask->GetCredential().IsValid());

        NN_UTIL_LOCK_GUARD(g_MutexForAuthTokenCacheManager);

        g_AuthTokenCacheManager.Set(pTask->GetCredential().userId, pTask->GetAuthToken());
    }

    void ServiceThread_CancelObserver(const TaskId& taskId) NN_NOEXCEPT
    {
        Task* pTask = g_TaskManager.Fetch(taskId);

        if (pTask)
        {
            pTask->Cancel();

            NN_DETAIL_ENS_DELETE_TASK(pTask);
        }

        {
            NN_UTIL_LOCK_GUARD(g_MutexForExecution);

            for (auto it = g_ExecutionContextList.begin(); it != g_ExecutionContextList.end(); ++it)
            {
                if (it->pRunningTask && it->pRunningTask->GetId() == taskId)
                {
                    it->pRunningTask->Cancel();
                    break;
                }
            }
        }
    }

    void ServiceThread_DestructObserver(const TaskId& taskId) NN_NOEXCEPT
    {
        Task* pTask = g_TaskManager.Fetch(taskId);

        if (pTask)
        {
            pTask->UnregisterAsyncContext();

            NN_DETAIL_ENS_DELETE_TASK(pTask);
        }

        // MEMO: 実行中処理のキャンセルを行うべきかどうかは要検討。
        {
            NN_UTIL_LOCK_GUARD(g_MutexForExecution);

            for (auto it = g_ExecutionContextList.begin(); it != g_ExecutionContextList.end(); ++it)
            {
                if (it->pRunningTask && it->pRunningTask->GetId() == taskId)
                {
                    it->pRunningTask->UnregisterAsyncContext();
                    break;
                }
            }
        }
    }
}

void StartServiceLoop(ServiceThreadContext& context) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(context.GetBuffer());

    ExecutionContext executionContext = {};

    RegisterExecutionContext(&executionContext);

    NN_UTIL_SCOPE_EXIT
    {
        UnregisterExecutionContext(&executionContext);
    };

    context.SetThread(nn::os::GetCurrentThread());

    context.GetStoppedEvent().Clear();
    context.GetStopEvent().Clear();
    context.GetStartedEvent().Signal();

    detail::util::EventHolder eventHolder(nn::os::EventClearMode_ManualClear);

    g_TaskManager.RegisterEvent(&eventHolder);

    NN_UTIL_SCOPE_EXIT
    {
        g_TaskManager.UnregisterEvent(&eventHolder);
    };

    HttpConnectionProvider connectionProvider;

    nn::os::TimerEvent connectionCleanupTimer(nn::os::EventClearMode_ManualClear);

    while (NN_STATIC_CONDITION(true))
    {
        nn::os::WaitAny(context.GetStopEvent().GetBase(), connectionCleanupTimer.GetBase(), eventHolder.Get().GetBase());

        if (context.GetStopEvent().TryWait())
        {
            break;
        }
        if (connectionCleanupTimer.TryWait())
        {
            connectionCleanupTimer.Clear();

            connectionProvider.GetConnection().Cleanup();
        }
        if (!eventHolder.Get().TryWait())
        {
            continue;
        }

        eventHolder.Get().Clear();

        while (NN_STATIC_CONDITION(true))
        {
            Task* pTask = nullptr;

            {
                NN_UTIL_LOCK_GUARD(g_MutexForExecution);

                pTask = g_TaskManager.Pop();

                if (!pTask)
                {
                    break;
                }

                executionContext.pRunningTask = pTask;
            }

            pTask->SetWorkBuffer(context.GetBuffer(), context.GetBufferSize());

            HttpTask* pHttpTask = DynamicCast<HttpTask*>(pTask);

            if (pHttpTask)
            {
                pHttpTask->SetHttpConnectionProvider(&connectionProvider);
            }

            AuthenticationTask* pAuthenticationTask = DynamicCast<AuthenticationTask*>(pTask);

            if (pAuthenticationTask)
            {
                GetAuthTokenCache(pAuthenticationTask);
            }

            pTask->Run();

            if (pAuthenticationTask)
            {
                UpdateAuthTokenCache(pAuthenticationTask);
            }

            {
                NN_UTIL_LOCK_GUARD(g_MutexForExecution);

                executionContext.pRunningTask = nullptr;
            }

            NN_DETAIL_ENS_DELETE_TASK(pTask);
        }

        connectionCleanupTimer.StartOneShot(LibrarySettings::GetHttpKeepAliveTimeout());
    }

    context.GetStartedEvent().Clear();
    context.GetStoppedEvent().Signal();
} // NOLINT(impl/function_size)

void StopServiceLoop(ServiceThreadContext& context) NN_NOEXCEPT
{
    context.GetStartedEvent().Wait();

    NN_SDK_ASSERT(!context.GetStopEvent().TryWait());

    context.GetStopEvent().Signal();

    {
        NN_UTIL_LOCK_GUARD(g_MutexForExecution);

        for (auto it = g_ExecutionContextList.begin(); it != g_ExecutionContextList.end(); ++it)
        {
            if (it->pThread == context.GetThread())
            {
                if (it->pRunningTask)
                {
                    it->pRunningTask->Cancel();
                }
                break;
            }
        }
    }

    context.GetStoppedEvent().Wait();
    context.SetThread(nullptr);
}

void RegisterTask(Task* pTask, AsyncContext* pContext) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pTask);
    NN_SDK_REQUIRES_NOT_NULL(pContext);

    pTask->RegisterAsyncContext(pContext);

    pContext->Register(pTask->GetId(), ServiceThread_CancelObserver, ServiceThread_DestructObserver);

    g_TaskManager.PushTail(pTask);
}

void UnregisterTaskAll() NN_NOEXCEPT
{
    while (NN_STATIC_CONDITION(true))
    {
        detail::core::Task* pTask = g_TaskManager.Pop();

        if (!pTask)
        {
            break;
        }

        pTask->Cancel();

        NN_DETAIL_ENS_DELETE_TASK(pTask);
    }
}

void SetAuthTokenToCache(const UserId& userId, const AuthToken& token) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(g_MutexForAuthTokenCacheManager);

    g_AuthTokenCacheManager.Set(userId, token);
}

void ClearAuthTokenCache() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(g_MutexForAuthTokenCacheManager);

    g_AuthTokenCacheManager.Clear();
}

}}}}
