﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eft2_Enum.h>
#include <nw/eft/eft2_UniformBlock.h>

#include <nw/eft/eft2_System.h>


namespace nw   {
namespace eft2 {



bool UniformBlockBuffer::g_IsCpuCacheFlush = true;
bool UniformBlockBuffer::g_IsGpuCacheFlush = true;
bool UniformBlockBuffer::g_IsEndianSwap    = false;

//---------------------------------------------------------------------------
//  無効化
//---------------------------------------------------------------------------
void UniformBlockID::Invalidate()
{
#if EFT_OPENGL
    m_BufferBindPoint                    = EFT_INVALID_UBO_BIND_POINT;
    m_BufferId                           = static_cast<u32>( EFT_INVALID_UBO_BUFFER_ID );
#endif
#if EFT_GX2
    m_BufferBindPoint                    = EFT_INVALID_UBO_BIND_POINT;
#endif
}

//---------------------------------------------------------------------------
//  終了処理
//---------------------------------------------------------------------------
void UniformBlockID::Finalize()
{
#if EFT_OPENGL
    if ( m_BufferId != EFT_INVALID_UBO_BUFFER_ID ) { glDeleteBuffers( 1, &m_BufferId ); }
    if ( m_BufferId != EFT_INVALID_UBO_BUFFER_ID ) { glDeleteBuffers( 1, &m_BufferId ); }
#endif
}

//---------------------------------------------------------------------------
// 頂点ユニフォームブロックの初期化
//---------------------------------------------------------------------------
bool UniformBlockID::Initialize( u32 bindPoint, const char* name )
{
#if EFT_OPENGL
    EFT_UNUSED_VARIABLE( name );
    m_BufferBindPoint = bindPoint;
    glGenBuffers( 1, &m_BufferId );
    glBindBuffer( GL_UNIFORM_BUFFER, m_BufferId );
    glBindBufferBase( GL_UNIFORM_BUFFER, m_BufferBindPoint, m_BufferId );
    EFT_GLERR_CHECK();
#elif EFT_GX2
    m_BufferBindPoint = bindPoint;
#else
    EFT_UNUSED_VARIABLE( bindPoint );
    EFT_UNUSED_VARIABLE( name );
#endif
    return true;
}

//---------------------------------------------------------------------------
//  ユニフォームブロックへのバッファセット
//---------------------------------------------------------------------------
void UniformBlockID::BindUniformBlock( const UniformBlockBuffer* uniformBlockBuffer, bool vertex, bool fragment )
{
#if EFT_OPENGL
    if ( vertex || fragment )
    {
        glBindBuffer( GL_UNIFORM_BUFFER, m_BufferId );
        glBufferData( GL_UNIFORM_BUFFER, uniformBlockBuffer->GetBufferSize(), uniformBlockBuffer->GetBuffer(), GL_DYNAMIC_DRAW );
        glBindBufferRange( GL_UNIFORM_BUFFER, m_BufferBindPoint, m_BufferId, 0, uniformBlockBuffer->GetBufferSize() );
        EFT_GLERR_CHECK();
    }
#elif EFT_GX2
    if ( vertex )   GX2SetVertexUniformBlock( m_BufferBindPoint,
                                              uniformBlockBuffer->GetBufferSize(),
                                              uniformBlockBuffer->GetBuffer() );
    if ( fragment ) GX2SetPixelUniformBlock(  m_BufferBindPoint,
                                              uniformBlockBuffer->GetBufferSize(),
                                              uniformBlockBuffer->GetBuffer() );
#else
    EFT_UNUSED_VARIABLE( uniformBlockBuffer );
    EFT_UNUSED_VARIABLE( vertex );
    EFT_UNUSED_VARIABLE( fragment );
#endif

}

//---------------------------------------------------------------------------
//  ユニフォームブロックバッファを有効化する
//---------------------------------------------------------------------------
void UniformBlockBuffer::Validate( bool endianSwap )
{
    //EFT_ASSERT( !m_IsValidate );
    EFT_NULL_ASSERT( m_UniformBlockBuffer );

#ifdef EFT_OGL
    EFT_UNUSED_VARIABLE( endianSwap );
#endif

#if EFT_GX2
    if ( endianSwap )
    {
        GX2EndianSwap( m_UniformBlockBuffer, m_UniformBlockBufferSize );
    }
    if ( g_IsCpuCacheFlush )
    {
        DCFlushRange( m_UniformBlockBuffer, m_UniformBlockBufferSize );
    }
    if ( g_IsGpuCacheFlush )
    {
        GX2Invalidate( GX2_INVALIDATE_UNIFORM_BLOCK, m_UniformBlockBuffer, m_UniformBlockBufferSize );
    }
#endif

    m_IsValidate = true;
}

//---------------------------------------------------------------------------
//! @brief  静的ユニフォームブロックバッファを有効化する
//---------------------------------------------------------------------------
void StaticUniformBlockBuffer::Validate()
{
    EFT_ASSERT( !m_IsValidate );
    EFT_NULL_ASSERT( m_UniformBlockBuffer );

#if EFT_GX2
    GX2EndianSwap( m_UniformBlockBuffer, m_UniformBlockBufferSize );
    DCFlushRange( m_UniformBlockBuffer, m_UniformBlockBufferSize );
#endif
#if EFT_OPENGL
    EFT_UNUSED_VARIABLE( system );
#endif
    m_IsValidate = true;
}

//---------------------------------------------------------------------------
//  動的ユニフォームブロックバッファを無効化する
//---------------------------------------------------------------------------
void DynamicUniformBlockBuffer::InValidate()
{
#if EFT_GX2
    MemUtil::ZeroRange( m_UniformBlockBuffer, m_UniformBlockBufferSize );
#endif
    m_IsValidate = false;
}


//---------------------------------------------------------------------------
//  テンポラリユニフォームブロックバッファを確保する
//---------------------------------------------------------------------------
void* TemporaryUniformBlockBuffer::Alloc( System* system, u32 bufferSize )
{
    m_UniformBlockBufferSize = bufferSize;
    m_UniformBlockBuffer     = system->AllocFromTempBuffer( bufferSize );
    return m_UniformBlockBuffer;
}


} // namespace eft2
} // namespace nw

