﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eft2_StripeConnection.h>
#include <nw/eft/eft2_Misc.h>
#include <nw/eft/eft2_MemUtil.h>
#include <nw/eft/eft2_Emitter.h>
#include <nw/eft/eft2_RenderContext.h>
#include <nw/eft/eft2_Render.h>
#include <nw/eft/eft2_System.h>
#include <nw/eft/eft2_Stripe.h>
#include <nw/eft/eft2_CallbackPtclCalcArg.h>

#ifndef EFT_DEGRADATION_SPEC

namespace nw   {
namespace eft2 {

ConnectionStripeSystem* sConnectionStripeSystem = NULL;

//------------------------------------------------------------------------------
//  CalculateHermiteInterpolatedCurveVec() の内部処理。 入力ベクトルの差分を半分にして返す。
//------------------------------------------------------------------------------
inline void GetSubAndHalfVector( nw::math::VEC3* pOutValue, const nw::math::VEC3& v0, const nw::math::VEC3& v1 )
{
    NW_NULL_ASSERT( pOutValue );
    *pOutValue = ( v0 - v1 ) * 0.5f;
}

//---------------------------------------------------------------------------
//  CalculateHermiteInterpolatedCurveVec() の内部処理。
//  3点のうち、真ん中の点へのベクトル2本を足し合わせて、係数を乗算して返す。
//---------------------------------------------------------------------------
inline void GetQuarterVectorFrom3Pos( nw::math::VEC3* pOutValue, const nw::math::VEC3& v0, const nw::math::VEC3& v1, const nw::math::VEC3& v2, float ratio )
{
    NW_NULL_ASSERT( pOutValue );
    *pOutValue = ( ( v1 - v0 ) + ( v1 - v2 ) ) * ratio;
}

//---------------------------------------------------------------------------
//  CalculateHermiteInterpolatedCurveVec() の内部処理。
//  方向ベクトルを補間する。
//---------------------------------------------------------------------------
inline void GetInterpolatedDirVector( nw::math::VEC3* pOutValue, const nw::math::VEC3& v0, const nw::math::VEC3& v1, float innerPos )
{
    NW_NULL_ASSERT( pOutValue );
    *pOutValue = ( v0 * ( 1.0f - innerPos ) ) + ( v1 * innerPos );
}

//------------------------------------------------------------------------------
//  連結式ストライプシステムの初期化処理を行います。
//------------------------------------------------------------------------------
void ConnectionStripeSystem::Initialize( nw::eft2::Heap* heap, nw::eft2::System* system, BufferMode bufferMode, u32 stripeNum )
{
    if ( sConnectionStripeSystem )
    {
        OutputError( "StripeSystem is Initialized.\n" );
    }

    // プレビュー生成用のメモリを確保します。
    void* buffer = heap->Alloc( sizeof(ConnectionStripeSystem) );
    if( !buffer )
    {
        nw::eft2::OutputWarning( "Memory Allocate Error!! : %d\n", sizeof(ConnectionStripeSystem) );
        return;
    }

    sConnectionStripeSystem = new (buffer) ConnectionStripeSystem( heap, system, bufferMode, stripeNum );
}

//------------------------------------------------------------------------------
//  連結式ストライプシステムの終了処理を行います。
//------------------------------------------------------------------------------
void ConnectionStripeSystem::Finalize( nw::eft2::Heap* heap )
{
    sConnectionStripeSystem->~ConnectionStripeSystem();
    heap->Free( sConnectionStripeSystem );
    sConnectionStripeSystem = NULL;
}

//------------------------------------------------------------------------------
//  連結式ストライプシステムで確保したワークサイズを取得します。
//------------------------------------------------------------------------------
u32 ConnectionStripeSystem::GetWorkSize()
{
    if ( !sConnectionStripeSystem )
    {
        return 0;
    }
    return sConnectionStripeSystem->_GetWorkSize();
}

//------------------------------------------------------------------------------
//  パーティクル生成コールバック
//------------------------------------------------------------------------------
bool ConnectionStripeSystem::_ParticleEmitCallback( ParticleEmitArg& arg )
{
    NW_UNUSED_VARIABLE( arg );
    return true;
}

//------------------------------------------------------------------------------
//  パーティクル削除コールバック
//------------------------------------------------------------------------------
bool ConnectionStripeSystem::_ParticleRemoveCallback( ParticleRemoveArg& arg )
{
    NW_UNUSED_VARIABLE( arg );
    return true;
}

//------------------------------------------------------------------------------
//  パーティクル計算コールバック
//------------------------------------------------------------------------------
void ConnectionStripeSystem::_ParticleCalcCallback( ParticleCalcArg& arg )
{
    NW_UNUSED_VARIABLE( arg );
}

//------------------------------------------------------------------------------
//  エミッタ生成後コールバック
//------------------------------------------------------------------------------
bool ConnectionStripeSystem::_EmitterInitializeCallback( EmitterInitializeArg& arg )
{
    // ポリゴン用バッファを確保
    sConnectionStripeSystem->AllocStripeVertexBuffer( arg.emitter );
    return true;
}

//------------------------------------------------------------------------------
//  エミッタ計算処理前コールバック
//------------------------------------------------------------------------------
void ConnectionStripeSystem::_EmitterPreCalcCallback( EmitterPreCalcArg& arg )
{
    NW_UNUSED_VARIABLE( arg );
}

//------------------------------------------------------------------------------
//  エミッタ計算処理後コールバック
//------------------------------------------------------------------------------
void ConnectionStripeSystem::_EmitterPostCalcCallback( EmitterPostCalcArg& arg )
{
    // 連結式ストライプのポリゴンを生成！
    if ( arg.emitter->GetCalcType() == EFT_EMITTER_CALC_TYPE_CPU )
    {
        sConnectionStripeSystem->MakeStripeVertexBuffer( arg.emitter );
    }
}

//------------------------------------------------------------------------------
//  エミッタ描画コールバック
//------------------------------------------------------------------------------
bool ConnectionStripeSystem::_EmitterDrawCallback( EmitterDrawArg& arg )
{
    // 連結式ストライプを描画！
    if ( arg.emitter->GetCalcType() == EFT_EMITTER_CALC_TYPE_CPU )
    {
        sConnectionStripeSystem->Draw( arg.emitter, arg.shaderType, arg.userParam );
        return true;
    }
    return false;
}

//------------------------------------------------------------------------------
//  エミッタ破棄後コールバック
//------------------------------------------------------------------------------
void ConnectionStripeSystem::_EmitterFinalizeCallback( EmitterFinalizeArg& arg )
{
    if( arg.emitter->stripeVertexBuffer )
    {
        FreeFromDynamicHeap( arg.emitter->stripeVertexBuffer, false );
    }
}

//---------------------------------------------------------------------------
//  コンストラクタ
//---------------------------------------------------------------------------
ConnectionStripeSystem::ConnectionStripeSystem( nw::eft2::Heap* heap, nw::eft2::System* system, BufferMode bufferMode, u32 stripeNum )
{
    mHeap   = heap;
    mSystem = system;

    mStripeArrayNum     = stripeNum;
    mStripeWorkSize     = sizeof(ConnectionStripe) * mStripeArrayNum;
    mStripeArray        = (ConnectionStripe*)mHeap->Alloc( sizeof(ConnectionStripe) * mStripeArrayNum );
    mStripeNum          = 0;
    mBufferMode         = bufferMode;

    for( u32 i = 0; i < mStripeArrayNum; i++ )
    {
        mStripeArray[i].used        = false;
        mStripeArray[i].numHistory  = 0;
        mStripeArray[i].next        = NULL;
    }

    // ストライプコールバックを設定する
    CallbackSet set;
    set.emitterPreCalc      = ConnectionStripeSystem::_EmitterPreCalcCallback;
    set.emitterInitialize   = ConnectionStripeSystem::_EmitterInitializeCallback;
    set.particleEmit        = ConnectionStripeSystem::_ParticleEmitCallback;
    set.particleRemove      = ConnectionStripeSystem::_ParticleRemoveCallback;
    set.particleCalc        = ConnectionStripeSystem::_ParticleCalcCallback;
    set.emitterPostCalc     = ConnectionStripeSystem::_EmitterPostCalcCallback;
    set.emitterDraw         = ConnectionStripeSystem::_EmitterDrawCallback;
    set.emitterFinalize     = ConnectionStripeSystem::_EmitterFinalizeCallback;
    set.renderStateSet      = _StripeRenderStateSetCallback;
    mSystem->SetEmitterPluginCallbackSet( EFT_EMITTER_PLUGIN_CALLBACK_ID_1, set );
}

//---------------------------------------------------------------------------
//  デストラクタ
//---------------------------------------------------------------------------
ConnectionStripeSystem::~ConnectionStripeSystem()
{
    mHeap->Free( mStripeArray );
}


//---------------------------------------------------------------------------
//  バッファをスワップする。
//---------------------------------------------------------------------------
void ConnectionStripeSystem::SwapBuffer( Emitter* emitter )
{
    emitter->stripeVertexBufferCurIdx = emitter->stripeVertexBufferStartIdx[emitter->bufferID];
    return;
}

//---------------------------------------------------------------------------
//! @brief        ポリゴン用ワークを確保します。
//---------------------------------------------------------------------------
void ConnectionStripeSystem::AllocStripeVertexBuffer( Emitter* emitter )
{
    if( !emitter )
    {
        return;
    }

    const u32 particleNum = emitter->ptclAttrFillMax;
    ResStripeConnection* pStripeData = reinterpret_cast< ResStripeConnection* >( emitter->emitterRes->emitterPluginData );

    // 要素数
    u32 element     = sizeof( ConnectionStripeVertexBuffer ) / sizeof( nw::math::VEC4* );

    // パーティクルの最大数：「エミッタに接続」「先端につなぐ」で一つ点が増えるので、その分1つ追加。
    const int maxParticleCount = particleNum + 1;
    const int divCount = static_cast< int >( pStripeData->numDivide );
    const int vertexBufferCount = ( maxParticleCount + ( maxParticleCount - 1 ) * divCount ) * 2;

    // 頂点バッファサイズ
    emitter->stripeVertexBufferNum = mBufferMode * vertexBufferCount;
    u32 bufferSize = emitter->stripeVertexBufferNum * element * sizeof( nw::math::VEC4 );

    // ポリゴン用バッファを確保
    if ( bufferSize > 0 )
    {
        nw::math::VEC4* vec4Buffer = reinterpret_cast<nw::math::VEC4*>( AllocFromDynamicHeap( bufferSize ) );
        if ( !vec4Buffer )
        {
            emitter->stripeVertexBuffer = NULL;
            return;
        }

        emitter->stripeVertexBuffer = vec4Buffer;

        // バッファを切り分ける
        if ( mBufferMode == EFT_BUFFER_MODE_DOUBLE )
        {
            emitter->stripeVertexBufferStartIdx[EFT_BUFFER_ID_0] = 0;
            emitter->stripeVertexBufferEndIdx[EFT_BUFFER_ID_0]   = emitter->stripeVertexBufferNum/2;
            emitter->stripeVertexBufferStartIdx[EFT_BUFFER_ID_1] = emitter->stripeVertexBufferNum/2;
            emitter->stripeVertexBufferEndIdx[EFT_BUFFER_ID_1]   = emitter->stripeVertexBufferNum;
            emitter->stripeVertexBufferStartIdx[EFT_BUFFER_ID_2] = 0;
            emitter->stripeVertexBufferEndIdx[EFT_BUFFER_ID_2]   = emitter->stripeVertexBufferNum/2;
        }
        else
        {
            emitter->stripeVertexBufferStartIdx[EFT_BUFFER_ID_0] = 0;
            emitter->stripeVertexBufferEndIdx[EFT_BUFFER_ID_0]   = emitter->stripeVertexBufferNum/3;
            emitter->stripeVertexBufferStartIdx[EFT_BUFFER_ID_1] = emitter->stripeVertexBufferNum/3;
            emitter->stripeVertexBufferEndIdx[EFT_BUFFER_ID_1]   = emitter->stripeVertexBufferNum/3*2;
            emitter->stripeVertexBufferStartIdx[EFT_BUFFER_ID_2] = emitter->stripeVertexBufferNum/3*2;
            emitter->stripeVertexBufferEndIdx[EFT_BUFFER_ID_2]   = emitter->stripeVertexBufferNum;
        }
    }
    else
    {
        emitter->stripeVertexBuffer = NULL;
    }
}

//---------------------------------------------------------------------------
//  頂点バッファに頂点情報をセットします。
//---------------------------------------------------------------------------
void SetVertexAttribute(
    ConnectionStripeVertexBuffer* pOutVertexBuffer,
    int index,
    int maxIndex,
    const nw::math::VEC3& pos,
    const nw::math::VEC3& dir,
    const nw::math::VEC3& emat,
    float scale )
{
    NW_NULL_ASSERT( pOutVertexBuffer );

    const int indexL = index * 2;
    const int indexR = indexL + 1;

    EFT_F32_VEC3_COPY( &pOutVertexBuffer->pos[ indexL ], &pos );
    EFT_F32_VEC3_COPY( &pOutVertexBuffer->pos[ indexR ], &pos );

    EFT_F32_VEC3_COPY( &pOutVertexBuffer->dir[ indexL ], &dir );
    EFT_F32_VEC3_COPY( &pOutVertexBuffer->dir[ indexR ], &dir );

    EFT_F32_VEC3_COPY( &pOutVertexBuffer->emat[ indexL ], &emat );
    EFT_F32_VEC3_COPY( &pOutVertexBuffer->emat[ indexR ], &emat );

    // MEMO: 連結式ストライプのバッファの詰め順は履歴式とは逆なので、スケール方向も逆にしておく。
    pOutVertexBuffer->pos[ indexL ].w = -scale;
    pOutVertexBuffer->pos[ indexR ].w = scale;
    pOutVertexBuffer->dir[ indexR ].w = pOutVertexBuffer->dir[ indexL ].w = ( static_cast< float >( index ) ) / ( maxIndex - 1 );    // テクスチャ座標
    pOutVertexBuffer->emat[ indexR ].w = pOutVertexBuffer->emat[ indexL ].w = static_cast< float >( index );
}

//---------------------------------------------------------------------------
//  頂点バッファに頂点情報をセット＋カウンタの更新のユーティリティ
//---------------------------------------------------------------------------
void AddVertexDataToBuffer(
    ConnectionStripeVertexBuffer* pOutVertexBuffer,
    int* pOutAttrIndex,
    u32* pOutRenderVertexCount,
    int vertexCount,
    const nw::math::VEC3& vertexWorldPos,
    const nw::math::VEC3& vertexWorldDir,
    const nw::math::VEC3& emitterMatY,
    float scaleValue )
{
    NW_NULL_ASSERT( pOutVertexBuffer );
    NW_NULL_ASSERT( pOutAttrIndex );
    NW_NULL_ASSERT( pOutRenderVertexCount );

    SetVertexAttribute( pOutVertexBuffer, *pOutAttrIndex, vertexCount, vertexWorldPos, vertexWorldDir, emitterMatY, scaleValue );
    ( *pOutRenderVertexCount ) += 2;
    ( *pOutAttrIndex )++;
}

//------------------------------------------------------------------------------
//  パーティクルのワールド座標を取得
//------------------------------------------------------------------------------
void GetParticleWorldPos( nw::math::VEC3* outPos, Emitter* emitter, const u32 ptclIndex )
{
    NW_NULL_ASSERT( outPos );
    NW_NULL_ASSERT( emitter );

    nw::math::VEC3 localPos;
    EFT_F32_VEC3_COPY( &localPos, &emitter->particlePosAttr[ ptclIndex ].localPos );
    if( emitter->IsParticleAttributeDoubleBuffer() &&
        emitter->particleData[ ptclIndex ].GetTime( emitter->time ) == 0 )
    {
        // ダブルバッファ時の初回フレームは、まだBackに値が入っていないのでFrontから取る
        emitter->TransformToWorldPos( outPos, localPos, &emitter->particleAttr[ ptclIndex ] );
    }
    else
    {
        emitter->TransformToWorldPos( outPos, localPos, &emitter->particleAttrBack[ ptclIndex ] );
    }
}

//------------------------------------------------------------------------------
//  パーティクルのスケールを取得
//------------------------------------------------------------------------------
inline void GetParticleScale( nw::math::VEC3* pOutScale, Emitter* pEmitter, int particleindex )
{
    NW_NULL_ASSERT( pOutScale );
    NW_NULL_ASSERT( pEmitter );

    nw::math::VEC4* random    = NULL;
    nw::math::VEC4* scaleTemp = NULL;
    const f32 ptclLife = pEmitter->particlePosAttr[ particleindex ].GetLife();
    const f32 ptclTime = pEmitter->particlePosAttr[ particleindex ].GetTime( pEmitter->time );

    if( pEmitter->IsParticleAttributeDoubleBuffer() &&
        pEmitter->particleData[ particleindex ].GetTime( pEmitter->time ) == 0 )
    {
        random = &pEmitter->particleAttr[ particleindex ].random;
        scaleTemp = &pEmitter->particleAttr[ particleindex ].scale;
    }
    else
    {
        random = &pEmitter->particleAttrBack[ particleindex ].random;
        scaleTemp = &pEmitter->particleAttrBack[ particleindex ].scale;
    }
    const nw::math::VEC3& s = pEmitter->emitterSet->GetParticleScaleForCalc();

    nw::math::VEC4 worldScale;
    CalcPtclScaleVecFromTime( &worldScale, pEmitter->emitterRes, scaleTemp, random, ptclLife, ptclTime );

    worldScale.x *= s.x;
    worldScale.y *= s.y;
    worldScale.z *= s.z;

    EFT_F32_VEC3_COPY( pOutScale, &worldScale );
}

//------------------------------------------------------------------------------
//  エミッタ行列Y軸を取得
//------------------------------------------------------------------------------
void GetParticleEmitterMatrixY( nw::math::VEC3* pOutEmatY, Emitter* pEmitter, ResStripeConnection* pStripeData, int particleIndex )
{
    NW_NULL_ASSERT( pOutEmatY );
    NW_NULL_ASSERT( pEmitter );
    NW_NULL_ASSERT( pStripeData );

    if( pStripeData->calcType == EFT_STRIPE_POLYGON_TYPE_BILLBOARD )
    {
        //------------------------------------------------------------------------------
        // ビルボードストライプ（※使わないので何もしない）
        //------------------------------------------------------------------------------
        return;
    }
    else if( pStripeData->calcType == EFT_STRIPE_POLYGON_TYPE_EMITTER_MATRIX || pStripeData->calcType == EFT_STRIPE_POLYGON_TYPE_EMITTER_UPRIGHT )
    {
        //------------------------------------------------------------------------------
        // エミッタ行列ストライプ／エミッタ上下に伸ばす
        //------------------------------------------------------------------------------
        if( pEmitter->GetResEmitter()->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_ALL )
        {
            const nw::math::MTX34& matrixSrt = pEmitter->GetMatrixSrt();
            *pOutEmatY = matrixSrt.GetColumn( 1 );
        }
        else
        {
            pOutEmatY->x = pEmitter->particleAttr[ particleIndex ].emitterMat0[ 1 ];
            pOutEmatY->y = pEmitter->particleAttr[ particleIndex ].emitterMat1[ 1 ];
            pOutEmatY->z = pEmitter->particleAttr[ particleIndex ].emitterMat2[ 1 ];
        }
    }
}

//---------------------------------------------------------------------------
//  ストライプのポリゴン生成を行います。
//---------------------------------------------------------------------------
void ConnectionStripeSystem::MakeStripeVertexBuffer( Emitter* pEmitter )
{
    if ( !pEmitter )
    {
        return;
    }

    ResStripeConnection* pStripeData = reinterpret_cast< ResStripeConnection* >( pEmitter->emitterRes->emitterPluginData );
    pEmitter->stripeVertexBufferRenderNum = 0;   // 一旦0でリセット

    // 必要な頂点数の計算
    // 末端処理のタイプ分け
    StripeTailType tailConnectionType = static_cast< StripeTailType >( pStripeData->connectionType );

    // パーティクルのトラバース
    // 分割数
    // 描画頂点数
    const s32 numDivide = pStripeData->numDivide;
    NW_UNUSED_VARIABLE( numDivide );

    if( !pEmitter->stripeVertexBuffer ){ return; }

    nw::math::VEC4* vec4Buffer = reinterpret_cast< nw::math::VEC4* >( pEmitter->stripeVertexBuffer );
    u32 segment = pEmitter->stripeVertexBufferNum;  //各要素数

    ConnectionStripeVertexBuffer ConnectionVertexBuffer;
    ConnectionStripeVertexBuffer* pVertexBuffer = &ConnectionVertexBuffer;
    pVertexBuffer->pos  = &vec4Buffer[ segment * 0 + pEmitter->stripeVertexBufferCurIdx ];
    pVertexBuffer->dir  = &vec4Buffer[ segment * 1 + pEmitter->stripeVertexBufferCurIdx ];
    pVertexBuffer->emat = &vec4Buffer[ segment * 2 + pEmitter->stripeVertexBufferCurIdx ];

    // パーティクルのソート
    // ソートした配列の先頭＆配列サイズ
    u32 particleNum = 0;
    System::ParticleSortInfo* pSortIndexArrayHead = NULL;
    {
        CpuCore core = pEmitter->GetEmitterSet()->GetSystem()->GetCurrentCore();
        pEmitter->GetEmitterSet()->GetSystem()->GetSortedPtclList( &pSortIndexArrayHead, &particleNum, pEmitter, EFT_SORT_TYPE_REV_ORDER, core );
        if( particleNum <= 0 )
        {
            return;
        }
    }

    // 描画する背骨の数（頂点数はこの二倍）
    u32 baseVertexCount = particleNum;
    if( tailConnectionType == EFT_STRIPE_TAIL_TYPE_EMITTER ||
        tailConnectionType == EFT_STRIPE_TAIL_TYPE_LOOP )
    {
        baseVertexCount += 1;
    }

    // 分割数を込にした、描画するすべての頂点数
    const float invDelta = 1.0f / ( numDivide + 1 );
    const int wholeVertexCount = baseVertexCount + ( baseVertexCount - 1 ) * numDivide;

    // スケールフェードイン／アウト
    const f32 scaleFadeValue = pEmitter->CalculateCurrentScaleFadeValue();
    const int loopCount = particleNum - 1;
    int attrIndex = 0;

    //---------------------------------------------------------------------------
    // 二回以上ループしないとポリゴンが成立しないので処理しない
    //---------------------------------------------------------------------------
    if( ( numDivide == 0 && baseVertexCount >= 2 ) || ( numDivide > 0 && particleNum >= 3 ) )
    {
        //---------------------------------------------------------------------------
        // 先端処理：結合タイプによる特殊処理
        //---------------------------------------------------------------------------
        {
            nw::math::VEC3 nowPos( 0, 0, 0 );
            nw::math::VEC3 nowDir( 0, 0, 0 );
            nw::math::VEC3 nextPos( 0, 0, 0 );
            nw::math::VEC3 nextDir( 0, 0, 0 );
            nw::math::VEC3 ematY( 0, 0, 0 );
            nw::math::VEC3 scale( 0, 0, 0 );
            float scaleValue = 0.0f;

            if( tailConnectionType == EFT_STRIPE_TAIL_TYPE_LOOP )
            {
                //---------------------------------------------------------------------------
                // 「先端に繋ぐ」
                //---------------------------------------------------------------------------
                const int idxNow = pSortIndexArrayHead[ loopCount ].index;
                const int idxNext = pSortIndexArrayHead[ 0 ].index;
                const int idxNext2 = pSortIndexArrayHead[ 1 ].index;

                GetParticleWorldPos( &nowPos, pEmitter, idxNow );
                GetParticleWorldPos( &nextPos, pEmitter, idxNext );

                GetParticleEmitterMatrixY( &ematY, pEmitter, pStripeData, idxNow );
                GetParticleScale( &scale, pEmitter, idxNow );
                scaleValue = scale.x * scaleFadeValue;

                if( numDivide > 0 )
                {
                    //---------------------------------------------------------------------------
                    // 分割あり
                    //---------------------------------------------------------------------------
                    nw::math::VEC3 next2Pos;
                    GetParticleWorldPos( &next2Pos, pEmitter, idxNext2 );

                    nowDir = nextPos - nowPos;  // MEMO: これだとほんのちょっと歪む
                    GetSubAndHalfVector( &nextDir, next2Pos, nowPos );
                }
                else
                {
                    //---------------------------------------------------------------------------
                    // 分割無し
                    //---------------------------------------------------------------------------
                    // 方向は前のパーティクルと比較した方がまだ自然。
                    nw::math::VEC3 next2Pos;
                    GetParticleWorldPos( &next2Pos, pEmitter, idxNext2 );

                    nowDir = nextPos - nowPos;
                    nextDir = next2Pos - nextPos;
                }

                // まず基準となる頂点について埋めておく。
                AddVertexDataToBuffer( pVertexBuffer, &attrIndex, &pEmitter->stripeVertexBufferRenderNum, wholeVertexCount, nowPos, nowDir, ematY, scaleValue );
            }
            else if( tailConnectionType == EFT_STRIPE_TAIL_TYPE_EMITTER )
            {
                //---------------------------------------------------------------------------
                // 「エミッタに繋ぐ」：エミッタ位置にも頂点を追加。
                //---------------------------------------------------------------------------
                // TODO: 分割無し時、エミッタにつなぐ部分が捻じれている
                const int idxNext = pSortIndexArrayHead[ 0 ].index;
                const int idxNext2 = pSortIndexArrayHead[ 1 ].index;

                GetParticleWorldPos( &nextPos, pEmitter, idxNext );

                GetParticleEmitterMatrixY( &ematY, pEmitter, pStripeData, idxNext );
                GetParticleScale( &scale, pEmitter, idxNext );
                scaleValue = scale.x * scaleFadeValue;

                if( numDivide > 0 )
                {
                    //---------------------------------------------------------------------------
                    // 分割あり
                    //---------------------------------------------------------------------------
                    const nw::math::MTX34& matrixSrt = pEmitter->GetMatrixSrt();
                    nowPos = matrixSrt.GetColumn( 3 );

                    nw::math::VEC3 next2Pos;
                    GetParticleWorldPos( &next2Pos, pEmitter, idxNext2 );

                    GetQuarterVectorFrom3Pos( &nowDir, nowPos, nextPos, next2Pos, 0.25f );
                    GetSubAndHalfVector( &nextDir, next2Pos, nowPos );
                }
                else
                {
                    //---------------------------------------------------------------------------
                    // 分割無し
                    //---------------------------------------------------------------------------
                    // 完全にが一周した
                    //「エミッタにつなぐ」
                    const nw::math::MTX34& matrixSrt = pEmitter->GetMatrixSrt();
                    nowPos = matrixSrt.GetColumn( 3 );

                    // 一つ前の位置
                    nowDir = nextPos - nowPos;  // 頂点方向（ワールド系）
                }

                // まず基準となる頂点について埋めておく。
                AddVertexDataToBuffer( pVertexBuffer, &attrIndex, &pEmitter->stripeVertexBufferRenderNum, wholeVertexCount, nowPos, nowDir, ematY, scaleValue );
            }

            // 分割数がある場合、間の分割数分の頂点を埋める。
            if( numDivide > 0 &&
                ( tailConnectionType == EFT_STRIPE_TAIL_TYPE_LOOP || tailConnectionType == EFT_STRIPE_TAIL_TYPE_EMITTER ))
            {
                for( int i = 1; i <= numDivide; ++i )
                {
                    const float innerPos = invDelta * i;
                    nw::math::VEC3 curvePos;
                    nw::math::VEC3 curveDir;

                    GetInterpolatedDirVector( &curveDir, nowDir, nextDir, innerPos );
                    HermiteInterpolationOnCubic( &curvePos, nowPos, nowDir, nextPos, nextDir, innerPos );

                    AddVertexDataToBuffer( pVertexBuffer, &attrIndex, &pEmitter->stripeVertexBufferRenderNum, wholeVertexCount, curvePos, curveDir, ematY, scaleValue );
                }
            }
        }

        //---------------------------------------------------------------------------
        // 共通処理：ソートされた順にパーティクルをポリゴンで繋ぐ
        //---------------------------------------------------------------------------
        {
            nw::math::VEC3 nowPos( 0, 0, 0 );
            nw::math::VEC3 nowDir( 0, 0, 0 );
            nw::math::VEC3 nextPos( 0, 0, 0 );
            nw::math::VEC3 nextDir( 0, 0, 0 );
            nw::math::VEC3 ematY( 0, 0, 0 );
            nw::math::VEC3 scale( 0, 0, 0 );

            // 最後の頂点のひとつ前までループして処理する
            for( int idx = 0; idx < loopCount; ++idx )
            {
                const int idxNow = pSortIndexArrayHead[ idx ].index;
                const int idxNext = pSortIndexArrayHead[ idx + 1 ].index;
                GetParticleWorldPos( &nowPos, pEmitter, idxNow );
                GetParticleWorldPos( &nextPos, pEmitter, idxNext );

                GetParticleEmitterMatrixY( &ematY, pEmitter, pStripeData, idxNow );
                GetParticleScale( &scale, pEmitter, idxNow );
                const float scaleValue = scale.x * scaleFadeValue;

                if( numDivide > 0 )
                {
                    //---------------------------------------------------------------------------
                    // エルミート補間を行うための処理分岐。
                    // ここに来た時点で、粒が3つ以上あるのは確定。
                    // この部分の処理は、粒が3つでも4つ以上でも正しく動作する想定。
                    //---------------------------------------------------------------------------
                    if( idx == 0 )
                    {
                        // 先頭部分（3点で近似補間）
                        const int idxNext2 = pSortIndexArrayHead[ idx + 2 ].index;
                        nw::math::VEC3 next2Pos;
                        GetParticleWorldPos( &next2Pos, pEmitter, idxNext2 );

                        nowDir = nextPos - nowPos;
                        GetSubAndHalfVector( &nextDir, next2Pos, nowPos );
                    }
                    else if( idx == loopCount - 1 )
                    {
                        // ストライプの末端のひとつ前（3点で近似補間）
                        const int idxPrev = pSortIndexArrayHead[ idx - 1 ].index;
                        nw::math::VEC3 prevPos;
                        GetParticleWorldPos( &prevPos, pEmitter, idxPrev );

                        GetSubAndHalfVector( &nowDir, nextPos, prevPos );
                        GetQuarterVectorFrom3Pos( &nextDir, nextPos, nowPos, prevPos, -0.25f );
                    }
                    else
                    {
                        // 中間部（4点で補間）
                        // 前に1点、次に2点存在するのが保証されている。
                        const int idxPrev = pSortIndexArrayHead[ idx - 1 ].index;
                        const int idxNext2 = pSortIndexArrayHead[ idx + 2 ].index;
                        nw::math::VEC3 prevPos;
                        nw::math::VEC3 next2Pos;
                        GetParticleWorldPos( &prevPos, pEmitter, idxPrev );
                        GetParticleWorldPos( &next2Pos, pEmitter, idxNext2 );

                        GetSubAndHalfVector( &nowDir, nextPos, prevPos );
                        GetSubAndHalfVector( &nextDir, next2Pos, nowPos );
                    }
                }
                else
                {
                    //---------------------------------------------------------------------------
                    // 分割無し
                    //---------------------------------------------------------------------------
                    GetParticleWorldPos( &nowPos, pEmitter, idxNow );
                    GetParticleWorldPos( &nextPos, pEmitter, idxNext );
                    nowDir = nextPos - nowPos;
                }

                // まず基準となる頂点について埋めておく。
                AddVertexDataToBuffer( pVertexBuffer, &attrIndex, &pEmitter->stripeVertexBufferRenderNum, wholeVertexCount, nowPos, nowDir, ematY, scaleValue );

                // 分割数がある場合、間の分割数分の頂点を埋める。
                if( numDivide > 0 )
                {
                    for( int i = 1; i <= numDivide; ++i )
                    {
                        const float innerPos = invDelta * i;
                        nw::math::VEC3 curvePos;
                        nw::math::VEC3 curveDir;

                        GetInterpolatedDirVector( &curveDir, nowDir, nextDir, innerPos );
                        HermiteInterpolationOnCubic( &curvePos, nowPos, nowDir, nextPos, nextDir, innerPos );

                        AddVertexDataToBuffer( pVertexBuffer, &attrIndex, &pEmitter->stripeVertexBufferRenderNum, wholeVertexCount, curvePos, curveDir, ematY, scaleValue );
                    }
                }
            }
        }

        //---------------------------------------------------------------------------
        // 端処理（最後尾）
        //---------------------------------------------------------------------------
        if( particleNum >= 2 )
        {
            nw::math::VEC3 nowPos( 0, 0, 0 );
            nw::math::VEC3 nowDir( 0, 0, 0 );
            nw::math::VEC3 ematY( 0, 0, 0 );
            nw::math::VEC3 scale( 0, 0, 0 );

            const int idxNow = pSortIndexArrayHead[ loopCount ].index;
            GetParticleEmitterMatrixY( &ematY, pEmitter, pStripeData, idxNow );
            GetParticleScale( &scale, pEmitter, idxNow );
            const float scaleValue = scale.x * scaleFadeValue;

            GetParticleWorldPos( &nowPos, pEmitter, idxNow );

            if( tailConnectionType == EFT_STRIPE_TAIL_TYPE_DEFAULT ||
                tailConnectionType == EFT_STRIPE_TAIL_TYPE_EMITTER )
            {
                //---------------------------------------------------------------------------
                // 「通常」＆「エミッタに繋ぐ」
                //---------------------------------------------------------------------------
                nw::math::VEC3 prevPos;
                const int idxPrev = pSortIndexArrayHead[ loopCount - 1 ].index;
                GetParticleWorldPos( &prevPos, pEmitter, idxPrev );
                nowDir = nowPos - prevPos;

                AddVertexDataToBuffer( pVertexBuffer, &attrIndex, &pEmitter->stripeVertexBufferRenderNum, wholeVertexCount, nowPos, nowDir, ematY, scaleValue );
            }
            else if( tailConnectionType == EFT_STRIPE_TAIL_TYPE_LOOP )
            {
                //---------------------------------------------------------------------------
                // 「先端につなぐ」
                //---------------------------------------------------------------------------
                const int idxNext = pSortIndexArrayHead[ 0 ].index;
                nw::math::VEC3 nextPos;
                GetParticleWorldPos( &nextPos, pEmitter, idxNext );
                nowDir = nextPos - nowPos;

                AddVertexDataToBuffer( pVertexBuffer, &attrIndex, &pEmitter->stripeVertexBufferRenderNum, wholeVertexCount, nowPos, nowDir, ematY, scaleValue );
            }
        }
    }

#if EFT_IS_CAFE
    MemUtil::FlushCache( pVertexBuffer->pos,  sizeof( nw::math::VEC4 ) * pEmitter->stripeVertexBufferRenderNum );
    MemUtil::FlushCache( pVertexBuffer->dir,  sizeof( nw::math::VEC4 ) * pEmitter->stripeVertexBufferRenderNum );
    MemUtil::FlushCache( pVertexBuffer->emat, sizeof( nw::math::VEC4 ) * pEmitter->stripeVertexBufferRenderNum );
#endif
}

//---------------------------------------------------------------------------
//  ストライプ描画処理を行います。
//---------------------------------------------------------------------------
void ConnectionStripeSystem::Draw( Emitter* emitter, ShaderType shaderType, void* userParam )
{
    // 描画するポリゴンバッファが未生成
    if ( emitter->stripeVertexBufferRenderNum == 0 ) { return; }

    System* system = emitter->emitterSet->GetSystem();

    struct ConnectedStripeUniformBlockParam
    {
        nw::math::VEC4 random;
        nw::math::VEC4 param0;
        nw::math::VEC4 color0;
        nw::math::VEC4 color1;
        nw::math::VEC4 param1;
    };

    // シェーダ切り替え
    Shader* shader = emitter->shader[shaderType];
    shader->Bind();

    // 履歴ストライプデータ
    ResStripeConnection* stripeData = reinterpret_cast< ResStripeConnection* >( emitter->emitterRes->emitterPluginData );

    // 描画設定
    ResRenderState* state = &emitter->emitterData->renderState;
    Rendercontext::SetRenderStateResource( state, system->GetColorBufferEnbaleFlag() );

    // テクスチャ設定
    for ( u32 i = 0; i < EFT_TEXTURE_SLOT_MAX; i++ )
    {
        TextureSlot slot = static_cast<TextureSlot>( i );
        TextureResource* texture = emitter->emitterRes->texture[i];

        if ( texture )
        {
            u32 fragLoc = shader->GetFragmentTextureSamplerLocation( slot );
            u32 vertLoc = shader->GetVertexTextureSamplerLocation( slot );

            if ( texture->Bind( fragLoc, vertLoc, slot ) )
            {
                emitter->emitterRes->textureSampler[i].Bind( fragLoc, vertLoc, slot );
            }
        }
    }

    /*
        TODO:サンプラの間借りを辞める。
    */

    // フレームバッファテクスチャの設定
    {
        u32 fragLoc = shader->GetFrameBufferTextureFragmentSamplerLocation();
        u32 vertLoc = shader->GetFrameBufferTextureVertexSamplerLocation();

        if ( system->GetFrameBufferTexture().Bind( fragLoc, vertLoc, EFT_TEXTURE_SLOT_EXT_FB ) )
        {
            emitter->emitterRes->textureSampler[0].Bind( fragLoc, vertLoc, EFT_TEXTURE_SLOT_EXT_FB );
        }
    }

    // デプスバッファテクスチャの設定
    {
        u32 fragLoc = shader->GetDepthBufferTextureFragmentSamplerLocation();
        u32 vertLoc = shader->GetDepthBufferTextureVertexSamplerLocation();

        if ( system->GetDepthBufferTexture().Bind( fragLoc, vertLoc, EFT_TEXTURE_SLOT_EXT_DB ) )
        {
            emitter->emitterRes->textureSampler[0].Bind( fragLoc, vertLoc, EFT_TEXTURE_SLOT_EXT_DB );
        }
    }

    // エミッタ動的UBOの設定
    shader->BindUniformBlock( Shader::EFT_UBO_TYPE_EMITTER_DYNAMIC, &emitter->dynamicUBO[emitter->bufferID] );

    // エミッタ静的UBOの設定
    shader->BindUniformBlock( Shader::EFT_UBO_TYPE_EMITTER_STATIC, &emitter->emitterRes->emitterStaticUbo );

    // カスタムシェーダコールバックの呼び出し
    {
        RenderStateSetArg rssArg;
        rssArg.emitter      = emitter;
        rssArg.userParam    = userParam;
        rssArg.shaderType   = shaderType;
        rssArg.isStreamOut  = false;

        mSystem->ResetCustomShaderSetting( shader );
        {
            // 描画設定後コールバックの呼び出し
            bool ret = InvokeRenderStateSetCallback( rssArg );
            if ( !ret ) { return; }

            // DrawPath コールバックの呼び出し
            if ( emitter->drawPathCallback ) { emitter->drawPathCallback( rssArg ); }
        }
        mSystem->ResetCustomShaderSetting( NULL );
    }

    // 描画
    {
        const u32 vertexNum = emitter->stripeVertexBufferRenderNum;
        if ( vertexNum > 0 )
        {
            // ストライプUBOを設定する
            if( emitter->stripeVertexBuffer )
            {
                TemporaryUniformBlockBuffer stripeTempUbo;
                ConnectedStripeUniformBlockParam* ubo = reinterpret_cast< ConnectedStripeUniformBlockParam* >( stripeTempUbo.Alloc( system, sizeof( ConnectedStripeUniformBlockParam ) ) );
                if ( !ubo ) { return; }

                ubo->param0.x = stripeData->headAlpha;
                ubo->param0.y = stripeData->tailAlpha;
                ubo->random = nw::math::VEC4( 1, 1, 1, 1 );
                ubo->color0 = nw::math::VEC4( 1, 1, 1, 1 ); // 連結式ストライプとしては白を入れておく（シェーダ側でエミッタカラーに乗算される）
                ubo->color1 = nw::math::VEC4( 1, 1, 1, 1 ); // 連結式ストライプとしては白を入れておく（シェーダ側でエミッタカラーに乗算される）
                ubo->param1.x = emitter->time;
                ubo->param1.y = static_cast< f32 >( vertexNum );
                ubo->param1.z = EFT_STRIPE_NORMAL_MODE;
                ubo->param1.w = static_cast< f32 >( emitter->emitterData->ptcl.life );

                stripeTempUbo.Validate();

                // ストライプデータのUBO設定
                shader->BindEmitterPluginUniformBlock( &stripeTempUbo );

                nw::math::VEC4* vec4Buffer = reinterpret_cast<nw::math::VEC4*>( emitter->stripeVertexBuffer );
                u32 segemnet = emitter->stripeVertexBufferNum;
                ConnectionStripeVertexBuffer ConnectionVertexBuffer;
                ConnectionStripeVertexBuffer* vertexBuffer = &ConnectionVertexBuffer;
                vertexBuffer->pos  = &vec4Buffer[emitter->stripeVertexBufferCurIdx];
                vertexBuffer->dir  = &vec4Buffer[segemnet + emitter->stripeVertexBufferCurIdx];
                vertexBuffer->emat = &vec4Buffer[segemnet * 2 + emitter->stripeVertexBufferCurIdx];

                // 頂点を設定
                VertexBuffer::BindExtBuffer( shader->GetAttribute( Shader::EFT_ATTR_IDX_POSITION ),
                    sizeof(nw::math::VEC4) * vertexNum, 4, sizeof(nw::math::VEC4), vertexBuffer->pos );
                VertexBuffer::BindExtBuffer( shader->GetEmitterPluginAttribute( Shader::EFT_EMTR_PLUGIN_ATTR_IDX_0 ),
                    sizeof(nw::math::VEC4) * vertexNum, 4, sizeof(nw::math::VEC4), vertexBuffer->dir );
                VertexBuffer::BindExtBuffer( shader->GetEmitterPluginAttribute( Shader::EFT_EMTR_PLUGIN_ATTR_IDX_1 ),
                    sizeof(nw::math::VEC4) * vertexNum, 4, sizeof(nw::math::VEC4), vertexBuffer->emat );

                // 描画
                Render::DrawPrimitive( Render::PRIM_TYPE_TRIANGLE_STRIP, 0, vertexNum );

                if( stripeData->option == EFT_STRIPE_CROSS_MODE )
                {
                    // クロスの場合、再度描画する！
                    // ストライプUBOを設定する
                    TemporaryUniformBlockBuffer stripeCrossTempUbo;
                    ConnectedStripeUniformBlockParam* ubo2 = reinterpret_cast< ConnectedStripeUniformBlockParam* >( stripeCrossTempUbo.Alloc( system, sizeof( ConnectedStripeUniformBlockParam ) ) );
                    if ( !ubo2 ) { return; }

                    ubo2->param0.x = stripeData->headAlpha;
                    ubo2->param0.y = stripeData->tailAlpha;
                    ubo2->random = nw::math::VEC4( 1, 1, 1, 1 );
                    ubo2->color0 = nw::math::VEC4( 1, 1, 1, 1 ); // 連結式ストライプとしては白を入れておく（シェーダ側でエミッタカラーに乗算される）
                    ubo2->color1 = nw::math::VEC4( 1, 1, 1, 1 ); // 連結式ストライプとしては白を入れておく（シェーダ側でエミッタカラーに乗算される）
                    ubo2->param1.x = emitter->time;
                    ubo2->param1.y = static_cast< f32 >( vertexNum );
                    ubo2->param1.z = EFT_STRIPE_CROSS_MODE;
                    ubo2->param1.w = static_cast< f32 >( emitter->emitterData->ptcl.life );

                    stripeCrossTempUbo.Validate();

                    // ストライプデータのUBO設定
                    shader->BindEmitterPluginUniformBlock( &stripeCrossTempUbo );

                    // 描画
                    Render::DrawPrimitive( Render::PRIM_TYPE_TRIANGLE_STRIP, 0, vertexNum );
                }
            }
        }
    }
}

} // namespace eft2
} // namespace nw

#endif      // EFT_DEGRADATION_SPEC
