﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eft2_ShaderRes.h>
#include <nw/eft/eft2_Enum.h>
#include <nw/eft/eft2_EmitterRes.h>

namespace nw   {
namespace eft2 {

#ifdef EFT_OGL
ShaderCodeArg ShaderResource::gOriginalVshCodeArg;
ShaderCodeArg ShaderResource::gOriginalFshCodeArg;

ShaderCodeArg ShaderResource::gCurrnetVshCodeArg;
ShaderCodeArg ShaderResource::gCurrnetFshCodeArg;

GLchar*       ShaderResource::gCustomShaderCode[];
GLint         ShaderResource::gCustomShaderCodeSize[];

//------------------------------------------------------------------------------
//  シェーダコードテキスト出力
//------------------------------------------------------------------------------
bool ShaderResource::WriteVtxShaderCode( const char* writeTextPath )
{
    FILE* fp = NULL;
    errno_t err;

    #if EFT_IS_WIN
    err = fopen_s( &fp, writeTextPath, "wb" );
    #else
    fp = fopen( writeTextPath, "wb" );
    #endif
    if ( err != 0 || !fp )
    {
        return false;
    }
    else
    {
        fseek(fp, 0L, SEEK_SET);
        fwrite( ShaderResource::gCurrnetVshCodeArg.code[0], ShaderResource::gCurrnetVshCodeArg.codeSize[0], 1, fp );
        fwrite( ShaderResource::gCurrnetVshCodeArg.code[1], ShaderResource::gCurrnetVshCodeArg.codeSize[1], 1, fp );
        fwrite( ShaderResource::gCurrnetVshCodeArg.code[2], ShaderResource::gCurrnetVshCodeArg.codeSize[2], 1, fp );
        fwrite( ShaderResource::gCurrnetVshCodeArg.code[3], ShaderResource::gCurrnetVshCodeArg.codeSize[3], 1, fp );
        fwrite( ShaderResource::gCurrnetVshCodeArg.code[4], ShaderResource::gCurrnetVshCodeArg.codeSize[4], 1, fp );
        fwrite( ShaderResource::gCurrnetVshCodeArg.code[5], ShaderResource::gCurrnetVshCodeArg.codeSize[5], 1, fp );
        fwrite( ShaderResource::gCurrnetVshCodeArg.code[6], ShaderResource::gCurrnetVshCodeArg.codeSize[6], 1, fp );
        fwrite( ShaderResource::gCurrnetVshCodeArg.code[7], ShaderResource::gCurrnetVshCodeArg.codeSize[7], 1, fp );
        fwrite( ShaderResource::gCurrnetVshCodeArg.code[8], ShaderResource::gCurrnetVshCodeArg.codeSize[8], 1, fp );
        fwrite( ShaderResource::gCurrnetVshCodeArg.code[9], ShaderResource::gCurrnetVshCodeArg.codeSize[9], 1, fp );
        fwrite( ShaderResource::gCurrnetVshCodeArg.code[10], ShaderResource::gCurrnetVshCodeArg.codeSize[10], 1, fp );
        fclose( fp );
        return true;
    }
}
bool ShaderResource::WriteFragShaderCode( const char* writeTextPath )
{
    FILE* fp = NULL;
    errno_t err;

    #if EFT_IS_WIN
    err = fopen_s( &fp, writeTextPath, "wb" );
    #else
    fp = fopen( writeTextPath, "wb" );
    #endif
    if ( err != 0 || !fp )
    {
        return false;
    }
    else
    {
        fseek(fp, 0L, SEEK_SET);
        fwrite( ShaderResource::gCurrnetFshCodeArg.code[0], ShaderResource::gCurrnetFshCodeArg.codeSize[0], 1, fp );
        fwrite( ShaderResource::gCurrnetFshCodeArg.code[1], ShaderResource::gCurrnetFshCodeArg.codeSize[1], 1, fp );
        fwrite( ShaderResource::gCurrnetFshCodeArg.code[2], ShaderResource::gCurrnetFshCodeArg.codeSize[2], 1, fp );
        fwrite( ShaderResource::gCurrnetFshCodeArg.code[3], ShaderResource::gCurrnetFshCodeArg.codeSize[3], 1, fp );
        fwrite( ShaderResource::gCurrnetFshCodeArg.code[4], ShaderResource::gCurrnetFshCodeArg.codeSize[4], 1, fp );
        fwrite( ShaderResource::gCurrnetFshCodeArg.code[5], ShaderResource::gCurrnetFshCodeArg.codeSize[5], 1, fp );
        fwrite( ShaderResource::gCurrnetFshCodeArg.code[6], ShaderResource::gCurrnetFshCodeArg.codeSize[6], 1, fp );
        fwrite( ShaderResource::gCurrnetFshCodeArg.code[7], ShaderResource::gCurrnetFshCodeArg.codeSize[7], 1, fp );
        fwrite( ShaderResource::gCurrnetFshCodeArg.code[8], ShaderResource::gCurrnetFshCodeArg.codeSize[8], 1, fp );
        fwrite( ShaderResource::gCurrnetFshCodeArg.code[9], ShaderResource::gCurrnetFshCodeArg.codeSize[9], 1, fp );
        fwrite( ShaderResource::gCurrnetFshCodeArg.code[10], ShaderResource::gCurrnetFshCodeArg.codeSize[10], 1, fp );
        fclose( fp );
        return true;
    }
}

//------------------------------------------------------------------------------
//      エラー出力
//------------------------------------------------------------------------------
bool _checkShaderStatus( GLuint hShader )
{
    GLint   ret = 0;
    char*   err;

    glGetShaderiv( hShader, GL_COMPILE_STATUS, &ret);

    if ( ret == 0 )
    {
        GLint infoLen = 0;
        glGetShaderiv( hShader, GL_INFO_LOG_LENGTH, &infoLen );

        if ( infoLen > 1 )
        {
            err = static_cast<char *> ( malloc( infoLen + 1 ) );
            glGetShaderInfoLog( hShader, infoLen, &infoLen, err );
            char temp[65];
            s32 loop = (infoLen/64+1);
            for( s32 i = 0; i < loop; i++ )
            {
                if ( i == (loop-1) )
                {
                    #if EFT_IS_WIN
                    strncpy_s( temp, &err[i*64], infoLen%64 );
                    #else
                    strncpy( temp, &err[i*64], infoLen%64 );
                    #endif
                    temp[infoLen%64+1] = 0;
                }
                else
                {
                    #if EFT_IS_WIN
                    strncpy_s( temp, &err[i*64], 64 );
                    #else
                    strncpy( temp, &err[i*64], 64 );
                    #endif
                    temp[64] = 0;
                }
                EFT_PRINT( "%s", temp );
            }
            EFT_PRINT( "\n", temp );
            free( err );
            return false;
        }

        return false;
    }

    return true;
}

bool _checkProgramStatus( GLuint hProgram )
{
    GLint ret = 0;
    char*  err;

    glGetProgramiv( hProgram, GL_LINK_STATUS, &ret);

    if ( ret == 0 )
    {
        GLint infoLen = 0;
        glGetProgramiv( hProgram, GL_INFO_LOG_LENGTH, &infoLen );

        if ( infoLen > 1 )
        {
            err = static_cast<char *> ( malloc( infoLen + 1 ) );
            glGetProgramInfoLog( hProgram, infoLen, &infoLen, err );
            char temp[65];
            s32 loop = (infoLen/64+1);
            for( s32 i = 0; i < loop; i++ )
            {
                if ( i == (loop-1) )
                {
                    #if EFT_IS_WIN
                    strncpy_s( temp, &err[i*64], infoLen%64 );
                    #else
                    strncpy( temp, &err[i*64], infoLen%64 );
                    #endif
                    temp[infoLen%64+1] = 0;
                }
                else
                {
                    #if EFT_IS_WIN
                    strncpy_s( temp, &err[i*64], 64 );
                    #else
                    strncpy( temp, &err[i*64], 64 );
                    #endif
                    temp[64] = 0;
                }
                EFT_PRINT( "%s", temp );
            }
            EFT_PRINT( "\n", temp );
            free( err );
            return false;
        }
    }
    return true;
}

//---------------------------------------------------------------------------
//! @brief  カスタムシェーダコード用のバッファを初期化します。
//---------------------------------------------------------------------------
void ShaderResource::InitializeCustomShader()
{
    for( u32 id = 0; id < EFT_CUSTOM_SHADER_ID_MAX; id++ )
    {
        gCustomShaderCode[id]       = NULL;
        gCustomShaderCodeSize[id]   = 0;
    }
}

//---------------------------------------------------------------------------
//  カスタムシェーダコードを設定します。
//---------------------------------------------------------------------------
void ShaderResource::SetCustomShaderCode( CustomShaderID id, GLchar* code, GLint codeSize )
{
    gCustomShaderCode[id]       = code;
    gCustomShaderCodeSize[id]   = codeSize;
}

//---------------------------------------------------------------------------
//  初期化
//---------------------------------------------------------------------------
void ShaderResource::Initialize( const EmitterResource* emitterResSet, const char* addCompileDef )
{
    // フィールド利用フラグを生成
    u32 fieldFlag = 0;
    if ( emitterResSet->fieldRandomData ) fieldFlag |=  EFT_SHADER_KEY_FIELD_RANDOM;

    // シェーダキーを生成
    shaderKey.Initialize(
        emitterResSet->emitterData,
        emitterResSet->emitterPluginIndex,
        emitterResSet->emitterPluginData,
        fieldFlag,
        emitterResSet->fieldCustomData,
        addCompileDef );
}

//---------------------------------------------------------------------------
//! @brief  シェーダコンパイルを行います
//! @param[in] emitterResSet TBD
//! @param[in] addCompileDef TBD
//! @param[in] isCancelCustomShader TBD
//---------------------------------------------------------------------------
void ShaderResource::Compile( const EmitterResource* emitterResSet, bool isCancelCustomShader )
{
    // シェーダコンパイル
    _Compile( shaderKey.GetCompileSetting(),
              emitterResSet->emitterData->shader.customShaderIndex,
              emitterResSet->emitterPluginIndex,
              emitterResSet->emitterData->shader.eftCombinerShaderIndex,
              isCancelCustomShader );
}

//---------------------------------------------------------------------------
//  シェーダコンパイルを行います
//---------------------------------------------------------------------------
void ShaderResource::_Compile( const char* compileSetting, u32 customShaderIndex, u32 emitterPluginShaderIndex, u32 eftCombinerShaderIndex, bool isCancelCustomShader )
{
    gCurrnetVshCodeArg = gOriginalVshCodeArg;
    gCurrnetFshCodeArg = gOriginalFshCodeArg;

    const char* setting = compileSetting;

    static char _nonCustomShaderCode[]          = "// CustomShader None.\n\n";
    static char _nonEmiterPluginShaderCode[]    = "// EmitterPluginShader None.\n\n";
    static char _nonStreamOutShaderCode[]       = "// StreamOutShader None.\n\n";
    //static char _nonMainVshShaderCode[]         = "// MainVshShader None.\n\n";
    static char _nonCombinerShaderCode[]        = "// CombinerShader None.\n\n";

    // エミッタプラグインシェーダを一旦OFF
    gCurrnetFshCodeArg.emPlugin         = _nonEmiterPluginShaderCode;
    gCurrnetFshCodeArg.emPluginSize     = static_cast<GLint>( strlen( _nonEmiterPluginShaderCode ) );

    // エミッタプラグインがある場合、コードを挟み込む
    if( emitterPluginShaderIndex > 0 )
    {
        const u32 index = emitterPluginShaderIndex - 1;
        gCurrnetVshCodeArg.emPlugin     = gCurrnetVshCodeArg.emitterPluginCode[index];
        gCurrnetVshCodeArg.emPluginSize = gCurrnetVshCodeArg.emitterPluginCodeSize[index];
    }
    else
    {
        gCurrnetVshCodeArg.emPlugin         = _nonEmiterPluginShaderCode;
        gCurrnetVshCodeArg.emPluginSize     = static_cast<GLint>( strlen( _nonEmiterPluginShaderCode ) );
    }

    // カスタム有効の場合は、カスタムコードを挟み込む
    gCurrnetVshCodeArg.custom     = _nonCustomShaderCode;
    gCurrnetVshCodeArg.customSize = static_cast<GLint>( strlen( _nonCustomShaderCode ) );
    gCurrnetFshCodeArg.custom     = _nonCustomShaderCode;
    gCurrnetFshCodeArg.customSize = static_cast<GLint>( strlen( _nonCustomShaderCode ) );

    if ( customShaderIndex != 0 && !isCancelCustomShader )
    {
        u32 index = customShaderIndex - 1;
        if ( gCurrnetVshCodeArg.customCode[index] )
        {
            gCurrnetVshCodeArg.custom     = gCurrnetVshCodeArg.customCode[index];
            gCurrnetVshCodeArg.customSize = gCurrnetVshCodeArg.customCodeSize[index];
        }
        else
        {
            gCurrnetVshCodeArg.custom     = _nonCustomShaderCode;
            gCurrnetVshCodeArg.customSize = static_cast<GLint>( strlen( _nonCustomShaderCode ) );
        }

        if ( gCurrnetFshCodeArg.customCode[index] )
        {
            gCurrnetFshCodeArg.custom     = gCurrnetFshCodeArg.customCode[index];
            gCurrnetFshCodeArg.customSize = gCurrnetFshCodeArg.customCodeSize[index];
        }
        else
        {
            gCurrnetFshCodeArg.custom     = _nonCustomShaderCode;
            gCurrnetFshCodeArg.customSize = static_cast<GLint>( strlen( _nonCustomShaderCode ) );
        }
    }
    else
    {
        // カスタムシェーダ設定無しの場合は、Genaralコードのみを付加する
        if ( gCurrnetVshCodeArg.customGeneralCode )
        {
            gCurrnetVshCodeArg.custom     = gCurrnetVshCodeArg.customGeneralCode;
            gCurrnetVshCodeArg.customSize = gCurrnetVshCodeArg.customGeneralCodeSize;
        }
        if ( gCurrnetFshCodeArg.customGeneralCode )
        {
            gCurrnetFshCodeArg.custom     = gCurrnetFshCodeArg.customGeneralCode;
            gCurrnetFshCodeArg.customSize = gCurrnetFshCodeArg.customGeneralCodeSize;
        }
    }

    // 頂点シェーダでコンバイナシェーダは使わない
    gCurrnetVshCodeArg.combiner     = _nonCombinerShaderCode;
    gCurrnetVshCodeArg.combinerSize = static_cast<GLint>( strlen( _nonCombinerShaderCode ) );

    // コンバイナシェーダの場合は、programを書き換える,
    // コンバイナシェーダを使用しない場合は、EffectMakerでs32型の-1が指定される.
    // ランタイム側ではこれをu32型にキャストする.
    if ( eftCombinerShaderIndex != (u32)-1 )
    {
        gCurrnetFshCodeArg.combiner     = gCurrnetFshCodeArg.combinerCode[eftCombinerShaderIndex];
        gCurrnetFshCodeArg.combinerSize = gCurrnetFshCodeArg.combinerCodeSize[eftCombinerShaderIndex];
    }
    else
    {
        gCurrnetFshCodeArg.combiner     = _nonCombinerShaderCode;
        gCurrnetFshCodeArg.combinerSize = static_cast<GLint>( strlen( _nonCombinerShaderCode ) );
    }

    // ストリームアウトシェーダを除去
    gCurrnetVshCodeArg.soDef            = _nonStreamOutShaderCode;
    gCurrnetVshCodeArg.soDefSize        = static_cast<GLint>( strlen( _nonStreamOutShaderCode ) );
    gCurrnetVshCodeArg.streamout        = _nonStreamOutShaderCode;
    gCurrnetVshCodeArg.streamoutSize    = static_cast<GLint>( strlen( _nonStreamOutShaderCode ) );
    gCurrnetVshCodeArg.soDec            = _nonStreamOutShaderCode;
    gCurrnetVshCodeArg.soDecSize        = static_cast<GLint>( strlen( _nonStreamOutShaderCode ) );
    gCurrnetVshCodeArg.soCustom         = _nonStreamOutShaderCode;
    gCurrnetVshCodeArg.soCustomSize     = static_cast<GLint>( strlen( _nonStreamOutShaderCode ) );
    gCurrnetFshCodeArg.soDef            = _nonStreamOutShaderCode;
    gCurrnetFshCodeArg.soDefSize        = static_cast<GLint>( strlen( _nonStreamOutShaderCode ) );
    gCurrnetFshCodeArg.streamout        = _nonStreamOutShaderCode;
    gCurrnetFshCodeArg.streamoutSize    = static_cast<GLint>( strlen( _nonStreamOutShaderCode ) );
    gCurrnetFshCodeArg.soDec            = _nonStreamOutShaderCode;
    gCurrnetFshCodeArg.soDecSize        = static_cast<GLint>( strlen( _nonStreamOutShaderCode ) );
    gCurrnetFshCodeArg.soCustom         = _nonStreamOutShaderCode;
    gCurrnetFshCodeArg.soCustomSize     = static_cast<GLint>( strlen( _nonStreamOutShaderCode ) );

    gCurrnetVshCodeArg.codeNum      = sizeof( gCurrnetVshCodeArg.code ) / sizeof( GLchar* );
    gCurrnetFshCodeArg.codeNum      = sizeof( gCurrnetFshCodeArg.code ) / sizeof( GLchar* );
    gCurrnetVshCodeArg.setting      = setting;
    gCurrnetFshCodeArg.setting      = setting;
    gCurrnetVshCodeArg.settingSize  = static_cast<GLint>( strlen( setting ) );
    gCurrnetFshCodeArg.settingSize  = static_cast<GLint>( strlen( setting ) );

    // TODO:特定スペックではオンラインで実機用に分岐する必要あり
    gCurrnetVshCodeArg.specDec      = gOriginalVshCodeArg.specDecGenericCode;
    gCurrnetFshCodeArg.specDec      = gOriginalFshCodeArg.specDecGenericCode;
    gCurrnetVshCodeArg.specDecSize  = gOriginalVshCodeArg.specDecGenericCodeSize;
    gCurrnetFshCodeArg.specDecSize  = gOriginalFshCodeArg.specDecGenericCodeSize;

#ifdef NW_PLATFORM_WIN32
    gCurrnetVshCodeArg.specDec      = gOriginalVshCodeArg.specDecGenericCode;
    gCurrnetFshCodeArg.specDec      = gOriginalFshCodeArg.specDecGenericCode;
    gCurrnetVshCodeArg.specDecSize  = gOriginalVshCodeArg.specDecGenericCodeSize;
    gCurrnetFshCodeArg.specDecSize  = gOriginalFshCodeArg.specDecGenericCodeSize;
#else
    gCurrnetVshCodeArg.specDec      = gOriginalVshCodeArg.specDecTargetCode;
    gCurrnetFshCodeArg.specDec      = gOriginalFshCodeArg.specDecTargetCode;
    gCurrnetVshCodeArg.specDecSize  = gOriginalVshCodeArg.specDecTargetCodeSize;
    gCurrnetFshCodeArg.specDecSize  = gOriginalFshCodeArg.specDecTargetCodeSize;
#endif


    gOriginalVshCodeArg.setting     = setting;
    gOriginalVshCodeArg.settingSize = static_cast<GLint>( strlen( setting ) );
    gOriginalFshCodeArg.setting     = setting;
    gOriginalFshCodeArg.settingSize = static_cast<GLint>( strlen( setting ) );

    // TODO:特定スペックではオンラインで実機用に分岐する必要あり
    gOriginalVshCodeArg.specDec     = gOriginalVshCodeArg.specDecGenericCode;
    gOriginalVshCodeArg.specDecSize = gOriginalVshCodeArg.specDecGenericCodeSize;
    gOriginalFshCodeArg.specDec     = gOriginalFshCodeArg.specDecGenericCode;
    gOriginalFshCodeArg.specDecSize = gOriginalFshCodeArg.specDecGenericCodeSize;

    // シェーダコンパイル
    if ( !_CompileCurrentShader( false ) )
    {
        isError = true;
        //EFT_ERR( "Shader Compile Error.\n" );
    }
    else
    {
        isError = false;
    }
}

//---------------------------------------------------------------------------
//! @brief  ストリームアウトシェーダの初期化
//---------------------------------------------------------------------------
void ShaderResource::InitializeStreamOutShader()
{
#ifndef EFT_DEGRADATION_SPEC
    static char _streamOutDefinition[]          = "#define COMPUTE_SHADER\n\n";
    static char _nonCustomShaderCode[]          = "// CustomShader None.\n\n";
    static char _nonEmiterPluginShaderCode[]    = "// EmitterPluginShader None.\n\n";
    static char _nonStreamOutShaderCode[]       = "// StreamOutShader None.\n\n";
    static char _nonMainVshShaderCode[]         = "// MainVshShader None.\n\n";
    static char _nonMainFshShaderCode[]         = "void main(){}\n";
    static char _nonCombinerShaderCode[]        = "// CombinerShader None.\n\n";
    static char _SettingShaderCode[]            = "// Setting None.\n\n";


    gCurrnetVshCodeArg = gOriginalVshCodeArg;
    gCurrnetFshCodeArg = gOriginalFshCodeArg;

    // TODO:特定スペックではオンラインで実機用に分岐する必要あり
    gCurrnetVshCodeArg.specDec      = gOriginalVshCodeArg.specDecGenericCode;
    gCurrnetFshCodeArg.specDec      = gOriginalFshCodeArg.specDecGenericCode;
    gCurrnetVshCodeArg.specDecSize  = gOriginalVshCodeArg.specDecGenericCodeSize;
    gCurrnetFshCodeArg.specDecSize  = gOriginalFshCodeArg.specDecGenericCodeSize;

    gCurrnetVshCodeArg.setting      = _SettingShaderCode;
    gCurrnetFshCodeArg.setting      = _SettingShaderCode;
    gCurrnetVshCodeArg.settingSize  = strlen( _SettingShaderCode );
    gCurrnetFshCodeArg.settingSize  = strlen( _SettingShaderCode );

    // カスタムシェーダGenaralコードのみを付加する
    if ( gCurrnetVshCodeArg.customGeneralCode )
    {
        gCurrnetVshCodeArg.custom     = gCurrnetVshCodeArg.customGeneralCode;
        gCurrnetVshCodeArg.customSize = gCurrnetVshCodeArg.customGeneralCodeSize;
    }
    if ( gCurrnetFshCodeArg.customGeneralCode )
    {
        gCurrnetFshCodeArg.custom     = gCurrnetFshCodeArg.customGeneralCode;
        gCurrnetFshCodeArg.customSize = gCurrnetFshCodeArg.customGeneralCodeSize;
    }

    // ストライプシェーダは無効
    gCurrnetVshCodeArg.emPlugin       = _nonEmiterPluginShaderCode;
    gCurrnetVshCodeArg.emPluginSize   = strlen( _nonEmiterPluginShaderCode );
    gCurrnetFshCodeArg.emPlugin       = _nonEmiterPluginShaderCode;
    gCurrnetFshCodeArg.emPluginSize   = strlen( _nonEmiterPluginShaderCode );

    // 通常頂点シェーダを除去して
    gCurrnetVshCodeArg.program      = _nonMainVshShaderCode;
    gCurrnetVshCodeArg.programSize  = strlen( _nonMainVshShaderCode );
    gCurrnetFshCodeArg.program      = _nonMainFshShaderCode;
    gCurrnetFshCodeArg.programSize  = strlen( _nonMainFshShaderCode );

    // コンバイナシェーダを除去する
    gCurrnetVshCodeArg.combiner      = _nonCombinerShaderCode;
    gCurrnetVshCodeArg.combinerSize  = strlen( _nonCombinerShaderCode );
    gCurrnetFshCodeArg.combiner      = _nonCombinerShaderCode;
    gCurrnetFshCodeArg.combinerSize  = strlen( _nonCombinerShaderCode );

    // ストリームアウトシェーダを付加する
    gCurrnetVshCodeArg.soDef            = reinterpret_cast<const GLchar*>( _streamOutDefinition );
    gCurrnetVshCodeArg.soDefSize        = strlen( _streamOutDefinition );
    gCurrnetFshCodeArg.streamout        = _nonStreamOutShaderCode;
    gCurrnetFshCodeArg.streamoutSize    = strlen( _nonStreamOutShaderCode );
    gCurrnetFshCodeArg.soDec            = _nonStreamOutShaderCode;
    gCurrnetFshCodeArg.soDecSize        = strlen( _nonStreamOutShaderCode );
    gCurrnetFshCodeArg.soCustom         = _nonStreamOutShaderCode;
    gCurrnetFshCodeArg.soCustomSize     = strlen( _nonStreamOutShaderCode );

    gCurrnetVshCodeArg.codeNum      = sizeof( gCurrnetVshCodeArg.code ) / sizeof( GLchar* );
    gCurrnetFshCodeArg.codeNum      = sizeof( gCurrnetFshCodeArg.code ) / sizeof( GLchar* );

    // シェーダコンパイル
    if ( !_CompileCurrentShader( true ) )
    {
        isError = true;
    }
    else
    {
        isError = false;
    }
#endif
}

//------------------------------------------------------------------------------
//  gCurrnetVshCodeArg/gCurrnetFshCodeArg の内容でシェーダコンパイルを行う
//------------------------------------------------------------------------------
bool ShaderResource::_CompileCurrentShader( bool isStreamOut )
{
#ifdef EFT_DEGRADATION_SPEC
    EFT_UNUSED_VARIABLE( isStreamOut );
#endif

    // コンパイルしたコードを出力
    // ShaderResource::WriteVtxShaderCode(  "temp.vsh" );
    // ShaderResource::WriteFragShaderCode( "temp.fsh" );

    // 頂点シェーダコンパイル
    vtxShader = glCreateShader( GL_VERTEX_SHADER );
    glShaderSource( vtxShader,
                    gCurrnetVshCodeArg.codeNum,
                    gCurrnetVshCodeArg.code,
                    gCurrnetVshCodeArg.codeSize );
    glCompileShader( vtxShader );
    if ( !_checkShaderStatus( vtxShader ) )
    {
        EFT_PRINT( "[EFT] Vertex Shader Compile Error!! ------------------------------\n");
        return false;
    }

    // フラグメントシェーダコンパイル
    fragShader = glCreateShader( GL_FRAGMENT_SHADER );
    glShaderSource( fragShader,
                    gCurrnetFshCodeArg.codeNum,
                    gCurrnetFshCodeArg.code,
                    gCurrnetFshCodeArg.codeSize );
    glCompileShader( fragShader );
    if ( !_checkShaderStatus( fragShader ) )
    {
        EFT_PRINT( "[EFT] Fragment Shader Compile Error!! ------------------------------\n");
        return false;
    }

    // リンク
    program = glCreateProgram();
    glAttachShader( program, vtxShader  );
    glAttachShader( program, fragShader );

#ifndef EFT_DEGRADATION_SPEC
    if( isStreamOut )
    {
        const static char *varyings[] = { "sysOutPos", "sysOutVec" };
        glTransformFeedbackVaryings( program, 2, varyings, GL_SEPARATE_ATTRIBS );
    }
#endif

    glLinkProgram( program );
    if ( !_checkProgramStatus( program ) )
    {
        EFT_PRINT( "[EFT] Shader Link Error!! ------------------------------\n");
        return false;
    }

    EFT_GLERR_CHECK();
    return true;
}

//---------------------------------------------------------------------------
//  シェーダコードを設定します。
//---------------------------------------------------------------------------
void ShaderResource::SetShaderCode( const ShaderCodeArg& vshArg, const ShaderCodeArg& fshArg )
{
    gOriginalVshCodeArg     = vshArg;
    gOriginalFshCodeArg     = fshArg;
}
#endif


#if EFT_GX2
//---------------------------------------------------------------------------
//  初期化
//---------------------------------------------------------------------------
void ShaderResource::Initialize( Heap* heap, void* binary, u32 vertexIdx, u32 fragIdx )
{
    attribsNum  = 0;
    isInitialized = false;
    fetchShaderBuf = NULL;

    vertexShader = GFDGetVertexShaderPointer( vertexIdx, binary );
    fragShader   = GFDGetPixelShaderPointer( fragIdx, binary );
    EFT_ASSERT(vertexShader != NULL && fragShader != NULL && "Shader file malformed");

}
#endif

//---------------------------------------------------------------------------
//  終了処理
//---------------------------------------------------------------------------
void ShaderResource::Finalize( Heap* heap )
{
#ifdef EFT_OGL
    EFT_UNUSED_VARIABLE( heap );
    if ( program    != EFT_INVALID_SHADER_ID ) glDeleteProgram( program );
    if ( vtxShader  != EFT_INVALID_SHADER_ID ) glDeleteShader( vtxShader );
    if ( fragShader != EFT_INVALID_SHADER_ID ) glDeleteShader( fragShader );
#endif
#if EFT_GX2
    if ( heap && fetchShaderBuf )
    {
        heap->Free( fetchShaderBuf );
        fetchShaderBuf = NULL;
    }
#endif
}

//---------------------------------------------------------------------------
//  無効化
//---------------------------------------------------------------------------
void ShaderResource::Invalidate()
{
#ifdef EFT_OGL
    program    = static_cast<u32>( EFT_INVALID_SHADER_ID );
    vtxShader  = static_cast<u32>( EFT_INVALID_SHADER_ID );
    fragShader = static_cast<u32>( EFT_INVALID_SHADER_ID );
    shaderKey.Invalidate();
#endif
#if EFT_GX2
    vertexShader   = NULL;
    fragShader    = NULL;
    fetchShaderBuf = NULL;
    attribsNum     = 0;
    for( u32 i = 0; i < cAttributeMax; i ++ )
    {
        attribsIndex[i] = EFT_INVALID_SHADER_ATTRIBUTE;
    }
#endif
}

//------------------------------------------------------------------------------
//  シェーダを有効にする
//------------------------------------------------------------------------------
void ShaderResource::Bind() const
{
#ifdef EFT_OGL
    glUseProgram( program );
    EFT_GLERR_CHECK();
#endif
#if EFT_GX2
    GX2SetShaders( &fetchShader, vertexShader, fragShader );
#endif
}


//------------------------------------------------------------------------------
//  ユニフォームロケーションを取得する
//------------------------------------------------------------------------------
u32 ShaderResource::GetUniformLocation( const char* name ) const
{
#ifdef EFT_OGL
    {
        GLuint uniformLoc = glGetUniformLocation( program, name );
        EFT_GLERR_CHECK();
        return uniformLoc;
    }
#else
    EFT_UNUSED_VARIABLE( name );
    return static_cast<u32>( EFT_INVALID_SHADER_UNIFORM_REGISTER_LOCATION );
#endif
}

//------------------------------------------------------------------------------
//  アトリビュートを取得する
//------------------------------------------------------------------------------
u32 ShaderResource::GetAttribute( const char* name, u32 index, VertexFormat fmt, u32 offset, bool instancingAttr, bool endianSwap )
{
#ifdef EFT_OGL
    {
        EFT_UNUSED_VARIABLE( index );
        EFT_UNUSED_VARIABLE( fmt );
        EFT_UNUSED_VARIABLE( offset );
        EFT_UNUSED_VARIABLE( instancingAttr );
        EFT_UNUSED_VARIABLE( endianSwap );
        GLuint attr = glGetAttribLocation( program, name );
        EFT_GLERR_CHECK();
        return attr;
    }
#endif
#if EFT_GX2
    {
        EFT_ASSERT( attribsNum < cAttributeMax );

        u32 attrLoc = GX2GetVertexAttribVarLocation( vertexShader, name );
        if ( attrLoc == EFT_INVALID_SHADER_ATTRIBUTE ) return EFT_INVALID_SHADER_ATTRIBUTE;

        attribsIndex[attrLoc] = index;
        GX2InitAttribStream( &attribs[attribsNum], attrLoc, index, offset * sizeof(f32), static_cast<GX2AttribFormat>(fmt) );

        if ( instancingAttr )
        {
            attribs[attribsNum].indexType    = GX2_ATTRIB_INDEX_INSTANCE_ID;
            attribs[attribsNum].aluDivisor   = 1;
        }
        if ( endianSwap )
        {
            attribs[attribsNum].endianSwap = GX2_ENDIANSWAP_NONE;
        }
        attribsNum++;
        return index;
    }
#endif
}

//------------------------------------------------------------------------------
//  フラグメントテクスチャ サンプラーロケーションを取得する
//------------------------------------------------------------------------------
u32 ShaderResource::GetFragmentSamplerLocation( const char* name )
{
#ifdef EFT_OGL
    u32 loc = glGetUniformLocation( program, name );
    EFT_GLERR_CHECK();
    return loc;
#endif
#if EFT_GX2
    s32 loc = GX2GetPixelSamplerVarLocation( fragShader, name );
    return static_cast<u32>(loc);
#endif
}

//------------------------------------------------------------------------------
//  頂点テクスチャ サンプラーロケーションを取得する
//------------------------------------------------------------------------------
u32 ShaderResource::GetVertexSamplerLocation( const char* name )
{
#ifdef EFT_OGL
    u32 loc = glGetUniformLocation( program, name );
    EFT_GLERR_CHECK();
    return loc;
#endif
#if EFT_GX2
    s32 loc = GX2GetVertexSamplerVarLocation( vertexShader, name );
    return static_cast<u32>(loc);
#endif
}

#if EFT_GX2
//------------------------------------------------------------------------------
//  シェーダをセットアップする
//------------------------------------------------------------------------------
void ShaderResource::SetupShader( Heap* heap )
{
    EFT_ASSERT( !fetchShaderBuf );

    // フェッチシェーダーのセットアップ
    u32 allocSize = GX2CalcFetchShaderSize( attribsNum );
    allocSize = nw::eft2::RoundUp( allocSize, 32 );
    fetchShaderBuf = heap->Alloc( GX2CalcFetchShaderSize( attribsNum ), GX2_SHADER_ALIGNMENT );
    EFT_ASSERT( fetchShaderBuf );

    GX2InitFetchShader( &fetchShader, fetchShaderBuf, attribsNum, attribs );
    DCFlushRange( fetchShaderBuf, GX2CalcFetchShaderSize( attribsNum ) );
    isInitialized = true;
}
#endif










#if EFT_GX2
//------------------------------------------------------------------------------
// Read Vertex Shader from buffer
//------------------------------------------------------------------------------
bool ShaderResourceBase::_DEMOGFDReadVertexShader( nw::eft2::Heap* heap, GX2VertexShader **ppShader, u32 index, const void *pData )
{
    GX2VertexShader *pHeader;
    void *pProgram;
    u32 ret;
    u32 headerSize;
    u32 programSize;

    // Check inputs
    if(pData == NULL || ppShader == NULL)
    {
        return FALSE;
    }

    // Check index number of shaders
    if(index >= GFDGetVertexShaderCount(pData))
    {
        return FALSE;
    }

    // Get the size
    headerSize = GFDGetVertexShaderHeaderSize(index, pData);
    headerSize =  nw::eft2::RoundUp( headerSize, 32 );
    programSize = GFDGetVertexShaderProgramSize(index, pData);
    programSize =  nw::eft2::RoundUp( programSize, 32 );

    if(!headerSize || !programSize)
        return FALSE;

    // Alloc header
    pHeader = (GX2VertexShader *)heap->Alloc( headerSize, PPC_IO_BUFFER_ALIGN );

    // Alloc shader program (Must be aligned on 256 byte boundary.)
    //    pProgram = (void *)DEMOGfxAllocMEM2(programSize, GX2_SHADER_ALIGNMENT);
    pProgram = heap->Alloc( programSize, GX2_SHADER_ALIGNMENT );

    // Get the shader structure and program
    // from file buffer into user aligned buffer which created above
    ret = GFDGetVertexShader(pHeader, pProgram, index, pData);

    if(ret)
    {
        // Flush CPU caches holding our shader so the GPU can use them
        DCFlushRange( pHeader->shaderPtr, pHeader->shaderSize );
        *ppShader = pHeader;
    }
    else
    {
        OSReport("Warning: Invalid Vertex Shader :%d", ret);
        // Free if fail
        if(pHeader)
            heap->Free(pHeader);
        if(pProgram)
            heap->Free(pProgram);
    }
    return ret;
}

// Read Pixel Shader from buffer
bool ShaderResourceBase::_DEMOGFDReadPixelShader( nw::eft2::Heap* heap, GX2PixelShader **ppShader, u32 index, const void *pData )
{
    GX2PixelShader *pHeader;
    void *pProgram;
    u32 ret;
    u32 headerSize;
    u32 programSize;

    // Check inputs
    if(pData == NULL || ppShader == NULL)
    {
        return FALSE;
    }

    // Check index number of shaders
    if(index >= GFDGetPixelShaderCount(pData))
    {
        return FALSE;
    }
    // Get the size
    headerSize = GFDGetPixelShaderHeaderSize(index, pData);
    headerSize =  nw::eft2::RoundUp( headerSize, 32 );
    programSize = GFDGetPixelShaderProgramSize(index, pData);
    programSize =  nw::eft2::RoundUp( programSize, 32 );

    if(!headerSize || !programSize)
        return FALSE;

    // Alloc header
    pHeader = (GX2PixelShader *)heap->Alloc( headerSize, PPC_IO_BUFFER_ALIGN );

    // Alloc shader program (Must be aligned on 256 byte boundary.)
    //    pProgram = (void *)DEMOGfxAllocMEM2(programSize, GX2_SHADER_ALIGNMENT);
    pProgram = heap->Alloc( programSize, GX2_SHADER_ALIGNMENT );

    // Get the shader structure and program
    // from file buffer into user aligned buffer which created above
    ret = GFDGetPixelShader(pHeader, pProgram, index, pData);

    if(ret)
    {
        // Flush CPU caches holding our shader so the GPU can use them
        DCFlushRange( pHeader->shaderPtr, pHeader->shaderSize );

        *ppShader = pHeader;
    }
    else
    {
        OSReport("Warning: Invalid Pixel Shader :%d", ret);
        // Free if fail
        if(pHeader)
            heap->Free(pHeader);
        if(pProgram)
            heap->Free(pProgram);
    }
    return ret;
}
#endif      // EFT_GX2


#ifdef EFT_OGL
//------------------------------------------------------------------------------
//  シェーダコンパイルを行う
//------------------------------------------------------------------------------
bool ShaderResourceBase::Compile( const GLchar* vshShaderCode, u32 vshShaderCodeSize,
                                 const GLchar* fshShaderCode, u32 fshShaderCodeSize )
{
    // 頂点シェーダコンパイル
    vtxShader = glCreateShader( GL_VERTEX_SHADER );
    glShaderSource(  vtxShader, 1, &vshShaderCode, reinterpret_cast<const GLint *>( &vshShaderCodeSize ) );
    glCompileShader( vtxShader );
    if ( !_checkShaderStatus( vtxShader ) )
    {
        EFT_PRINT( "[EFT] Vertex Shader Compile Error!! ------------------------------\n");
        return false;
    }

    // フラグメントシェーダコンパイル
    fragShader = glCreateShader( GL_FRAGMENT_SHADER );
    glShaderSource(  fragShader, 1, &fshShaderCode, reinterpret_cast<const GLint *>( &fshShaderCodeSize ) );
    glCompileShader( fragShader );
    if ( !_checkShaderStatus( fragShader ) )
    {
        EFT_PRINT( "[EFT] Fragment Shader Compile Error!! ------------------------------\n");
        return false;
    }

    // リンク
    program = glCreateProgram();
    glAttachShader( program, vtxShader  );
    glAttachShader( program, fragShader );
    glLinkProgram( program );
    if ( !_checkProgramStatus( program ) )
    {
        EFT_PRINT( "[EFT] Shader Link Error!! ------------------------------\n");
        return false;
    }

    EFT_GLERR_CHECK();
    return true;
}
#endif

//---------------------------------------------------------------------------
//  終了処理
//---------------------------------------------------------------------------
bool ShaderResourceBase::Finalize( Heap* heap )
{
#ifdef EFT_OGL
    EFT_UNUSED_VARIABLE( heap );
    if ( program    != EFT_INVALID_SHADER_ID ) glDeleteProgram( program );
    if ( vtxShader  != EFT_INVALID_SHADER_ID ) glDeleteShader( vtxShader );
    if ( fragShader != EFT_INVALID_SHADER_ID ) glDeleteShader( fragShader );
#endif

    return true;
}

//------------------------------------------------------------------------------
//  アトリビュートを取得する
//------------------------------------------------------------------------------
u32 ShaderResourceBase::GetAttribute( const char* name, u32 index, VertexFormat fmt, u32 offset, bool instancingAttr, bool endianSwap )
{
#ifdef EFT_OGL
    EFT_UNUSED_VARIABLE( index );
    EFT_UNUSED_VARIABLE( fmt );
    EFT_UNUSED_VARIABLE( offset );
    EFT_UNUSED_VARIABLE( instancingAttr );
    EFT_UNUSED_VARIABLE( endianSwap );

    GLuint attr = glGetAttribLocation( program, name );
    EFT_GLERR_CHECK();
    return attr;
#endif
#if EFT_GX2
    EFT_ASSERT( attribsNum < cAttributeMax );

    u32 attrLoc = GetAttributeLocation( name );
    if ( attrLoc == EFT_INVALID_SHADER_ATTRIBUTE ) return EFT_INVALID_SHADER_ATTRIBUTE;

    attribsIndex[attrLoc] = index;
    GX2InitAttribStream( &attribs[attribsNum], attrLoc, index, offset * sizeof(f32), static_cast<GX2AttribFormat>(fmt) );

    if ( instancingAttr )
    {
        attribs[attribsNum].indexType    = GX2_ATTRIB_INDEX_INSTANCE_ID;
        attribs[attribsNum].aluDivisor   = 1;
    }
    if ( endianSwap )
    {
        attribs[attribsNum].endianSwap = GX2_ENDIANSWAP_NONE;
    }
    attribsNum++;
    return index;
#endif
}

//------------------------------------------------------------------------------
//  ユニフォームロケーションを取得する
//------------------------------------------------------------------------------
u32 ShaderResourceBase::GetVertexUniformLocation( const char* name )
{
#ifdef EFT_OGL
    s32 loc = glGetUniformLocation( program, name );
    EFT_GLERR_CHECK();
    return static_cast<u32>(loc);
#endif
#if EFT_GX2
    return GX2GetVertexUniformVarOffset( vertexShader, name );
#endif
}

//------------------------------------------------------------------------------
//  ユニフォームロケーションを取得する
//------------------------------------------------------------------------------
u32 ShaderResourceBase::GetFragmentUniformLocation( const char* name )
{
#ifdef EFT_OGL
    return GetVertexUniformLocation( name );
#endif
#if EFT_GX2
    return GX2GetPixelUniformVarOffset( fragShader, name );
#endif
}

//---------------------------------------------------------------------------
//  フラグメントサンプラーロケーションを取得する
//---------------------------------------------------------------------------
u32 ShaderResourceBase::GetFragmentSamplerLocation( const char* name )
{
#ifdef EFT_OGL
    s32 loc = glGetUniformLocation( program, name );
    EFT_GLERR_CHECK();
    return static_cast<u32>(loc);
#endif
#if EFT_GX2
    s32 loc = GX2GetPixelSamplerVarLocation( fragShader, name );
    return static_cast<u32>(loc);
#endif

}

//---------------------------------------------------------------------------
//  シェーダーをセットアップします
//---------------------------------------------------------------------------
void ShaderResourceBase::SetupShader( Heap* heap )
{
#ifdef EFT_OGL
    EFT_UNUSED_VARIABLE( heap );
#endif
#if EFT_GX2
    EFT_ASSERT( !fetchShaderBuf );

    // フェッチシェーダーのセットアップ
    fetchShaderBuf = heap->Alloc( GX2CalcFetchShaderSize( attribsNum ), GX2_SHADER_ALIGNMENT );
    EFT_ASSERT( fetchShaderBuf );

    GX2InitFetchShader( &fetchShader, fetchShaderBuf, attribsNum, attribs );

    // シェーダバインドをDL化
    displayList = reinterpret_cast<u8*>(
        heap->Alloc( DISPLAY_LSIT_SIZE, GX2_SHADER_ALIGNMENT ) );
    EFT_ASSERT( displayList );
    memset( displayList, 0, DISPLAY_LSIT_SIZE );
    DCFlushRange( displayList, DISPLAY_LSIT_SIZE );
    DCInvalidateRange( displayList, DISPLAY_LSIT_SIZE );
    GX2BeginDisplayList( displayList, DISPLAY_LSIT_SIZE );
    {
        GX2SetShaders( &fetchShader, vertexShader, fragShader );
    }
    displayListSize = GX2EndDisplayList( displayList );
    EFT_ASSERT( displayListSize<DISPLAY_LSIT_SIZE );

#endif
}

//------------------------------------------------------------------------------
//  シェーダを有効にする
//------------------------------------------------------------------------------
void ShaderResourceBase::Bind() const
{
#ifdef EFT_OGL
    glUseProgram( program );
    EFT_GLERR_CHECK();
#endif
#if EFT_GX2
    GX2CallDisplayList( displayList, displayListSize );
#endif
}

#if EFT_GX2
//------------------------------------------------------------------------------
//      シェーダをセットアップする
//------------------------------------------------------------------------------
bool ShaderResourceBase::CreateShader( Heap* heap, const void* gshBuffer, u32 gshBufferSize )
{
    attribsNum  = 0;
    isInitialized = false;
    fetchShaderBuf = NULL;

    BOOL fOKv = _DEMOGFDReadVertexShader(   heap, &vertexShader, 0, gshBuffer );
    BOOL fOKp = _DEMOGFDReadPixelShader(    heap, &fragShader, 0, gshBuffer );

    ASSERT(fOKv && fOKp && "Shader file malformed");
    return true;
}

bool ShaderResourceBase::CreateShader( Heap* heap, const void* gshBuffer, u32 gshBufferSize, u32 index )
{
    attribsNum  = 0;
    isInitialized = false;
    fetchShaderBuf = NULL;

    BOOL fOKv = _DEMOGFDReadVertexShader(   heap, &vertexShader,  index, gshBuffer );
    BOOL fOKp = _DEMOGFDReadPixelShader(    heap, &fragShader,   index, gshBuffer );

    ASSERT(fOKv && fOKp && "Shader file malformed");
    return true;
}
//------------------------------------------------------------------------------
//      アトリビュートロケーションを取得する
//------------------------------------------------------------------------------
u32 ShaderResourceBase::GetAttributeLocation( const char* name )
{
    s32 loc = GX2GetVertexAttribVarLocation( vertexShader, name );
    return static_cast<u32>(loc);
}
#endif



} // namespace eft2
} // namespace nw

