﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eft2_ShaderManager.h>
#include <nw/eft/eft2_EmitterRes.h>
#include <nw/eft/eft2_Heap.h>
#include <nw/eft/eft2_Shader.h>
#include <nw/eft/eft2_Misc.h>

namespace nw   {
namespace eft2 {

#if EFT_GX2
BOOL _SetTagDAT( u32 Offset ) { return Offset | GFD_TAG_DAT; }
u32  _CleanTag( u32 Offset )  { return Offset & ~GFD_TAG_MASK; }

void _RevertBlock(u32 nBytesBlock, char *pData)
{
    if(pData == NULL) return;

    // GFDBlockRelocationHeaderはGFDBlockの最後にあり、サイズが固定です。
    u32 size = sizeof(GFDBlockRelocationHeader);

    GFDBlockRelocationHeader *pTrailer =  (GFDBlockRelocationHeader *)(pData + nBytesBlock - size);

    // 既に配置されている場合は、配置アドレスをオフセット値に戻す。
    if(pTrailer-> basePatchAddress != 0)
    {
        u32 MainOffset   = _CleanTag( pTrailer->dataOffset );
        u32 PTableOffset = _CleanTag( pTrailer->patchTableOffset );
        u32 NPatches     = pTrailer->patchTableOffsetNumber;
        u32 i;
        u32 Offset;

        for(i = 0; i < NPatches; i++)
        {
            Offset = *((u32*) (pData + PTableOffset + 4*i));
            if(Offset != 0)
            {
                u32 *pPatchLoc;
                pPatchLoc = ((u32*) (pData +  _CleanTag(Offset)) );

                u32 tableOffset = *pPatchLoc - pTrailer->basePatchAddress;
                *pPatchLoc = (u32)( _SetTagDAT( tableOffset ) );
            }
        }
        pTrailer->basePatchAddress = 0;
    }
}

//---------------------------------------------------------------------------
//  シェーダバイナリを配置前に戻す。
//---------------------------------------------------------------------------
void _RevertGX2ShaderBinary( void* pData )
{
    char *pDataStruct;
    GFDBlockHeader* pBlockHeader;
    char* pHeader = NULL;

    pDataStruct = (char*)pData + ((GFDHeader*)pData)->size; // jump over the header
    pBlockHeader = (GFDBlockHeader *)pDataStruct;

    while ( GFD_BLOCK_TYPE_END != pBlockHeader->type )
    {
        pDataStruct += pBlockHeader->size;

        switch(pBlockHeader->type)
        {
        case GFD_BLOCK_TYPE_GX2_PSH_HEADER:
        case GFD_BLOCK_TYPE_GX2_VSH_HEADER:
            pHeader = (char*)pDataStruct;
            _RevertBlock(pBlockHeader->dataSize,(char *)pHeader);
            break;
        default:
            break;
        }
        pDataStruct += pBlockHeader->dataSize;
        pBlockHeader = (GFDBlockHeader *)pDataStruct;
    }
}
#endif

//---------------------------------------------------------------------------
//  コンストラクタ
//---------------------------------------------------------------------------
ShaderManager::ShaderManager()
{
    m_ShaderNum      = 0;
    m_ShaderMaxNum   = 0;
    m_ShaderArray    = NULL;

#if EFT_GX2
    m_ShaderBinary          = NULL;
    m_TotalVertexShader     = 0;
    m_TotalPixelShader      = 0;
    m_TotalVertexShaderSize = 0;
    m_TotalPixelShaderSize  = 0;

#endif
}

//---------------------------------------------------------------------------
//  初期化処理
//---------------------------------------------------------------------------
void ShaderManager::Initialize( Heap* heap, u32 shaderMaxNum )
{
    m_ShaderNum      = 0;
    m_ShaderMaxNum   = shaderMaxNum;
    m_ShaderArray    = reinterpret_cast<Shader*>( heap->Alloc( sizeof(Shader) * m_ShaderMaxNum ) );
    EFT_NULL_ASSERT( m_ShaderArray );

    for ( u32 i = 0; i < m_ShaderMaxNum; i++ )
    {
        m_ShaderArray[i].Invalidate();
    }
}

//---------------------------------------------------------------------------
//  終了処理
//---------------------------------------------------------------------------
void ShaderManager::Finalize( Heap* heap )
{
    if ( m_ShaderArray )
    {
        for ( u32 i = 0; i < m_ShaderNum; i++ )
        {
            m_ShaderArray[i].Finalize( heap );
        }
        heap->Free( m_ShaderArray );
    }
}

//---------------------------------------------------------------------------
//  エミッタリソースセットを追加する
//---------------------------------------------------------------------------
#ifdef EFT_OGL
#ifndef EFT_DEGRADATION_SPEC
u32 ShaderManager::AddEmitterResSet( const EmitterResource* emResSet, const char* addCompileDef, bool delayCompile )
{
    EFT_ASSERT( m_ShaderNum < m_ShaderMaxNum );
    EFT_NULL_ASSERT( m_ShaderArray );

    u32 shaderNo = m_ShaderNum;

    m_ShaderArray[shaderNo].Invalidate();
    m_ShaderArray[shaderNo].Initialize( emResSet, addCompileDef );

    if ( !delayCompile )
    {
        if ( m_ShaderArray[shaderNo].CompileShader() )
        {
            m_ShaderArray[shaderNo].InitializeLocation();
            #ifdef EFT_USE_UNIFORM_BLOCK
            m_ShaderArray[shaderNo].BindingUniformBlock();
            #endif
        }
        else
        {
            OutputError( "SahderCompile Error.\n" );
        }
    }
    m_ShaderNum++;
    return shaderNo;
}
#else
u32 ShaderManager::AddEmitterResSet( const EmitterResource* emResSet, const char* addCompileDef, bool delayCompile )
{
    EFT_ASSERT( m_ShaderNum < m_ShaderMaxNum );
    EFT_NULL_ASSERT( m_ShaderArray );

    // 既に登録されているシェーダがないか検索し、同シェーダがあればそのIndexを返す
    u32 fieldFlag = 0;
    if ( emResSet->fieldRandomData ) fieldFlag |=  EFT_SHADER_KEY_FIELD_RANDOM;

    // シェーダキーを生成
    ShaderKey tempShaderKey;
    tempShaderKey.Initialize(
        emResSet->emitterData,
        emResSet->emitterPluginIndex,
        emResSet->emitterPluginData,
        fieldFlag,
        emResSet->fieldCustomData,
        addCompileDef );

    for ( u32 i = 0; i < m_ShaderNum; i++ )
    {
        if ( tempShaderKey.IsEqual( m_ShaderArray[i].GetShaderResource()->GetShaderKey() ) )
        {
            return i;
        }
    }

    // シェーダを新規登録
    u32 shaderNo = m_ShaderNum;
    m_ShaderArray[shaderNo].Invalidate();
    m_ShaderArray[shaderNo].Initialize( emResSet, addCompileDef );

    if ( !delayCompile )
    {
        if ( m_ShaderArray[shaderNo].CompileShader() )
        {
            m_ShaderArray[shaderNo].InitializeLocation();
        }
        else
        {
            OutputError( "SahderCompile Error.\n" );
        }
    }

    m_ShaderNum++;
    return shaderNo;
}
#endif

#endif

#if EFT_GX2
//---------------------------------------------------------------------------
//  シェーダバイナリを設定する
//---------------------------------------------------------------------------
void ShaderManager::SetShaderBinary( Heap* heap, void* binary )
{
    EFT_UNUSED_VARIABLE( heap );
    EFT_NULL_ASSERT( m_ShaderArray );
    EFT_ASSERT( m_ShaderNum <= m_ShaderMaxNum );

    m_TotalVertexShader = GFDGetVertexShaderCount( binary );
    m_TotalPixelShader  = GFDGetPixelShaderCount(  binary );

    m_ShaderBinary = binary;

    for ( int i = 0; i < m_TotalVertexShader; i++ )
    {
        m_TotalVertexShaderSize += GFDGetVertexShaderHeaderSize( i, m_ShaderBinary );
        m_TotalVertexShaderSize += GFDGetVertexShaderProgramSize( i, m_ShaderBinary );
    }

    for ( int i = 0; i < m_TotalPixelShader; i++ )
    {
        m_TotalPixelShaderSize += GFDGetPixelShaderHeaderSize( i, m_ShaderBinary );
        m_TotalPixelShaderSize += GFDGetPixelShaderProgramSize( i, m_ShaderBinary );
    }
}

//---------------------------------------------------------------------------
//  シェーダセットをシェーダ管理配列に追加する
//---------------------------------------------------------------------------
u32 ShaderManager::AddShaderSet( Heap* heap, u32 vertexIdx, u32 fragIdx )
{
    EFT_NULL_ASSERT( m_ShaderBinary );

    // 既に登録されているものを探す。
    for ( u32 i = 0; i < m_ShaderNum; i++ )
    {
        if ( m_ShaderArray[i].GetVertexShaderId()   == vertexIdx &&
             m_ShaderArray[i].GetFragmentShaderId() == fragIdx )
        return i;
    }

    EFT_ASSERT( m_ShaderNum < m_ShaderMaxNum );

    // 未登録時は新規に登録する。
    m_ShaderArray[m_ShaderNum].Invalidate();
    m_ShaderArray[m_ShaderNum].InitializeShader( heap, m_ShaderBinary, vertexIdx, fragIdx );
    m_ShaderArray[m_ShaderNum].InitializeLocation();
    m_ShaderArray[m_ShaderNum].BindingUniformBlock();
    m_ShaderArray[m_ShaderNum].SetupShader( heap );

    u32 retId = m_ShaderNum;
    m_ShaderNum++;

    return retId;
}

//---------------------------------------------------------------------------
//  シェーダバイナリを配置前に戻す。
//---------------------------------------------------------------------------
void ShaderManager::Unrelocate( void* pData )
{
    _RevertGX2ShaderBinary( pData );
}
#endif

//---------------------------------------------------------------------------
//  シェーダを取得します。
//---------------------------------------------------------------------------
Shader* ShaderManager::GetShader( u32 index )
{
    EFT_ASSERT( index < m_ShaderMaxNum );
    EFT_NULL_ASSERT( m_ShaderArray );

#if EFT_GX2
    EFT_ASSERT( m_ShaderArray[index].IsCompiled() );
#endif
#ifdef EFT_OGL
    if ( !m_ShaderArray[index].IsCompiled() )
    {
        if ( m_ShaderArray[index].CompileShader() )
        {
            m_ShaderArray[index].InitializeLocation();
            #ifdef EFT_USE_UNIFORM_BLOCK
            m_ShaderArray[index].BindingUniformBlock();
            #endif
        }
        else
        {
            OutputError( "[EFT] Shader Compile Failed.\n");
        }
    }
#endif
    return &m_ShaderArray[index];
}

} // namespace eft2
} // namespace nw

