﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


//------------------------------------------------------------------------------
#include <nw/eft/eft2_EmitterCalc.h>
#include <nw/eft/eft2_RenderContext.h>
#include <nw/eft/eft2_CurlNoiseData.h>
#include <nw/eft/eft2_System.h>
#include <nw/eft/eft2_Emitter.h>

namespace nw   {
namespace eft2 {


#if ( EFT_OPENGL || EFT_GX2 )

//---------------------------------------------------------------------------
//  アトリビュートを設定します。
//---------------------------------------------------------------------------
inline void _SetAttrPtr( u32 attr, nw::math::VEC4* ptr, u32 first, u32 count, u32 stride )
{
    if ( attr != EFT_INVALID_SHADER_ATTRIBUTE )
    {
        VertexBuffer::BindExtBuffer( attr, sizeof( nw::math::VEC4 ) * count, 4, stride, &ptr[first] );
    }
}

//---------------------------------------------------------------------------
//  テクスチャを設定します。
//---------------------------------------------------------------------------
inline void _SetTexture( System* system, Emitter* emitter, Shader* shader, TextureSampler* samplerClamp, TextureSampler* smaplerRepeat )
{
    // テクスチャ設定
    for ( u32 i = 0; i < EFT_TEXTURE_SLOT_MAX; i++ )
    {
        TextureSlot slot = static_cast<TextureSlot>( i );
        TextureResource* texture = emitter->emitterRes->texture[i];

        if ( texture )
        {
            u32 fragLoc = shader->GetFragmentTextureSamplerLocation( slot );
            u32 vertLoc = shader->GetVertexTextureSamplerLocation( slot );

            if ( texture->Bind( fragLoc, vertLoc, slot ) )
            {
                emitter->emitterRes->textureSampler[i].Bind( fragLoc, vertLoc, slot );
            }
        }
    }

    // フレームバッファテクスチャの設定
    {
        u32 fragLoc = shader->GetFrameBufferTextureFragmentSamplerLocation();
        u32 vertLoc = shader->GetFrameBufferTextureVertexSamplerLocation();

        if ( system->GetFrameBufferTexture().Bind( fragLoc, vertLoc, EFT_TEXTURE_SLOT_EXT_FB ) )
        {
            samplerClamp->Bind( fragLoc, vertLoc, EFT_TEXTURE_SLOT_EXT_FB );
        }
    }

    // デプスバッファテクスチャの設定
    {
        u32 fragLoc = shader->GetDepthBufferTextureFragmentSamplerLocation();
        u32 vertLoc = shader->GetDepthBufferTextureVertexSamplerLocation();

        if ( system->GetDepthBufferTexture().Bind( fragLoc, vertLoc, EFT_TEXTURE_SLOT_EXT_DB ) )
        {
            samplerClamp->Bind( fragLoc, vertLoc, EFT_TEXTURE_SLOT_EXT_DB );
        }
    }

    // カールノイズテクスチャを設定
    {
        u32 fragLoc = shader->GetCurlNoiseTextureFragmentSamplerLocation();
        u32 vertLoc = shader->GetCurlNoiseTextureVertexSamplerLocation();

        if ( GetCurlNoiseTexture().Bind( fragLoc, vertLoc, EFT_TEXTURE_SLOT_EXT_CN ) )
        {
            smaplerRepeat->Bind( fragLoc, vertLoc, EFT_TEXTURE_SLOT_EXT_CN );
        }
    }
}


//---------------------------------------------------------------------------
//  パーティクル View Sort
//---------------------------------------------------------------------------
// ソート
struct _sortInfo
{
    f32             param;
    u32             index;
};

int _CompareParticleGreater( const void *a, const void  *b )
{
    const _sortInfo *compA = reinterpret_cast<const _sortInfo*>( a );
    const _sortInfo *compB = reinterpret_cast<const _sortInfo*>( b );

    if ( compA->param > compB->param ) { return -1; }
    else                               { return  1; }
}

int _CompareParticleLess( const void *a, const void  *b )
{
    const _sortInfo *compA = reinterpret_cast<const _sortInfo*>( a );
    const _sortInfo *compB = reinterpret_cast<const _sortInfo*>( b );

    if ( compA->param < compB->param ) { return -1; }
    else                               { return  1; }
}


//---------------------------------------------------------------------------
//  エミッタストリームアウト計算処理をします。
//---------------------------------------------------------------------------
bool EmitterCalc::CalcStreamOut( Emitter* emitter, Shader* shader, u32 globalCounter, bool calcStreamOut, void* userParam )
{
    // 1フレーム中に複数回（複数パス/複数画面描画など）StreamOutが計算されるのを防止
    if ( emitter->soCounter == globalCounter ) { return false; }

    // ポーズ対応1.フレームレートが0のときは計算しない
    // ポーズ対応2.Drawの引数で強制的に計算しない設定にされた場合も同様（ユーザー側でCalcを呼ばない場合に対応）
    if ( emitter->frameRate == 0.0f || !calcStreamOut ) { return false; }

#if EFT_GX2
    GX2SetDepthOnlyControl( GX2_FALSE, GX2_FALSE, GX2_COMPARE_LEQUAL );
    GX2SetRasterizerClipControl(GX2_DISABLE, GX2_ENABLE);
    GX2SetStreamOutEnable(GX2_ENABLE);
#endif

#if EFT_OPENGL
    // VBAを無効化する
    glBindVertexArray(0);

    // 属性位置無効化.
    for (int i = 0; i < 16; ++i) { glDisableVertexAttribArray(i); }
    glBindBuffer( GL_ARRAY_BUFFER, GL_NONE );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, GL_NONE );
#endif

    // シェーダをバインド
    shader->Bind();

    // エミッタ動的UBOの設定
    shader->BindUniformBlock( Shader::EFT_UBO_TYPE_EMITTER_DYNAMIC, &emitter->dynamicUBO[emitter->bufferID] );

    // エミッタ静的UBOの設定
    shader->BindUniformBlock( Shader::EFT_UBO_TYPE_EMITTER_STATIC, &emitter->emitterRes->emitterStaticUbo );

    // フィールドUBOの設定
    if( emitter->emitterRes->fieldUniformBlock )
    {
        shader->BindUniformBlock( Shader::EFT_UBO_TYPE_EMITTER_FIELD, &emitter->emitterRes->fieldStaticUbo );
    }

    // カールノイズテクスチャを設定
    {
        u32 fragLoc = shader->GetCurlNoiseTextureFragmentSamplerLocation();
        u32 vertLoc = shader->GetCurlNoiseTextureVertexSamplerLocation();

        if ( GetCurlNoiseTexture().Bind( fragLoc, vertLoc, EFT_TEXTURE_SLOT_EXT_CN ) )
        {
            m_TextureSamplerForCurlNoise.Bind( fragLoc, vertLoc, EFT_TEXTURE_SLOT_EXT_CN );
        }
    }

    // カスタムシェーダコールバックの呼び出し
    {
        RenderStateSetArg rssArg;
        rssArg.emitter      = emitter;
        rssArg.userParam    = userParam;
        rssArg.shaderType   = EFT_SHADER_TYPE_NORMAL;
        rssArg.isStreamOut  = true;

        m_System->ResetCustomShaderSetting( shader );
        {
            // 描画設定後コールバックの呼び出し
            InvokeRenderStateSetCallback( rssArg );
        }
        m_System->ResetCustomShaderSetting( NULL );
    }

#if EFT_GX2
    VertexBuffer::BindExtBuffer( 8, sizeof(ParticlePosAttribute) * emitter->ptclNum,
        0, sizeof(ParticlePosAttribute), &emitter->particlePosAttr[0] );
    VertexBuffer::BindExtBuffer( 9, sizeof(ParticleAttribute) * emitter->ptclNum,
        0, sizeof(ParticleAttribute), &emitter->particleAttr[0] );
#endif

#if EFT_OPENGL
    // パーティクル発生位置を設定する。
    u32 posAttr   = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_LOCAL_POSITION );
    u32 vecAttr   = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_LOCAL_VECTOR );
    u32 sclAttr   = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_SCALE );
    u32 rndAttr   = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_RANDOM );
    u32 rotAttr   = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_ROTATE );
    u32 col0Attr  = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_COLOR0 );
    u32 col1Attr  = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_COLOR1 );
    u32 em0Attr   = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_EMTR_SRT_MTX0 );
    u32 em1Attr   = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_EMTR_SRT_MTX1 );
    u32 em2Attr   = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_EMTR_SRT_MTX2 );
    u32 em0RTAttr = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_EMTR_RT_MTX0 );
    u32 em1RTAttr = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_EMTR_RT_MTX1 );
    u32 em2RTAttr = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_EMTR_RT_MTX2 );

    _SetAttrPtr( posAttr,   &emitter->particlePosAttr->localPos,    0, emitter->ptclNum, sizeof( ParticlePosAttribute ) );
    _SetAttrPtr( vecAttr,   &emitter->particlePosAttr->localVec,    0, emitter->ptclNum, sizeof( ParticlePosAttribute ) );
    _SetAttrPtr( sclAttr,   &emitter->particleAttr->scale,          0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( rndAttr,   &emitter->particleAttr->random,         0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( rotAttr,   &emitter->particleAttr->initRotate,     0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( col0Attr,  &emitter->particleAttr->initColor0,     0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( col1Attr,  &emitter->particleAttr->initColor1,     0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( em0Attr,   &emitter->particleAttr->emitterMat0,    0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( em1Attr,   &emitter->particleAttr->emitterMat1,    0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( em2Attr,   &emitter->particleAttr->emitterMat2,    0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( em0RTAttr, &emitter->particleAttr->emitterRTMat0,  0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( em1RTAttr, &emitter->particleAttr->emitterRTMat1,  0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( em2RTAttr, &emitter->particleAttr->emitterRTMat2,  0, emitter->ptclNum, sizeof( ParticleAttribute ) );
#endif

    // StreamOut用のバッファを設定する。
    {
        // Flip
        emitter->streamOutFlip = 1 - emitter->streamOutFlip;

        // in
        emitter->streamOutPos.BindInBuffer( shader->GetStreamOutAttribute( Shader::EFT_STREAM_ATTR_IDX_POSITION ), emitter->streamOutFlip );
        emitter->streamOutVec.BindInBuffer( shader->GetStreamOutAttribute( Shader::EFT_STREAM_ATTR_IDX_VECTOR ), emitter->streamOutFlip );

        // out
        emitter->streamOutPos.BindOutBuffer( 0, emitter->streamOutFlip );
        emitter->streamOutVec.BindOutBuffer( 1, emitter->streamOutFlip );
    }

#if EFT_OPENGL
    glEnable( GL_RASTERIZER_DISCARD );
    glBeginTransformFeedback( GL_POINTS );

    Render::DrawPrimitive( Render::PRIM_TYPE_POINTS, emitter->ptclNum );

    emitter->streamOutPos.UnBind( 0 );
    emitter->streamOutVec.UnBind( 1 );

    glEndTransformFeedback();
    glDisable( GL_RASTERIZER_DISCARD );
    glBindBuffer( GL_ARRAY_BUFFER, 0 );
#endif

#if EFT_GX2
    Render::DrawInstancePrimitive( Render::PRIM_TYPE_POINTS, 1, emitter->ptclNum );
    emitter->streamOutPos.UnBind( 0 );
    emitter->streamOutVec.UnBind( 1 );
    GX2SetStreamOutEnable( GX2_DISABLE );
    GX2SetRasterizerClipControl( GX2_ENABLE,
                                 static_cast<GX2Boolean>( emitter->emitterSet->GetSystem()->GetZCllipFlagByStreamOut() ) );
#endif

    // 1フレーム内２度実行を回避するため、StreamOutカウンタ更新
    emitter->soCounter = globalCounter;

    return true;
}


//---------------------------------------------------------------------------
//  ソートなし描画処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalc::EntryParticleNoSort( Emitter* emitter, Shader* shader, Primitive* prim, Render::PrimitiveType primType )
{
#if EFT_GX2
    VertexBuffer::BindExtBuffer( 8, sizeof(ParticlePosAttribute) * emitter->ptclNum,
        0, sizeof(ParticlePosAttribute), &emitter->particlePosAttr[0] );
    VertexBuffer::BindExtBuffer( 9, sizeof(ParticleAttribute) * emitter->ptclNum,
        0, sizeof(ParticleAttribute), &emitter->particleAttr[0] );
#endif

    u32 inPosAttr = shader->GetStreamOutAttribute( Shader::EFT_STREAM_ATTR_IDX_POSITION );
    u32 inVecAttr = shader->GetStreamOutAttribute( Shader::EFT_STREAM_ATTR_IDX_VECTOR );

#if EFT_OPENGL
    u32 posAttr   = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_LOCAL_POSITION );
    u32 vecAttr   = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_LOCAL_VECTOR );
    u32 diffAttr  = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_LOCAL_POS_DELTA );
    u32 sclAttr   = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_SCALE );
    u32 rndAttr   = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_RANDOM );
    u32 rotAttr   = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_ROTATE );
    u32 col0Attr  = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_COLOR0 );
    u32 col1Attr  = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_COLOR1 );
    u32 em0Attr   = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_EMTR_SRT_MTX0 );
    u32 em1Attr   = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_EMTR_SRT_MTX1 );
    u32 em2Attr   = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_EMTR_SRT_MTX2 );
    u32 em0RTAttr = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_EMTR_RT_MTX0 );
    u32 em1RTAttr = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_EMTR_RT_MTX1 );
    u32 em2RTAttr = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_EMTR_RT_MTX2 );

    _SetAttrPtr( posAttr,   &emitter->particlePosAttr->localPos,  0, emitter->ptclNum, sizeof( ParticlePosAttribute ) );
    _SetAttrPtr( vecAttr,   &emitter->particlePosAttr->localVec,  0, emitter->ptclNum, sizeof( ParticlePosAttribute ) );
    _SetAttrPtr( diffAttr,  &emitter->particlePosAttr->localDiff, 0, emitter->ptclNum, sizeof( ParticlePosAttribute ) );
    _SetAttrPtr( sclAttr,   &emitter->particleAttr->scale,        0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( rndAttr,   &emitter->particleAttr->random,       0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( rotAttr,   &emitter->particleAttr->initRotate,   0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( col0Attr,  &emitter->particleAttr->initColor0,   0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( col1Attr,  &emitter->particleAttr->initColor1,   0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( em0Attr,   &emitter->particleAttr->emitterMat0,  0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( em1Attr,   &emitter->particleAttr->emitterMat1,  0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( em2Attr,   &emitter->particleAttr->emitterMat2,  0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( em0RTAttr, &emitter->particleAttr->emitterRTMat0,  0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( em1RTAttr, &emitter->particleAttr->emitterRTMat1,  0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( em2RTAttr, &emitter->particleAttr->emitterRTMat2,  0, emitter->ptclNum, sizeof( ParticleAttribute ) );
#endif

    if ( emitter->emitterData->emitter.calcType == EFT_EMITTER_CALC_TYPE_GPU_SO )
    {
        if ( inPosAttr  != EFT_INVALID_SHADER_ATTRIBUTE )
        {
            emitter->streamOutPos.BindInBuffer( inPosAttr, 1 - emitter->streamOutFlip );
        }
        if ( inVecAttr  != EFT_INVALID_SHADER_ATTRIBUTE )
        {
            emitter->streamOutVec.BindInBuffer( inVecAttr, 1 - emitter->streamOutFlip );
        }
    }

    if ( prim )
    {
        Render::DrawInstanceIndexedPrimitive( primType, prim->GetIndexNum(),
            static_cast<void *>( prim->GetIndexBuffer() ), emitter->ptclNum );
    }
    else
    {
        Render::DrawInstancePrimitive( primType, EFT_PARTICLE_PRIM_VERTICES_NUM, emitter->ptclNum );
    }

    return true;
}

//---------------------------------------------------------------------------
//  ソート済みパーティクル描画処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalc::EntrySortedParticle( Emitter* emitter, Shader* shader, Primitive* prim, Render::PrimitiveType primType, const ParticleSortType sortType )
{
    // Zソート用のバッファを確保
    if ( emitter->ptclNum > m_System->GetSortParticleCount() )
    {
        OutputWarning( "ptclNum is exceeded the number of Particle sort count.\n" );
        return false;
    }

    _sortInfo* sortInfo = reinterpret_cast<_sortInfo *>( m_System->GetParticleSortBuffer() );
    if ( !sortInfo )
    {
        OutputWarning( "The sort information buffer can't get.\n" );
        return false;
    }

    ViewParam* view = m_System->GetViewParam();

    switch ( sortType )
    {
    case EFT_SORT_TYPE_ORDER:
        {
            for( u32 i = 0; i < emitter->ptclNum; i++ )
            {
                sortInfo[i].param = emitter->particleData[i].createTime;
                sortInfo[i].index = i;
            }
        }
        qsort( sortInfo, emitter->ptclNum, sizeof(_sortInfo), _CompareParticleLess );
        break;
    case EFT_SORT_TYPE_REV_ORDER:
        {
            for( u32 i = 0; i < emitter->ptclNum; i++ )
            {
                sortInfo[i].param = emitter->particleData[i].createTime;
                sortInfo[i].index = i;
            }
        }
        qsort( sortInfo, emitter->ptclNum, sizeof(_sortInfo), _CompareParticleGreater );
        break;
    case EFT_SORT_TYPE_Z_SORT:
        {
            for( u32 i = 0; i < emitter->ptclNum; i++ )
            {
                nw::math::VEC3 worldPos;
                nw::math::VEC3 localPos( emitter->particlePosAttr[ i ].localPos.x, emitter->particlePosAttr[ i ].localPos.y, emitter->particlePosAttr[ i ].localPos.z );
                emitter->TransformToWorldPos( &worldPos, localPos, &emitter->particleAttr[ i ] );
                // Zを計算
                sortInfo[i].param =
                    view->viewMat.f._20 * worldPos.x +
                    view->viewMat.f._21 * worldPos.y +
                    view->viewMat.f._22 * worldPos.z +
                    view->viewMat.f._23;
                sortInfo[i].index = i;
            }
        }
        qsort( sortInfo, emitter->ptclNum, sizeof(_sortInfo), _CompareParticleLess );
        break;
    }

    u32 inPosAttr = shader->GetStreamOutAttribute( Shader::EFT_STREAM_ATTR_IDX_POSITION );
    u32 inVecAttr = shader->GetStreamOutAttribute( Shader::EFT_STREAM_ATTR_IDX_VECTOR );

#if EFT_OPENGL
    u32 posAttr   = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_LOCAL_POSITION );
    u32 vecAttr   = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_LOCAL_VECTOR );
    u32 diffAttr  = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_LOCAL_POS_DELTA );
    u32 sclAttr   = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_SCALE );
    u32 rndAttr   = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_RANDOM );
    u32 rotAttr   = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_ROTATE );
    u32 col0Attr  = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_COLOR0 );
    u32 col1Attr  = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_COLOR1 );
    u32 em0Attr   = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_EMTR_SRT_MTX0 );
    u32 em1Attr   = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_EMTR_SRT_MTX1 );
    u32 em2Attr   = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_EMTR_SRT_MTX2 );
    u32 em0RTAttr = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_EMTR_RT_MTX0 );
    u32 em1RTAttr = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_EMTR_RT_MTX1 );
    u32 em2RTAttr = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_EMTR_RT_MTX2 );

    _SetAttrPtr( posAttr,   &emitter->particlePosAttr->localPos,  0, emitter->ptclNum, sizeof( ParticlePosAttribute ) );
    _SetAttrPtr( vecAttr,   &emitter->particlePosAttr->localVec,  0, emitter->ptclNum, sizeof( ParticlePosAttribute ) );
    _SetAttrPtr( diffAttr,  &emitter->particlePosAttr->localDiff, 0, emitter->ptclNum, sizeof( ParticlePosAttribute ) );
    _SetAttrPtr( sclAttr,   &emitter->particleAttr->scale,        0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( rndAttr,   &emitter->particleAttr->random,       0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( rotAttr,   &emitter->particleAttr->initRotate,   0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( col0Attr,  &emitter->particleAttr->initColor0,   0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( col1Attr,  &emitter->particleAttr->initColor1,   0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( em0Attr,   &emitter->particleAttr->emitterMat0,  0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( em1Attr,   &emitter->particleAttr->emitterMat1,  0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( em2Attr,   &emitter->particleAttr->emitterMat2,  0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( em0RTAttr, &emitter->particleAttr->emitterRTMat0,  0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( em1RTAttr, &emitter->particleAttr->emitterRTMat1,  0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( em2RTAttr, &emitter->particleAttr->emitterRTMat2,  0, emitter->ptclNum, sizeof( ParticleAttribute ) );
#endif

#if EFT_GX2
    VertexBuffer::BindExtBuffer( 8, sizeof(ParticlePosAttribute) * emitter->ptclNum,
        0, sizeof(ParticlePosAttribute), &emitter->particlePosAttr[0] );
    VertexBuffer::BindExtBuffer( 9, sizeof(ParticleAttribute) * emitter->ptclNum,
        0, sizeof(ParticleAttribute), &emitter->particleAttr[0] );
#endif

    if ( emitter->emitterData->emitter.calcType == EFT_EMITTER_CALC_TYPE_GPU_SO )
    {
        if ( inPosAttr  != EFT_INVALID_SHADER_ATTRIBUTE )
        {
            emitter->streamOutPos.BindInBuffer( inPosAttr, 1 - emitter->streamOutFlip );
        }
        if ( inVecAttr  != EFT_INVALID_SHADER_ATTRIBUTE )
        {
            emitter->streamOutVec.BindInBuffer( inVecAttr, 1 - emitter->streamOutFlip );
        }
    }

    // ソートされた内容に従って１パーティクルづつ描画する。
    if ( prim )
    {
#if EFT_GX2
        for( u32 i = 0; i < emitter->ptclNum; i++ )
        {
            Render::DrawBaseInstancePrimitive( primType, prim->GetIndexNum(),
                static_cast<void *>( prim->GetIndexBuffer() ), 1, sortInfo[i].index );
        }
#endif
#if EFT_OPENGL  // OpenGL版で正しく描画されないので現状ソートされない
        Render::DrawInstanceIndexedPrimitive( primType, prim->GetIndexNum(),
            static_cast<void *>( prim->GetIndexBuffer() ), emitter->ptclNum );
#endif
    }
    else
    {
        for( u32 i = 0; i < emitter->ptclNum; i++ )
        {
            const u32 index = sortInfo[i].index;
            ParticleData* ptclData = &emitter->particleData[ index ];
            const f32 time = ptclData->GetTime( emitter->time );
            const f32 life = ptclData->GetLife();

            if( time <= life && ptclData->createID != 0 && ptclData->life > 0 )
            {
                Render::DrawBaseInstancePrimitive( primType, EFT_PARTICLE_PRIM_VERTICES_NUM,
                    static_cast< void* >( m_IndexBuffer.GetVertexBuffer() ), EFT_PARTICLE_PRIM_PRIMITIVE_NUM, sortInfo[i].index );
            }
        }
    }
    return true;
}


//---------------------------------------------------------------------------
//  エミッタ描画処理をします。
//---------------------------------------------------------------------------
bool EmitterCalc::Draw( Emitter* emitter, ShaderType shaderType, void* userParam )
{
    EFT_ASSERT( emitter );

    if ( !emitter->emitterRes->visibility && emitter->groupID == (EFT_GROUP_MAX-1) ) return false;

    // エミッタ描画コールバック
    {
        EmitterDrawArg edArg;
        edArg.emitter       = emitter;
        edArg.shaderType    = shaderType;
        edArg.userParam     = userParam;

        if( InvokeEmitterDrawCallback( edArg ) )
        {
            return true;
        }
    }

    // シェーダ切り替え
    Shader* shader = emitter->shader[shaderType];
    if ( !shader )
    {
        Warning( emitter, EFT_WARNING_SHADER_IS_NOT_EXIST );
        return false;
    }
    shader->Bind();

    DrawEmitterUsingBoundShader( emitter, shaderType, userParam );

    return true;
}

bool EmitterCalc::DrawEmitterUsingBoundShader( Emitter* emitter, ShaderType shaderType, void* userParam )
{
    Shader* shader = emitter->shader[shaderType];
    System* system = emitter->emitterSet->GetSystem();

    // 描画設定
    ResRenderState* state = &emitter->emitterData->renderState;
    Rendercontext::SetRenderStateResource( state, system->GetColorBufferEnbaleFlag() );

    // テクスチャ設定
    _SetTexture( system, emitter, shader, &m_TextureSamplerForColorAndDepth, &m_TextureSamplerForCurlNoise );

    // エミッタ動的UBOの設定
    shader->BindUniformBlock( Shader::EFT_UBO_TYPE_EMITTER_DYNAMIC, &emitter->dynamicUBO[emitter->bufferID] );

    // エミッタ静的UBOの設定
    shader->BindUniformBlock( Shader::EFT_UBO_TYPE_EMITTER_STATIC, &emitter->emitterRes->emitterStaticUbo );

    // フィールドUBOの設定
    if( emitter->emitterRes->fieldUniformBlock )
    {
        shader->BindUniformBlock( Shader::EFT_UBO_TYPE_EMITTER_FIELD, &emitter->emitterRes->fieldStaticUbo );
    }

    // カスタムシェーダコールバックの呼び出し
    {
        RenderStateSetArg rssArg;
        rssArg.emitter      = emitter;
        rssArg.userParam    = userParam;
        rssArg.shaderType   = shaderType;
        rssArg.isStreamOut  = false;

        m_System->ResetCustomShaderSetting( shader );
        {
            // 描画設定後コールバックの呼び出し
            bool ret = InvokeRenderStateSetCallback( rssArg );
            if( !ret ){ return false; }

            // DrawPath コールバックの呼び出し
            if ( emitter->drawPathCallback ) { emitter->drawPathCallback( rssArg ); }
        }
        m_System->ResetCustomShaderSetting( NULL );
    }

    Render::PrimitiveType primType  = Render::PARTICLE_PRIM_TYPE;

    // 頂点アトリビュート設定
    Primitive* prim = emitter->emitterRes->primitive;
    if ( !prim )
    {
        // パーティクルの場合
        m_PositionBuffer.BindBuffer( shader->GetAttribute( Shader::EFT_ATTR_IDX_POSITION ), sizeof( nw::math::VEC4 ) * EFT_PARTICLE_PRIM_VERTICES_NUM, sizeof( nw::math::VEC4 ) );
    }
    else
    {
        // プリミティブの場合
        prim->BindPosionBuffer(     shader->GetAttribute( Shader::EFT_ATTR_IDX_POSITION ) );
        prim->BindTextureCrdBuffer( shader->GetAttribute( Shader::EFT_ATTR_IDX_TEXTURE_COORD0 )   );
        prim->BindColorBuffer(      shader->GetAttribute( Shader::EFT_ATTR_IDX_VERTEX_COLOR0 )    );
        if ( shader->GetAttribute( Shader::EFT_ATTR_IDX_NORMAL ) != EFT_INVALID_SHADER_ATTRIBUTE )
        {
            prim->BindNormalBuffer( shader->GetAttribute( Shader::EFT_ATTR_IDX_NORMAL ) );
        }
        if ( shader->GetAttribute( Shader::EFT_ATTR_IDX_TANGENT ) != EFT_INVALID_SHADER_ATTRIBUTE )
        {
            prim->BindTangentBuffer( shader->GetAttribute( Shader::EFT_ATTR_IDX_TANGENT ) );
        }

        primType = Render::PRIM_TYPE_TRIANGLES;
    }

    // インスタンシング設定ON
    shader->EnableInstanced();

    if( emitter->emitterData->emitter.sortType == EFT_SORT_TYPE_NONE )
    {
        // ソートしないで描画
        EntryParticleNoSort( emitter, shader, prim, primType );
    }
    else
    {
        // パーティクルをソートして描画
        EntrySortedParticle( emitter, shader, prim, primType, emitter->emitterData->emitter.sortType );
    }

    // インスタンシング設定OFF
    shader->DisableInstanced();


#if EFT_OPENGL
    u32 inPosAttr   = shader->GetStreamOutAttribute( Shader::EFT_STREAM_ATTR_IDX_POSITION );
    u32 inVecAttr   = shader->GetStreamOutAttribute( Shader::EFT_STREAM_ATTR_IDX_VECTOR );

    if ( inPosAttr  != EFT_INVALID_SHADER_ATTRIBUTE )
    {
        glVertexAttribDivisor( inPosAttr,  0 );
        glDisableVertexAttribArray( inPosAttr );
        glBindBuffer( GL_ARRAY_BUFFER, 0 );
    }
    if ( inVecAttr  != EFT_INVALID_SHADER_ATTRIBUTE )
    {
        glVertexAttribDivisor( inVecAttr,  0 );
        glDisableVertexAttribArray( inVecAttr );
        glBindBuffer( GL_ARRAY_BUFFER, 0 );
    }
    EFT_GLERR_CHECK();
#endif

    return true;
}

#endif      //  #if ( EFT_OPENGL || EFT_GX2 )




#if defined( NW_PLATFORM_SMART_DEVICE )
#if EFT_OPENGLES2
//---------------------------------------------------------------------------
//! @brief        エミッタストリームアウト計算処理をします。
//---------------------------------------------------------------------------
bool EmitterCalc::CalcStreamOut( Emitter* emitter, Shader* shader, u32 globalCounter, bool calcStreamOut, void* userParam )
{
    EFT_UNUSED_VARIABLE( emitter );
    EFT_UNUSED_VARIABLE( shader );
    EFT_UNUSED_VARIABLE( globalCounter );
    EFT_UNUSED_VARIABLE( calcStreamOut );
    EFT_UNUSED_VARIABLE( userParam );
    return true;
}

//---------------------------------------------------------------------------
//  テクスチャを設定します。
//---------------------------------------------------------------------------
inline void _SetTexture( System* system, Emitter* emitter, Shader* shader )
{
    EFT_UNUSED_VARIABLE( system );

    for ( u32 i = 0; i < EFT_TEXTURE_SLOT_MAX; i++ )
    {
        TextureSlot slot = static_cast<TextureSlot>( i );
        TextureResource* texture = emitter->emitterRes->texture[i];
        const TextureSampler*  sampler = &emitter->emitterRes->textureSampler[i];

        if ( texture )
        {
            u32 fragLoc = shader->GetFragmentTextureSamplerLocation( slot );
            u32 vertLoc = shader->GetVertexTextureSamplerLocation( slot );
            texture->Bind( fragLoc, vertLoc, slot );
#if EFT_IS_WIN
            // サンプラオブジェクト解除
            glBindSampler( slot, 0 );
#endif
            const_cast<TextureSampler*>( sampler )->Setup( sampler->GetTextureSamplerRes() );
        }
    }
}

//---------------------------------------------------------------------------
//  アトリビュートを設定します。
//---------------------------------------------------------------------------
inline void _SetAttrPtr( u32 attr, nw::math::VEC4* ptr, u32 first, u32 count, u32 stride )
{
    if ( attr != EFT_INVALID_SHADER_ATTRIBUTE )
    {
        VertexBuffer::BindExtBuffer( attr, sizeof( nw::math::VEC4 ) * count, 4, stride, &ptr[first] );
    }
}

//---------------------------------------------------------------------------
//  エミッタ描画処理をします。
//---------------------------------------------------------------------------
bool EmitterCalc::Draw( Emitter* emitter, ShaderType shaderType, void* userParam )
{
    EFT_UNUSED_VARIABLE( emitter );
    EFT_UNUSED_VARIABLE( shaderType );
    EFT_UNUSED_VARIABLE( userParam );

    EFT_ASSERT( emitter );

    if ( !emitter->emitterRes->visibility && emitter->groupID == (EFT_GROUP_MAX-1) ) return false;

    // シェーダ切り替え
    Shader* shader = emitter->shader[shaderType];
    if ( !shader )
    {
        Warning( emitter, EFT_WARNING_SHADER_IS_NOT_EXIST );
        return false;
    }
    shader->Bind();

    System* system = emitter->emitterSet->GetSystem();

    // 描画設定
    ResRenderState* state = &emitter->emitterData->renderState;
    Rendercontext::SetRenderStateResource( state, system->GetColorBufferEnbaleFlag() );

    // テクスチャ設定
    _SetTexture( system, emitter, shader );//, &m_TextureSampler );

    // ビューレジスタの設定
    ViewParam* view = system->GetCoppiedViewParam();
    u32 vwMatrixLoc = shader->GetRegisterLocation( Shader::EFT_REGISTER_LOC_VIEW_MATRIX );
    if( vwMatrixLoc != 0xFFFFFFFF ) glUniformMatrix4fv( vwMatrixLoc, 1, GL_FALSE, view->viewMat );
    u32 pjMatrixLoc = shader->GetRegisterLocation( Shader::EFT_REGISTER_LOC_PROJ_MATRIX );
    if( pjMatrixLoc != 0xFFFFFFFF ) glUniformMatrix4fv( pjMatrixLoc, 1, GL_FALSE, view->projMat );
    u32 vpMatrixLoc = shader->GetRegisterLocation( Shader::EFT_REGISTER_LOC_VIEW_PROJ_MATRIX );
    if( vpMatrixLoc != 0xFFFFFFFF ) glUniformMatrix4fv( vpMatrixLoc, 1, GL_FALSE, view->vpMat );
    u32 bldMatrixLoc = shader->GetRegisterLocation( Shader::EFT_REGISTER_LOC_BILLBOARD_MATRIX );
    if( bldMatrixLoc != 0xFFFFFFFF ) glUniformMatrix4fv( bldMatrixLoc, 1, GL_FALSE, view->bldMat );
    u32 eyePosLoc = shader->GetRegisterLocation( Shader::EFT_REGISTER_LOC_CAMERA_POS );
    if( eyePosLoc != 0xFFFFFFFF ) glUniform4fv( eyePosLoc, 1, view->eyePos );
    u32 eyeVecLoc = shader->GetRegisterLocation( Shader::EFT_REGISTER_LOC_CAMERA_VEC );
    if( eyeVecLoc != 0xFFFFFFFF ) glUniform4fv( eyeVecLoc, 1, view->eyeVec );
    u32 eyeVecParamLoc = shader->GetRegisterLocation( Shader::EFT_REGISTER_LOC_VIEW_PARAM );
    if( eyeVecParamLoc != 0xFFFFFFFF ) glUniform4fv( eyeVecParamLoc, 1, view->viewParam );
    EFT_GLERR_CHECK();

    // エミッタレジスタの設定
    EmitterDynamicUniformBlock* emDynamicUbo =
        reinterpret_cast<EmitterDynamicUniformBlock*>( emitter->dynamicUBO[emitter->bufferID].GetBuffer() );
    u32 emParam0Loc = shader->GetRegisterLocation( Shader::EFT_REGISTER_LOC_EMITTER_PARAM0 );
    if( emParam0Loc != 0xFFFFFFFF ) glUniform4fv( emParam0Loc, 1, emDynamicUbo->emitterParam0 );
    u32 emParam1Loc = shader->GetRegisterLocation( Shader::EFT_REGISTER_LOC_EMITTER_PARAM1 );
    if( emParam1Loc != 0xFFFFFFFF ) glUniform4fv( emParam1Loc, 1, emDynamicUbo->emitterParam1 );
    u32 emColor0Loc = shader->GetRegisterLocation( Shader::EFT_REGISTER_LOC_EMITTER_COLOR0 );
    if( emColor0Loc != 0xFFFFFFFF ) glUniform4fv( emColor0Loc, 1, emDynamicUbo->emitterColor0 );
    u32 emColor1Loc = shader->GetRegisterLocation( Shader::EFT_REGISTER_LOC_EMITTER_COLOR1 );
    if( emColor1Loc != 0xFFFFFFFF ) glUniform4fv( emColor1Loc, 1, emDynamicUbo->emitterColor1 );
    u32 emSRTMatLoc = shader->GetRegisterLocation( Shader::EFT_REGISTER_LOC_EMITTER_SRT_MATIRX );
    if( emSRTMatLoc != 0xFFFFFFFF ) glUniformMatrix4fv( emSRTMatLoc, 1, GL_FALSE, emDynamicUbo->emitterMatrix );
    u32 emRTMatLoc = shader->GetRegisterLocation( Shader::EFT_REGISTER_LOC_EMITTER_RT_MATIRX );
    if( emRTMatLoc != 0xFFFFFFFF ) glUniformMatrix4fv( emRTMatLoc, 1, GL_FALSE, emDynamicUbo->emitterMatrixRT );
    EFT_GLERR_CHECK();

    // テクスチャ座標アニメパラメータ設定
    nw::math::VEC4 uniformParam;
    u32 tex0ScrollParam = shader->GetShaderResource()->GetUniformLocation( "tex0ScrollParam" );
    if( tex0ScrollParam != 0xFFFFFFFF )
    {
        uniformParam.x = emitter->emitterData->staticUbo.texShiftAnim0.scrollAdd.x;
        uniformParam.y = emitter->emitterData->staticUbo.texShiftAnim0.scrollAdd.y;
        uniformParam.z = emitter->emitterData->staticUbo.texShiftAnim0.scaleAdd.x;
        uniformParam.w = emitter->emitterData->staticUbo.texShiftAnim0.rotAdd;
        glUniform4fv( tex0ScrollParam, 1, uniformParam );
    }
    u32 tex0PatternParam = shader->GetShaderResource()->GetUniformLocation( "tex0PatternParam" );
    if( tex0PatternParam != 0xFFFFFFFF )
    {
        uniformParam.x = emitter->emitterData->staticUbo.texShiftAnim0.uvDiv.x;
        uniformParam.y = emitter->emitterData->staticUbo.texShiftAnim0.uvDiv.y;
        uniformParam.z = emitter->emitterData->staticUbo.texShiftAnim0.uvScale.x;
        uniformParam.w = emitter->emitterData->staticUbo.texShiftAnim0.uvScale.y;
        glUniform4fv( tex0PatternParam, 1, uniformParam );
    }

    u32 tex1ScrollParam = shader->GetShaderResource()->GetUniformLocation( "tex1ScrollParam" );
    if( tex1ScrollParam != 0xFFFFFFFF )
    {
        uniformParam.x = emitter->emitterData->staticUbo.texShiftAnim1.scrollAdd.x;
        uniformParam.y = emitter->emitterData->staticUbo.texShiftAnim1.scrollAdd.y;
        uniformParam.z = emitter->emitterData->staticUbo.texShiftAnim1.scaleAdd.x;
        uniformParam.w = emitter->emitterData->staticUbo.texShiftAnim1.rotAdd;
        glUniform4fv( tex1ScrollParam, 1, uniformParam );
    }
    u32 tex1PatternParam = shader->GetShaderResource()->GetUniformLocation( "tex1PatternParam" );
    if( tex1PatternParam != 0xFFFFFFFF )
    {
        uniformParam.x = emitter->emitterData->staticUbo.texShiftAnim1.uvDiv.x;
        uniformParam.y = emitter->emitterData->staticUbo.texShiftAnim1.uvDiv.y;
        uniformParam.z = emitter->emitterData->staticUbo.texShiftAnim1.uvScale.x;
        uniformParam.w = emitter->emitterData->staticUbo.texShiftAnim1.uvScale.y;
        glUniform4fv( tex1PatternParam, 1, uniformParam );
    }

    // パーティクルパラメータ
    u32 particleParam = shader->GetShaderResource()->GetUniformLocation( "particleParam" );
    if( particleParam != 0xFFFFFFFF )
    {
        uniformParam.x = emitter->emitterData->staticUbo.coefficient0;
        uniformParam.y = emitter->emitterData->staticUbo.coefficient1;
        uniformParam.z = emitter->emitterData->staticUbo.center.x;
        uniformParam.w = emitter->emitterData->staticUbo.center.y;
        glUniform4fv( particleParam, 1, uniformParam );
    }

    // 頂点アトリビュート設定
    Render::PrimitiveType primType  = Render::PARTICLE_PRIM_TYPE;
    Primitive* prim = emitter->emitterRes->primitive;
    if ( !prim )
    {
        // パーティクルの場合
        m_PositionBuffer.BindBuffer( shader->GetAttribute( Shader::EFT_ATTR_IDX_POSITION ), sizeof( nw::math::VEC4 ) * EFT_PARTICLE_PRIM_VERTICES_NUM, sizeof( nw::math::VEC4 ) );
    }
    else
    {
        // プリミティブの場合
        prim->BindPosionBuffer(     shader->GetAttribute( Shader::EFT_ATTR_IDX_POSITION ) );
        prim->BindTextureCrdBuffer( shader->GetAttribute( Shader::EFT_ATTR_IDX_TEXTURE_COORD0 )   );
        prim->BindColorBuffer(      shader->GetAttribute( Shader::EFT_ATTR_IDX_VERTEX_COLOR0 )    );
        if ( shader->GetAttribute( Shader::EFT_ATTR_IDX_NORMAL ) != EFT_INVALID_SHADER_ATTRIBUTE )
        {
            prim->BindNormalBuffer( shader->GetAttribute( Shader::EFT_ATTR_IDX_NORMAL ) );
        }
        if ( shader->GetAttribute( Shader::EFT_ATTR_IDX_TANGENT ) != EFT_INVALID_SHADER_ATTRIBUTE )
        {
            prim->BindTangentBuffer( shader->GetAttribute( Shader::EFT_ATTR_IDX_TANGENT ) );
        }
        primType = Render::PRIM_TYPE_TRIANGLES;
    }
    EFT_GLERR_CHECK();

    // パーティクルアトリビュート取得
    u32 localPosLoc         = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_LOCAL_POSITION );
    u32 localVecLoc         = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_LOCAL_VECTOR );
    u32 localPosDiffLoc     = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_LOCAL_POS_DELTA );
    u32 particleRotLoc      = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_ROTATE );
    u32 particleSclLoc      = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_SCALE );
    u32 particleCol0Loc     = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_COLOR0 );
    u32 particleCol1Loc     = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_COLOR1 );
    u32 particleEmSRT0Loc   = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_EMTR_SRT_MTX0 );
    u32 particleEmSRT1Loc   = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_EMTR_SRT_MTX1 );
    u32 particleEmSRT2Loc   = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_EMTR_SRT_MTX2 );
    u32 particleEmRT0Loc    = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_EMTR_RT_MTX0 );
    u32 particleEmRT1Loc    = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_EMTR_RT_MTX1 );
    u32 particleEmRT2Loc    = shader->GetParticleAttribute( Shader::EFT_PTCL_ATTR_IDX_EMTR_RT_MTX2 );

#ifdef EFT_USE_REGISTER_DRAW
    //-------------------------------------
    // レジスタ描画
    //-------------------------------------
    for( u32 i = 0; i < emitter->ptclNum; i++ )
    {
        if ( emitter->particleData[i].createID != 0 )
        {
            if( localPosLoc       != 0xFFFFFFFF ) glUniform4fv( localPosLoc,       1, (f32*)&emitter->particlePosAttr[i].localPos );
            if( localVecLoc       != 0xFFFFFFFF ) glUniform4fv( localVecLoc,       1, (f32*)&emitter->particlePosAttr[i].localVec );
            if( localPosDiffLoc   != 0xFFFFFFFF ) glUniform4fv( localPosDiffLoc,   1, (f32*)&emitter->particlePosAttr[i].localDiff );
            if( particleRotLoc    != 0xFFFFFFFF ) glUniform4fv( particleRotLoc,    1, (f32*)&emitter->particleAttr[i].initRotate );
            if( particleSclLoc    != 0xFFFFFFFF ) glUniform4fv( particleSclLoc,    1, (f32*)&emitter->particleAttr[i].scale );
            if( particleCol0Loc   != 0xFFFFFFFF ) glUniform4fv( particleCol0Loc,   1, (f32*)&emitter->particleAttr[i].initColor0 );
            if( particleCol1Loc   != 0xFFFFFFFF ) glUniform4fv( particleCol1Loc,   1, (f32*)&emitter->particleAttr[i].initColor1 );
            if( particleEmSRT0Loc != 0xFFFFFFFF ) glUniform4fv( particleEmSRT0Loc, 1, (f32*)&emitter->particleAttr[i].emitterMat0 );
            if( particleEmSRT1Loc != 0xFFFFFFFF ) glUniform4fv( particleEmSRT1Loc, 1, (f32*)&emitter->particleAttr[i].emitterMat1 );
            if( particleEmSRT2Loc != 0xFFFFFFFF ) glUniform4fv( particleEmSRT2Loc, 1, (f32*)&emitter->particleAttr[i].emitterMat2 );
            if( particleEmRT0Loc  != 0xFFFFFFFF ) glUniform4fv( particleEmRT0Loc,  1, (f32*)&emitter->particleAttr[i].emitterRTMat0 );
            if( particleEmRT1Loc  != 0xFFFFFFFF ) glUniform4fv( particleEmRT1Loc,  1, (f32*)&emitter->particleAttr[i].emitterRTMat1 );
            if( particleEmRT2Loc  != 0xFFFFFFFF ) glUniform4fv( particleEmRT2Loc,  1, (f32*)&emitter->particleAttr[i].emitterRTMat2 );

            if ( prim )
            {
                Render::DrawIndexedPrimitive( primType, prim->GetIndexNum(), static_cast<void *>( prim->GetIndexBuffer() ) );
            }
            else
            {
                Render::DrawPrimitive( primType, EFT_PARTICLE_PRIM_VERTICES_NUM );
            }
        }
    }
    EFT_GLERR_CHECK();
#endif

#ifdef EFT_USE_INSTANS_DRAW
    //-------------------------------------
    // インスタンシング描画
    //-------------------------------------
    _SetAttrPtr( localPosLoc,        &emitter->particlePosAttr->localPos,    0, emitter->ptclNum, sizeof( ParticlePosAttribute ) );
    _SetAttrPtr( localVecLoc,        &emitter->particlePosAttr->localVec,    0, emitter->ptclNum, sizeof( ParticlePosAttribute ) );
    _SetAttrPtr( localPosDiffLoc,    &emitter->particlePosAttr->localDiff,   0, emitter->ptclNum, sizeof( ParticlePosAttribute ) );
    _SetAttrPtr( particleSclLoc,     &emitter->particleAttr->scale,          0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( particleRotLoc,     &emitter->particleAttr->initRotate,     0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( particleCol0Loc,    &emitter->particleAttr->initColor0,     0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( particleCol1Loc,    &emitter->particleAttr->initColor1,     0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( particleEmSRT0Loc,  &emitter->particleAttr->emitterMat0,    0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( particleEmSRT1Loc,  &emitter->particleAttr->emitterMat1,    0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( particleEmSRT2Loc,  &emitter->particleAttr->emitterMat2,    0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( particleEmRT0Loc,   &emitter->particleAttr->emitterRTMat0,  0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( particleEmRT1Loc,   &emitter->particleAttr->emitterRTMat1,  0, emitter->ptclNum, sizeof( ParticleAttribute ) );
    _SetAttrPtr( particleEmRT2Loc,   &emitter->particleAttr->emitterRTMat2,  0, emitter->ptclNum, sizeof( ParticleAttribute ) );

    // インスタンシング設定ON
    shader->EnableInstanced();

    if ( prim )
    {
        Render::DrawInstanceIndexedPrimitive( primType, prim->GetIndexNum(),
            static_cast<void *>( prim->GetIndexBuffer() ), emitter->ptclNum );
    }
    else
    {
        Render::DrawInstancePrimitive( primType, EFT_PARTICLE_PRIM_VERTICES_NUM, emitter->ptclNum );
    }

    // インスタンシング設定OFF
    shader->DisableInstanced();

    EFT_GLERR_CHECK();
#endif

    return true;
}
#endif

// NW_PLATFORM_SMART_DEVICE
#endif

} // namespace eft2
} // namespace nw
