﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eft2_Emitter.h>
#include <nw/eft/eft2_CurlNoiseData.h>
#include <nw/eft/eft2_MemUtil.h>

namespace nw  {
namespace eft2 {

CurlNoiseTbl        g_CurlNoiseTbl32;
CurlNoiseTexture    g_CurlNoiseTexture;
    // カールノイズテーブル
static const s8 g_curlNoiseTbl32[ 32 * 32 * 32 * 3  ] =
{
    #include "eft2_CurlNoiseDataTable.cpp"
};

//---------------------------------------------------
//  CurlNoiseフィールド コールバック
//---------------------------------------------------
void _CalcParticleBehavior_FieldCurlNoise( Emitter* emitter, nw::math::VEC4* pos, nw::math::VEC4* vec, const ParticleAttribute* attr )
{
    const ResFieldCurlNoiseData* dat = emitter->emitterRes->fieldCurlNoiseData;
    const f32 rate = emitter->frameRate;
    const f32 emitterTime    = emitter->time;

    const f32 spdMulX = dat->fieldCurlNoiseInfluence.x;        // 影響の強さX
    const f32 spdMulY = dat->fieldCurlNoiseInfluence.y;        // 影響の強さY
    const f32 spdMulZ = dat->fieldCurlNoiseInfluence.z;        // 影響の強さZ
    const f32 sizeMul = dat->fieldCurlNoiseScale;              // 渦のスケール

    const f32 ofsXSpd = dat->fieldCurlNoiseSpeed.x;            // 渦アニメ速度X
    const f32 ofsYSpd = dat->fieldCurlNoiseSpeed.y;            // 渦アニメ速度Y
    const f32 ofsZSpd = dat->fieldCurlNoiseSpeed.z;            // 渦アニメ速度Z
    f32 ofsBase = dat->fieldCurlNoiseBase;               // ノイズのオフセット
    if ( dat->isFieldCurlNoiseBaseRandom & 0x01 ) ofsBase *= attr->random.x;  // ノイズのオフセットランダム

    // パーティクル位置から受ける力を計算
    nw::math::VEC3 ptclPos;
    EFT_F32_VEC3_COPY( &ptclPos, pos );
    if( dat->isWorldCoordinate > 0 )
    {
        nw::math::VEC3 inPos( ptclPos );
        emitter->TransformToWorldPos( &ptclPos, inPos, attr );
    }

    nw::math::VEC3 tex_coord;
    tex_coord.x = ( ptclPos.x * sizeMul + ofsXSpd * emitterTime );
    tex_coord.y = ( ptclPos.y * sizeMul + ofsYSpd * emitterTime );
    tex_coord.z = ( ptclPos.z * sizeMul + ofsZSpd * emitterTime );
    tex_coord.x += ofsBase;
    tex_coord.y += ofsBase;
    tex_coord.z += ofsBase;

    // サンプリング（ 32 x 32 x 32 の空間から ）
    nw::math::VEC3 curlNoise;
    const s32 ipType  = ( dat->isFieldCurlNoiseInterpolation & 0x01 );                 // 補間タイプ(0:補間なし 1以上:補間あり)
    if( ipType == 0 )
    {
        curlNoise = g_CurlNoiseTbl32.GetCurlNoiseS8( static_cast< s32 >( tex_coord.x ), static_cast< s32 >( tex_coord.y ), static_cast< s32 >( tex_coord.z ) );
    }
    else
    {
        curlNoise = g_CurlNoiseTbl32.GetCurlNoiseS8Interpolate( tex_coord.x, tex_coord.y, tex_coord.z );
    }

    if( dat->isWorldCoordinate > 0 )
    {
        nw::math::VEC3 inDir( curlNoise );
        emitter->TransformToLocalVec( &curlNoise, inDir, attr );
    }

    // 力を速度に加算
    vec->x += curlNoise.x * spdMulX * rate;
    vec->y += curlNoise.y * spdMulY * rate;
    vec->z += curlNoise.z * spdMulZ * rate;
}

//---------------------------------------------------
//  CurlNoise パラメータ初期化処理
//---------------------------------------------------
void InitializeCurlNoise()
{
    g_CurlNoiseTbl32.Initialize( 32, (void*)&g_curlNoiseTbl32[0] );
#ifndef EFT_DEGRADATION_SPEC
    g_CurlNoiseTexture.Initialize();
#endif
}

//---------------------------------------------------
//  CurlNoise パラメータ終了処理
//---------------------------------------------------
void FinalizeCurlNoise()
{
#ifndef EFT_DEGRADATION_SPEC
    g_CurlNoiseTexture.Finalize();
#endif
}

//---------------------------------------------------------------------------
//  CurlNoise テクスチャでヒープから確保したメモリサイズを取得する
//---------------------------------------------------------------------------
u32 GetCurlNoiseTextureAllocedSize()
{
#ifndef EFT_DEGRADATION_SPEC
    return g_CurlNoiseTexture.allocedSize;
#else
    EFT_ERR( "Not Implemented." );
    return 0;
#endif
}

//---------------------------------------------------------------------------
//  カールノイズテクスチャを取得します。
//---------------------------------------------------------------------------
TextureBase& GetCurlNoiseTexture()
{
    return g_CurlNoiseTexture.GetTexture();
}


//---------------------------------------------------------------------------
//  初期化処理
//---------------------------------------------------------------------------
void CurlNoiseTexture::Initialize()
{
#ifdef EFT_DEGRADATION_SPEC
    EFT_ERR( "Not Implemented." );
#else

    #define   _CURL_NOISE_TEXTURE_WIDTH     32
    #define   _CURL_NOISE_TEXTURE_HEIGHT    32
    #define   _CURL_NOISE_TEXTURE_DEPTH     32

#if EFT_OPENGL
    {
        // Heap
        allocedSize = _CURL_NOISE_TEXTURE_WIDTH * _CURL_NOISE_TEXTURE_HEIGHT * _CURL_NOISE_TEXTURE_DEPTH * 4;
        textureImage= AllocFromStaticHeap( allocedSize );

        u32 idx = 0;

        u32 table_size = sizeof( g_curlNoiseTbl32 );
        table_size = 0;

        u8 r = 0;
        u8 g = 0;
        u8 b = 0;
        u8 a = 0;

        // RGBAテーブルへ置き換え
        for ( u32 k = 0; k < _CURL_NOISE_TEXTURE_DEPTH; k++ )
        {
            for ( u32 i = 0; i < _CURL_NOISE_TEXTURE_HEIGHT; i++ )
            {
                for ( u32 j = 0; j < _CURL_NOISE_TEXTURE_WIDTH; j++ )
                {
                    u32 height = i * _CURL_NOISE_TEXTURE_WIDTH;
                    u32 depth  = k * ( _CURL_NOISE_TEXTURE_WIDTH * _CURL_NOISE_TEXTURE_HEIGHT );

                    r = static_cast< u8 >( g_curlNoiseTbl32[ idx ]     + 128 );
                    g = static_cast< u8 >( g_curlNoiseTbl32[ idx + 1 ] + 128 );
                    b = static_cast< u8 >( g_curlNoiseTbl32[ idx + 2 ] + 128 );
                    a = 255;
                    idx +=3;

                    ( (u32 *)textureImage )[ depth + height + j] =
                        ( a << 24 ) | ( b << 16 ) | ( g <<  8 ) | ( r <<  0 );

                }
            }
        }

        glGenTextures( 1, &textureid );
        glBindTexture( GL_TEXTURE_3D, textureid );
        glPixelStorei( GL_UNPACK_ALIGNMENT, 1 );
        glTexImage3D(   GL_TEXTURE_3D, 0, GL_RGBA,
            _CURL_NOISE_TEXTURE_WIDTH, _CURL_NOISE_TEXTURE_HEIGHT, _CURL_NOISE_TEXTURE_DEPTH,
            0, GL_RGBA, GL_UNSIGNED_BYTE, textureImage );

        texture.Initialize( textureid, EFT_TEXTURE_TYPE_3D );

        EFT_GLERR_CHECK();
    }
#endif

#if EFT_GX2
    {
        textureImage = NULL;
        GX2InitTexture( &gx2Texture,
            _CURL_NOISE_TEXTURE_WIDTH,                  // width
            _CURL_NOISE_TEXTURE_HEIGHT,                 // height
            _CURL_NOISE_TEXTURE_DEPTH,                  // depth
            0,                                          // num mips
            GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM,   // format(8888固定)
            GX2_SURFACE_DIM_3D );                 // dim

        gx2Texture.surface.tileMode = GX2_TILE_MODE_LINEAR_ALIGNED;
        GX2CalcSurfaceSizeAndAlignment( &gx2Texture.surface );
        GX2InitTextureRegs( &gx2Texture );

        allocedSize = gx2Texture.surface.imageSize;
        textureImage = AllocFromStaticHeap( allocedSize, gx2Texture.surface.alignment );
        GX2InitTexturePtrs( &gx2Texture, textureImage, 0 );

        // RGBAテーブルへ置き換え
        u32 idx = 0;
        for ( u32 k = 0; k < _CURL_NOISE_TEXTURE_DEPTH; k++ )
        {
            for ( u32 i = 0; i < _CURL_NOISE_TEXTURE_HEIGHT; i++ )
            {
                for ( u32 j = 0; j < _CURL_NOISE_TEXTURE_WIDTH; j++ )
                {
                    u32 height = i * gx2Texture.surface.pitch;
                    u32 depth  = k * ( gx2Texture.surface.pitch * _CURL_NOISE_TEXTURE_HEIGHT );

                    u8 b = (u8)(g_curlNoiseTbl32[idx]   + 128);
                    u8 g = (u8)(g_curlNoiseTbl32[idx+1] + 128);
                    u8 r = (u8)(g_curlNoiseTbl32[idx+2] + 128);
                    u8 a = 255;

                    ( (u32 *)textureImage )[ depth + height + j] =
                        ( r << 24 ) | ( g << 16 ) | ( b <<  8 ) | ( a <<  0 );
                    idx += 3;
                }
            }
        }

        // Flush it out
        DCFlushRange( gx2Texture.surface.imagePtr,
            gx2Texture.surface.imageSize );

        texture.Initialize( &gx2Texture, EFT_TEXTURE_TYPE_3D );
    }
#endif
#endif      // EFT_DEGRADATION_SPEC
}

//---------------------------------------------------------------------------
//  終了処理
//---------------------------------------------------------------------------
void CurlNoiseTexture::Finalize()
{
#ifdef EFT_OGL
    if ( textureid != 0xFFFFFFFF )
    {
        glDeleteTextures( 1, &textureid );
    }
#endif

    if ( textureImage )
    {
        FreeFromStaticHeap( textureImage );
    }
}

//---------------------------------------------------------------------------
//  無効化
//---------------------------------------------------------------------------
void CurlNoiseTexture::Invalidate()
{
    textureImage    = NULL;
#ifdef EFT_OGL
    textureid       = 0xFFFFFFFF;
#endif
}



}
}
