﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_InitializeForScreenShotServer.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/capsrv/capsrv_AlbumControl.h>

#include "screenshotsrv/visrv_ScreenShotServerObject.h"

#define NN_CAPSRV_STRONG_REQUIRES(expr) \
    NN_SDK_REQUIRES(expr);  \
    NN_ABORT_UNLESS(expr);

namespace nn{ namespace capsrv{ namespace server{

    namespace {
        bool g_IsInitialized = false;
        NN_ALIGNAS(4096) char g_ScreenShotRequestQueueWorkMemory[4096];
        NN_ALIGNAS(4096) char g_ScreenShotControlRequestQueueWorkMemory[4096];
    }

    nn::Result InitializeForScreenShotServer(nn::os::ThreadType* pScreenShotWorkerThread) NN_NOEXCEPT
    {
        NN_CAPSRV_STRONG_REQUIRES(!g_IsInitialized);

        g_ScreenShotEnvironmentInfo.Initialize();

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());

        g_ScreenShotCaptureModule.Initialize();
        g_ScreenShotContextManager.Initialize();
        g_TransitionCaptureModule.Initialize();
        g_ScreenShotSequenceManager.Initialize();
        g_OverlayNotifier.Initialize();

        g_ScreenShotControlRequestQueue.Initialize(
            ScreenShotRequestWaitHolderTag_ScreenShotControl,
            g_ScreenShotControlRequestQueueWorkMemory,
            sizeof(g_ScreenShotControlRequestQueueWorkMemory)
        );
        g_ScreenShotRequestQueue.Initialize(
            ScreenShotRequestWaitHolderTag_ScreenShot,
            g_ScreenShotRequestQueueWorkMemory,
            sizeof(g_ScreenShotRequestQueueWorkMemory)
        );

        g_ScreenShotWorker.Initialize(pScreenShotWorkerThread);

        NN_ABORT_UNLESS_RESULT_SUCCESS(g_ScreenShotControlServerManager.Initialize());
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_ScreenShotServerManager.Initialize());

        g_ScreenShotControlServerManager.StartServer();
        g_ScreenShotServerManager.StartServer();
        g_IsInitialized = true;
        NN_RESULT_SUCCESS;
    }

    void FinalizeForScreenShotServer() NN_NOEXCEPT
    {
        NN_CAPSRV_STRONG_REQUIRES(g_IsInitialized);

        // ServerManager: Worker, Queue よりも先に Finalize すること
        g_ScreenShotServerManager.StopServer();
        g_ScreenShotControlServerManager.StopServer();
        g_ScreenShotServerManager.Finalize();
        g_ScreenShotControlServerManager.Finalize();

        // Worker: Queue よりも先に Finalize すること
        g_ScreenShotWorker.StopWorker();
        g_ScreenShotWorker.Finalize();

        // Queue:
        g_ScreenShotRequestQueue.Finalize();
        g_ScreenShotControlRequestQueue.Finalize();

        // Other:
        g_OverlayNotifier.Finalize();
        g_ScreenShotSequenceManager.Finalize();
        g_TransitionCaptureModule.Finalize();
        g_ScreenShotContextManager.Finalize();
        g_ScreenShotCaptureModule.Finalize();

        nn::capsrv::FinalizeAlbumControl();

        g_ScreenShotEnvironmentInfo.Finalize();
        g_IsInitialized = false;
    }

    void ScreenShotControlServerThreadFunction(void*) NN_NOEXCEPT
    {
        NN_CAPSRV_STRONG_REQUIRES(g_IsInitialized);
        g_ScreenShotControlServerManager.RunServer();
    }

    void ScreenShotControlRealtimeServerThreadFunction(void*) NN_NOEXCEPT
    {
        NN_CAPSRV_STRONG_REQUIRES(g_IsInitialized);
        g_ScreenShotControlServerManager.RunRealtimeServer();
    }

    void ScreenShotServerThreadFunction(void*) NN_NOEXCEPT
    {
        NN_CAPSRV_STRONG_REQUIRES(g_IsInitialized);
        g_ScreenShotServerManager.RunServer();
    }

    void ScreenShotWorkerThreadFunction(void*) NN_NOEXCEPT
    {
        NN_CAPSRV_STRONG_REQUIRES(g_IsInitialized);
        g_ScreenShotWorker.RunWorker();
    }

}}}
