﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "decodersrv_SoftwareJpegDecoderRgba8.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/capsrv/capsrv_Result.h>
#include "capsrvServer_JpegErrorHandler.h"

#ifdef NN_CAPSRV_USE_LIBJPEG_TURBO
#include "capsrvServer_SoftwareJpegEncoderType-encoder.libjpegTurbo.h"
#else
#include "capsrvServer_SoftwareJpegEncoderType-encoder.libjpeg.h"
#endif

#define NN_CAPSRV_STRONG_REQUIRES(cond) \
    NN_SDK_REQUIRES(cond);  \
    NN_ABORT_UNLESS(cond);

#define NN_CAPSRV_STRONG_ASSERT(cond)   \
    NN_SDK_ASSERT(cond);    \
    NN_RESULT_THROW_UNLESS(cond, nn::capsrv::ResultAlbumError());

namespace nn{ namespace capsrv{ namespace server{ namespace jpeg{

    namespace{
        static const int RgbaComponentCount = 4;
        static const int RgbComponentCount  = 3;
        static const int ImageSizeHorizontalUnit = 16;
        static const int ImageSizeVerticalUnit = 4;
    }

    nn::Result SoftwareJpegDecoderRgba8::DecodeRgba8BufferFromBuffer(
        SoftwareJpegDecoderBufferOutputInfo& outputInfo,
        const SoftwareJpegDecoderBufferInputInfo& inputInfo,
        void* pWorkBuffer,
        size_t workBufferSize
    ) NN_NOEXCEPT
    {
        NN_CAPSRV_STRONG_REQUIRES((inputInfo.width % ImageSizeHorizontalUnit) == 0);
        NN_CAPSRV_STRONG_REQUIRES((inputInfo.height % ImageSizeVerticalUnit) == 0);
        NN_CAPSRV_STRONG_REQUIRES(outputInfo.pBuffer != nullptr);
        NN_CAPSRV_STRONG_REQUIRES(outputInfo.pOutHeight != nullptr);
        NN_CAPSRV_STRONG_REQUIRES(outputInfo.pOutWidth != nullptr);
        NN_CAPSRV_STRONG_REQUIRES(outputInfo.bufferSize >= static_cast<size_t>(RgbaComponentCount * inputInfo.width * inputInfo.height));

        NN_CAPSRV_LOG_DEV("Decoding option\n");
        NN_CAPSRV_LOG_DEV("  FancyUpsampling: %s\n", inputInfo.isFancyUpsamplingEnabled ? "yes" : "no");
        NN_CAPSRV_LOG_DEV("  BlockSmoothing : %s\n", inputInfo.isBlockSmoothingEnabled ? "yes" : "no");

        char* pWork = reinterpret_cast<char*>(pWorkBuffer);
        char* pWorkEnd = pWork + workBufferSize;

        uint8_t* pRgbBuffer = nullptr;
        size_t rgbBufferStride = 0;
        size_t rgbBufferSize = 0;
        {
            int rgbBufferWidth = ((inputInfo.width + ImageSizeHorizontalUnit - 1) / ImageSizeHorizontalUnit) * ImageSizeHorizontalUnit;
            rgbBufferStride = RgbComponentCount * rgbBufferWidth;
            rgbBufferSize   = rgbBufferStride * ImageSizeVerticalUnit;
            NN_RESULT_THROW_UNLESS(
                workBufferSize >= rgbBufferSize,
                ResultInternalJpegWorkMemoryShortage()
            );
            pRgbBuffer = reinterpret_cast<uint8_t*>(pWork);
            pWork += rgbBufferSize;
        }

        JpegEncoderType::jpeg_decompress_struct cinfo = {};
        JpegEncoderType::jpeg_workbuf workbuf = {pWork, static_cast<size_t>(pWorkEnd - pWork)};
        cinfo.workbuf = &workbuf;

        JpegErrorHandler jerr = {};
        cinfo.err = jpeg_std_error(&jerr);
        jerr.error_exit = JpegErrorHandler::HandleError;
        jerr.encoderResult = nn::ResultSuccess();

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
#pragma warning(push)
#pragma warning(disable: 4611)
#endif
        if (setjmp(jerr.jmpContext) == 0)
#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
#pragma warning(pop)
#endif
        {
            jpeg_create_decompress(&cinfo);
            NN_UTIL_SCOPE_EXIT{
                jpeg_destroy_decompress(&cinfo);
            };

            // 入力データ設定
            jpeg_mem_src(&cinfo, const_cast<unsigned char*>(reinterpret_cast<const unsigned char*>(inputInfo.pBuffer)), static_cast<unsigned long>(inputInfo.bufferSize));
            NN_RESULT_THROW_UNLESS(jpeg_read_header(&cinfo, true) == JPEG_HEADER_OK, nn::capsrv::ResultAlbumInvalidFileData());

            NN_CAPSRV_LOG_DEV("Decoding File info:\n");
            NN_CAPSRV_LOG_DEV("  image size  : w%d x h%d\n", cinfo.image_width, cinfo.image_height);
            NN_CAPSRV_LOG_DEV("  #components : %d\n", cinfo.num_components);
            NN_CAPSRV_LOG_DEV("  color space : %d\n", cinfo.jpeg_color_space);
            NN_CAPSRV_LOG_DEV("  has JFIF    : %s\n", cinfo.saw_JFIF_marker ? "yes" : "no");
            if(cinfo.saw_JFIF_marker)
            {
                NN_CAPSRV_LOG_DEV("    version      : %d.%d\n", cinfo.JFIF_major_version, cinfo.JFIF_minor_version);
                NN_CAPSRV_LOG_DEV("    density unit : %d\n", cinfo.density_unit);
                NN_CAPSRV_LOG_DEV("    density      : x%d y%d%d\n", cinfo.X_density, cinfo.Y_density);
            }
            NN_CAPSRV_LOG_DEV("  has Adobe marker : %s\n", cinfo.saw_Adobe_marker ? "yes" : "no");
            if(cinfo.saw_Adobe_marker)
            {
                NN_CAPSRV_LOG_DEV("    transform    : %d\n", cinfo.Adobe_transform);
            }

            // 画像サイズをチェック
            NN_RESULT_THROW_UNLESS(cinfo.image_width == inputInfo.width, nn::capsrv::ResultAlbumInvalidFileData());
            NN_RESULT_THROW_UNLESS(cinfo.image_height == inputInfo.height, nn::capsrv::ResultAlbumInvalidFileData());

            // パラメータ設定
            cinfo.out_color_space = JpegEncoderType::J_COLOR_SPACE::JCS_RGB;
            cinfo.dct_method = JpegEncoderType::J_DCT_METHOD::JDCT_ISLOW;
            cinfo.do_fancy_upsampling = inputInfo.isFancyUpsamplingEnabled ? true : false;
            cinfo.do_block_smoothing = inputInfo.isBlockSmoothingEnabled ? true : false;

            // デコード処理
            NN_RESULT_THROW_UNLESS(jpeg_start_decompress(&cinfo), nn::capsrv::ResultAlbumInvalidFileData());
            NN_CAPSRV_STRONG_ASSERT(cinfo.output_width == inputInfo.width);
            NN_CAPSRV_STRONG_ASSERT(cinfo.output_height == inputInfo.height);
            NN_CAPSRV_STRONG_ASSERT(cinfo.out_color_components == 3);
            NN_CAPSRV_STRONG_ASSERT(cinfo.output_components == cinfo.out_color_components);

            {
                uint8_t* pDst = reinterpret_cast<uint8_t*>(outputInfo.pBuffer);

                JpegEncoderType::JSAMPROW lines[ImageSizeVerticalUnit] = {};
                for(int i = 0; i < ImageSizeVerticalUnit; i++)
                {
                    lines[i] = pRgbBuffer + rgbBufferStride * i;
                }

                while(cinfo.output_scanline < inputInfo.height)
                {
                    auto n = jpeg_read_scanlines(&cinfo, lines, ImageSizeVerticalUnit);
                    NN_CAPSRV_STRONG_ASSERT(n <= ImageSizeVerticalUnit);

                    for(int i = 0; i < n; i++)
                    {
                        const uint8_t* pSrc = lines[i];
                        for(int j = 0; j < inputInfo.width; j++)
                        {
                            *(pDst++) = *(pSrc++);  // R
                            *(pDst++) = *(pSrc++);  // G
                            *(pDst++) = *(pSrc++);  // B
                            *(pDst++) = 255;        // A
                        }
                    }
                }
            }
            NN_RESULT_THROW_UNLESS(jpeg_finish_decompress(&cinfo), nn::capsrv::ResultAlbumInvalidFileData());
        }
        else
        {
            NN_RESULT_THROW(nn::capsrv::ResultAlbumInvalidFileData());
        }

        *outputInfo.pOutWidth  = static_cast<int>(cinfo.output_width);
        *outputInfo.pOutHeight = static_cast<int>(cinfo.output_height);
        NN_RESULT_SUCCESS;
    }// NOLINT(impl/function_size)

}}}}
