﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_ScreenShotAttributeUtility.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/capsrv/capsrv_Result.h>

namespace nn{ namespace capsrv{ namespace server{ namespace detail{

    namespace {

        // ScreenShotSize
        nn::Result GetWidthAndHeight(uint32_t* pOutWidth, uint32_t* pOutHeight, nn::capsrv::ScreenShotSizeType size) NN_NOEXCEPT
        {
            switch(size)
            {
            case nn::capsrv::ScreenShotSize_1280x720:
                *pOutWidth = 1280;
                *pOutHeight = 720;
                NN_RESULT_SUCCESS;
            case nn::capsrv::ScreenShotSize_1920x1080:
                *pOutWidth = 1920;
                *pOutHeight = 1080;
                NN_RESULT_SUCCESS;
            default: NN_RESULT_THROW(nn::capsrv::ResultScreenShotInvalidSize());
            }
        }

        nn::Result GetScreenShotSize(nn::capsrv::ScreenShotSizeType* pOutValue, uint32_t width, uint32_t height) NN_NOEXCEPT
        {
            if(width == 1280 && height == 720)
            {
                *pOutValue = nn::capsrv::ScreenShotSize_1280x720;
                NN_RESULT_SUCCESS;
            }
            else if(width == 1920 && height == 1080)
            {
                *pOutValue = nn::capsrv::ScreenShotSize_1920x1080;
                NN_RESULT_SUCCESS;
            }
            else
            {
                NN_RESULT_THROW(nn::capsrv::ResultScreenShotInvalidSize());
            }
        }
    }

    nn::Result ScreenShotAttributeUtility::ConvertToMakerNoteInfo(
        MakerNoteInfo* pOutValue,
        MakerNoteVersionType makerNoteVersion,
        EncryptedApplicationId applicationId,
        AlbumFileDateTime dateTime,
        const nn::capsrv::ScreenShotAttribute& src,
        AppletData* pAppletData,
        ApplicationData* pApplicationData,
        SystemReservedInfo* pSystemReservedInfo
    ) NN_NOEXCEPT
    {
        uint32_t dataWidth = 0;
        uint32_t dataHeight = 0;
        NN_RESULT_DO(GetWidthAndHeight(&dataWidth, &dataHeight, src.size));

        uint32_t movieDataWidth = 0;
        uint32_t movieDataHeight = 0;
        NN_RESULT_DO(GetWidthAndHeight(&movieDataWidth, &movieDataHeight, src.movieSize));

        std::memset(pOutValue, 0, sizeof(*pOutValue));

        pOutValue->version = makerNoteVersion;
        // pOutValue->signature; 署名はゼロ埋めのまま
        pOutValue->encryptedApplicationId = applicationId;
        pOutValue->dateTime = dateTime;
        pOutValue->fileDescription = src.description;
        pOutValue->dataWidth       = dataWidth;
        pOutValue->dataHeight      = dataHeight;
        pOutValue->movieDataWidth  = movieDataWidth;
        pOutValue->movieDataHeight = movieDataHeight;
        pOutValue->dataOrientation = src.orientation;
        pOutValue->frameCount               = src.frameCount;
        pOutValue->frameRateNumerator       = src.frameRateNumerator;
        pOutValue->frameRateDenominator     = src.frameRateDenominator;
        pOutValue->dataDurationMilliseconds = src.dataDurationMilliseconds;
        pOutValue->keyFrameInterval         = src.keyFrameInterval;
        pOutValue->isCopyrightComposited     = src.IsCopyrightImageComposited() ? 1 : 0;
        pOutValue->hasUneditableArea         = src.HasUneditableArea() ? 1 : 0;
        pOutValue->uneditableAreaCoordinateX = src.uneditableAreaCoordinateX;
        pOutValue->uneditableAreaCoordinateY = src.uneditableAreaCoordinateY;
        pOutValue->uneditableAreaWidth       = src.uneditableAreaWidth;
        pOutValue->uneditableAreaHeight      = src.uneditableAreaHeight;

        if(pAppletData)
        {
            std::memcpy(&pOutValue->appletData, pAppletData, sizeof(AppletData));
        }

        if(pApplicationData)
        {
            std::memcpy(&pOutValue->applicationData, pApplicationData, sizeof(ApplicationData));
        }

        if(pSystemReservedInfo)
        {
            std::memcpy(&pOutValue->systemReservedInfo, pSystemReservedInfo, sizeof(SystemReservedInfo));
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result ScreenShotAttributeUtility::ConvertFromMakerNoteInfo(
        nn::capsrv::ScreenShotAttribute* pOutValue,
        const MakerNoteInfo& src
    ) NN_NOEXCEPT
    {
        nn::capsrv::ScreenShotSizeType screenshotSize = {};
        NN_RESULT_DO(GetScreenShotSize(&screenshotSize, src.dataWidth, src.dataHeight));

        nn::capsrv::ScreenShotSizeType movieSize = {};
        NN_RESULT_DO(GetScreenShotSize(&movieSize, src.movieDataWidth, src.movieDataHeight));

        uint32_t flags
            = (src.isCopyrightComposited ? nn::capsrv::detail::ScreenShotAttributeFlag_IsCopyrightImageComposited : 0)
            | (src.hasUneditableArea ? nn::capsrv::detail::ScreenShotAttributeFlag_HasUneditableArea : 0)
            ;

        pOutValue->size = screenshotSize;
        pOutValue->orientation = src.dataOrientation;
        pOutValue->description = src.fileDescription;
        pOutValue->frameCount  = src.frameCount;
        pOutValue->frameRateNumerator       = src.frameRateNumerator;
        pOutValue->frameRateDenominator     = src.frameRateDenominator;
        pOutValue->dataDurationMilliseconds = src.dataDurationMilliseconds;
        pOutValue->keyFrameInterval         = src.keyFrameInterval;
        pOutValue->uneditableAreaCoordinateX = src.uneditableAreaCoordinateX;
        pOutValue->uneditableAreaCoordinateY = src.uneditableAreaCoordinateY;
        pOutValue->uneditableAreaWidth       = src.uneditableAreaWidth;
        pOutValue->uneditableAreaHeight      = src.uneditableAreaHeight;
        pOutValue->flags = flags;
        pOutValue->movieSize = movieSize;
        NN_RESULT_SUCCESS;
    }

}}}}
