﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_OverlayThumbnailHolder.h"

#include <mutex>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/capsrv/capsrv_ThumbnailFormat.h>
#include "../capsrvServer_ResultPrivate.h"

namespace nn{ namespace capsrv{ namespace server{ namespace detail{

    const int OverlayThumbnailHolder::ThumbnailCount;

    OverlayThumbnailHolder::OverlayThumbnailHolder() NN_NOEXCEPT
    {
        std::memset(this, 0, sizeof(*this));
    }

    size_t OverlayThumbnailHolder::GetThumbnailDataSize() NN_NOEXCEPT
    {
        return ThumbnailImageDataSize_Default;
    }

    size_t OverlayThumbnailHolder::GetRequiredMemorySize() NN_NOEXCEPT
    {
        return ThumbnailCount * GetThumbnailDataSize();
    }

    size_t OverlayThumbnailHolder::GetRequiredMemoryAlignment() NN_NOEXCEPT
    {
        return 8; // とりあえずアクセス効率よさそうな alignof(int64_t) くらい。
    }

    void OverlayThumbnailHolder::SetMemory(void* memory, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!m_IsInitialized);
        m_Memory = memory;
        m_MemorySize = size;
    }

    void OverlayThumbnailHolder::Initialize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!m_IsInitialized);
        NN_SDK_REQUIRES_NOT_NULL(m_Memory);
        NN_SDK_REQUIRES_ALIGNED(m_Memory, GetRequiredMemoryAlignment());
        NN_SDK_REQUIRES_GREATER_EQUAL(m_MemorySize, GetRequiredMemorySize());

        char* pMemoryHead = reinterpret_cast<char*>(m_Memory);
        ptrdiff_t offset = 0;
        for(int i = 0; i < ThumbnailCount; i++)
        {
            m_ThumbnailList[i].m_ImageBuffer = pMemoryHead + offset;
            m_ThumbnailList[i].m_FileId = {};
            m_ThumbnailList[i].m_LockState = ThumbnailLockState_Unlocked;

            // 画像の初期値は白一色
            std::memset(m_ThumbnailList[i].m_ImageBuffer, 0xFF, GetThumbnailDataSize());

            offset += GetThumbnailDataSize();
        }
        NN_SDK_ASSERT_LESS_EQUAL(static_cast<size_t>(offset), m_MemorySize);

        m_LockStateMutex.Initialize();
        m_LockStateConditionVariable.Initialize();

        m_IsInitialized = true;
    }

    void OverlayThumbnailHolder::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);

        for(int i = 0; i < ThumbnailCount; i++)
        {
            NN_SDK_REQUIRES_EQUAL(m_ThumbnailList[i].m_LockState, ThumbnailLockState_Unlocked);
        }

        m_IsInitialized = false;
    }

    nn::Result OverlayThumbnailHolder::GetThumbnailImage(
        AlbumFileId* pOutAlbumFileId,
        size_t* pOutDataSize,
        void* pOutDataBuffer,
        size_t bufferSize,
        int index
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        NN_SDK_REQUIRES_NOT_NULL(pOutAlbumFileId);
        NN_SDK_REQUIRES_NOT_NULL(pOutDataSize);
        NN_SDK_REQUIRES_NOT_NULL(pOutDataBuffer);
        NN_SDK_REQUIRES_RANGE(index, 0, ThumbnailCount);

        NN_RESULT_THROW_UNLESS(
            bufferSize >= GetThumbnailDataSize(),
            nn::capsrv::ResultAlbumReadBufferShortage()
        );

        std::lock_guard<decltype(m_LockStateMutex)> lock(m_LockStateMutex);

        auto& e = m_ThumbnailList[index];
        WaitThumbnailUnlockedImpl(e);

        // NOTE:
        //   この関数は Mutex を取りっぱなしで読むので Lock-Unlock 操作しない。

        std::memcpy(pOutDataBuffer, e.m_ImageBuffer, GetThumbnailDataSize());
        *pOutDataSize = GetThumbnailDataSize();
        *pOutAlbumFileId = e.m_FileId;
        NN_RESULT_SUCCESS;
    }

    void* OverlayThumbnailHolder::BeginUpdateThumbnailImage(const AlbumFileId& fileId, int index) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        NN_SDK_REQUIRES_RANGE(index, 0, ThumbnailCount);

        std::lock_guard<decltype(m_LockStateMutex)> lock(m_LockStateMutex);

        auto& e = m_ThumbnailList[index];
        WaitThumbnailUnlockedImpl(e);

        LockThumbnailImpl(e);
        e.m_FileId = fileId;
        return e.m_ImageBuffer;
    }

    void OverlayThumbnailHolder::EndUpdateThumbnailImage(int index) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        NN_SDK_REQUIRES_RANGE(index, 0, ThumbnailCount);


        std::lock_guard<decltype(m_LockStateMutex)> lock(m_LockStateMutex);

        auto& e = m_ThumbnailList[index];
        UnlockThumbnailImpl(e);
    }

    void OverlayThumbnailHolder::WaitThumbnailUnlockedImpl(Thumbnail& e) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_LockStateMutex.IsLockedByCurrentThread());

        while(e.m_LockState != ThumbnailLockState_Unlocked)
        {
            m_LockStateConditionVariable.Wait(m_LockStateMutex);
        }
    }

    void OverlayThumbnailHolder::LockThumbnailImpl(Thumbnail& e) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_LockStateMutex.IsLockedByCurrentThread());
        NN_SDK_REQUIRES_EQUAL(e.m_LockState, ThumbnailLockState_Unlocked);

        e.m_LockState = ThumbnailLockState_Locked;
    }

    void OverlayThumbnailHolder::UnlockThumbnailImpl(Thumbnail& e) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_LockStateMutex.IsLockedByCurrentThread());
        NN_SDK_REQUIRES_EQUAL(e.m_LockState, ThumbnailLockState_Locked);

        e.m_LockState = ThumbnailLockState_Unlocked;
        m_LockStateConditionVariable.Signal();
    }

}}}}
