﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/capsrv/capsrv_AlbumFileDateTime.h>
#include <nn/capsrv/capsrv_AlbumFileDescription.h>
#include <nn/capsrv/capsrv_AppletData.h>
#include <nn/capsrv/capsrv_ApplicationData.h>
#include <nn/capsrv/capsrv_SystemReservedInfo.h>
#include "capsrvServer_MakerNoteEntry.h"
#include "capsrvServer_MakerNoteEncryption.h"
#include "capsrvServer_ScreenShotFileSignature.h"
#include "capsrvServer_Signature.h"
#include "capsrvServer_EncryptedApplicationId.h"
#include "../capsrvServer_Config.h"
#include "capsrvServer_MakerNoteValuePropertyUtility.h"

namespace nn{ namespace capsrv{ namespace server{ namespace detail{

    typedef void MakerNoteTag;

    enum MakerNoteVersion
    {
        MakerNoteVersion_Version0 = 0,
        MakerNoteVersion_Version1 = 1,  // NX Addon 4.0.0 (FW4.0.0-0.0) 以降
        MakerNoteVersion_Version2 = 2,  // NX Addon 5.1.0 (FW5.0.0-1.0) 以降
        MakerNoteVersion_Version3 = 3,  // NX Addon 6.0.0 (FW6.0.0-0.0) 以降
        MakerNoteVersion_End,
    };

    // 各 MakerNoteVersion のポリシー型
    struct MakerNoteVersionPolicyV0
    {
        static const MakerNoteVersionType Version = MakerNoteVersion_Version0;

        static const bool IsErrorOnUnknownEntryEnabled = true;
        static const size_t    SizeMin = 44;
        static const size_t    SizeMax = 44;

        static const ptrdiff_t SignatureOffset = 8;
        static const size_t    SignatureSize   = 16;
        static const ScreenShotFileSignatureVersion SignatureVersion = ScreenShotFileSignatureVersion_Version0;

        static const ptrdiff_t EncryptionStartOffset = 0;
        typedef MakerNoteEncryptionPolicyNoEncryption EncryptionPolicy;

        NN_STATIC_ASSERT(SizeMax == SizeMin);
    };

    struct MakerNoteVersionPolicyV1
    {
        static const MakerNoteVersionType Version = MakerNoteVersion_Version1;

        static const bool IsErrorOnUnknownEntryEnabled = true;
        static const size_t    SizeMin = 1176;
        static const size_t    SizeMax = 1176;

        static const ptrdiff_t SignatureOffset = 8; // version(4) + signatureTag(2) + signatureSize(2)
        static const size_t    SignatureSize   = 16;
        static const ScreenShotFileSignatureVersion SignatureVersion = ScreenShotFileSignatureVersion_Version1;

        static const ptrdiff_t EncryptionStartOffset = 24; // version(4) + signatureEntry(2 + 2 + 16)
        typedef MakerNoteEncryptionPolicyV1 EncryptionPolicy;

        NN_STATIC_ASSERT(SizeMax == SizeMin);
        NN_STATIC_ASSERT((SizeMin - EncryptionStartOffset) % EncryptionPolicy::UnitSize == 0);
    };

    struct MakerNoteVersionPolicyV2
    {
        static const MakerNoteVersionType Version = MakerNoteVersion_Version2;

        static const bool IsErrorOnUnknownEntryEnabled = false;
        static const size_t    SizeMin = 1176; // 適当
        static const size_t    SizeMax = MakerNoteSizeLimit;

        static const ptrdiff_t SignatureOffset = 8; // version(4) + signatureTag(2) + signatureSize(2)
        static const size_t    SignatureSize   = 16;
        static const ScreenShotFileSignatureVersion SignatureVersion = ScreenShotFileSignatureVersion_Version1;

        static const ptrdiff_t EncryptionStartOffset = 24; // version(4) + signatureEntry(2 + 2 + 16)
        typedef MakerNoteEncryptionPolicyV1 EncryptionPolicy;
    };

    struct MakerNoteVersionPolicyV3
    {
        static const MakerNoteVersionType Version = MakerNoteVersion_Version3;

        static const bool IsErrorOnUnknownEntryEnabled = false;
        static const size_t    SizeMin = 1176; // 適当
        static const size_t    SizeMax = MakerNoteSizeLimit;

        static const ptrdiff_t SignatureOffset = 8; // version(4) + signatureTag(2) + signatureSize(2)
        static const size_t    SignatureSize   = 16;
        static const ScreenShotFileSignatureVersion SignatureVersion = ScreenShotFileSignatureVersion_Version1;

        static const ptrdiff_t EncryptionStartOffset = 24; // version(4) + signatureEntry(2 + 2 + 16)
        typedef MakerNoteEncryptionPolicyV1 EncryptionPolicy;
    };

#define NN_CAPSRV_DETAIL_FOREACH_MAKERNOTEVERSIONPOLICY(macro)    \
    macro(nn::capsrv::server::detail::MakerNoteVersionPolicyV0)    \
    macro(nn::capsrv::server::detail::MakerNoteVersionPolicyV1)    \
    macro(nn::capsrv::server::detail::MakerNoteVersionPolicyV2)    \
    macro(nn::capsrv::server::detail::MakerNoteVersionPolicyV3)    \
    //

    //-------------------------------------------------------
    // MakerNoteEntry の定義
    //-------------------------------------------------------

    enum MakerNoteEntryTag
    {

        // 署名。
        // version >= 0。
        MakerNoteEntryTag_Signature              = 0x0000,

        // 暗号化されたアプリケーションID。
        // version >= 0。
        MakerNoteEntryTag_EncryptedApplicationId = 0x0001,

        // 撮影日時。
        // version >= 1。
        MakerNoteEntryTag_DateTime               = 0x0002,

        // ファイル種別詳細。
        // version >= 1。
        MakerNoteEntryTag_FileDescription        = 0x0003,

        // Ocean 用拡張領域。
        // version >= 1。
        MakerNoteEntryTag_AppletData             = 0x0004,

        // 欠番（使用禁止）
        MakerNoteEntryTag_Unused0                = 0x0005,

        // ScreenShotData の幅。
        // version >= 1。
        MakerNoteEntryTag_DataWidth              = 0x0006,

        // ScreenShotData の高さ。
        // version >= 1。
        MakerNoteEntryTag_DataHeight             = 0x0007,

        // Data の回転。
        // version >= 1。
        MakerNoteEntryTag_DataOrientation        = 0x0008,

        // フレーム数。
        // version >= 1。
        MakerNoteEntryTag_FrameCount             = 0x0009,

        // フレームレートの分子。
        // version >= 1。
        MakerNoteEntryTag_FrameRateNumerator     = 0x000A,

        // フレームレートの分母。
        // version >= 1。
        MakerNoteEntryTag_FrameRateDenominator   = 0x000B,

        // 再生時間(ms)。
        // version >= 1。
        MakerNoteEntryTag_DataDurationMilliseconds = 0x000C,

        // キーフレーム間隔(frames)。
        // version >= 1。
        MakerNoteEntryTag_KeyFrameInterval       = 0x000D,

        // 権利表記の有無。
        // version >= 2。
        MakerNoteEntryTag_CopyrightComposition   = 0x000E,

        // 編集不可範囲（矩形領域）
        // version >= 2。
        MakerNoteEntryTag_UneditableArea            = 0x000F,
        MakerNoteEntryTag_UneditableAreaCoordinateX = 0x0010,
        MakerNoteEntryTag_UneditableAreaCoordinateY = 0x0011,
        MakerNoteEntryTag_UneditableAreaWidth       = 0x0012,
        MakerNoteEntryTag_UneditableAreaHeight      = 0x0013,

        // Application 用拡張領域。
        // version >= 2。
        MakerNoteEntryTag_ApplicationData        = 0x0014,

        // MovieData の幅。
        // version >= 2。
        MakerNoteEntryTag_MovieDataWidth  = 0x0015,

        // MovieData の高さ。
        // version >= 2。
        MakerNoteEntryTag_MovieDataHeight = 0x0016,

        // SystemReservedInfo の大きさ。
        // version >= 3。
        MakerNoteEntryTag_SystemReservedInfo = 0x0017,

        MakerNoteEntryTag_Max,

    };

    typedef nn::capsrv::AlbumFileDateTime MakerNoteDateTimeType;
    typedef nn::capsrv::AlbumFileDescriptionType MakerNoteFileDescriptionType;
    typedef nn::capsrv::AppletData MakerNoteAppletDataType;
    typedef uint32_t MakerNoteDataWidthType;
    typedef uint32_t MakerNoteDataHeightType;
    typedef uint32_t MakerNoteDataOrientationType;
    typedef uint32_t MakerNoteFrameCountType;
    typedef uint32_t MakerNoteFrameRateNumeratorType;
    typedef uint32_t MakerNoteFrameRateDenominatorType;
    typedef uint32_t MakerNoteDataDurationType;
    typedef uint32_t MakerNoteKeyFrameIntervalType;
    typedef uint32_t MakerNoteCopyrightCompositionType;
    typedef uint32_t MakerNoteUneditableAreaType;
    typedef uint32_t MakerNoteUneditableAreaCoordinateXType;
    typedef uint32_t MakerNoteUneditableAreaCoordinateYType;
    typedef uint32_t MakerNoteUneditableAreaWidthType;
    typedef uint32_t MakerNoteUneditableAreaHeightType;
    typedef nn::capsrv::ApplicationData MakerNoteApplicationDataType;
    typedef nn::capsrv::SystemReservedInfo MakerNoteSystemReservedInfoType;

    struct MakerNoteInfo
    {
    public:
        // バージョン
        MakerNoteVersionType version;

        // 署名
        Signature signature;

        // 暗号化されたアプリケーション ID
        EncryptedApplicationId encryptedApplicationId;

        // 撮影日時
        MakerNoteDateTimeType dateTime;

        // ファイル種別詳細
        MakerNoteFileDescriptionType fileDescription;

        // Applet 用拡張領域
        MakerNoteAppletDataType appletData;

        // Application 用拡張領域
        MakerNoteApplicationDataType applicationData;

        // ScreenShotData の幅(px)
        MakerNoteDataWidthType dataWidth;

        // ScreenShotData の高さ(px)
        MakerNoteDataHeightType dataHeight;

        // MovieData の幅(px)
        MakerNoteDataWidthType movieDataWidth;

        // MovieData の高さ(px)
        MakerNoteDataWidthType movieDataHeight;

        // Data の回転
        MakerNoteDataOrientationType dataOrientation;

        // Data のフレーム数(frame)
        MakerNoteFrameCountType frameCount;

        // Data のフレームレートの分子(frame)
        MakerNoteFrameRateNumeratorType frameRateNumerator;

        // Data のフレームレートの分母(sec)
        MakerNoteFrameRateDenominatorType frameRateDenominator;

        // Data の再生時間(ms)
        MakerNoteDataDurationType dataDurationMilliseconds;

        // Data のキーフレーム間隔(frame)
        MakerNoteKeyFrameIntervalType keyFrameInterval;

        // Data の権利表記の有無
        MakerNoteCopyrightCompositionType isCopyrightComposited;

        // Data の編集不可領域の有無
        MakerNoteUneditableAreaType hasUneditableArea;

        // Data の編集不可領域
        MakerNoteUneditableAreaCoordinateXType uneditableAreaCoordinateX;
        MakerNoteUneditableAreaCoordinateYType uneditableAreaCoordinateY;
        MakerNoteUneditableAreaWidthType       uneditableAreaWidth;
        MakerNoteUneditableAreaHeightType      uneditableAreaHeight;

        // SystemReservedInfo 用領域
        MakerNoteSystemReservedInfoType systemReservedInfo;
    };

    NN_STATIC_ASSERT(sizeof(MakerNoteInfo) <= 1024 * 4);


    class MakerNoteValueProperty
        : public MakerNoteValuePropertyUtility
    {
    public:
        struct SignatureProperty
            : TagDefinition<MakerNoteEntryTag_Signature>
            , ValueDefinition<MakerNoteInfo, Signature, &MakerNoteInfo::signature, MakerNoteValueClass_Bytes>
            , DefaultValueUnexpected<Signature>
            , EntryCountMinMaxFrom<MakerNoteVersion_Version0, 1, 1, void>
        {
        };

        struct EncryptedApplicationIdProperty
            : TagDefinition<MakerNoteEntryTag_EncryptedApplicationId>
            , ValueDefinition<MakerNoteInfo, EncryptedApplicationId, &MakerNoteInfo::encryptedApplicationId, MakerNoteValueClass_Bytes>
            , DefaultValueUnexpected<EncryptedApplicationId>
            , EntryCountMinMaxFrom<MakerNoteVersion_Version0, 1, 1, void>
        {
        };

        struct DateTimeProperty
            : TagDefinition<MakerNoteEntryTag_DateTime>
            , ValueDefinition<MakerNoteInfo, MakerNoteDateTimeType, &MakerNoteInfo::dateTime, MakerNoteValueClass_Bytes>
            , DefaultValueZeroClear<MakerNoteVersion_Version0, MakerNoteDateTimeType, void>
            , EntryCountMinMaxFrom<MakerNoteVersion_Version1, 1, 1, void>
        {
        };

        struct FileDescriptionProperty
            : TagDefinition<MakerNoteEntryTag_FileDescription>
            , ValueDefinition<MakerNoteInfo, MakerNoteFileDescriptionType, &MakerNoteInfo::fileDescription, MakerNoteValueClass_Integer>
            , DefaultValueInteger<MakerNoteVersion_Version0, MakerNoteFileDescriptionType, AlbumFileDescription_ScreenShot, void>
            , EntryCountMinMaxFrom<MakerNoteVersion_Version1, 1, 1, void>
        {
        };

        struct AppletDataProperty
            : TagDefinition<MakerNoteEntryTag_AppletData>
            , ValueDefinition<MakerNoteInfo, MakerNoteAppletDataType, &MakerNoteInfo::appletData, MakerNoteValueClass_Bytes>
            , DefaultValueZeroClear<MakerNoteVersion_Version0, MakerNoteAppletDataType, void>
            , EntryCountMinMaxFrom<MakerNoteVersion_Version1, 1, 1, void>
        {
        };

        struct DataWidthProperty
            : TagDefinition<MakerNoteEntryTag_DataWidth>
            , ValueDefinition<MakerNoteInfo, MakerNoteDataWidthType, &MakerNoteInfo::dataWidth, MakerNoteValueClass_Integer>
            , DefaultValueInteger<MakerNoteVersion_Version0, MakerNoteDataWidthType, 1280, void>
            , EntryCountMinMaxFrom<MakerNoteVersion_Version1, 1, 1, void>
        {
        };

        struct DataHeightProperty
            : TagDefinition<MakerNoteEntryTag_DataHeight>
            , ValueDefinition<MakerNoteInfo, MakerNoteDataHeightType, &MakerNoteInfo::dataHeight, MakerNoteValueClass_Integer>
            , DefaultValueInteger<MakerNoteVersion_Version0, MakerNoteDataHeightType, 720, void>
            , EntryCountMinMaxFrom<MakerNoteVersion_Version1, 1, 1, void>
        {
        };

        struct DataOrientationProperty
            : TagDefinition<MakerNoteEntryTag_DataOrientation>
            , ValueDefinition<MakerNoteInfo, MakerNoteDataOrientationType, &MakerNoteInfo::dataOrientation, MakerNoteValueClass_Integer>
            , DefaultValueInteger<MakerNoteVersion_Version0, MakerNoteDataOrientationType, 0, void>
            , EntryCountMinMaxFrom<MakerNoteVersion_Version1, 1, 1, void>
        {
        };

        struct FrameCountProperty
            : TagDefinition<MakerNoteEntryTag_FrameCount>
            , ValueDefinition<MakerNoteInfo, MakerNoteFrameCountType, &MakerNoteInfo::frameCount, MakerNoteValueClass_Integer>
            , DefaultValueInteger<MakerNoteVersion_Version0, MakerNoteFrameCountType, 1, void>
            , EntryCountMinMaxFrom<MakerNoteVersion_Version1, 1, 1, void>
        {
        };

        struct FrameRateNumeratorProperty
            : TagDefinition<MakerNoteEntryTag_FrameRateNumerator>
            , ValueDefinition<MakerNoteInfo, MakerNoteFrameRateNumeratorType, &MakerNoteInfo::frameRateNumerator, MakerNoteValueClass_Integer>
            , DefaultValueInteger<MakerNoteVersion_Version0, MakerNoteFrameRateNumeratorType, 0, void>
            , EntryCountMinMaxFrom<MakerNoteVersion_Version1, 1, 1, void>
        {
        };

        struct FrameRateDenominatorProperty
            : TagDefinition<MakerNoteEntryTag_FrameRateDenominator>
            , ValueDefinition<MakerNoteInfo, MakerNoteFrameRateDenominatorType, &MakerNoteInfo::frameRateDenominator, MakerNoteValueClass_Integer>
            , DefaultValueInteger<MakerNoteVersion_Version0, MakerNoteFrameRateDenominatorType, 0, void>
            , EntryCountMinMaxFrom<MakerNoteVersion_Version1, 1, 1, void>
        {
        };

        struct DataDurationProperty
            : TagDefinition<MakerNoteEntryTag_DataDurationMilliseconds>
            , ValueDefinition<MakerNoteInfo, MakerNoteDataDurationType, &MakerNoteInfo::dataDurationMilliseconds, MakerNoteValueClass_Integer>
            , DefaultValueInteger<MakerNoteVersion_Version0, MakerNoteDataDurationType, 0, void>
            , EntryCountMinMaxFrom<MakerNoteVersion_Version1, 1, 1, void>
        {
        };

        struct KeyFrameIntervalProperty
            : TagDefinition<MakerNoteEntryTag_KeyFrameInterval>
            , ValueDefinition<MakerNoteInfo, MakerNoteKeyFrameIntervalType, &MakerNoteInfo::keyFrameInterval, MakerNoteValueClass_Integer>
            , DefaultValueInteger<MakerNoteVersion_Version0, MakerNoteKeyFrameIntervalType, 0, void>
            , EntryCountMinMaxFrom<MakerNoteVersion_Version1, 1, 1, void>
        {
        };

        struct CopyrightCompositionProperty
            : TagDefinition<MakerNoteEntryTag_CopyrightComposition>
            , ValueDefinition<MakerNoteInfo, MakerNoteCopyrightCompositionType, &MakerNoteInfo::isCopyrightComposited, MakerNoteValueClass_Integer>
            , DefaultValueInteger<MakerNoteVersion_Version0, MakerNoteCopyrightCompositionType, 0, void>
            , EntryCountMinMaxFrom<MakerNoteVersion_Version2, 1, 1, void>
        {
        };

        struct UneditableAreaProperty
            : TagDefinition<MakerNoteEntryTag_UneditableArea>
            , ValueDefinition<MakerNoteInfo, MakerNoteUneditableAreaType, &MakerNoteInfo::hasUneditableArea, MakerNoteValueClass_Integer>
            , DefaultValueInteger<MakerNoteVersion_Version0, MakerNoteUneditableAreaType, 0, void>
            , EntryCountMinMaxFrom<MakerNoteVersion_Version2, 1, 1, void>
        {
        };

        struct UneditableAreaCoordinateXProperty
            : TagDefinition<MakerNoteEntryTag_UneditableAreaCoordinateX>
            , ValueDefinition<MakerNoteInfo, MakerNoteUneditableAreaCoordinateXType, &MakerNoteInfo::uneditableAreaCoordinateX, MakerNoteValueClass_Integer>
            , DefaultValueInteger<MakerNoteVersion_Version0, MakerNoteUneditableAreaCoordinateXType, 0, void>
            , EntryCountMinMaxFrom<MakerNoteVersion_Version2, 1, 1, void>
        {
        };

        struct UneditableAreaCoordinateYProperty
            : TagDefinition<MakerNoteEntryTag_UneditableAreaCoordinateY>
            , ValueDefinition<MakerNoteInfo, MakerNoteUneditableAreaCoordinateYType, &MakerNoteInfo::uneditableAreaCoordinateY, MakerNoteValueClass_Integer>
            , DefaultValueInteger<MakerNoteVersion_Version0, MakerNoteUneditableAreaCoordinateYType, 0, void>
            , EntryCountMinMaxFrom<MakerNoteVersion_Version2, 1, 1, void>
        {
        };

        struct UneditableAreaWidthProperty
            : TagDefinition<MakerNoteEntryTag_UneditableAreaWidth>
            , ValueDefinition<MakerNoteInfo, MakerNoteUneditableAreaWidthType, &MakerNoteInfo::uneditableAreaWidth, MakerNoteValueClass_Integer>
            , DefaultValueInteger<MakerNoteVersion_Version0, MakerNoteUneditableAreaWidthType, 0, void>
            , EntryCountMinMaxFrom<MakerNoteVersion_Version2, 1, 1, void>
        {
        };

        struct UneditableAreaHeightProperty
            : TagDefinition<MakerNoteEntryTag_UneditableAreaHeight>
            , ValueDefinition<MakerNoteInfo, MakerNoteUneditableAreaHeightType, &MakerNoteInfo::uneditableAreaHeight, MakerNoteValueClass_Integer>
            , DefaultValueInteger<MakerNoteVersion_Version0, MakerNoteUneditableAreaHeightType, 0, void>
            , EntryCountMinMaxFrom<MakerNoteVersion_Version2, 1, 1, void>
        {
        };

        struct ApplicationDataProperty
            : TagDefinition<MakerNoteEntryTag_ApplicationData>
            , ValueDefinition<MakerNoteInfo, MakerNoteApplicationDataType, &MakerNoteInfo::applicationData, MakerNoteValueClass_Bytes>
            , DefaultValueZeroClear<MakerNoteVersion_Version0, MakerNoteApplicationDataType, void>
            , EntryCountMinMaxFrom<MakerNoteVersion_Version2, 1, 1, void>
        {
        };

        struct MovieDataWidthProperty
            : TagDefinition<MakerNoteEntryTag_MovieDataWidth>
            , ValueDefinition<MakerNoteInfo, MakerNoteDataWidthType, &MakerNoteInfo::movieDataWidth, MakerNoteValueClass_Integer>
            , DefaultValueInteger<MakerNoteVersion_Version0, MakerNoteDataWidthType, 0xFFFFFFFF, void>
            , EntryCountMinMaxFrom<MakerNoteVersion_Version2, 1, 1, void>
        {
        };

        struct MovieDataHeightProperty
            : TagDefinition<MakerNoteEntryTag_MovieDataHeight>
            , ValueDefinition<MakerNoteInfo, MakerNoteDataHeightType, &MakerNoteInfo::movieDataHeight, MakerNoteValueClass_Integer>
            , DefaultValueInteger<MakerNoteVersion_Version0, MakerNoteDataHeightType, 0xFFFFFFFF, void>
            , EntryCountMinMaxFrom<MakerNoteVersion_Version2, 1, 1, void>
        {
        };

        struct SystemReservedInfoProperty
            : TagDefinition<MakerNoteEntryTag_SystemReservedInfo>
            , ValueDefinition<MakerNoteInfo, MakerNoteSystemReservedInfoType, &MakerNoteInfo::systemReservedInfo, MakerNoteValueClass_Bytes>
            , DefaultValueZeroClear<MakerNoteVersion_Version0, MakerNoteSystemReservedInfoType, void>
            , EntryCountMinMaxFrom<MakerNoteVersion_Version3, 1, 1, void>
        {
        };

    };
}}}}

// ここは MakerNoteInfo 構造体内のメンバ順ではなく、
// MakerNoteEntryTag の番号順に並べること。
#define NN_CAPSRV_DETAIL_FOREACH_MAKERNOTEPROPERTY(macro)    \
    macro(nn::capsrv::server::detail::MakerNoteValueProperty::SignatureProperty)    \
    macro(nn::capsrv::server::detail::MakerNoteValueProperty::EncryptedApplicationIdProperty)    \
    macro(nn::capsrv::server::detail::MakerNoteValueProperty::DateTimeProperty)    \
    macro(nn::capsrv::server::detail::MakerNoteValueProperty::FileDescriptionProperty) \
    macro(nn::capsrv::server::detail::MakerNoteValueProperty::AppletDataProperty)    \
    macro(nn::capsrv::server::detail::MakerNoteValueProperty::DataWidthProperty)    \
    macro(nn::capsrv::server::detail::MakerNoteValueProperty::DataHeightProperty)    \
    macro(nn::capsrv::server::detail::MakerNoteValueProperty::DataOrientationProperty)    \
    macro(nn::capsrv::server::detail::MakerNoteValueProperty::FrameCountProperty)    \
    macro(nn::capsrv::server::detail::MakerNoteValueProperty::FrameRateNumeratorProperty)    \
    macro(nn::capsrv::server::detail::MakerNoteValueProperty::FrameRateDenominatorProperty) \
    macro(nn::capsrv::server::detail::MakerNoteValueProperty::DataDurationProperty) \
    macro(nn::capsrv::server::detail::MakerNoteValueProperty::KeyFrameIntervalProperty) \
    macro(nn::capsrv::server::detail::MakerNoteValueProperty::CopyrightCompositionProperty) \
    macro(nn::capsrv::server::detail::MakerNoteValueProperty::UneditableAreaProperty) \
    macro(nn::capsrv::server::detail::MakerNoteValueProperty::UneditableAreaCoordinateXProperty) \
    macro(nn::capsrv::server::detail::MakerNoteValueProperty::UneditableAreaCoordinateYProperty) \
    macro(nn::capsrv::server::detail::MakerNoteValueProperty::UneditableAreaWidthProperty) \
    macro(nn::capsrv::server::detail::MakerNoteValueProperty::UneditableAreaHeightProperty) \
    macro(nn::capsrv::server::detail::MakerNoteValueProperty::ApplicationDataProperty) \
    macro(nn::capsrv::server::detail::MakerNoteValueProperty::MovieDataWidthProperty)    \
    macro(nn::capsrv::server::detail::MakerNoteValueProperty::MovieDataHeightProperty)   \
    macro(nn::capsrv::server::detail::MakerNoteValueProperty::SystemReservedInfoProperty)   \
    //
