﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_CalculateJpegMac.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_IntUtil.h>
#include <nn/crypto/crypto_HmacSha256Generator.h>
#include <nn/capsrv/capsrv_Result.h>

#include "../capsrvServer_Config.h"
#include "capsrvServer_MakerNoteInfo.h"
#include "../capsrvServer_ResultPrivate.h"

namespace nn{ namespace capsrv{ namespace server{ namespace detail{

    namespace {

        class MacVersion0
        {
        public:
            // @brief Hmac の長さ（バイト）
            static const size_t HmacLength = 32;

            // @brief Hmac 鍵の長さ（バイト）
            static const size_t HmacKeyLength = 32;

            // @brief Hmac 鍵のバイト列（ランダムに生成したもの）
            static const uint8_t HmacKey[HmacKeyLength];

        public:
            static nn::Result CalculateImpl(Signature* pOutValue, const void* jpegData, size_t jpegDataSize, ptrdiff_t signatureOffset, size_t signatureSize) NN_NOEXCEPT
            {
                nn::crypto::HmacSha256Generator generator;
                generator.Initialize(HmacKey, HmacKeyLength);

                const char* p = reinterpret_cast<const char*>(jpegData);
                Signature zeroSignature = {};

                // 署名部分より前
                generator.Update(p, signatureOffset);
                // 署名部分はゼロ埋めしたもので計算
                generator.Update(&zeroSignature, sizeof(Signature));
                // 署名部分の後
                size_t processedSize = static_cast<size_t>(signatureOffset) + sizeof(Signature);
                p += static_cast<ptrdiff_t>(processedSize);
                size_t remainSize = jpegDataSize - processedSize;
                generator.Update(p, remainSize);

                // 署名の方が短いので一時変数で受ける
                // NOTE: 長さが同じになったら直接 pOutValue で受けられる
                NN_STATIC_ASSERT(sizeof(Signature) < HmacLength);
                char mac[HmacLength] = {};
                generator.GetMac(mac, sizeof(mac));

                // 先頭部分だけコピー
                std::memcpy(pOutValue, mac, sizeof(Signature));
                NN_RESULT_SUCCESS;
            }

        };

        const uint8_t MacVersion0::HmacKey[MacVersion0::HmacKeyLength] = {
            0x28, 0x7A, 0xAB, 0xF9, 0xFE, 0xD3, 0x4D, 0x4E,
            0x99, 0x5C, 0xC7, 0xBE, 0x0D, 0x91, 0x4A, 0x32,
            0x21, 0xB9, 0x82, 0x2A, 0x45, 0x52, 0x46, 0x49,
            0xB0, 0xA2, 0xCB, 0xCD, 0xD4, 0xB9, 0x8E, 0x4A
        };

        //--------------------------------------------

        nn::Result (*g_SignerFunctionVersion1)(Signature* pOutValue, const void* data, size_t dataSize) = nullptr;

        class MacVersion1
        {
        public:
            static nn::Result CalculateImpl(Signature* pOutValue, const void* jpegData, size_t jpegDataSize, ptrdiff_t signatureOffset, size_t signatureSize) NN_NOEXCEPT
            {
                NN_SDK_ASSERT_NOT_NULL(g_SignerFunctionVersion1);
                NN_RESULT_THROW_UNLESS(g_SignerFunctionVersion1 != nullptr, ResultInternalError());

                // sha256 を計算
                nn::crypto::Sha256Generator hasher;
                hasher.Initialize();

                const char* p = reinterpret_cast<const char*>(jpegData);
                Signature zeroSignature = {};

                // 署名部分より前
                hasher.Update(p, signatureOffset);
                // 署名部分はゼロ埋めしたもので計算
                hasher.Update(&zeroSignature, sizeof(Signature));
                // 署名部分の後
                size_t processedSize = static_cast<size_t>(signatureOffset) + sizeof(Signature);
                p += static_cast<ptrdiff_t>(processedSize);
                size_t remainSize = jpegDataSize - processedSize;
                hasher.Update(p, remainSize);

                char hash[nn::crypto::Sha256Generator::HashSize] = {};
                hasher.GetHash(hash, sizeof(hash));

                Signature value = {};
                NN_RESULT_DO(g_SignerFunctionVersion1(&value, hash, sizeof(hash)));
                *pOutValue = value;
                NN_RESULT_SUCCESS;
            }
        };

#ifdef NN_CAPSRV_DEBUG_ENABLE_FILE_SIGNATURE_FOR_UNIT_TEST
        class MacForTesting
        {
        public:
            // 自動テスト用。単に sha256 を計算して先頭 16 バイトを返す
            static nn::Result CalculateImpl(Signature* pOutValue, const void* jpegData, size_t jpegDataSize, ptrdiff_t signatureOffset, size_t signatureSize) NN_NOEXCEPT
            {
                // sha256 を計算
                nn::crypto::Sha256Generator hasher;
                hasher.Initialize();

                const char* p = reinterpret_cast<const char*>(jpegData);
                Signature zeroSignature = {};

                // 署名部分より前
                hasher.Update(p, signatureOffset);
                // 署名部分はゼロ埋めしたもので計算
                hasher.Update(&zeroSignature, sizeof(Signature));
                // 署名部分の後
                size_t processedSize = static_cast<size_t>(signatureOffset) + sizeof(Signature);
                p += static_cast<ptrdiff_t>(processedSize);
                size_t remainSize = jpegDataSize - processedSize;
                hasher.Update(p, remainSize);

                char hash[nn::crypto::Sha256Generator::HashSize] = {};
                hasher.GetHash(hash, sizeof(hash));

                // 先頭部分だけコピー
                std::memcpy(pOutValue, hash, sizeof(Signature));
                NN_RESULT_SUCCESS;
            }
        };
#endif


        template<typename Impl>
        class MacCalculator
        {
        public:
            static nn::Result CalculateSignRange(ptrdiff_t* pOutOffset, size_t* pOutSize, int64_t makerNoteOffset, int64_t makerNoteSize, ptrdiff_t signatureOffsetInMakerNote) NN_NOEXCEPT
            {
                NN_CAPSRV_LOG_DEV("calculate sign range(mnote: offset %lld, size %lld)\n", makerNoteOffset, makerNoteSize);

                // makerNote のサイズチェック
                NN_RESULT_THROW_UNLESS(makerNoteSize >= static_cast<int64_t>(sizeof(Signature)), ResultAlbumInvalidFileData());
                NN_RESULT_THROW_UNLESS(makerNoteSize <= MakerNoteSizeLimit, ResultAlbumInvalidFileData());

                // makerNote は EXIF の範囲内にある
                NN_CAPSRV_LOG_DEV("  checking maker-note is in exif\n");
                NN_RESULT_THROW_UNLESS(makerNoteOffset >= 0, ResultAlbumInvalidFileData());
                NN_RESULT_THROW_UNLESS(makerNoteOffset <= AlbumManagerThumbnailLoadSize - makerNoteSize, ResultAlbumInvalidFileData());

                ptrdiff_t signatureOffset = static_cast<ptrdiff_t>(makerNoteOffset) + signatureOffsetInMakerNote;
                size_t signatureSize = sizeof(Signature);

                NN_CAPSRV_LOG_DEV("  -> offset %lld, size %llu\n", signatureOffset, signatureSize);
                *pOutOffset = signatureOffset;
                *pOutSize = signatureSize;
                NN_RESULT_SUCCESS;
            }

            static nn::Result CalculateSignature(Signature* pOutValue, const void* jpegData, size_t jpegDataSize, ptrdiff_t signatureOffset, size_t signatureSize) NN_NOEXCEPT
            {
                NN_RESULT_THROW_UNLESS(jpegData != nullptr, ResultInternalError());
                NN_RESULT_THROW_UNLESS(nn::util::IsIntValueRepresentable<int64_t>(jpegDataSize), ResultInternalError());
                NN_RESULT_THROW_UNLESS(signatureOffset + signatureSize <= static_cast<int64_t>(jpegDataSize), ResultAlbumInvalidFileData());

                Signature signature = {};
                // 置き換え前はゼロ埋めされていることを確認
                NN_RESULT_THROW_UNLESS(0 == std::memcmp(reinterpret_cast<const char*>(jpegData) + signatureOffset, &signature, sizeof(signature)), ResultAlbumInvalidFileData());

                NN_RESULT_DO(Impl::CalculateImpl(&signature, jpegData, jpegDataSize, signatureOffset, signatureSize));

                *pOutValue = signature;
                NN_RESULT_SUCCESS;
            }

            static nn::Result CheckSignature(const void* jpegData, size_t jpegDataSize, ptrdiff_t signatureOffset, size_t signatureSize) NN_NOEXCEPT
            {
                NN_RESULT_THROW_UNLESS(jpegData != nullptr, ResultInternalError());
                NN_RESULT_THROW_UNLESS(nn::util::IsIntValueRepresentable<int64_t>(jpegDataSize), ResultInternalError());
                NN_RESULT_THROW_UNLESS(signatureOffset + signatureSize <= static_cast<int64_t>(jpegDataSize), ResultAlbumInvalidFileData());

                Signature signature = {};
                NN_RESULT_DO(Impl::CalculateImpl(&signature, jpegData, jpegDataSize, signatureOffset, signatureSize));

                NN_RESULT_THROW_UNLESS(0 == std::memcmp(reinterpret_cast<const char*>(jpegData) + signatureOffset, &signature, sizeof(signature)), ResultInternalFileDataVerificationInconsistentSignature());
                NN_RESULT_SUCCESS;
            }

        };

        template<typename Impl>
        nn::Result CalculateJpegMacImpl(
            Signature* pOutValue,
            int64_t* pOutOffset,
            const void* pJpegData,
            size_t jpegDataSize,
            int64_t makerNoteOffset,
            int64_t makerNoteSize,
            ptrdiff_t signatureOffsetInMakerNote
            ) NN_NOEXCEPT
        {
            ptrdiff_t signatureOffset = 0;
            size_t signatureSize = 0;
            NN_RESULT_DO(MacCalculator<Impl>::CalculateSignRange(&signatureOffset, &signatureSize, makerNoteOffset, makerNoteSize, signatureOffsetInMakerNote));

            Signature signature = {};
            NN_RESULT_DO(MacCalculator<Impl>::CalculateSignature(&signature, pJpegData, jpegDataSize, signatureOffset, signatureSize));

            *pOutValue = signature;
            *pOutOffset = signatureOffset;
            NN_RESULT_SUCCESS;
        }

        template<typename Impl>
        nn::Result CalculateAndCheckJpegMacImpl(
            const void* pJpegData,
            size_t jpegDataSize,
            int64_t makerNoteOffset,
            int64_t makerNoteSize,
            ptrdiff_t signatureOffsetInMakerNote
            ) NN_NOEXCEPT
        {
            ptrdiff_t signatureOffset = 0;
            size_t signatureSize = 0;
            NN_RESULT_DO(MacCalculator<Impl>::CalculateSignRange(&signatureOffset, &signatureSize, makerNoteOffset, makerNoteSize, signatureOffsetInMakerNote));

#ifdef NN_CAPSRV_DEBUG_ENABLE_FILE_SIGNATURE_FOR_UNIT_TEST
            if(MacCalculator<MacForTesting>::CheckSignature(pJpegData, jpegDataSize, signatureOffset, signatureSize).IsSuccess())
            {
                NN_CAPSRV_LOG_WARN("screenshot signature for testing enabled!!\n");
                NN_RESULT_SUCCESS;
            }
#endif

            NN_RESULT_DO(MacCalculator<Impl>::CheckSignature(pJpegData, jpegDataSize, signatureOffset, signatureSize));
            NN_RESULT_SUCCESS;
        }

    }

    namespace {

        ScreenShotFileSignatureVersion GetFileSignatureVersionImpl(uint64_t makerNoteVersion) NN_NOEXCEPT
        {
            switch(makerNoteVersion)
            {
            #define NN_CAPSRV_DETAIL_CASE_MAKERNOTEVERSION(policy) \
            case policy::Version: return policy::SignatureVersion;

                NN_CAPSRV_DETAIL_FOREACH_MAKERNOTEVERSIONPOLICY(NN_CAPSRV_DETAIL_CASE_MAKERNOTEVERSION);

            #undef NN_CAPSRV_DETAIL_CASE_MAKERNOTEVERSION
            default:
                return ScreenShotFileSignatureVersion_Invlid;
            }
        }

        ptrdiff_t GetSignatureOffsetInMakerNoteImpl(uint64_t makerNoteVersion) NN_NOEXCEPT
        {
            switch(makerNoteVersion)
            {
            #define NN_CAPSRV_DETAIL_CASE_MAKERNOTEVERSION(policy) \
            case policy::Version: return policy::SignatureOffset;

                NN_CAPSRV_DETAIL_FOREACH_MAKERNOTEVERSIONPOLICY(NN_CAPSRV_DETAIL_CASE_MAKERNOTEVERSION);

            #undef NN_CAPSRV_DETAIL_CASE_MAKERNOTEVERSION
            default: NN_UNEXPECTED_DEFAULT;
            }
        }

    }

    nn::Result CalculateJpegMac(
        Signature* pOutValue,
        int64_t* pOutOffset,
        const void* pJpegData,
        size_t jpegDataSize,
        uint64_t makerNoteVersion,
        int64_t makerNoteOffset,
        int64_t makerNoteSize
        ) NN_NOEXCEPT
    {
        switch(GetFileSignatureVersionImpl(makerNoteVersion))
        {
        case ScreenShotFileSignatureVersion_Version0:
            return CalculateJpegMacImpl<MacVersion0>(pOutValue, pOutOffset, pJpegData, jpegDataSize, makerNoteOffset, makerNoteSize, GetSignatureOffsetInMakerNoteImpl(makerNoteVersion));
        case ScreenShotFileSignatureVersion_Version1:
            return CalculateJpegMacImpl<MacVersion1>(pOutValue, pOutOffset, pJpegData, jpegDataSize, makerNoteOffset, makerNoteSize, GetSignatureOffsetInMakerNoteImpl(makerNoteVersion));
        default:
            NN_RESULT_THROW(ResultAlbumInvalidFileData());
        }
    }

    nn::Result CalculateAndCheckJpegMac(
        const void* pJpegData,
        size_t jpegDataSize,
        int64_t makerNoteOffset,
        int64_t makerNoteSize
        ) NN_NOEXCEPT
    {
        // 少なくとも MakerNote の末尾が jpegData の中に納まっていることの確認
        //   makerNoteOffset + makerNoteSize <= jpegDataSize
        NN_RESULT_THROW_UNLESS(makerNoteOffset >= 0, ResultInternalError());
        NN_RESULT_THROW_UNLESS(makerNoteSize >= 0, ResultInternalError());
        NN_RESULT_THROW_UNLESS(nn::util::IsIntValueRepresentable<int64_t>(jpegDataSize), ResultInternalError());
        NN_RESULT_THROW_UNLESS(makerNoteOffset <= static_cast<int64_t>(jpegDataSize), ResultAlbumInvalidFileData());
        NN_RESULT_THROW_UNLESS(makerNoteSize <= static_cast<int64_t>(jpegDataSize) - makerNoteOffset, ResultAlbumInvalidFileData());

        uint32_t makerNoteVersion = 0;
        std::memcpy(&makerNoteVersion, reinterpret_cast<const char*>(pJpegData) + makerNoteOffset, sizeof(makerNoteVersion));

        switch(GetFileSignatureVersionImpl(makerNoteVersion))
        {
        case ScreenShotFileSignatureVersion_Version0:
            return CalculateAndCheckJpegMacImpl<MacVersion0>(pJpegData, jpegDataSize, makerNoteOffset, makerNoteSize, GetSignatureOffsetInMakerNoteImpl(makerNoteVersion));
        case ScreenShotFileSignatureVersion_Version1:
            return CalculateAndCheckJpegMacImpl<MacVersion1>(pJpegData, jpegDataSize, makerNoteOffset, makerNoteSize, GetSignatureOffsetInMakerNoteImpl(makerNoteVersion));
        default:
            NN_RESULT_THROW(ResultAlbumInvalidFileData());
        }
    }

    void SetScreenShotFileSignFunction(
        ScreenShotFileSignatureVersion fileSignatureVersion,
        nn::Result (*function)(Signature* pOutValue, const void* data, size_t dataSize)
    ) NN_NOEXCEPT
    {
        switch(fileSignatureVersion)
        {
        case ScreenShotFileSignatureVersion_Version1:
            {
                g_SignerFunctionVersion1 = function;
                return;
            }
        default: NN_UNEXPECTED_DEFAULT;
        }
    }


}}}}
