﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_EnvironmentInfo.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include "capsrvServer_Config.h"

namespace nn{ namespace capsrv{ namespace server{

    namespace {
        static const char* SettingName = "capsrv";

        bool GetBooleanSettingValue(const char* key, bool defaultValue) NN_NOEXCEPT
        {
            bool value = false;
            auto size = nn::settings::fwdbg::GetSettingsItemValue(&value, sizeof(value), SettingName, key);
            if(size == 0)
            {
                return defaultValue;
            }
            return value;
        }

        std::unique_ptr<char[]> GetStringSettingValue(size_t sizeMax, const char* key) NN_NOEXCEPT
        {
            auto p = std::unique_ptr<char[]>(new char[sizeMax]);
            auto realSize = nn::settings::fwdbg::GetSettingsItemValue(p.get(), sizeMax, SettingName, key);
            if (!(realSize <= sizeMax))
            {
                return nullptr;
            }
            return p;
        }
    }

    EnvironmentInfo::EnvironmentInfo() NN_NOEXCEPT
    {
        // 安全側に設定
        m_IsProdMode = true;
        m_IsRawScreenShotReadStreamEnabled = false;
        m_IsAlbumScreenShotFileSupported = false;
        m_IsAlbumMovieFileSupported = false;

        m_IsAlbumScreenShotFileDataVerificationEnabled = true;
        m_IsAlbumMovieFileSignVerificationEnabled = true;
        m_IsAlbumMovieFileHashVerificationEnabled = true;

        m_MakerNoteVersion = 0;
        m_NandScreenShotFileCountLimit = 0;
        m_NandMovieFileCountLimit = 0;
        m_SdScreenShotFileCountLimit = 0;
        m_SdMovieFileCountLimit = 0;
    }

    void EnvironmentInfo::SetDefault() NN_NOEXCEPT
    {
        *this = EnvironmentInfo();
    }

    void EnvironmentInfo::Initialize() NN_NOEXCEPT
    {
        bool isProdMode = !nn::settings::fwdbg::IsDebugModeEnabled();

        m_IsProdMode = isProdMode;
        m_IsRawScreenShotReadStreamEnabled = !isProdMode;

        m_IsAlbumScreenShotFileSupported = GetBooleanSettingValue("enable_album_screenshot_file_support", true);
        m_IsAlbumMovieFileSupported      = GetBooleanSettingValue("enable_album_movie_file_support", true);

        m_IsAlbumScreenShotFileDataVerificationEnabled = GetBooleanSettingValue("enable_album_screenshot_filedata_verification", true);
        m_IsAlbumMovieFileSignVerificationEnabled = GetBooleanSettingValue("enable_album_movie_filesign_verification", true);
        m_IsAlbumMovieFileHashVerificationEnabled = GetBooleanSettingValue("enable_album_movie_filehash_verification", true);

        m_MakerNoteVersion = detail::MakerNoteVersion_Version3;

        m_NandScreenShotFileCountLimit = DefaultNandScreenShotFileCountLimit;
        m_NandMovieFileCountLimit      = DefaultNandMovieFileCountLimit;
        m_SdScreenShotFileCountLimit   = DefaultSdScreenShotFileCountLimit;
        m_SdMovieFileCountLimit        = DefaultSdMovieFileCountLimit;

        {
            if (GetBooleanSettingValue("enable_album_directory_change", false))
            {
                auto p = GetStringSettingValue(256, "album_directory_path");
                if (p.get())
                {
                    this->m_pPcHostAlbumDirectoryPath = std::move(p);
                }
            }
        }

        Dump();
    }

    void EnvironmentInfo::Finalize() NN_NOEXCEPT
    {
        SetDefault();
    }

    //---------------------------------------------------------

    bool EnvironmentInfo::IsRawScreenShotReadStreamEnabled() const NN_NOEXCEPT
    {
        return m_IsRawScreenShotReadStreamEnabled;
    }

    bool EnvironmentInfo::IsAlbumScreenShotFileSupported() const NN_NOEXCEPT
    {
        return m_IsAlbumScreenShotFileSupported;
    }

    bool EnvironmentInfo::IsAlbumMovieFileSupported() const NN_NOEXCEPT
    {
        return m_IsAlbumMovieFileSupported;
    }

    bool EnvironmentInfo::IsAlbumExtraScreenShotFileSupported() const NN_NOEXCEPT
    {
        return true;
    }

    bool EnvironmentInfo::IsAlbumExtraMovieFileSupported() const NN_NOEXCEPT
    {
        return true;
    }

    bool EnvironmentInfo::IsAlbumExtraMovieFileSupportedForNandStorage() const NN_NOEXCEPT
    {
        // TODO:
        //  現時点では固定値を返す
        return false;
    }

    bool EnvironmentInfo::IsAlbumFileContentsSupported(AlbumFileContentsType contents) const NN_NOEXCEPT
    {
        switch (contents)
        {
            case AlbumFileContents_ScreenShot:
            {
                return this->IsAlbumScreenShotFileSupported();
            }
            case AlbumFileContents_Movie:
            {
                return this->IsAlbumMovieFileSupported();
            }
            case AlbumFileContents_ExtraScreenShot:
            {
                return this->IsAlbumExtraScreenShotFileSupported();
            }
            case AlbumFileContents_ExtraMovie:
            {
                return this->IsAlbumExtraMovieFileSupported();
            }
            default:
            {
                return false;
            }
        }
    }

    bool EnvironmentInfo::IsAlbumScreenShotFileDataVerificationEnabled() const NN_NOEXCEPT
    {
        return m_IsAlbumScreenShotFileDataVerificationEnabled;
    }

    bool EnvironmentInfo::IsAlbumMovieFileSignVerificationEnabled() const NN_NOEXCEPT
    {
        return m_IsAlbumMovieFileSignVerificationEnabled;
    }

    bool EnvironmentInfo::IsAlbumMovieFileHashVerificationEnabled() const NN_NOEXCEPT
    {
        return m_IsAlbumMovieFileHashVerificationEnabled;
    }

    detail::MakerNoteVersionType EnvironmentInfo::GetMakerNoteVersion() const NN_NOEXCEPT
    {
        return m_MakerNoteVersion;
    }

    int64_t EnvironmentInfo::GetFileCountLimit(AlbumStorageType storage, AlbumFileContentsType contents) const NN_NOEXCEPT
    {
        if (storage == AlbumStorage_Nand)
        {
            switch (contents)
            {
                case AlbumFileContents_ScreenShot:
                {
                    return m_NandScreenShotFileCountLimit;
                }
                case AlbumFileContents_Movie:
                {
                    return m_NandMovieFileCountLimit;
                }
                case AlbumFileContents_ExtraScreenShot:
                {
                    return m_NandScreenShotFileCountLimit;
                }
                case AlbumFileContents_ExtraMovie:
                {
                    return m_NandMovieFileCountLimit;
                }
                default:
                {
                    return 0;
                }
            }
        }
        else if (storage == AlbumStorage_Sd)
        {
            switch (contents)
            {
                case AlbumFileContents_ScreenShot:
                {
                    return m_SdScreenShotFileCountLimit;
                }
                case AlbumFileContents_Movie:
                {
                    return m_SdMovieFileCountLimit;
                }
                case AlbumFileContents_ExtraScreenShot:
                {
                    return m_SdScreenShotFileCountLimit;
                }
                case AlbumFileContents_ExtraMovie:
                {
                    return m_SdMovieFileCountLimit;
                }
                default:
                {
                    return 0;
                }
            }
        }
        else
        {
            return 0;
        }
    }

    const char* EnvironmentInfo::GetPcHostAlbumDirectoryPath() const NN_NOEXCEPT
    {
        // 製品機では常に PC-Host は指定できないようにしておく
        return m_IsProdMode ? nullptr : m_pPcHostAlbumDirectoryPath.get();
    }


    //---------------------------------------------------------

    void EnvironmentInfo::Dump() const NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_DEV("[env]EnvironmentInfo:\n");
        NN_CAPSRV_LOG_DEV("[env]  raw-screenshot-read-stream : %s\n", m_IsRawScreenShotReadStreamEnabled ? "enable" : "disable");
        NN_CAPSRV_LOG_DEV("[env]  support-screenshot-file : %s\n", m_IsAlbumScreenShotFileSupported ? "yes" : "no");
        NN_CAPSRV_LOG_DEV("[env]  support-movie-file      : %s\n", m_IsAlbumMovieFileSupported ? "yes" : "no");
        NN_CAPSRV_LOG_DEV("[env]  enable-screenshot-filedata-verification : %s\n", m_IsAlbumScreenShotFileDataVerificationEnabled ? "yes" : "no");
        NN_CAPSRV_LOG_DEV("[env]  enable-movie-filesign-verification      : %s\n", m_IsAlbumMovieFileSignVerificationEnabled ? "yes" : "no");
        NN_CAPSRV_LOG_DEV("[env]  enable-movie-filehash-verification      : %s\n", m_IsAlbumMovieFileHashVerificationEnabled ? "yes" : "no");
        NN_CAPSRV_LOG_DEV("[env]  maker-note-version : %d\n", m_MakerNoteVersion);
        NN_CAPSRV_LOG_DEV("[env]  file-count-limits:\n");
        NN_CAPSRV_LOG_DEV("[env]    nand-screenshot : %lld\n", m_NandScreenShotFileCountLimit);
        NN_CAPSRV_LOG_DEV("[env]    nand-movie      : %lld\n", m_NandMovieFileCountLimit);
        NN_CAPSRV_LOG_DEV("[env]    sd-screenshot   : %lld\n", m_SdScreenShotFileCountLimit);
        NN_CAPSRV_LOG_DEV("[env]    sd-movie        : %lld\n", m_SdMovieFileCountLimit);

        {
            auto p = this->GetPcHostAlbumDirectoryPath();
            NN_CAPSRV_LOG_INFO("[env]  album-directory-path : %s\n", p == nullptr ? "<default>" : p);
            NN_UNUSED(p);
        }
    }
}}}

