﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_AlbumControlServiceImpl.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/capsrv/capsrv_Result.h>
#include <mutex>

#include "../capsrvServer_Config.h"
#include "capsrvServer_AlbumServerObject.h"
#include "capsrvServer_AlbumControlSessionFactory.h"

#if defined(NN_CAPSRV_USE_HIPC)

#define NN_CAPSRV_SERVER_DO(Function)  \
    NN_RESULT_DO(g_AlbumErrorConverter.Convert(Function()));

#define NN_CAPSRV_SERVER_SYNC_DO(Function)  \
    NN_RESULT_DO(g_AlbumErrorConverter.Convert((::std::unique_lock<decltype(g_AlbumGlobalMutex)>(g_AlbumGlobalMutex), Function())));

#elif defined(NN_CAPSRV_USE_DIRECT_FUNCTION_CALL)

#define NN_CAPSRV_SERVER_DO(Function)  \
    NN_RESULT_DO(g_AlbumErrorConverter.Convert(Function()));
#define NN_CAPSRV_SERVER_SYNC_DO  NN_CAPSRV_SERVER_DO

#endif

namespace nn{ namespace capsrv{ namespace server{

    //---------------------------
    // caps:c
    //---------------------------

    nn::Result AlbumControlServiceImpl::NotifyAlbumStorageIsAvailable(
        AlbumStorageType storage
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_C(NotifyAlbumSotrageIsAvailable);
        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                NN_RESULT_DO(g_AlbumManager.ResetMountStatus(storage));
                NN_RESULT_DO(g_AlbumManager.RefreshStorageCache(storage));
                NN_RESULT_SUCCESS;
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlServiceImpl::NotifyAlbumStorageIsUnavailable(
        AlbumStorageType storage
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_C(NotifyAlbumStorageIsUnavailable);
        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                NN_RESULT_DO(g_AlbumManager.ForceUnmounted(storage));
                NN_RESULT_SUCCESS;
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlServiceImpl::RegisterAppletResourceUserId(
        nn::applet::AppletResourceUserId aruid,
        nn::ncm::ApplicationId applicationId
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_C(RegisterAppletResouceUserId);
        NN_CAPSRV_SERVER_DO((
            [&]() -> nn::Result {
                NN_RESULT_DO(g_ApplicationResourceManager.Register(aruid, applicationId));
                NN_RESULT_SUCCESS;
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlServiceImpl::UnregisterAppletResourceUserId(
        nn::applet::AppletResourceUserId aruid,
        nn::ncm::ApplicationId applicationId
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_C(UnregisterAppletResourceUserId);
        NN_CAPSRV_SERVER_DO((
            [&]() -> nn::Result {
                g_ApplicationResourceManager.Unregister(aruid, applicationId);
                NN_RESULT_SUCCESS;
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlServiceImpl::GetApplicationIdFromAruid(nn::sf::Out<nn::ncm::ApplicationId> outApplicationId, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_C(GetApplicationIdFromAruid);
        nn::ncm::ApplicationId applicationId = {};
        NN_CAPSRV_SERVER_DO((
            [&]() -> nn::Result {
                NN_RESULT_DO(g_ApplicationResourceManager.GetApplicationIdFromAruid(&applicationId, aruid));
                NN_RESULT_SUCCESS;
            }
        ));
        outApplicationId.Set(applicationId);
        NN_RESULT_SUCCESS;
    }


    nn::Result AlbumControlServiceImpl::CheckApplicationIdRegistered(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_C(CheckApplicationIdRegistered);
        NN_CAPSRV_SERVER_DO((
            [&]() -> nn::Result {
                NN_RESULT_DO(g_ApplicationResourceManager.CheckApplicationIdRegistered(applicationId));
                NN_RESULT_SUCCESS;
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlServiceImpl::GenerateCurrentAlbumFileId(
        nn::sf::Out<nn::capsrv::AlbumFileId> outFileId,
        nn::ncm::ProgramId programId,
        nn::capsrv::AlbumFileContentsType contentType
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_C(GenerateCurrentAlbumFileId);
        nn::capsrv::AlbumFileId fileId = {};
        NN_CAPSRV_SERVER_DO((
            [&]() -> nn::Result {
                NN_RESULT_DO(g_AlbumFileIdGenerator.GenerateCurrentAlbumFileId(
                    &fileId,
                    { programId.value },
                    contentType,
                    &g_AlbumManager
                ));
                NN_RESULT_SUCCESS;
            }
        ));
        outFileId.Set(fileId);
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlServiceImpl::GenerateApplicationAlbumEnrty(
        nn::sf::Out<nn::capsrv::ApplicationAlbumEntry> outEntry,
        const nn::capsrv::AlbumEntry& srcEntry,
        nn::ncm::ApplicationId applicationId
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_C(GenerateApplicationAlbumEnrty);
        nn::capsrv::ApplicationAlbumEntry appEntry = {};
        NN_CAPSRV_SERVER_DO((
            [&]() -> nn::Result {
                detail::ApplicationAlbumEntryKey key = {};
                NN_RESULT_DO(g_ApplicationResourceManager.GetApplicationAlbumEntryKeyFromApplicationId(&key, applicationId));
                appEntry = detail::GetApplicationAlbumEntryFromAlbumEntry(srcEntry, key);
                NN_RESULT_SUCCESS;
            }
        ));
        outEntry.Set(appEntry);
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlServiceImpl::SaveAlbumScreenShotFileEx(
        const nn::capsrv::AlbumFileId& fileId,
        const nn::sf::InBuffer& fileData,
        std::uint64_t makerNoteVersion,
        std::int64_t makerNoteOffset,
        std::int64_t makerNoteSize
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_C(SaveAlbumScreenShotFile);
        const void* pData = fileData.GetPointerUnsafe();
        size_t size = fileData.GetSize();
        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                NN_RESULT_DO(g_AlbumManager.SaveScreenShotFile(&fileId, pData, size, makerNoteVersion, makerNoteOffset, makerNoteSize));
                NN_RESULT_SUCCESS;
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlServiceImpl::SetOverlayScreenShotThumbnailData(
        const nn::capsrv::AlbumFileId& fileId,
        const nn::sf::InBuffer& thumbData
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_C(SetOverlayScreenShotThumbnailData);
        const void* pData = thumbData.GetPointerUnsafe();
        size_t size = thumbData.GetSize();

        NN_RESULT_THROW_UNLESS(size >= detail::OverlayThumbnailHolder::GetThumbnailDataSize(), ResultAlbumError());

        NN_CAPSRV_SERVER_DO((
            [&]() -> nn::Result {
                auto pHolder = &g_OverlayThumbnailHolder;
                auto pBuffer = pHolder->BeginUpdateThumbnailImage(fileId, nn::capsrv::server::detail::OverlayThumbnailIndex_ScreenShot);
                std::memcpy(pBuffer, pData, detail::OverlayThumbnailHolder::GetThumbnailDataSize());
                pHolder->EndUpdateThumbnailImage(nn::capsrv::server::detail::OverlayThumbnailIndex_ScreenShot);
                NN_RESULT_SUCCESS;
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlServiceImpl::SetOverlayMovieThumbnailData(
        const nn::capsrv::AlbumFileId& fileId,
        const nn::sf::InBuffer& thumbData
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_C(SetOverlayMovieThumbnailData);
        const void* pData = thumbData.GetPointerUnsafe();
        size_t size = thumbData.GetSize();

        NN_RESULT_THROW_UNLESS(size >= detail::OverlayThumbnailHolder::GetThumbnailDataSize(), ResultAlbumError());

        NN_CAPSRV_SERVER_DO((
            [&]() -> nn::Result {
                auto pHolder = &g_OverlayThumbnailHolder;
                auto pBuffer = pHolder->BeginUpdateThumbnailImage(fileId, nn::capsrv::server::detail::OverlayThumbnailIndex_Movie);
                std::memcpy(pBuffer, pData, detail::OverlayThumbnailHolder::GetThumbnailDataSize());
                pHolder->EndUpdateThumbnailImage(nn::capsrv::server::detail::OverlayThumbnailIndex_Movie);
                NN_RESULT_SUCCESS;
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlServiceImpl::OpenControlSession(nn::sf::Out<nn::sf::SharedPointer<nn::capsrv::sf::IAlbumControlSession>> outValue, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_C(OpenControlSession);
        NN_UNUSED(aruid);

        nn::sf::SharedPointer<sf::IAlbumControlSession> p;
        NN_RESULT_DO(AlbumControlSessionFactory::Create(&p));

        outValue.Set(p);
        NN_RESULT_SUCCESS;
    }

}}}
