﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_MovieWriteStreamEntry.h"

#include <limits>
#include <algorithm>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_IntUtil.h>
#include <nn/capsrv/capsrv_Result.h>
#include <nn/capsrv/movie/capsrv_MovieConfig.h>
#include <nn/capsrv/movie/capsrv_MovieMetaData.h>
#include "capsrvServer_AlbumMovieUtility.h"
#include "../../capsrv_Macro.h"
#include "../capsrvServer_ResultPrivate.h"
#include "../albumsrv/capsrvServer_AlbumServerObject.h"
#include "../detail/capsrvServer_CalculateJpegMac.h"

#define NN_CAPSRV_LOG_MOVIE_STATE(...) NN_CAPSRV_LOG_MOVIE_DEV("[state]" __VA_ARGS__)
#define NN_CAPSRV_LOG_MOVIE_FILE(...) NN_CAPSRV_LOG_MOVIE_DEV("[file]" __VA_ARGS__)

namespace nn{ namespace capsrv{ namespace server{ namespace album{

    MovieWriteStreamEntry::WriteFileStream::WriteFileStream() NN_NOEXCEPT
        : m_FileHandle()
        , m_FileSize()
        , m_IsFileOpened()
    {
    }

    bool MovieWriteStreamEntry::WriteFileStream::IsOpened() const NN_NOEXCEPT
    {
        return m_IsFileOpened;
    }

    nn::Result MovieWriteStreamEntry::WriteFileStream::Open(AlbumCacheDelta* pOutDelta, const AlbumFileId& fileId, const EnvironmentInfo& env) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!m_IsFileOpened);
        NN_RESULT_THROW_UNLESS(!m_IsFileOpened, nn::capsrv::ResultInternalError());

        const int64_t initialFileSize = MovieFileInitialSize;

        NN_CAPSRV_LOG_MOVIE_FILE("opening movie file: initial size %lld MB\n", initialFileSize / 1024 / 1024);

        NN_CAPSRV_PROCESS_START();
        AlbumFileHandle fileHandle = {};
        AlbumFilePath filepath = {};
        NN_RESULT_DO(AlbumFileManipulator::GetFilePath(&filepath, fileId, AlbumStorageDirection_Destination, env));
        NN_RESULT_DO(AlbumFileManipulator::OpenAlbumFileForWriteStream(&fileHandle, pOutDelta, filepath, initialFileSize));
        NN_CAPSRV_PROCESS_ROLLBACK(AlbumFileManipulator::CloseAlbumFile(fileHandle));

        NN_CAPSRV_PROCESS_SUCCESS();
        m_FileHandle   = fileHandle;
        m_FileSize     = initialFileSize;
        m_IsFileOpened = true;
        NN_RESULT_SUCCESS;
    }

    void MovieWriteStreamEntry::WriteFileStream::Close() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsFileOpened);
        if(!m_IsFileOpened)
        {
            return;
        }

        NN_CAPSRV_LOG_MOVIE_FILE("closing movie file\n");
        AlbumFileManipulator::CloseAlbumFile(m_FileHandle);

        m_FileHandle = {};
        m_FileSize = 0;
        m_IsFileOpened = false;
    }

    nn::Result MovieWriteStreamEntry::WriteFileStream::ReadFile(size_t* pOutSize, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsFileOpened);
        NN_RESULT_THROW_UNLESS(m_IsFileOpened, nn::capsrv::ResultInternalError());

        NN_RESULT_DO(AlbumFileManipulator::ReadFile(pOutSize, m_FileHandle, offset, buffer, size));

        NN_RESULT_SUCCESS;
    }

    nn::Result MovieWriteStreamEntry::WriteFileStream::WriteFile(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsFileOpened);
        NN_RESULT_THROW_UNLESS(m_IsFileOpened, nn::capsrv::ResultInternalError());

        int64_t requiredFileSize = offset + size;
        if(requiredFileSize > m_FileSize)
        {
            int64_t newFileSize = ((requiredFileSize + MovieFileUnitSize - 1) / MovieFileUnitSize) * MovieFileUnitSize;
            NN_CAPSRV_LOG_MOVIE_FILE("expanding movie file: %lld MB -> %lld MB\n", m_FileSize / 1024 / 1024, newFileSize / 1024 / 1024);
            NN_RESULT_DO(AlbumFileManipulator::ResizeFile(m_FileHandle, newFileSize));
            m_FileSize = newFileSize;
        }
        NN_SDK_ASSERT(requiredFileSize <= m_FileSize);

        NN_RESULT_DO(AlbumFileManipulator::WriteFile(m_FileHandle, offset, buffer, size));

        NN_RESULT_SUCCESS;
    }

    nn::Result MovieWriteStreamEntry::WriteFileStream::SetFileSize(int64_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsFileOpened);
        NN_RESULT_THROW_UNLESS(m_IsFileOpened, nn::capsrv::ResultInternalError());

        NN_CAPSRV_LOG_MOVIE_FILE("setting movie file size: %lld MB(%lld bytes)\n", size / 1024 / 1024, size);

        NN_RESULT_DO(AlbumFileManipulator::ResizeFile(m_FileHandle, size));

        m_FileSize = size;

        NN_RESULT_SUCCESS;
    }

    //------------------------------------

    MovieWriteStreamEntry::MovieWriteStreamEntry() NN_NOEXCEPT
    {
        m_State = MovieWriteStreamState_Unused;
        m_BrokenReason = nn::ResultSuccess();

        m_ClientSessionId = {};
        m_StreamId = {};
        m_FileId = {};
        m_Version = 0;

        m_FileStream = {};
        m_DataOffset = 0;
        m_DataSize = 0;
        m_MetaOffset = 0;
        m_MetaSize = 0;
        m_HashSignOffset = 0;
        m_HashSignSize = 0;
        m_VersionOffset = 0;
        m_VersionSize = 0;
        m_WrittenDataBlockFlag = {};
    }

    //--------------------------------------------------------

    int64_t MovieWriteStreamEntry::GetFileSizeLimit() NN_NOEXCEPT
    {
        return MovieSizeLimit;
    }

    MovieWriteStreamState MovieWriteStreamEntry::GetState() const NN_NOEXCEPT
    {
        return m_State;
    }

    SessionId MovieWriteStreamEntry::GetClientSessionId() const NN_NOEXCEPT
    {
        return m_ClientSessionId;
    }

    MovieStreamId MovieWriteStreamEntry::GetStreamId() const NN_NOEXCEPT
    {
        return m_StreamId;
    }

    const AlbumFileId& MovieWriteStreamEntry::GetAlbumFileId() const NN_NOEXCEPT
    {
        return m_FileId;
    }

    bool MovieWriteStreamEntry::IsBroken() const NN_NOEXCEPT
    {
        return m_State == MovieWriteStreamState_Broken;
    }

    nn::Result MovieWriteStreamEntry::GetBrokenReason() const NN_NOEXCEPT
    {
        return m_BrokenReason;
    }

    //--------------------------------------------------------

    void MovieWriteStreamEntry::NotifyStorageUnmounted(AlbumStorageType storage) NN_NOEXCEPT
    {
        if(m_FileId.storage == storage)
        {
            CloseFileInternal();
            if(!IsBroken())
            {
                MarkAsBroken(ResultAlbumIsNotMounted());
            }
        }
    }

    //--------------------------------------------------------
    // 状態遷移

    nn::Result MovieWriteStreamEntry::OpenFile(
        AlbumCacheDelta* pOutCacheDelta,
        SessionId clientSessionId,
        MovieStreamId streamId,
        const AlbumFileId& fileId,
        const EnvironmentInfo& env
    ) NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_MOVIE_STATE("%lld: -> empty\n", streamId);
        NN_SDK_REQUIRES_EQUAL(m_State, MovieWriteStreamState_Unused);
        NN_SDK_REQUIRES(!m_FileStream.IsOpened());
        NN_SDK_REQUIRES(m_BrokenReason.IsSuccess());

        NN_CAPSRV_PROCESS_START();

        NN_RESULT_THROW_UNLESS(fileId.contents == nn::capsrv::AlbumFileContents_Movie || fileId.contents == nn::capsrv::AlbumFileContents_ExtraMovie, ResultAlbumInvalidFileContents());

        NN_RESULT_DO(m_HashSignature.Initialize());
        NN_CAPSRV_PROCESS_ROLLBACK(m_HashSignature.Finalize());

        AlbumCacheDelta delta = {};
        NN_RESULT_DO(m_FileStream.Open(&delta, fileId, env));
        NN_CAPSRV_PROCESS_ROLLBACK(m_FileStream.Close());

        // TODO: ここで所有権を獲得

        NN_CAPSRV_PROCESS_SUCCESS();
        m_State = MovieWriteStreamState_Empty;
        m_ClientSessionId = clientSessionId;
        m_StreamId = streamId;
        m_FileId = fileId;
        m_Version = {};
        m_DataOffset = 0;
        m_DataSize = 0;
        m_MetaOffset = 0;
        m_MetaSize = 0;
        m_HashSignOffset = 0;
        m_HashSignSize = 0;
        m_VersionOffset = 0;
        m_VersionSize = 0;
        m_WrittenDataBlockFlag = {};
        *pOutCacheDelta = delta;
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieWriteStreamEntry::StartDataSection() NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_MOVIE_STATE("%lld: -> writing-data\n", m_StreamId);
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, MovieWriteStreamState_Unused);
        NN_RESULT_THROW_UNLESS(m_State != MovieWriteStreamState_Broken, ResultAlbumDenied());
        NN_RESULT_THROW_UNLESS(m_State != MovieWriteStreamState_WritingData, ResultAlbumAlreadyOpened());
        NN_RESULT_THROW_UNLESS(m_State == MovieWriteStreamState_Empty, ResultAlbumDenied());

        NN_RESULT_DO(MarkAsBrokenIfFailure(m_HashSignature.StartDataSection()));

        m_DataOffset = 0; // 必ず先頭から書き出す
        m_DataSize   = 0; // サイズ 0 から始める
        m_WrittenDataBlockFlag = {}; // 始めは何も書かれていない
        m_State = MovieWriteStreamState_WritingData;
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieWriteStreamEntry::EndDataSection() NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_MOVIE_STATE("%lld: -> data-complete\n", m_StreamId);
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, MovieWriteStreamState_Unused);
        NN_RESULT_THROW_UNLESS(m_State != MovieWriteStreamState_Broken, ResultAlbumDenied());
        NN_RESULT_THROW_UNLESS(m_State != MovieWriteStreamState_DataComplete, ResultAlbumAlreadyOpened());
        NN_RESULT_THROW_UNLESS(m_State == MovieWriteStreamState_WritingData, ResultAlbumDenied());

        // DataSize までの全ブロックが書込み済でなければならない
        int64_t writtenBlockCount = (m_DataSize + MovieDataBlockSize - 1) / MovieDataBlockSize;
        NN_CAPSRV_LOG_MOVIE_STATE("%lld:   data-size: %lld bytes(%lld blocks)\n", m_StreamId, m_DataSize, writtenBlockCount);
        NN_SDK_ASSERT_MINMAX(writtenBlockCount, 0, MovieDataBlockCountLimit);
        NN_RESULT_THROW_UNLESS(writtenBlockCount >= 0 && writtenBlockCount <= MovieDataBlockCountLimit, ResultInternalError());
        for(int64_t i = 0; i < writtenBlockCount; i++)
        {
            NN_RESULT_THROW_UNLESS(m_WrittenDataBlockFlag[i] == true, ResultAlbumInvalidFileData());
        }
        // DataSize 以降は未書き込みになっているはず
        for(int64_t i = writtenBlockCount; i < MovieDataBlockCountLimit; i++)
        {
            NN_SDK_ASSERT(m_WrittenDataBlockFlag[i] == false);
            NN_RESULT_THROW_UNLESS(m_WrittenDataBlockFlag[i] == false, ResultInternalError());
        }

        NN_RESULT_DO(MarkAsBrokenIfFailure(m_HashSignature.EndDataSection()));

        m_State = MovieWriteStreamState_DataComplete;
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieWriteStreamEntry::StartMetaSection() NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_MOVIE_STATE("%lld: -> writing-meta\n", m_StreamId);
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, MovieWriteStreamState_Unused);
        NN_RESULT_THROW_UNLESS(m_State != MovieWriteStreamState_Broken, ResultAlbumDenied());
        NN_RESULT_THROW_UNLESS(m_State != MovieWriteStreamState_WritingMeta, ResultAlbumAlreadyOpened());
        NN_RESULT_THROW_UNLESS(m_State == MovieWriteStreamState_DataComplete, ResultAlbumDenied());

        NN_RESULT_DO(MarkAsBrokenIfFailure(m_HashSignature.StartMetaSection()));

        m_MetaOffset = m_DataOffset + m_DataSize; // Data の直後から書き始める
        m_MetaSize   = 0;
        m_State = MovieWriteStreamState_WritingMeta;
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieWriteStreamEntry::EndMetaSection() NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_MOVIE_STATE("%lld: -> meta-complete\n", m_StreamId);
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, MovieWriteStreamState_Unused);
        NN_RESULT_THROW_UNLESS(m_State != MovieWriteStreamState_Broken, ResultAlbumDenied());
        NN_RESULT_THROW_UNLESS(m_State != MovieWriteStreamState_MetaComplete, ResultAlbumAlreadyOpened());
        NN_RESULT_THROW_UNLESS(m_State == MovieWriteStreamState_WritingMeta, ResultAlbumDenied());

        // 書いたことを確認
        NN_RESULT_THROW_UNLESS(m_MetaSize == MovieMetaDataSize, ResultAlbumInvalidFileData());

        NN_RESULT_DO(MarkAsBrokenIfFailure(m_HashSignature.EndMetaSection()));

        m_State = MovieWriteStreamState_MetaComplete;
        NN_RESULT_SUCCESS;
    }

    void MovieWriteStreamEntry::CloseFileInternal() NN_NOEXCEPT
    {
        if(m_FileStream.IsOpened())
        {
            m_FileStream.Close();
            m_HashSignature.Finalize();
        }
    }

    nn::Result MovieWriteStreamEntry::FinishFile(void* workMemory, size_t workMemorySize) NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_MOVIE_STATE("%lld: -> complete\n", m_StreamId);
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, MovieWriteStreamState_Unused);
        NN_RESULT_THROW_UNLESS(m_State != MovieWriteStreamState_Broken, ResultAlbumDenied());
        NN_RESULT_THROW_UNLESS(m_State != MovieWriteStreamState_Complete, ResultAlbumAlreadyOpened());
        NN_RESULT_THROW_UNLESS(m_State == MovieWriteStreamState_MetaComplete, ResultAlbumDenied());

        NN_RESULT_DO(MarkAsBrokenIfFailure(
            m_HashSignature.VerifyHashListSize(m_DataSize, m_MetaSize)
        ));

        // 署名書込
        m_HashSignOffset = m_MetaOffset + m_MetaSize; // Meta の直後から書き始める
        m_HashSignSize = 0;
        NN_RESULT_DO(MarkAsBrokenIfFailure(
            WriteHashSignImpl(workMemory, workMemorySize)
        ));

        // バージョン書込
        m_VersionOffset = m_HashSignOffset + m_HashSignSize;
        m_VersionSize = 0;
        NN_RESULT_DO(MarkAsBrokenIfFailure(
            WriteVersionImpl()
        ));

        int64_t fileSize = m_VersionOffset + m_VersionSize;
        NN_RESULT_DO(MarkAsBrokenIfFailure(
            m_FileStream.SetFileSize(fileSize)
        ));

        CloseFileInternal();

        m_State = MovieWriteStreamState_Complete;
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieWriteStreamEntry::CommitFile() NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_MOVIE_STATE("%lld: -> committed\n", m_StreamId);
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, MovieWriteStreamState_Unused);
        NN_RESULT_THROW_UNLESS(m_State != MovieWriteStreamState_Broken, ResultAlbumDenied());
        NN_RESULT_THROW_UNLESS(m_State == MovieWriteStreamState_Complete, ResultAlbumDenied());
        NN_SDK_REQUIRES(m_BrokenReason.IsSuccess());

        // TODO:ここで所有権を解放

        m_ClientSessionId = {};
        m_StreamId = {};
        m_FileId = {};
        m_Version = {};
        m_DataOffset = 0;
        m_DataSize = 0;
        m_MetaOffset = 0;
        m_MetaSize = 0;
        m_HashSignOffset = 0;
        m_HashSignSize = 0;
        m_VersionOffset = 0;
        m_VersionSize = 0;
        m_WrittenDataBlockFlag = {};
        m_State = MovieWriteStreamState_Unused;
        NN_RESULT_SUCCESS;
    }

    void MovieWriteStreamEntry::DiscardFile(
        AlbumCacheDelta* pOutCacheDelta,
        bool isDeleteSuppressed,
        const EnvironmentInfo& env
    ) NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_MOVIE_STATE("%lld: -> discarded\n", m_StreamId);
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, MovieWriteStreamState_Unused);

        AlbumFilePath filepath = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(AlbumFileManipulator::GetFilePath(&filepath, m_FileId, AlbumStorageDirection_Destination, env));

        CloseFileInternal();
        AlbumCacheDelta delta = {};
        if(!isDeleteSuppressed)
        {
            (void)AlbumFileManipulator::DeleteAlbumFile(&delta, filepath);
        }

        // TODO: ここで所有権を解放（削除済だが…）

        m_ClientSessionId = {};
        m_StreamId = {};
        m_FileId = {};
        m_Version = {};
        m_DataOffset = 0;
        m_DataSize = 0;
        m_MetaOffset = 0;
        m_MetaSize = 0;
        m_HashSignOffset = 0;
        m_HashSignSize = 0;
        m_VersionOffset = 0;
        m_VersionSize = 0;
        m_WrittenDataBlockFlag = {};
        m_State = MovieWriteStreamState_Unused;
        m_BrokenReason = nn::ResultSuccess();
        *pOutCacheDelta = delta;
    }

    void MovieWriteStreamEntry::MarkAsBroken(nn::Result brokenReason) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_State != MovieWriteStreamState_Unused);
        NN_SDK_REQUIRES(m_State != MovieWriteStreamState_Broken);
        NN_SDK_REQUIRES(m_BrokenReason.IsSuccess());
        m_State = MovieWriteStreamState_Broken;
        m_BrokenReason = brokenReason;
    }

    nn::Result MovieWriteStreamEntry::MarkAsBrokenIfFailure(nn::Result result) NN_NOEXCEPT
    {
        if(result.IsFailure())
        {
            MarkAsBroken(result);
        }
        return result;
    }

    //--------------------------------------------------------
    // ファイル操作

    nn::Result MovieWriteStreamEntry::ReadData(size_t* pOutReadSize, void* buffer, size_t size, int64_t offset) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, MovieWriteStreamState_Unused);
        NN_RESULT_THROW_UNLESS(
            m_State == MovieWriteStreamState_Empty ||
            m_State == MovieWriteStreamState_WritingData ||
            m_State == MovieWriteStreamState_DataComplete ||
            m_State == MovieWriteStreamState_WritingMeta ||
            m_State == MovieWriteStreamState_MetaComplete,
            ResultAlbumDenied()
        );

        // アクセスが UnitSize 単位になっているかチェック
        NN_RESULT_THROW_UNLESS(static_cast<int64_t>(size) % MovieDataBlockSize == 0, ResultAlbumOutOfRange());
        NN_RESULT_THROW_UNLESS(static_cast<int64_t>(offset) % MovieDataBlockSize == 0, ResultAlbumOutOfRange());

        if(size == 0)
        {
            *pOutReadSize = 0;
            NN_RESULT_SUCCESS;
        }

        NN_CAPSRV_PROCESS_START();
        std::memset(buffer, 0, size);
        NN_CAPSRV_PROCESS_ROLLBACK(std::memset(buffer, 0, size));

        // オーバーフローチェック
        //   0 <= offset
        NN_RESULT_THROW_UNLESS(offset >= 0, ResultAlbumOutOfRange());
        //   0 < size <= INT64_MAX
        NN_RESULT_THROW_UNLESS(nn::util::IsIntValueRepresentable<int64_t>(size), ResultAlbumOutOfRange());
        const int64_t sSize = static_cast<int64_t>(size);
        NN_SDK_ASSERT(sSize > 0);
        //   0 < offset + size <= INT64_MAX
        NN_RESULT_THROW_UNLESS(sSize <= std::numeric_limits<int64_t>::max() - offset, ResultAlbumOutOfRange());
        //   0 < DataOffset + offset + size <= INT64_MAX
        NN_RESULT_THROW_UNLESS(m_DataOffset <= std::numeric_limits<int64_t>::max() - offset - sSize, ResultAlbumOutOfRange());

        // ブロック単位でハッシュ検査しながら読込
        int64_t blockIndexBase  = offset / MovieDataBlockSize;
        int64_t blockIndexCount = sSize / MovieDataBlockSize;
        for(int64_t i = 0; i < blockIndexCount; i++)
        {
            auto pBlock = reinterpret_cast<char*>(buffer) + MovieDataBlockSize * i;
            int64_t blockIndex = blockIndexBase + i;

            if(blockIndex >= 0 && blockIndex < MovieDataBlockCountLimit && m_WrittenDataBlockFlag[blockIndex] == true)
            {
                // 書込み済領域であればファイルから読み込み
                int64_t offsetInData = blockIndex * MovieDataBlockSize;
                int64_t offsetInFile = m_DataOffset + offsetInData;
                int64_t sizeToRead = MovieDataBlockSize;                      // dst 側がオーバーランしないこと
                sizeToRead = std::min(sizeToRead, m_DataSize - offsetInData); // src 側がデータ部分の外側にでないこと
                size_t readSize = 0;
                NN_RESULT_DO(MarkAsBrokenIfFailure(
                    m_FileStream.ReadFile(&readSize, offsetInFile, pBlock, sizeToRead)
                ));
                if(readSize < MovieDataBlockSize)
                {
                    // BlockSize に足りなかった部分を 0 埋め
                    // （ Data 末尾は BlockSize の整数倍でなくてよい）
                    std::memset(pBlock + readSize, 0, MovieDataBlockSize - readSize);
                }

                // ハッシュを読み込み
                AlbumMovieUtility::BlockHash hash = {};
                NN_RESULT_DO(MarkAsBrokenIfFailure(
                    m_HashSignature.GetDataHash(&hash, blockIndex)
                ));
                // ハッシュを検査
                NN_RESULT_DO(AlbumMovieUtility::ValidateBlockData(pBlock, MovieDataBlockSize, hash));
            }
            else
            {
                // 範囲外 or 未書込み領域ならゼロ埋め
                std::memset(pBlock, 0, MovieDataBlockSize);
            }

        }

        NN_CAPSRV_PROCESS_SUCCESS();
        *pOutReadSize = size;
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieWriteStreamEntry::GetDataSize(int64_t* pOutValue) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, MovieWriteStreamState_Unused);
        NN_RESULT_THROW_UNLESS(
            m_State == MovieWriteStreamState_Empty ||
            m_State == MovieWriteStreamState_WritingData ||
            m_State == MovieWriteStreamState_DataComplete ||
            m_State == MovieWriteStreamState_WritingMeta ||
            m_State == MovieWriteStreamState_MetaComplete ||
            m_State == MovieWriteStreamState_Complete,
            ResultAlbumDenied()
        );

        *pOutValue = m_DataSize;
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieWriteStreamEntry::WriteData(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, MovieWriteStreamState_Unused);
        NN_RESULT_THROW_UNLESS(m_State == MovieWriteStreamState_WritingData, ResultAlbumDenied());

        // アクセスが UnitSize 単位になっているかチェック
        NN_RESULT_THROW_UNLESS(static_cast<int64_t>(size) % MovieDataBlockSize == 0, ResultAlbumOutOfRange());
        NN_RESULT_THROW_UNLESS(static_cast<int64_t>(offset) % MovieDataBlockSize == 0, ResultAlbumOutOfRange());

        if(size == 0)
        {
            NN_RESULT_SUCCESS;
        }

        // オーバーフローチェック
        //   0 <= offset
        NN_RESULT_THROW_UNLESS(offset >= 0, ResultAlbumOutOfRange());
        //   0 < size <= INT64_MAX
        NN_RESULT_THROW_UNLESS(nn::util::IsIntValueRepresentable<int64_t>(size), ResultAlbumOutOfRange());
        const int64_t sSize = static_cast<int64_t>(size);
        NN_SDK_ASSERT(sSize > 0);
        //   0 < offset + size <= INT64_MAX
        NN_RESULT_THROW_UNLESS(sSize <= std::numeric_limits<int64_t>::max() - offset, ResultAlbumOutOfRange());
        //   0 < DataOffset + offset + size <= INT64_MAX
        NN_RESULT_THROW_UNLESS(m_DataOffset <= std::numeric_limits<int64_t>::max() - offset - sSize, ResultAlbumOutOfRange());

        // ファイルサイズ上限チェック
        NN_RESULT_THROW_UNLESS(m_DataOffset + offset + sSize <= GetFileSizeLimit(), ResultAlbumFileSizeLimit());

        // ブロックインデックス範囲チェック
        int64_t blockIndexBase  = offset / MovieDataBlockSize;
        int64_t blockIndexCount = sSize / MovieDataBlockSize;
        NN_RESULT_THROW_UNLESS(blockIndexBase >= 0, ResultAlbumOutOfRange());
        NN_RESULT_THROW_UNLESS(blockIndexBase + blockIndexCount <= MovieDataBlockCountLimit, ResultAlbumOutOfRange());

        // ファイルに書き出し
        NN_RESULT_DO(MarkAsBrokenIfFailure(
            m_FileStream.WriteFile(m_DataOffset + offset, buffer, size)
        ));

        // ハッシュを登録
        for(int64_t pos = 0; pos < sSize; pos += MovieDataBlockSize)
        {
            auto p = reinterpret_cast<const char*>(buffer) + pos;
            NN_RESULT_DO(MarkAsBrokenIfFailure(
                m_HashSignature.RegisterData(offset + pos, p, MovieDataBlockSize)
            ));
        }

        // 書込み済フラグを設定
        for(int64_t i = 0; i < blockIndexCount; i++)
        {
            int64_t blockIndex = blockIndexBase + i;
            m_WrittenDataBlockFlag.Set(blockIndex, true);
        }

        int64_t appendedSize = std::max<int64_t>(0, offset + sSize - m_DataSize);
        m_DataSize += appendedSize;
        NN_SDK_ASSERT(m_DataSize <= GetFileSizeLimit() - m_DataOffset);

        NN_RESULT_SUCCESS;
    }

    nn::Result MovieWriteStreamEntry::SetDataSize(int64_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, MovieWriteStreamState_Unused);
        NN_RESULT_THROW_UNLESS(m_State == MovieWriteStreamState_WritingData, ResultAlbumDenied());

        // ブロック単位になっていること
        NN_RESULT_THROW_UNLESS(static_cast<int64_t>(size) % MovieDataBlockSize == 0, ResultAlbumOutOfRange());

        if(size == m_DataSize)
        {
            NN_RESULT_SUCCESS;
        }

        // オーバーフローチェック
        //   0 <= size (<= INT64_MAX)
        NN_RESULT_THROW_UNLESS(size >= 0, ResultAlbumOutOfRange());
        //   0 < DataOffset + size <= INT64_MAX
        NN_RESULT_THROW_UNLESS(m_DataOffset <= std::numeric_limits<int64_t>::max() - size, ResultAlbumOutOfRange());

        // ファイルサイズ上限チェック
        NN_RESULT_THROW_UNLESS(m_DataOffset + size <= GetFileSizeLimit(), ResultAlbumFileSizeLimit());

        // ブロック数チェック
        int64_t newBlockCount = size / MovieDataBlockSize;
        NN_RESULT_THROW_UNLESS(newBlockCount <= MovieDataBlockCountLimit, ResultAlbumOutOfRange());

        // ハッシュリストのサイズを変更
        NN_RESULT_DO(MarkAsBrokenIfFailure(
            m_HashSignature.ResizeData(size)
        ));

        // 範囲外になった部分の書込み済フラグをクリア
        for(int64_t i = newBlockCount; i < MovieDataBlockCountLimit; i++)
        {
            m_WrittenDataBlockFlag.Set(i, false);
        }

        m_DataSize = size;
        NN_SDK_ASSERT(m_DataSize <= GetFileSizeLimit() - m_DataOffset);

        NN_RESULT_SUCCESS;

    }

    nn::Result MovieWriteStreamEntry::WriteMeta(
        const void* meta,
        size_t size,
        uint64_t makerNoteVersion,
        int64_t makerNoteOffset,
        int64_t makerNoteSize,
        void* workMemory,
        size_t workMemorySize
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, MovieWriteStreamState_Unused);
        NN_RESULT_THROW_UNLESS(m_State == MovieWriteStreamState_WritingMeta, ResultAlbumDenied());

        NN_SDK_REQUIRES_NOT_NULL(workMemory);
        NN_SDK_REQUIRES_GREATER_EQUAL(workMemorySize, MovieDataBlockSize);
        NN_RESULT_THROW_UNLESS(workMemory != nullptr, ResultInternalError());
        NN_RESULT_THROW_UNLESS(workMemorySize >= MovieDataBlockSize, ResultInternalError());

        // Meta データのサイズは固定値。
        NN_RESULT_THROW_UNLESS(size == MovieMetaDataSize, ResultAlbumOutOfRange());
        // 2 回目の呼出は受け付けない
        NN_RESULT_THROW_UNLESS(m_MetaSize == 0, ResultAlbumDenied());

        // 静止画の署名を計算
        nn::capsrv::server::detail::Signature imageSignature = {};
        int64_t imageSignatureOffset = 0;
        {
            auto pMetaData = reinterpret_cast<const nn::capsrv::movie::MovieMetaData*>(meta);
            uint64_t imageDataSize = 0;
            std::memcpy(&imageDataSize, &pMetaData->infoData.imageDataSize, sizeof(uint64_t));
            NN_RESULT_THROW_UNLESS(imageDataSize <= sizeof(pMetaData->imageData), ResultAlbumInvalidFileData());

            NN_RESULT_DO(
                nn::capsrv::server::detail::CalculateJpegMac(&imageSignature, &imageSignatureOffset, meta, imageDataSize, makerNoteVersion, makerNoteOffset, makerNoteSize)
            );
        }

        NN_SDK_ASSERT(MovieDataBlockSize >= imageSignatureOffset + sizeof(imageSignature));
        NN_RESULT_THROW_UNLESS(MovieDataBlockSize >= imageSignatureOffset + sizeof(imageSignature), ResultInternalError());

        // ファイルに書き出し
        NN_STATIC_ASSERT(MovieMetaDataSize == 2 * MovieDataBlockSize);
        { // 前半
            int64_t pos = m_MetaOffset;
            char* buf = reinterpret_cast<char*>(workMemory);

            std::memcpy(buf, meta, MovieDataBlockSize);
            // 署名部分を置き換え
            std::memcpy(buf + imageSignatureOffset, &imageSignature, sizeof(imageSignature));

            NN_RESULT_DO(MarkAsBrokenIfFailure(
                m_FileStream.WriteFile(pos, buf, MovieDataBlockSize)
            ));
            NN_RESULT_DO(MarkAsBrokenIfFailure(
                m_HashSignature.RegisterMeta(pos - m_MetaOffset, buf, MovieDataBlockSize)
            ));
        }
        { // 後半
            int64_t pos = m_MetaOffset + MovieDataBlockSize;
            const char* buf = reinterpret_cast<const char*>(meta) + MovieDataBlockSize;

            NN_RESULT_DO(MarkAsBrokenIfFailure(
                m_FileStream.WriteFile(pos, buf, MovieDataBlockSize)
            ));
            NN_RESULT_DO(MarkAsBrokenIfFailure(
                m_HashSignature.RegisterMeta(pos - m_MetaOffset, buf, MovieDataBlockSize)
            ));
        }

        // 動画バージョン番号（末尾）をコピーしておく
        NN_STATIC_ASSERT(sizeof(m_Version) == sizeof(uint64_t));
        NN_STATIC_ASSERT(sizeof(nn::capsrv::movie::MovieMetaInfoData::version) == sizeof(uint64_t));
        std::memcpy(&m_Version, reinterpret_cast<const char*>(meta) + MovieMetaDataSize - sizeof(uint64_t), sizeof(uint64_t));

        m_MetaSize = MovieMetaDataSize;
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieWriteStreamEntry::WriteHashSignImpl(void* workMemory, size_t workMemorySize) NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_MOVIE_DEV("writing signature(work=%lld bytes)\n", workMemorySize);
        NN_SDK_REQUIRES_EQUAL(m_State, MovieWriteStreamState_MetaComplete);
        NN_RESULT_THROW_UNLESS(workMemorySize >= MovieDataBlockHashSize, ResultAlbumWorkMemoryError());

        size_t validWorkMemorySize = (workMemorySize / MovieDataBlockHashSize) * MovieDataBlockHashSize;

        NN_CAPSRV_LOG_MOVIE_DEV("  starting sign calculation\n");
        int64_t hashListSize = 0;
        NN_RESULT_DO(m_HashSignature.StartSignCalculation(&hashListSize));

        int64_t remain = hashListSize;
        while(remain > 0)
        {
            size_t readSize = 0;
            size_t sizeToRead = std::min(static_cast<size_t>(remain), validWorkMemorySize);
            NN_CAPSRV_LOG_MOVIE_DEV("  processing (%lld-%lld)\n", m_HashSignSize, m_HashSignSize + sizeToRead);
            NN_RESULT_DO(m_HashSignature.ProcessSignCalculationForNextHash(&readSize, workMemory, sizeToRead));
            NN_SDK_ASSERT_EQUAL(readSize, sizeToRead);

            // ファイルサイズチェック
            NN_CAPSRV_LOG_MOVIE_DEV("  checking file size limit\n");
            NN_RESULT_THROW_UNLESS(static_cast<int64_t>(readSize) <= GetFileSizeLimit() - (m_HashSignOffset + m_HashSignSize), ResultAlbumFileSizeLimit());

            NN_CAPSRV_LOG_MOVIE_DEV("  writing hash (%lld-%lld)\n", m_HashSignSize, m_HashSignSize + readSize);
            NN_RESULT_DO(m_FileStream.WriteFile(m_HashSignOffset + m_HashSignSize, workMemory, readSize));
            m_HashSignSize += static_cast<int64_t>(readSize);
            remain -= static_cast<int64_t>(readSize);
            NN_SDK_ASSERT_GREATER_EQUAL(remain, 0);
        }

        NN_CAPSRV_LOG_MOVIE_DEV("  ending sign calculation\n");
        NN_RESULT_DO(m_HashSignature.EndSignCalculation());

        {
            size_t size = 0;
            NN_CAPSRV_LOG_MOVIE_DEV("  getting signature\n");
            NN_RESULT_DO(m_HashSignature.GetSignature(&size, workMemory, validWorkMemorySize));

            // ファイルサイズチェック
            NN_CAPSRV_LOG_MOVIE_DEV("  checking file size limit\n");
            NN_RESULT_THROW_UNLESS(static_cast<int64_t>(size) <= GetFileSizeLimit() - (m_HashSignOffset + m_HashSignSize), ResultAlbumFileSizeLimit());

            NN_CAPSRV_LOG_MOVIE_DEV("  writing signature\n");
            NN_RESULT_DO(m_FileStream.WriteFile(m_HashSignOffset + m_HashSignSize, workMemory, size));
            m_HashSignSize += static_cast<int64_t>(size);
        }

        NN_CAPSRV_LOG_MOVIE_DEV("  -> success\n");
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieWriteStreamEntry::WriteVersionImpl() NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_MOVIE_DEV("writing version(%lld)\n", m_Version);

        // ファイルサイズチェック
        NN_CAPSRV_LOG_MOVIE_DEV("  checking file size limit\n");
        NN_RESULT_THROW_UNLESS(sizeof(m_Version) <= GetFileSizeLimit() - m_VersionOffset, ResultAlbumFileSizeLimit());

        NN_CAPSRV_LOG_MOVIE_DEV("  writing version\n");
        NN_RESULT_DO(m_FileStream.WriteFile(m_VersionOffset, &m_Version, sizeof(m_Version)));

        m_VersionSize = sizeof(m_Version);

        NN_CAPSRV_LOG_MOVIE_DEV("  -> success\n");
        NN_RESULT_SUCCESS;
    }


}}}}
