﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_AlbumFileIdGenerator.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/time.h>
#include <nn/settings/system/settings_Capture.h>
#include <nn/ncm/ncm_ContentMetaId.h>

#include <nn/capsrv/capsrv_Result.h>
#include "capsrvServer_AlbumManager.h"
#include "../albumsrv/capsrvServer_AlbumServerObject.h"

namespace nn{ namespace capsrv{ namespace server{ namespace album{

    namespace {
        // @brief 指定されたストレージがマウントされているか調べる
        // @param[out] pOutValue マウントされていた場合、storage が代入される変数のポインタ
        // @param[in] storage 調べるストレージ
        // @post 成功した場合、 *pOutValue == storage
        // @retval nn::ResultSuccess storage がマウントされていた場合
        // @retval nn::capsrv::ResultAlbumIsNotMounted マウントされていなかった場合
        nn::Result CheckStorageMounted(AlbumStorageType* pOutValue, AlbumStorageType storage, album::AlbumManager* pAlbumManager) NN_NOEXCEPT
        {
            static const char StorageNames[][8] = {
                "NAND",
                "SD",
            };
            NN_UNUSED(StorageNames);

            if(pAlbumManager->EnsureMounted(storage).IsSuccess())
            {
                NN_CAPSRV_LOG_DEV("    Checking storage %s ... Mounted\n", StorageNames[storage]);
                *pOutValue = storage;
                NN_RESULT_SUCCESS;
            }
            else
            {
                NN_CAPSRV_LOG_DEV("    Checking storage %s ... NotMounted\n", StorageNames[storage]);
                NN_RESULT_THROW(ResultAlbumIsNotMounted());
            }
        }
    }


    void AlbumFileIdGenerator::Initialize() NN_NOEXCEPT
    {
        m_PreviousTime = {};
        m_Id = 0;
    }

    void AlbumFileIdGenerator::Finalize() NN_NOEXCEPT
    {
    }

    nn::Result AlbumFileIdGenerator::GetAutoSavingStorage(AlbumStorageType* pOutValue, bool isExtra, album::AlbumManager* pAlbumManager) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        NN_SDK_REQUIRES_NOT_NULL(pAlbumManager);

        // Extra ファイルの場合はまず最初に SD カードをチェックし、
        // さらに NAND も許可されているなら NAND もチェックする。
        if (isExtra)
        {
            if(CheckStorageMounted(pOutValue, AlbumStorage_Sd, pAlbumManager).IsSuccess())
            {
                NN_RESULT_SUCCESS;
            }

            if (g_EnvironmentInfo.IsAlbumExtraMovieFileSupportedForNandStorage() && CheckStorageMounted(pOutValue, AlbumStorage_Nand, pAlbumManager).IsSuccess())
            {
                NN_RESULT_SUCCESS;
            }

            // 見つからなかったら失敗
            NN_RESULT_THROW(ResultAlbumIsNotMounted());
        }

        // 通常アルバムファイルの場合は、本体設定に応じて SD カードをチェック
        auto primaryStorage = nn::settings::system::GetPrimaryAlbumStorage();
        if(primaryStorage == nn::settings::system::PrimaryAlbumStorage_SdCard)
        {
            if(CheckStorageMounted(pOutValue, AlbumStorage_Sd, pAlbumManager).IsSuccess())
            {
                NN_RESULT_SUCCESS;
            }
        }

        // 通常アルバムファイルなら NAND は常にチェック
        {
            if(CheckStorageMounted(pOutValue, AlbumStorage_Nand, pAlbumManager).IsSuccess())
            {
                NN_RESULT_SUCCESS;
            }
        }

        // 見つからなかったら失敗
        NN_RESULT_THROW(ResultAlbumIsNotMounted());
    }

    nn::Result AlbumFileIdGenerator::GenerateCurrentAlbumFileId(
        AlbumFileId* pOutValue,
        nn::ncm::ApplicationId applicationId,
        AlbumFileContentsType contents,
        album::AlbumManager* pAlbumManager
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        NN_SDK_REQUIRES_NOT_NULL(pAlbumManager);
        NN_SDK_REQUIRES(applicationId.value != ncm::ApplicationId::GetInvalidId().value);

        AlbumFileId fileId = {};
        fileId.applicationId = applicationId;
        fileId.contents      = contents;
        NN_RESULT_DO(GetAutoSavingStorage(&fileId.storage, IsExtraAlbumFileContents(contents), pAlbumManager));
        NN_RESULT_DO(GenerateCurrentTimestamp(&fileId.time));

        *pOutValue = fileId;
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumFileIdGenerator::GenerateCurrentTimestamp(AlbumFileDateTime* pOutValue) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);

        nn::time::PosixTime posixTime = {};
        NN_RESULT_TRY(nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime))
            NN_RESULT_CATCH_ALL
            {
                posixTime.value = 0;
            }
        NN_RESULT_END_TRY;

        if(m_PreviousTime == posixTime)
        {
            if(m_Id < 99)
            {
                m_Id++;
            }
        }
        else
        {
            m_PreviousTime = posixTime;
            m_Id           = 0;
        }
        NN_SDK_ASSERT_RANGE(m_Id, 0, 100);

        nn::time::CalendarTime dateTime = {};
        NN_RESULT_DO(nn::time::ToCalendarTime(&dateTime, nullptr, posixTime));


        AlbumFileDateTime result = {};
        result.year   = static_cast<uint16_t>(dateTime.year);
        result.month  = static_cast<uint8_t>(dateTime.month);
        result.day    = static_cast<uint8_t>(dateTime.day);
        result.hour   = static_cast<uint8_t>(dateTime.hour);
        result.minute = static_cast<uint8_t>(dateTime.minute);
        result.second = static_cast<uint8_t>(dateTime.second);
        result.id     = static_cast<uint8_t>(m_Id);

        *pOutValue = result;
        NN_RESULT_SUCCESS;
    }




}}}}
