﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrv_MemoryPool.h"
#include "capsrv_MemoryPool-module.nvnflinger.h"

#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include "capsrv_ResultCapture.h"
#include "capsrv_CaptureModule-module.nvnflinger.h"

namespace nn{ namespace capsrv{ namespace capture{

    size_t MemoryPoolData::GetRequiredAlignment() NN_NOEXCEPT
    {
        return 1024; // TORIAEZU:
    }

    size_t MemoryPoolData::GetRequiredUnitSize() NN_NOEXCEPT
    {
        return 128 * 1024; // TORIAEZU:
    }

    MemoryPoolData::MemoryPoolData() NN_NOEXCEPT
        : m_State(State_NotInitialized)
        , m_hMemory(0)
        , m_pMemory(nullptr)
        , m_Size(0)
    {
    }

    bool MemoryPoolData::IsInitialized() const NN_NOEXCEPT
    {
        return m_State != State_NotInitialized;
    }

    nn::Result MemoryPoolData::Initialize(CaptureModule* pModule, void* pMemory, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pMemory);
        NN_SDK_REQUIRES_ALIGNED(pMemory, GetRequiredAlignment());
        NN_SDK_REQUIRES_GREATER(size, 0u);
        NN_SDK_REQUIRES_EQUAL(size % GetRequiredUnitSize(), 0u);
        NN_SDK_REQUIRES(!IsInitialized());

        size_t alignment = GetRequiredAlignment();
        NvRmDeviceHandle hDevice = pModule->GetVicDevice();
        NvRmMemKind kind = NvRmMemKind_Generic_16Bx2; // TORIAEZU:


        NVRM_DEFINE_MEM_HANDLE_ATTR(memAttributes);
        NVRM_MEM_HANDLE_SET_ATTR(
            memAttributes,
            static_cast<NvU32>(alignment),
            NvOsMemAttribute_WriteBack,
            static_cast<NvU32>(size),
            NVRM_MEM_TAG_DDK2D_MISC
        );
        NVRM_MEM_HANDLE_SET_KIND_ATTR(memAttributes, kind);
        NVRM_MEM_HANDLE_SET_VA_ATTR(memAttributes, reinterpret_cast<NvU64>(pMemory));

        NvError err = NvRmMemHandleAllocAttr(hDevice, &memAttributes, &m_hMemory);
        NN_RESULT_THROW_UNLESS(
            err == NvSuccess,
            ResultCaptureAllocationFailed()
        );
        m_pMemory = pMemory;
        m_Size = size;

        m_State = State_Initialized;
        NN_RESULT_SUCCESS;
    }

    nn::Result MemoryPoolData::InitializeAsImported(const MemoryPoolImportData& src) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsInitialized());

        m_hMemory = src.hMem;
        m_pMemory = src.memory;
        m_Size    = src.size;
        m_State   = State_Imported;
        NN_RESULT_SUCCESS;
    }

    void MemoryPoolData::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        if(m_State == State_Initialized)
        {
            NvRmMemHandleFree(m_hMemory);
        }
        m_hMemory = 0;
        m_pMemory = nullptr;
        m_Size = 0;
    }

    NvRmMemHandle MemoryPoolData::GetHandle() const NN_NOEXCEPT
    {
        return m_hMemory;
    }

    uintptr_t MemoryPoolData::GetMemoryAddress() const NN_NOEXCEPT
    {
        return reinterpret_cast<uintptr_t>(m_pMemory);
    }

    //---------------------------------------------

    size_t MemoryPool::GetRequiredAlignment() NN_NOEXCEPT
    {
        return MemoryPoolData::GetRequiredAlignment();
    }

    size_t MemoryPool::GetRequiredUnitSize() NN_NOEXCEPT
    {
        return MemoryPoolData::GetRequiredUnitSize();
    }

    MemoryPool::MemoryPool() NN_NOEXCEPT
    {
        NN_CAPSRV_CAPTURE_DATASTORAGE_CHECK(DataType, m_DataStorage);
        std::memset(this, 0, sizeof(*this));
    }

    nn::Result MemoryPool::Initialize(CaptureModule* pModule, void* pMemory, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsInitialized());
        m_pData = new(&m_DataStorage) DataType();
        return m_pData->Initialize(pModule, pMemory, size);
    }

    nn::Result MemoryPool::InitializeAsImported(const MemoryPoolImportData& src) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsInitialized());
        m_pData = new(&m_DataStorage) DataType();
        return m_pData->InitializeAsImported(src);
    }

    void MemoryPool::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        m_pData->Finalize();
        m_pData->~DataType();
        m_pData = nullptr;
    }

}}}
